/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: TelnetClientTask.java,v 1.5 2003/11/06 21:05:05 evertonm Exp $

package bera.telnet;

import java.net.Socket;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.InputStream;
import java.io.BufferedReader;
import java.io.OutputStreamWriter;
import java.io.BufferedWriter;

import bera.grabline.LineReader;

import org.apache.log4j.Logger;

class TelnetClientTask extends Thread implements TelnetClient {

    static private Logger logger = Logger.getLogger(TelnetClientTask.class);

    private Socket         clientSocket;
    private TelnetHandler  telnetHandler;
    private BufferedReader bufferedReader;
    private BufferedWriter bufferedWriter;

    TelnetClientTask(Socket socket, TelnetHandler handler) {
	clientSocket  = socket;
	telnetHandler = handler;
    }

    public BufferedReader getBufferedReader() {
	return bufferedReader;
    }

    public BufferedWriter getBufferedWriter() {
	return bufferedWriter;
    }

    public void run() {

	try {
	    bufferedReader = new BufferedReader(new InputStreamReader(clientSocket.getInputStream()));
	}
	catch (IOException e) {
	    logger.error("could not create read stream: " + e);
	    return;
	}

	try {
	    bufferedWriter = new BufferedWriter(new OutputStreamWriter(clientSocket.getOutputStream()));
	}
	catch (IOException e) {
	    logger.error("could not create write stream: " + e);
	    return;
	}

	LineReader reader;
	synchronized (telnetHandler) {
	    reader = telnetHandler.add(this);
	}

	if (reader == null) {
	    logger.error("could not add client to telnet handler - disconnecting");
	    close();
	    return;
	}

	for (;;) {
	    if (clientSocket.isClosed()) {
		logger.debug("connection closed");
		break;
	    }

	    String prompt;
	    synchronized (telnetHandler) {
		prompt = telnetHandler.getPrompt(this);
	    }

	    String line;
	    try {
		// We don't know whether LineReader.readLine is MT-safe
		synchronized (reader) {
		    line = reader.readLine(prompt);
		}
	    }
	    catch (IOException e) {
		logger.error("error reading from socket: " + e);
		break;
	    }

	    // EOF?
	    if (line == null) {
		logger.debug("client closed connection");
		break;
	    }

	    synchronized (telnetHandler) {
		telnetHandler.input(this, line);
	    }
	}

	synchronized (telnetHandler) {
	    telnetHandler.remove(this);
	}
    }

    public void output(String buf) {
	try {
	    bufferedWriter.write(buf, 0, buf.length());
	}
	catch (IOException e) {
	    logger.error("failure writing to socket: " + e);
	}
    }

    public void outputln(String buf) {
	output(buf);

	try {
	    bufferedWriter.newLine();
	}
	catch (IOException e) {
	    logger.error("failure writing newline to socket: " + e);
	}
    }

    public void flush() {
	try {
	    bufferedWriter.flush();
	}
	catch (IOException e) {
	    logger.error("failure flushing socket: " + e);
	}
    }

    public void close() {
	try {
	    clientSocket.close();
	}
	catch (IOException e) {
	    logger.error("failure closing socket: " + e);
	}
    }
}

