package freenet.client;

import java.io.*;
import java.net.BindException;
import freenet.*;
import freenet.node.NodeReference;
import freenet.interfaces.Interface;
import freenet.session.LinkManager;
import freenet.crypt.Global;
import freenet.support.*;
import freenet.message.*;
import freenet.presentation.FNPRawMessage;
import freenet.presentation.FreenetProtocol;
import freenet.thread.*;

/**
 * This is an implementation of Core the for use by the clients. It is 
 * runnable, and when started will listen for connections, feeding them
 * to a ClientMessageHandler.
 *
 * @author oskar
 **/

public class ClientCore extends Core {

    public ClientMessageHandler cmh;
    public Peer me;

    /* Clients are always static. */
    public boolean getTransience() {return true;}

    public NodeReference getNodeReference() {
        return new NodeReference(me, null, null);
    }
    
      /**
     * Create a new ClientCore.
     * @param myAddress  The address (port) that this client should listen for
     *                   messages on.
     * @param lm         The linkmanager with which to manage sessions.
     * @param p          The presentation protocol to use for this client.
     */
    public static ClientCore newInstance(Address myAddress, 
                                         LinkManager lm, Presentation p) 
        throws ListenException {

        DSAAuthentity auth = new DSAAuthentity(Global.DSAgroupC, 
                                               Core.randSource);
        return newInstance(auth, myAddress, lm, p);
    }

   /**
     * Create a new ClientCore.
     * @param privKey    The private key to use for communication
     * @param myAddress  The address (port) that this client should listen for
     *                   messages on.
     * @param lm         The linkmanager with which to manage sessions.
     * @param p          The presentation protocol to use for this client.
     */
    public static ClientCore newInstance(Authentity privKey, 
                                         Address myAddress, 
                                         LinkManager lm, Presentation p) 
        throws ListenException {
        
        TransportHandler th = new TransportHandler();
        th.register(myAddress.transport());
        SessionHandler sh = new SessionHandler();
        sh.register(lm, 100);
        PresentationHandler ph = new PresentationHandler();
        ph.register(p, 100);
        Peer me = new Peer(privKey.getIdentity(), myAddress, lm, p);
        return new ClientCore(privKey, th, sh, ph, me);
                              /* New plan - clients have no interfaces
                              new Interface[] {
                                  new StandardInterface(myAddress.listenPart(),
                                                        sh, ph, 0, 3)},
                              */
    }

    private ClientCore(Authentity privKey,
                       TransportHandler th, SessionHandler sh, 
                       PresentationHandler ph, Peer me) 
	throws ListenException {
        
        super(privKey, me.getIdentity(), th, sh, ph); 
        this.me = me; 
        this.cmh = new ClientMessageHandler(this);

        cmh.addType( FNPRawMessage.class, 
                     VoidMessage.messageName, 
                     VoidMessage.class);
        cmh.addType( FNPRawMessage.class, 
                     DataReply.messageName,      
                     DataReply.class      );
        cmh.addType( FNPRawMessage.class, 
                     DataNotFound.messageName,   
                     DataNotFound.class   );
        cmh.addType( FNPRawMessage.class, 
                     QueryRejected.messageName,  
                     QueryRejected.class  );
        cmh.addType( FNPRawMessage.class, 
                     QueryAborted.messageName,   
                     QueryAborted.class   );
        cmh.addType( FNPRawMessage.class, 
                     QueryRestarted.messageName, 
                     QueryRestarted.class );
        cmh.addType( FNPRawMessage.class, 
                     StoreData.messageName,      
                     StoreData.class      );
        cmh.addType( FNPRawMessage.class, 
                     InsertReply.messageName,    
                     InsertReply.class    );
        cmh.addType( FNPRawMessage.class, 
                     Accepted.messageName,       
                     Accepted.class       );
        cmh.addType( FNPRawMessage.class, 
                     DataInsert.messageName,     
                     DataInsert.class     );

    }

    /**
     * Starts the client listening for Connections and Messages.
     */
    public void acceptConnections() {
        ThreadGroup tg = new ThreadGroup(toString());
        ThreadFactory tf = (ThreadFactory) new FastThreadFactory(tg, 
                                                                 100);

        // REDFLAG: 20 ok? What value makes sense here?
        OpenConnectionManager ocm = new OpenConnectionManager(tf, 20);
        
        begin(tg, new Ticker(cmh, tf), ocm, null, false);
        join();
    }

    public String toString() {
        return "Client core serving: " + cmh;
    }
}


