package freenet.config;

import freenet.FieldSet;
import freenet.support.*;
import freenet.support.io.*;
import freenet.Core;
import java.util.*;
import java.io.*;

/**
 * Reads and stores Freenet parameters
 **/
public class Params extends FieldSet implements PropertySet {

    /*
      public static void main(String[] args) {
      try {
      Params p = new Params(".freenetrc", args);
      System.out.println("Params: " + p.params);
      System.out.println("Args: " + p.argVector);
      }
      catch (IOException e) {
      System.out.println ("Error reading .freenetrc: " + e);
      }
      }
    */


    //=== support classes =====================================================
    
    private static class TrimFilter implements Filter {
        public String filter(String s) {
            return s.trim();
        }
    }

    //=== main code ===========================================================

    private final static TrimFilter trimFilter = new TrimFilter();
    
    private Hashtable opts   = new Hashtable();
    private Vector argVector = new Vector();

    public Params() {
        super();
    }

    public Params(Params fs) {
	super(fs);
	opts = new Hashtable(fs.opts);
    }

    public Params(FieldSet fs) {
        super(fs);
    }

    public Params(Option[] options) {
        super();
        addOptions(options);
    }

    public Params(Option[] options, Params fs) {
	super(fs);
	addOptions(options);
	opts.putAll(fs.opts);
    }

    public Params(Option[] options, FieldSet fs) {
        super(fs);
        addOptions(options);
    }

    public void addOptions(Option[] options) {
        for (int i=0; i < options.length; ++i) {
	    addOption(options[i].name(), options[i]);
	    addOption(""+options[i].abbrev(), options[i]);
        }
        searchOptions();
    }

    protected void addOption(String name, Option option) {
	int x = name.indexOf('.');
	if(x >= 0)
	    ((Params)makeSet(name.substring(0,x))).addOption(name.substring(x+1), option);
	else
	    opts.put(name, option);
    }
    
    public void readArgs(String[] args) {
        for (int i = 0 ; i < args.length ; i++) {
            argVector.addElement(args[i]);
        }
        searchOptions();
    }

    /**
     * Iterates over the argVector, taking out any recognized options and
     * placing them in the fieldset.
     */
    protected void searchOptions() {
        boolean moreOptions = true;
        Vector newArgs = new Vector(argVector.size());

        for (Enumeration e = argVector.elements() ; e.hasMoreElements() ;) {

            String arg = (String) e.nextElement();

            if (arg == null) {
                continue;
            }

            Vector opt = new Vector();
            String rarg = null; // residual arg
            if (moreOptions && arg.startsWith("--")) { // word arg
                Option o = getOption(arg.substring(2));
                if (o != null)
                    opt.addElement(o);
                else 
                    rarg = arg;
            } else if (moreOptions && arg.startsWith("-")) {
                int n = arg.length();
                StringBuffer rab = new StringBuffer("-");
                for (int i = 1 ; i < n ; i++) {
                    Option o = // there is no Character.toString(a) !!
                        (Option) opts.get("" + arg.charAt(i));
		    // one char so don't recurse
                    if (o != null)
                        opt.addElement(o);
                    else
                        rab.append(arg.charAt(i));
                }
                if (rab.length() > 1)
                    rarg = rab.toString();
            } else {
                rarg = arg;
            }

            if (rarg != null) {
                newArgs.addElement(rarg);
            }

            for (Enumeration f = opt.elements() ; f.hasMoreElements() ;) {
                Option o = (Option) f.nextElement();
                if (o.numArgs == 0) {
                    put(o.name(),"true");
                } else {
                    StringBuffer sb = new StringBuffer();
                    if (e.hasMoreElements()) {
                        sb.append((String) e.nextElement());
                    }
                    for (int j = o.numArgs - 1; 
                         e.hasMoreElements() && j > 0;
                         j--) {                    
                        sb.append('\t').append((String) e.nextElement());
                        // i changed this to use a tab, not a comma  -tc
                    }
                    put(o.name(), sb.toString());
                }
            }
        }
        argVector = newArgs;
    }

    public void readParams(String[] files) 
        throws FileNotFoundException, IOException {

        if (files.length == 0) throw new IllegalArgumentException("no files");

        boolean foundOne = false;
        
        for (int i=0; i < files.length; ++i) {
            try {
                readParams(files[i]);
                foundOne = true;
            }
            catch (FileNotFoundException e) {}
        }

        if (!foundOne) {
            StringBuffer sb = new StringBuffer();
            if (files.length > 1) sb.append("any of ");
            sb.append(files[0]);
            for (int i=1; i < files.length; ++i) {
                sb.append(", ").append(files[i]);
            }
            throw new FileNotFoundException("Could not find " + sb);
        }
    }

    public void readParams(String filename) 
        throws FileNotFoundException, IOException {
        readParams(new FileInputStream(filename));
    }

    public void readParams(InputStream is) 
        throws IOException {
        BufferedReader bs = new CommentedBufferedReader(new 
                InputStreamReader(is), "[#%");
        
        parseParams(bs, '=', '.', trimFilter, trimFilter );

        bs.close();
    }

    //=== accessors ===========================================================

    /**
     * Returns true if the param is set or there is a registered default.
     */
    public boolean containsKey(String name) {
        return (getParam(name) != null || getOption(name) != null);
    }
    
    /** @return  number of command-line arguments not eaten by a switch */
    public int getNumArgs() {
        return argVector.size();
    }

    /** @return  command-line argument at the given position */
    public String getArg(int pos) {
        return pos < argVector.size() ? (String) argVector.elementAt(pos)
                                      : null;
    }

    /** @return  all command-line arguments as one vector */
    public String[] getArgs() {
        String[] r = new String[argVector.size()];
        argVector.copyInto(r);
        return r;
    }

    /** This is for breaking down a single-arg switch where the arg is
      * a comma-separated String.  Hence the default value, if applicable,
      * should also be a comma-separated String.
      * @throws  ClassCastException if the default is not a String
      */
    public String[] getList(String name) throws ClassCastException {
        Option o = getOption(name);
        String s = getParam(name);
        if (s == null || s.trim().equals(""))
            s = (o == null ? null : (String) o.defaultValue());
        if (s == null || s.trim().equals(""))
            return new String[0];
        StringTokenizer st = new StringTokenizer(s, ",");
        String[] list = new String[st.countTokens()];
        for (int i=0; st.hasMoreTokens(); ++i)
            list[i] = st.nextToken().trim();
        return list;
    }

    /** This is for getting back the args to a switch that took more than one
      * arg.  The intermediate form is a tab-separated String.  Hence the
      * default value, if applicable, should also be a tab-separated String.
      * @throws  ClassCastException  if the default is not a String
      */
    public String[] getMultipleArgs(String name) throws ClassCastException {
        Option o = getOption(name);
        String s = getParam(name);
        if (s == null || "".equals(s))
            s = (o == null || o.defaultValue() == null ? "" : (String) o.defaultValue());
        StringTokenizer st = new StringTokenizer(s, "\t");
        String[] list = new String[st.countTokens()];
        for (int i=0; st.hasMoreTokens(); ++i)
            list[i] = st.nextToken().trim();
        return list;
    }

    /** @return  the parameter if found, or the default
      * @throws  ClassCastException if the default is not a long, or the
      *                             parameter is not parseable as a long
      */
    public long getLong(String name) throws ClassCastException {
        try {
            long def = 0;
            Option o = getOption(name);
            if (o != null) def = parseLong(o.defaultValue().toString());
            return getParam(name) == null ? def : parseLong(getParam(name));
        }
        catch (NumberFormatException e) { throw new ClassCastException(); }
    }

    /** @return the parameter
     *  @throws NumberFormatException if the string is not parseable
     */
    static long parseLong(String s) throws NumberFormatException {
	long res = 1;
	int x = s.length()-1;
	int idx;
	try {
	    long[] l = {1000, 1<<10, 
			1000*1000, 1<<20, 
			1000*1000*1000, 1<<30,
			1000*1000*1000*1000, 1<<40,
			1000*1000*1000*1000*1000, 1<<50,
			1000*1000*1000*1000*1000*1000, 1<<60};
	    while(x>=0 && 
		  ((idx = "kKmMgGtTpPeE".indexOf(s.charAt(x))) != -1)) {
		x--;
		res *= l[idx];
	    }
	    res *= Double.parseDouble(s.substring(0,x+1));
	} catch (ArithmeticException e) { res = Long.MAX_VALUE; }
	return res;
    }
    
    /** @return  the parameter if found, or the default
      * @throws  ClassCastException if the default is not an int AND the
      *                             parameter is not parseable as an int
      */
    public int getInt(String name) throws ClassCastException {
        try {
            int def = 0;
            Option o = getOption(name);
            if (o != null) def = ((Integer) o.defaultValue()).intValue();
            return getParam(name) == null ? def : Integer.parseInt(getParam(name));
        }
        catch (NumberFormatException ee) { throw new ClassCastException(); }
    }

    /** @return  the parameter if found, or the default
      * @throws  ClassCastException if the default is not a short, or the
      *                             parameter is not parseable as a short
      */
    public short getShort(String name) throws ClassCastException {
        try {
            short def = 0;
            Option o = getOption(name);
            if (o != null) def = ((Short) o.defaultValue()).shortValue();
            return getParam(name) == null ? def : Short.parseShort(getParam(name));
        }
        catch (NumberFormatException e) { throw new ClassCastException(); }
    }

    public float getFloat(String name) throws ClassCastException {
        try {
            float def = 0;
            Option o = getOption(name);
            if (o != null)
                def = ((Float) o.defaultValue()).floatValue();
            return getParam(name) == null ? def : 
                Float.valueOf(getParam(name)).floatValue();
        }
        catch (NumberFormatException e) { throw new ClassCastException(); }
    }

    public double getDouble(String name) throws ClassCastException {
        try {
            double def = 0;
            Option o = getOption(name);
            if (o != null)
                def = ((Double) o.defaultValue()).doubleValue();
            return getParam(name) == null ? def : 
                Double.valueOf(getParam(name)).doubleValue();
        }
        catch (NumberFormatException e) { throw new ClassCastException(); }
    }

    /** @return  the parameter if found, or the default
      * @throws  ClassCastException if the default is not a Boolean
      */
    public boolean getBoolean(String name) throws ClassCastException {
        boolean def = false;
        Option o = getOption(name);
        if (o != null) def = ((Boolean) o.defaultValue()).booleanValue();
        String temp = getParam(name);
        if (temp == null) return def;
        return def ? !(temp.equalsIgnoreCase("no") || temp.equalsIgnoreCase("false"))
                   : (temp.equalsIgnoreCase("yes") || temp.equalsIgnoreCase("true"));
    }

    /** @return  the parameter if found, or the default
      * @throws  ClassCastException if the default is not a String
      */
    public String getString(String name)
	throws ClassCastException {
	return getString(name, false);
    }
    
    /** @return  the parameter if found, or the default
     *  @param name the parameter name
     *  @param translate whether to convert to a string rather than throwing
     *  @throws  ClassCastException if the default is not a String
     */
    public String getString(String name, boolean translate) 
	throws ClassCastException {
        String ret = getParam(name);
        if (ret == null || ret.equals("")) {
            Option o = getOption(name);
            if (o != null) ret = translate ? (o.defaultValue().toString()) :
		((String) o.defaultValue());
        }
        return ret;
    }

    /** Returns the raw string value of the parameter, if it was found
      * during readArgs() or readParams().
      */
    public String getParam(String name) {
        FieldSet fieldSet = this;
        StringTokenizer st = new StringTokenizer(name, ".");
        String fname;
        while (true) {
            fname = st.nextToken();
            if (st.hasMoreElements()) {
                fieldSet = getSet(fname);
                if (fieldSet == null) return null;
            }
            else {
                return fieldSet.get(fname);
            }
        }
    }

    /**
     * @return  the option associated with the parameter, if it is available.
     */
    public Option getOption(String name) {
	int x = name.indexOf('.');
	if(x >= 0) {
	    Params p = (Params)getSet(name.substring(0,x));
	    if(p == null)
		return null;
	    return p.getOption(name.substring(x+1));
	}
	return (Option) opts.get(name);
    }
    
    /**
     * Add all options to this Vector
     */
    public void getOptions(Vector v) {
	Enumeration e = opts.elements();
        for (;e.hasMoreElements();)
	    v.add((Option)e.nextElement());
	for (e=fields.elements();e.hasMoreElements();) {
	    Object o = e.nextElement();
	    if(o instanceof Params)
		((Params)o).getOptions(v);
	}
    }

    /**
     * @return  all Options added to this Params
     */
    public Option[] getOptions() {
	Vector v = new Vector(opts.size());
	getOptions(v);
        Option[] optArray = new Option[v.size()];
	for(int x=0;x<v.size();x++)
	    optArray[x] = (Option)v.elementAt(x);
	return optArray;
    }
    
    ////////////////////

    /**
     * Like parseFields, but using a system encoding. 
     * But unlike FieldSet.parseFields, we are not expecting trailing garbage.
     * We ignore lines that don't include the separator, rather than
     * stopping parsing at that point. This is to try to be a bit fault
     * tolerant with bad config files.
     */
    protected String parseParams(BufferedReader br, char equals, char sub,
                                 Filter nameF, Filter valueF) 
        throws IOException {
        
        String s = br.readLine();
        int i;
        while(s != null) {
	    if((i = s.indexOf(equals)) != -1) {
		String fullName = s.substring(0, i);
		String value = valueF.filter(s.substring(i + 1));
		readField(fullName, value, sub, nameF);
		s = br.readLine();
	    }
        }
        return s;
    }

    protected FieldSet newFieldSet() {
	return new Params();
    }

    public String toString() {
	String s = "Params: "+opts.size()+" Options: \n";
	int x=0;
	for(Enumeration e = opts.keys();e.hasMoreElements();) {
	    Object key = e.nextElement();
	    Object value = opts.get(key);
	    s += (x++) + ": " + key.toString() + ": " + value.toString()+"\n";
	}
	s += "End of Local Options.\n";
	x = 0;
	for(Enumeration e = fields.keys();e.hasMoreElements();) {
	    Object key = e.nextElement();
	    Object value = fields.get(key);
	    s += (x++) + ": " + key.toString() + ": " + value.toString()+"\n";
	}
	s += super.toString();
	s += "End.\n";
	return s;
    }
}



