package freenet.fs.acct;

import freenet.support.*;
import java.io.*;
import java.util.Enumeration;
import java.util.NoSuchElementException;

/**
 * An AccountingInitializer must be executed against the
 * AccountingTable at startup time, to determine the state
 * of affairs when the application terminated.  Transaction
 * annotations are collected and incomplete synchronized writes
 * are resolved at this time.  The AccountingInitializer
 * may then be used to instantiate an AccountingProcess.
 * @author tavin
 */
public class AccountingInitializer {


    public static void dump(AccountingInitializer init, PrintWriter out)
                                                    throws IOException {
        Enumeration bes;

        bes = init.getBlockElements();
        out.println("Data blocks");
        out.println("-----------");
        while (bes.hasMoreElements()) {
            BlockElement be = (BlockElement) bes.nextElement();
            out.println(be.getBlockNumber());
        }
        out.println();
        
        bes = init.freeBlocks.elements();
        out.println("Free blocks");
        out.println("-----------");
        while (bes.hasMoreElements()) {
            BlockElement be = (BlockElement) bes.nextElement();
            out.println(be.getBlockNumber());
        }
        out.println();

        bes = init.controlBlocks.elements();
        out.println("Control blocks");
        out.println("--------------");
        while (bes.hasMoreElements()) {
            ControlBlock cb = (ControlBlock) bes.nextElement();
            out.println(cb.toString());
        }
        out.println();

        bes = init.getBlockTransactions();
        out.println("Live transactions");
        out.println("-----------------");
        while (bes.hasMoreElements()) {
            BlockTransaction btx = (BlockTransaction) bes.nextElement();
            out.println(btx.toString());
        }
        out.println();
    }
    
    

    private final AccountingTable acct;
    
    
    private final OrderedVector freeBlocks = new OrderedVector();
    
    private final OrderedVector controlBlocks = new OrderedVector();
    
    private final OrderedVector txBlocks =
        new OrderedVector(new BlockTransaction.ComparatorByID());
    

    public AccountingInitializer(AccountingTable acct) throws IOException {
        
        this.acct = acct;

        ControlBlock cb;

        // initial scan:  erasures and control blocks
        for (int i=0; i<acct.getBlockCount(); ++i) {
            DataInput din = acct.readBlock(i);
            if (din == null) {  // erased block
                freeBlocks.insert(new BlockElement(i));
            }
            else {
                cb = ControlBlock.read(i, din);
                if (cb != null)
                    controlBlocks.insert(cb);
            }
        }

        // now put the control blocks in order of writing,
        // and discard any belonging to an incomplete write

        ControlBlock liveHead = null, deadHead = null;
        Enumeration cbe = controlBlocks.elements();
        
        while (cbe.hasMoreElements()) {
            cb = (ControlBlock) cbe.nextElement();
            if (cb.antecedent == -1) {
                liveHead = cb;
            }
            else {
                int i = controlBlocks.search(new BlockElement(cb.antecedent));
                if (i == -1)
                    deadHead = cb;
                else
                    ((ControlBlock) controlBlocks.elementAt(i)).next = cb;
            }
        }

        cb = deadHead;
        while (cb != null) {
            acct.destroyBlock(cb.getBlockNumber());
            int i = controlBlocks.search(cb);
            controlBlocks.removeElementAt(i);
            freeBlocks.insert(new BlockElement(cb.getBlockNumber()));
            cb = cb.next;
        }
        
        // now parse the legit control blocks

        SynchronizationBlock bsync = null;
        
        cb = liveHead;
        while (cb != null) {
            if (cb.type == ControlBlock.SYNCHRONIZATION_BLOCK) {
                if (cb.next != null) {
                    throw new AccountingException("found synchronization block "
                                                 +"that was not last in the chain");
                }
                bsync = (SynchronizationBlock) cb;
                break;
            }
            else if (cb.type == ControlBlock.ANNOTATION_BLOCK) {
                Enumeration bte = ((AnnotationBlock) cb).getAnnotations();
                while (bte.hasMoreElements()) {
                    BlockTransaction btxNew = (BlockTransaction) bte.nextElement();
                    int i = txBlocks.insert(btxNew, false);
                    BlockTransaction btx = (BlockTransaction) txBlocks.elementAt(i);
                    // concatenate annotations with the same id
                    if (btx != btxNew)
                        btx.annotate(btxNew.extractAnnotation());
                }
            }
            else throw new AccountingException("unknown control block type: 0x"
                                               +Integer.toHexString(cb.type));
            cb = cb.next;
        }
            
        // finally, resolve the block sync if necessary
        if (bsync != null) {
            int[] newBlocks = bsync.getNewBlocks();
            boolean halfComplete = true;
            for (int i=0; i<newBlocks.length; ++i) {
                if (freeBlocks.search(new BlockElement(newBlocks[i])) != -1) {
                    halfComplete = false;
                    break;
                }
            }
            if (halfComplete) {
                // well, the fact that we found the synchronization record
                // means it didn't finish completely, so..
                int[] oldBlocks = bsync.getOldBlocks();
                for (int i=0; i<oldBlocks.length; ++i) {
                    acct.destroyBlock(oldBlocks[i]);
                    freeBlocks.insert(new BlockElement(oldBlocks[i]), false);
                }
                cb = liveHead;
                while (cb != null) {
                    acct.destroyBlock(cb.getBlockNumber());
                    freeBlocks.insert(new BlockElement(cb.getBlockNumber()));
                    cb = cb.next;
                }
                // collected annotation data is void
                txBlocks.removeAllElements();
                controlBlocks.removeAllElements();
            }
            else {
                // wipe out the incomplete blocks and the sync block
                for (int i=0; i<newBlocks.length; ++i) {
                    acct.destroyBlock(newBlocks[i]);
                    freeBlocks.insert(new BlockElement(newBlocks[i]), false);
                }
                acct.destroyBlock(bsync.getBlockNumber());
                freeBlocks.insert(new BlockElement(bsync.getBlockNumber()));
                controlBlocks.removeElement(bsync);
            }
        }
    }

    public void initialize(AccountingStruct struct) throws IOException {
        
        OrderedVector txBlockSort = new OrderedVector();
        Enumeration bte = getBlockTransactions();
        while (bte.hasMoreElements()) {
            BlockTransaction btx = (BlockTransaction) bte.nextElement();
            if (btx.getBlockNumber() == -1)
                struct.found(btx);
            else
                txBlockSort.insert(btx);
        }

        Enumeration bes = getBlockElements();
        while (bes.hasMoreElements()) {
            
            BlockElement be = (BlockElement) bes.nextElement();
            
            DataInput din = acct.readBlock(be.getBlockNumber());
            // throw away the zeroes at the beginning
            din.skipBytes(ControlBlock.CONTROL_TYPE_WIDTH);
            
            int i = txBlockSort.search(be);
            if (i == -1) {
                struct.found(be.getBlockNumber(), din);
            }
            else {
                BlockTransaction btx = (BlockTransaction) txBlockSort.elementAt(i);
                struct.found(be.getBlockNumber(), din, btx);
            }
        }
    }

    final AccountingTable getAccountingTable() {
        return acct;
    }

    final BlockList getFreeBlocks() {
        return new BlockList(freeBlocks.size(), freeBlocks.elements());
    }

    final BlockList getControlBlocks() {
        return new BlockList(controlBlocks.size(), controlBlocks.elements());
    }

    public final Enumeration getBlockTransactions() {
        return txBlocks.elements();
    }
    
    public final Enumeration getBlockElements() {
        return new BlockElementEnumeration();
    }

    private final class BlockElementEnumeration implements Enumeration {

        // todo:  use a bloom filter?
        
        private BlockElement be;
        
        BlockElementEnumeration() {
            step(-1);
        }

        private void step(int n) {
            while (++n < acct.getBlockCount()) {
                be = new BlockElement(n);
                if (freeBlocks.search(be) == -1 && controlBlocks.search(be) == -1)
                    { return; }
            }
            be = null;
        }
        
        public final boolean hasMoreElements() {
            return be != null;
        }

        public final Object nextElement() {
            if (be == null) throw new NoSuchElementException();
            try {
                return be;
            }
            finally {
                step(be.getBlockNumber());
            }
        }
    }
}


