package freenet.fs.dir;

import freenet.support.DoublyLinkedList;
import freenet.support.DoublyLinkedListImpl;
import freenet.support.DoublyLinkedListImpl.Item;
import java.io.*;
import java.util.Enumeration;

import freenet.Core;

/**
 * A Buffer for tunneling data through a limited storage area.
 * After the writer starts writing to the output stream, readers
 * may pick up input streams until the first time the buffer laps.
 * @author tavin
 */
public class CircularBuffer implements Buffer {

    protected final Buffer lapbuf;
    protected final long vlength;

    protected int lap = -1;
    protected boolean aborted = false;
    
    protected DoublyLinkedList readers = new DoublyLinkedListImpl();

    /**
     * @param lapbuf   buffer to do laps over
     * @param vlength  virtual length -- total length of tunneled data
     */
    public CircularBuffer(Buffer lapbuf, long vlength) {
        this.lapbuf = lapbuf;
        this.vlength = vlength;
    }

    
    public final Ticket ticket() {
        return lapbuf.ticket();
    }


    public final void touch() {
        //throw new DirectoryException("now what's the point of that?");
    }

    public final void commit() {
        throw new DirectoryException("you can't commit a circular buffer");
    }
    
    public final void release() {
        lapbuf.release();
    }
    

    public final boolean failed() {
        return aborted || lapbuf.failed();
    }

    public final long length() {
        return vlength;
    }

    /**
     * This will grant input streams only until the buffer has lapped.
     * It will block until the output stream has been obtained.
     */
    public InputStream getInputStream() throws IOException,
                                               BufferException {
        return new CircularInputStream();
    }

    private class CircularInputStream extends FilterInputStream {

        private long rlim;
        private int rlap = -1;

        private Item it;

        private boolean constructed = false;
     
        private CircularInputStream() throws IOException, BufferException {
            super(null);
            synchronized (CircularBuffer.this) {
		atStart = true;
                if (lap > 0)
                    throw new BufferException("buffer already lapped - this is commonly caused by really small datastores, try the default 250MB");
                it = new CircularInputStreamItem(this);
                readers.push(it);
                constructed = true;
                nextLap();
            }
        }

	boolean atStart = false;

        public int read() throws IOException {
            if (aborted) throw new BufferException("buffer write aborted");
            if (rlim <= 0 && !nextLap())
                return -1;
            int rv = in.read();
//              System.out.println("Read:  " + rv + "\trlap: " + rlap + "\t\trlim: " + rlim);
            if (rv != -1) --rlim;
	    if(atStart) {
		if(rv != -1) atStart = false;
		synchronized(CircularBuffer.this) {
		    CircularBuffer.this.notify();
		}
	    }
            return rv;
        }

        public int read(byte[] buf, int off, int len) throws IOException {
            if (aborted) throw new BufferException("buffer write aborted");
            if (rlim <= 0 && !nextLap())
                return -1;
            int rv = in.read(buf, off, (int) Math.min(len, rlim));
	    Core.logger.log(this, "VLength "+vlength+" rlim="+rlim+" rlap="+
			    rlap+" lap="+lap+" rv="+rv, Core.logger.DEBUG);
            if (rv != -1) rlim -= rv;
	    if(atStart) {
		if(rv != -1) atStart = false;
		synchronized(CircularBuffer.this) {
		    CircularBuffer.this.notify();
		}
	    }
            return rv;
        }

        public int available() throws IOException {
            if (aborted) throw new BufferException("buffer write aborted");
            return in.available();
        }
        
        public long skip(long n) throws IOException {
            if (aborted) throw new BufferException("buffer write aborted");
	    return super.skip(n); // FIXME?
        }

        public void close() throws IOException {
            try {
                in.close();
            } finally {
                kick();
            }
        }

	// Do not create a new reader at the beginning until the old writer has moved on
	// Do not create a new writer at the beginning until the old reader has moved on
        private boolean nextLap() throws IOException {
            rlim = Math.min(lapbuf.length(),
                            vlength - (1+rlap) * lapbuf.length());
            if (rlim <= 0) {
                return false;
            }
            synchronized (CircularBuffer.this) {
                ++rlap;
                while (!failed() && ((rlap > lap) || outAtStart)) {
		    Core.logger.log(this, "Waiting in nextLap: rlap="+rlap+
				    " lap="+lap+" outAtStart="+outAtStart,
				    Core.logger.DEBUG);
                    try { 
			CircularBuffer.this.wait(200); // avoid problems with buggy JVMs !
		    }
                    catch (InterruptedException e) {}
                }
                if (aborted)
                    throw new BufferException("buffer write aborted");
		atStart = true;
		InputStream i = in;
                in = lapbuf.getInputStream();
		if(i!=null) i.close();
                CircularBuffer.this.notifyAll();
            }
            return true;
        }

        private void kick() {
            synchronized(CircularBuffer.this) {
                readers.remove(it);
            }
        }

        public void finalize() throws Throwable {
            if (constructed)
                kick();
        }
    }

    private class CircularInputStreamItem extends Item {
        CircularInputStream in;
        public CircularInputStreamItem(CircularInputStream in) {
            this.in = in;
        }
    }

    /**
     * This will grant one and only one output stream.  When the stream
     * laps the buffer, the buffer will be removed from the directory.
     */
    public OutputStream getOutputStream() throws IOException,
                                                 BufferException {
        return new CircularOutputStream();
    }    

    boolean outAtStart = false;

    private class CircularOutputStream extends FilterOutputStream {

        private long wlim;

        private CircularOutputStream() throws IOException, BufferException {
            super(null);
            synchronized (CircularBuffer.this) {
                if (lap != -1)
                    throw new BufferException("buffer already written");
                nextLap();
            }
        }

        public void write(int b) throws IOException {
            if (wlim <= 0) nextLap();
//              System.out.println("Wrote: " + (b & 0xff) + "\tlap: " + 
//                                 lap + "\t\twlim:" + wlim);
            out.write(b);
            --wlim;
	    if(outAtStart) {
		outAtStart = false;
		synchronized(CircularBuffer.this) {
		    CircularBuffer.this.notify();
		}
	    }
        }

        public void write(byte[] buf, int off, int len) throws IOException {
            while (len > 0) {
                if (wlim <= 0) nextLap();
                int n = (int) Math.min(len, wlim);
                out.write(buf, off, n);
                off += n;
                len -= n;
                wlim -= n;
		if(outAtStart) {
		    outAtStart = false;
		    synchronized(CircularBuffer.this) {
			CircularBuffer.this.notify();
		    }
		}
            }
        }
        
        private void nextLap() throws IOException {
            wlim = Math.min(lapbuf.length(),
                            vlength - (1+lap) * lapbuf.length());
            if (wlim <= 0) {
                throw new EOFException();
            }
            synchronized (CircularBuffer.this) {
                // first we wait for all readers to start writing
                // on the previous lap.
                while (shouldWait()) {
		    Core.logger.log(this, "Waiting in nextLap",
				    Core.logger.DEBUG);
                    try {
                        CircularBuffer.this.wait(200);
			// Avoid problems with buggy JVMs
                    } catch (InterruptedException e) {}
		}
                // then we tell the readers they can start reading on 
                // this one (when we release).
                ++lap;
		outAtStart = true;
		OutputStream o = out;
                out = lapbuf.getOutputStream();
		if(o!=null) o.close();
                CircularBuffer.this.notifyAll();
            }
        }

        // Called synced on CircularBuffer //
        private boolean shouldWait() {
            for (Enumeration e = readers.elements() ;e.hasMoreElements();){
		CircularInputStreamItem i = (CircularInputStreamItem) e.nextElement();
                int rlap = i.in.rlap;
                if (rlap < lap) {
		    // we can't go to the next until they enter ours
		    Core.logger.log(this, "Writer waiting because rlap="+rlap
				      +" lap="+lap, Core.logger.DEBUG);
                    return true;
		}
		if(i.in.atStart) {
		    Core.logger.log(this, "Writer waiting because reader "+
				      "atStart", Core.logger.DEBUG);
		    return true; 
		    // we can't go to the next until they are no longer at position 0
		}
            } // at least with NativeFSDir, that causes them to skip a lap
            return false;
        }

        public void close() throws IOException {
            try {
		Core.logger.log(this, "closing circularbuffer", new
				DirectoryException("debug"), 
				Core.logger.DEBUG);
                out.close();
            }
            finally {
                kick();
            }
        }

        private void kick() {
            if ((lap + 1) * lapbuf.length() <= vlength || wlim > 0) {
                synchronized (CircularBuffer.this) {
                    aborted = true;
                    Core.logger.log(this, "Aborting because only " +
                                    lap + " laps and " + wlim + " left of " +
                                    vlength + " bytes were written.",
                                    Core.logger.MINOR);
		    // MINOR because it can happen due to eg connections
		    // being closed prematurely
                    CircularBuffer.this.notifyAll();
                }
            }
        }

        protected void finalize() throws Throwable {
            kick();
        }
    }
}




