package freenet.fs.dir;

import freenet.fs.acct.Fragment;
import freenet.node.Node;
import freenet.support.Fields;
import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.Enumeration;
import java.util.Date;
import java.text.DateFormat;

/**
 * Debug console for the FSDirectory.
 * @author tavin
 */
public class FSConsole extends HttpServlet {

    Node node;
    
    public final void init() {
        this.node = (Node) getServletContext().getAttribute("freenet.node.Node");
    }
    
    public void doGet(HttpServletRequest req, HttpServletResponse resp)
                                                    throws IOException {

        FSDirectory dir = (FSDirectory) node.dir;

        resp.addHeader("Cache-control", "no-cache");
        resp.addHeader("Pragma", "no-cache");
        resp.addHeader("Expires", "Thu, 01 Jan 1970 00:00:00 GMT");
        resp.setContentType("text/html");

        PrintWriter out = resp.getWriter();
        
        out.println("<!DOCTYPE html");
        out.println("  PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"");
        out.println("  \"DTD/xhtml1-frameset.dtd\">");
        out.println("<html>");
        out.println("<head><title>FileSystem Console: " + (new Date())
                                                        + "</title></head>");
        out.println("<body>");

        synchronized (dir.semaphore()) {
        
            out.println("<p><code>");
            out.println("Version: " + FSDirectoryRoot.VERSION + "<br />");
            out.println("Free space: " + dir.available() + "<br />");
            out.println("Dirty: " + dir.dirty() + "<br />");
            out.println("Accounting ranges: "
                        + (dir.acctRanges == null
                           ? "none" : Fragment.rangeList(dir.acctRanges)));
            out.println("</code></p>");
            
            dumpFiles(dir, out);
            dumpLRUFiles(dir, out);
        }

        out.println("</body>");
        out.println("</html>");

        resp.flushBuffer();
    }

    private static void printKeyLink(byte[] val,
                                     HttpServletRequest req,
                                     HttpServletResponse resp) throws IOException {
        String hex = Fields.bytesToHex(val);
        String uri = req.getContextPath() + req.getServletPath() + '/' + hex;
        PrintWriter out = resp.getWriter();
        out.println("<td><input type=\"checkbox\" name=\"keys\" value=\""
                    + hex + "\" /></td>");
        out.println("<td><code><a href=\"" + uri + "\">"
                    + hex + "</a></code></td>");
    }

    private static void dumpFiles(FSDirectory dir, PrintWriter out) {
        out.println("<table>");
        out.print("<tr style=\"background-color: grey\"><th colspan=\"3\">");
        out.print("Files");
        out.println("</th></tr>");
        Enumeration keys = dir.keys(true);
        while (keys.hasMoreElements()) {
            FileNumber fn = (FileNumber) keys.nextElement();
            Buffer buffer = dir.fetch(fn);
            try {
                out.println("<tr>");
                out.println("<td><code>"+fn+"</code></td>");
                out.println("<td><code>@</code></td>");
                out.println("<td><code>");
                out.println(Fragment.rangeList(buffer.ticket().ranges));
                out.println("</code></td>");
                out.println("</tr>");
            }
            finally {
                buffer.release();
            }
        }
        out.println("</table>");
    }

    private static void dumpLRUFiles(FSDirectory dir, PrintWriter out) {
        DateFormat df = DateFormat.getDateTimeInstance();
        out.println("<table>");
        out.print("<tr style=\"background-color: grey\"><th colspan=\"3\">");
        out.print("LRU files (oldest last)");
        out.println("</th></tr>");
        Enumeration keys = dir.lruKeys(false);
        while (keys.hasMoreElements()) {
            FileNumber fn = (FileNumber) keys.nextElement();
            Buffer buffer = dir.fetch(fn);
            try {
                out.println("<tr>");
                out.println("<td><code>"+fn+"</code></td>");
                out.println("<td><code>@</code></td>");
                out.println("<td><code>");
                out.println(df.format(new Date(buffer.ticket().timestamp)));
                out.println("</code></td>");
                out.println("</tr>");
            }
            finally {
                buffer.release();
            }
        }
        out.println("</table>");
    }
}



