package freenet.fs.dir;

import freenet.fs.acct.Fragment;
import freenet.fs.acct.sys.AccountingTreeNode;
import freenet.support.*;
import freenet.support.Comparable;
import freenet.support.BinaryTree.Node;
import freenet.support.sort.*;
import java.util.Vector;


/**
 * Maintains a map of the internal Ticket structures that track allocations.
 * Two maps are actually kept:  a small one for the most recently used tickets
 * that is retained in memory, and a large one for the rest of the tickets
 * that is kept on disk only.
 * @author tavin
 */
class TicketManager {

    private final FSDirectory dir;

    private final TicketMap liveTicketMap;
    private final TicketMap mainTicketMap;

    private final LRUCache liveTicketLRU;

    private long lastID = -1;
    
                     
    /**
     * @param liveTicketMap  the in-memory map for recently active tickets
     * @param mainTicketMap  the on-disk map for the rest of them
     *
     * Both ticket maps should be already initialized.
     *
     * @param lruSize  number of committed tickets to keep in memory before
     *                 pushing back to the main map.
     */
    TicketManager(FSDirectory dir, int lruSize,
                  TicketMap liveTicketMap, TicketMap mainTicketMap) {

        this.dir = dir;
        this.liveTicketLRU = new LRUCache(lruSize);
        this.liveTicketMap = liveTicketMap;
        this.mainTicketMap = mainTicketMap;

        Walk walk = liveTicketMap.treeWalk(true);
        
        Vector v = new Vector();
        synchronized (v) {

            Node node;
            while (null != (node = (Node) walk.getNext())) {
                Ticket ticket = (Ticket) node.getObject();
                if (lastID < ticket.ticketID) {
                    lastID = ticket.ticketID;
                }
                if (ticket.timestamp != -1) {
                    ticket.cel = new TicketMemory(node);
                    v.addElement(ticket.cel);
                }
            }
            
            // sort on access time
            HeapSorter.heapSort(new VectorSorter(v));

            for (int i=0; i<v.size(); ++i) {
                TicketMemory tmem = (TicketMemory) v.elementAt(i);
                dir.demote(tmem.ticket().timestamp, tmem.ticket().fn);
                liveTicketLRU.cache(tmem);
            }
        }
    }


    /**
     * Deletes a ticket.  The ticket cannot have any active locks.
     * @return  the deleted ticket, or null if there was no ticket
     *          matching the given ticket ID
     * @throws DirectoryException
     *         if there are active locks on the ticket
     */
    Ticket delete(long ticketID) {

        Ticket liveTicket, mainTicket;

        Node node = liveTicketMap.treeSearch(new Ticket(ticketID));
        if (node == null) {
            liveTicket = null;
        }
        else {
            liveTicket = (Ticket) node.getObject();
            if (liveTicket.users > 0) {
                throw new DirectoryException("cannot delete an active ticket");
            }
            liveTicketMap.treeRemove(node);
            if (liveTicket.cel != null) {
                liveTicketLRU.uncache(liveTicket.cel);
                liveTicket.cel = null;  // break circular ref loop
            }
        }
        
        mainTicket = mainTicketMap.remove(ticketID);

        return liveTicket == null ? mainTicket : liveTicket;
    }


    /**
     * @return  the next ticket ID that may be used to call create()
     */
    final long getNextID() {
        return ++lastID;
    }

    /**
     * Create a new ticket.  A ticket-lock is automatically granted.
     */
    TicketLock create(long ticketID, Fragment[] ranges, FileNumber fn) {
        Ticket ticket = new Ticket(ticketID, -1, ranges, fn);
        liveTicketMap.put(ticket);
        ++ticket.users;
        return new TicketLockImpl(ticket);
    }

    /**
     * Lock an existing ticket.
     */
    TicketLock lock(long ticketID) {
        Ticket ticket = liveTicketMap.get(ticketID);
        if (ticket == null) {
            ticket = mainTicketMap.get(ticketID);
            if (ticket == null) {
                return null;
            }
            Node node = new AccountingTreeNode(ticket);
            liveTicketMap.treeInsert(node, false);
        }
        if (ticket.users++ == 0 && ticket.timestamp != -1) {
            if (ticket.cel != null) {
                liveTicketLRU.uncache(ticket.cel);
                ticket.cel = null;
            }
            dir.promote(ticket.timestamp, ticket.fn);
        }
        return new TicketLockImpl(ticket);
    }

    private final class TicketLockImpl implements TicketLock {

        private final Ticket ticket;
        private boolean released = false;
        
        TicketLockImpl(Ticket ticket) {
            this.ticket = ticket;
        }

        public final Ticket ticket() {
            return ticket;
        }

        public void touch() {
            if (ticket.timestamp == -1) {
                throw new DirectoryException("ticket not committed");
            }
            Node node = new AccountingTreeNode(ticket);
            ticket.timestamp = System.currentTimeMillis();
            liveTicketMap.treeInsert(node, true);
        }
        
        public void commit() {
            if (ticket.timestamp != -1) {
                throw new DirectoryException("ticket previously committed");
            }
            Node node = new AccountingTreeNode(ticket);
            ticket.timestamp = System.currentTimeMillis();
            liveTicketMap.treeInsert(node, true);
        }
        
        public void release() {
            if (!released) {
                released = true;
                if (--ticket.users == 0 && ticket.timestamp != -1) {
                    dir.demote(ticket.timestamp, ticket.fn);
                    Node n = liveTicketMap.treeSearch(ticket);
                    ticket.cel = new TicketMemory(n);
                    liveTicketLRU.cache(ticket.cel);
                }
            }
        }
    }
        



    // the TicketMemory is used to maintain an LRU access record
    // for tickets in the live ticket map.

    private final class TicketMemory extends DoublyLinkedListImpl.Item
                                     implements Cacheable, Comparable {
    
        Node ticketNode;
    
        TicketMemory(Node ticketNode) {
            this.ticketNode = ticketNode;
        }
    
        final Ticket ticket() {
            return (Ticket) ticketNode.getObject();
        }
        
        public final void drop() {
            // move the ticket from the live map to the main map
            try {
                Ticket ticket = ticket();
                liveTicketMap.treeRemove(ticketNode);
                mainTicketMap.put(ticket);
            }
            finally {
                ticketNode = null;  // break circular ref loop
            }
        }
        
        public final int compareTo(Object o) {
            return compareTo((TicketMemory) o);
        }
    
        public final int compareTo(TicketMemory that) {
            // order by access time
            Ticket thisTicket = this.ticket();
            Ticket thatTicket = that.ticket();
            return thisTicket.timestamp == thatTicket.timestamp
                   ? 0 : (thisTicket.timestamp > thatTicket.timestamp ? 1 : -1);
        }
    }
}


