package freenet.interfaces;

import freenet.*;
import freenet.thread.ThreadFactory;
import freenet.transport.*;
import freenet.support.*;
import freenet.config.Params;
import java.net.InetAddress;
import java.io.IOException;

/**
 * A BaseLocalNIOInterface is connected to by clients
 * and can do allowed hosts checking.
 *
 */
public abstract class BaseLocalNIOInterface extends NIOInterface {

    protected int[][] allowedHosts;
    
    private static final int intAddress(String addr) { 
	try {
	    return intAddress(InetAddress.getByName(addr));
	} catch (java.net.UnknownHostException e) {
	    return 0;
	}
    }

    private static final int intAddress(InetAddress addr) { 
	boolean logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
	if(logDEBUG) Core.logger.log(LocalInterface.class, "intAddress("+
				     addr.toString()+")", Logger.DEBUG);
	byte[] b = addr.getAddress();
	if(logDEBUG) Core.logger.log(LocalInterface.class, "Address: "+(((int)b[0]) & 0xff) +
				     "."+(((int)b[1]) & 0xff)+"."+(((int)b[2]) & 0xff)+"."+
				     (((int)b[3]) & 0xff)+" ("+
				     b.length+")", Logger.DEBUG);
	long x = ((((long)b[0]) & 0xff) << 24) + ((((int)b[1]) & 0xff) << 16) + 
	    ((((int)b[2]) & 0xff) << 8) + (((int)b[3]) & 0xff);
	if(logDEBUG) Core.logger.log(LocalInterface.class, "Returning "+
				     Fields.longToHex(x), Logger.DEBUG);
	return (int) x;
    }
    
    private static final int mask(int addr, int maskbits) {
	int mbits = 32 - maskbits;
	int power;
	if(mbits < 32)
	    power = 1 << mbits;
	else
	    power = 0; // 1 << 32 = 1 !! - check before you "fix" this
	int ones = power - 1;
	int mask = ~ones;
	//int mask = ~((1 << (32-maskbits)) -1);
	int out = addr & mask;
	return out;
    }
    
    int runningConnections = 0; // number of connections running
    int lowRunningConnections; // reenable interface when go below this
    int highRunningConnections; // disable interface when go above this
   
    /**
     * @param  allowedHosts         set of Addresses to do an equalsHost() check with;
     *                      null means to allow all hosts
     * @param  listenAddr           Description of the Parameter
     * @exception  ListenException  Description of the Exception
     */
    public BaseLocalNIOInterface(ListeningAddress listenAddr,
				 int[][] allowedHosts,
				 int lowRunningConnections, 
				 int highRunningConnections)
        throws ListenException {
        super(listenAddr);
        this.allowedHosts = allowedHosts;
	this.lowRunningConnections = lowRunningConnections;
	this.highRunningConnections = highRunningConnections;
	this.listener = getListener(listenAddr);
    }
    
    private NIOListener getListener(ListeningAddress listenAddr) 
	throws ListenException {
	return new tcpNIOListener((tcpTransport)listenAddr.getTransport(),
				  (tcpListeningAddress)listenAddr, false);
    }
    
    protected synchronized void reopenListener() throws ListenException {
	if(listening) return;
	Core.logger.log(this, "Restarting listening on "+this,
			Logger.NORMAL);
	this.listener = getListener(listenAddr);
	register(loop);
	listening = true;
    }
    
    /**
     * @param  allowedHosts         set of Addresses to do an equalsHost() check with;
     *                      null means to allow all hosts
     * @param  listenAddr           Description of the Parameter
     * @exception  ListenException  Description of the Exception
     */
    public BaseLocalNIOInterface(ListeningAddress listenAddr,
				 String allowedHosts,
				 int lowRunningConnections, 
				 int highRunningConnections)
        throws ListenException {
	super(listenAddr);
        int[][] allowedHostsAddr = null;

	if (allowedHosts == null || allowedHosts.trim().equals("")) {
	    allowedHosts = "127.0.0.0/8";
	}

	if (allowedHosts.trim().equals("*")) {
	    allowedHosts = "0.0.0.0/0";
	}
	
	if(Core.logger.shouldLog(Logger.DEBUG))
	    Core.logger.log(this, "New BaseLocalNIOInterface: "+listenAddr+
			    ","+allowedHosts, Logger.DEBUG);
	
	String[] hosts = Fields.commaList(allowedHosts);

	allowedHostsAddr = new int [hosts.length][2];
	
	for (int i = 0; i < hosts.length; ++i) {
	    int host, subnet, div = hosts[i].indexOf('/');
	    if (div == -1) {
		subnet = 32;
		host = intAddress(hosts[i]);
	    } else {
		subnet = Integer.parseInt(hosts[i].substring(div+1));
		host = intAddress(hosts[i].substring(0,div));
	    }
	    allowedHostsAddr[i][0] = mask(host, subnet);
	    allowedHostsAddr[i][1] = subnet;
	}
	this.allowedHosts = allowedHostsAddr;
	this.lowRunningConnections = lowRunningConnections;
	this.highRunningConnections = highRunningConnections;
	this.listener = getListener(listenAddr);
    }


    /**
     *  Description of the Method
     *
     * @param  conn                             Description of the Parameter
     * @exception  RejectedConnectionException  Description of the Exception
     */
    protected void dispatch(Connection conn) throws RejectedConnectionException {
	boolean allow = false;
	Address ha = conn.getPeerAddress();
	if(Core.logger.shouldLog(Core.logger.DEBUG))
	    Core.logger.log(this, 
			    "Dispatching connection on a BaseLocalNIOInterface from " +
			    ha.toString(), Core.logger.DEBUG);
	//insert some code to make sure ha is a tcpAddress and
	//handle things correctly when it's not. --thelema
	int inta;
	try {
	    inta = intAddress(((tcpAddress) ha).getHost());
	} catch (java.net.UnknownHostException e) {
	    Core.logger.log(this, "Unknown Host on incoming connection!!",
			    Core.logger.ERROR);
	    throw new RejectedConnectionException("unknown host on incoming connection!");
	}
	
	for (int i = 0; !allow && i < allowedHosts.length; i++) {
	    int subnet = allowedHosts[i][0];
	    int maskbits = allowedHosts[i][1];
	    allow |= (mask(inta, maskbits) == subnet);
	    if(Core.logger.shouldLog(Core.logger.DEBUG))
		Core.logger.log(this, "Trying "+Fields.intToHex(subnet)+
				":"+Fields.intToHex(maskbits)+" for "+
				Fields.intToHex(inta), Core.logger.DEBUG);
	}
	
	if (allow) {
	    handleConnection(conn);
	} else {
	    if(Core.logger.shouldLog(Core.logger.DEBUG))
		Core.logger.log(this, "Rejecting local connection",
				Core.logger.DEBUG);
	    throw new RejectedConnectionException("host not allowed: " + ha);
	}
    }
    
    protected abstract void handleConnection(Connection conn);
}



