package freenet.message;
import freenet.*;
import freenet.node.*;
import freenet.support.*;
import java.util.*;
import freenet.node.states.request.*;

public abstract class Request extends NodeMessage implements HTLMessage {

    public final Key searchKey;
    public long hopsToLive;
    public long stateTime = -1;
    
    protected final NodeReference requestSource;

    public Request(long id, long htl, Key key, NodeReference ref) {
        super(id);
        searchKey     = key;
        hopsToLive    = htl;
        requestSource = ref;
    }

    public Request(long id, long htl, Key key,
                   NodeReference ref, FieldSet otherFields) {
	
        super(id, otherFields);
        searchKey     = key;
        hopsToLive    = htl;
        requestSource = ref;
    }
   
    public Request(ConnectionHandler source, RawMessage raw) 
                                            throws InvalidMessageException {
        super(source, raw);
        String keyString  = otherFields.get("SearchKey");
        String hopsString = otherFields.get("HopsToLive");
        FieldSet nodeRef  = otherFields.getSet("Source");
        if (hopsString == null || hopsString.equals(""))
            throw new InvalidMessageException("Can't find HopsToLive field");
        if (keyString == null || keyString.equals("")) 
            throw new InvalidMessageException("Can't find SearchKey field");
        if (nodeRef == null)
            throw new InvalidMessageException("Can't find Source field");
        try {
            searchKey     = Key.readKey(keyString);
            hopsToLive    = Fields.stringToLong(hopsString);
            // there is no reason to verify this.
            requestSource = new NodeReference(nodeRef, false, 
                                              source.peerIdentity());
        }
        catch (KeyException k) {
            throw new InvalidMessageException("Failed to load key: " + k);
        }
        catch (NumberFormatException e) {
            throw new InvalidMessageException("Failed to read number " + e);
        }
        catch (BadReferenceException e) {
            throw new InvalidMessageException("Failed to read NodeReference: " 
                                              + e);
        }
        otherFields.remove("HopsToLive");
        otherFields.remove("SearchKey");
        otherFields.remove("Source");
    }
    
    public RawMessage toRawMessage(Presentation t) {
        RawMessage raw=super.toRawMessage(t);
        raw.fs.add("SearchKey",  searchKey.toString());
        raw.fs.add("HopsToLive", Fields.longToString(hopsToLive));
        raw.fs.add("Source",     requestSource.getFieldSet(false));
        return raw;
    }

    public int hopsToLive()  {
        return (int) hopsToLive;
    }

    public NodeReference getSource() {
        return requestSource;
    }
}



