package freenet.message;
import freenet.*;
import freenet.node.*;
import freenet.crypt.RandomSource;
import freenet.support.Logger;
import freenet.support.Fields;

/**
 * This message is sent after the DataSend has been completed and
 * causes a routing table reference to be added for the dataSource.
 *
 * @author oskar
 **/

public class StoreData extends NodeMessage {

    public static final String messageName = "StoreData";

    private final NodeReference dataSource;
    
    private int hopsSinceReset;

    private long requestsPerHour;

    public StoreData(long idnum) {
        this(idnum, null, 0, 0);
    }

    public StoreData(long idnum, 
                     NodeReference dataSource,
                     long requestsPerHour, int hopsSinceReset) {
        super(idnum, null);
        this.dataSource = dataSource;
        this.requestsPerHour = requestsPerHour;
	this.hopsSinceReset = hopsSinceReset;
    }

    public StoreData(ConnectionHandler source, RawMessage raw) 
        throws InvalidMessageException {
        
        super(source, raw);
        try {
            FieldSet dsource = otherFields.getSet("DataSource");
            dataSource = (dsource == null ? 
                          null : 
                          new NodeReference(dsource));
            
            requestsPerHour = -1;
            String rateAsString = otherFields.get("RequestRate");
            if (rateAsString != null) {
                requestsPerHour = Fields.stringToLong(rateAsString);
            }

            String hopsAsString = otherFields.get("HopsSinceReset");
	    if(hopsAsString != null) {
		try {
		    hopsSinceReset = Fields.hexToInt(hopsAsString);
		} catch (NumberFormatException e) {
		    hopsSinceReset = 0;
		}
	    } else hopsSinceReset = 0;
	    
	    otherFields.remove("HopsSinceReset");
            otherFields.remove("DataSource");
            otherFields.remove("RequestRate");
        }
        catch (BadReferenceException e) {
            throw new InvalidMessageException("Bad Reference");
        }
        catch (NumberFormatException nfe) {
            throw new InvalidMessageException("Failed to read number " + nfe);
        }
    }

    public RawMessage toRawMessage(Presentation t) {
        RawMessage raw = super.toRawMessage(t);
        if (dataSource != null) {
            raw.fs.add("DataSource", dataSource.getFieldSet());
        }
        if (requestsPerHour != -1) {
            raw.fs.add("RequestRate",Fields.longToString(requestsPerHour));
        }
	raw.fs.add("HopsSinceReset", Fields.intToHex(hopsSinceReset));
	return raw;
    }
    
    public int hopsSinceReset() {
	return hopsSinceReset;
    }

    public boolean shouldCache(RandomSource r, float cacheProbPerHop) {
	return r.nextFloat() < cacheProbability(cacheProbPerHop);
    }
    
    public float cacheProbability(float cacheProbPerHop) {
	return (float)Math.pow((float)cacheProbPerHop, hopsSinceReset);
    }
    
    public final NodeReference dataSource() {
        return dataSource;
    }
    
    // for now -1, means unknown
    public final long requestsPerHour() {
        return requestsPerHour;
    }
    
    public final String getMessageName() {
        return messageName;
    }
    
    public String toString() {
        return "freenet.Message: "+messageName+"@(hopsSinceReset="+
            hopsSinceReset+",requestsPerHour="+requestsPerHour+
            ",dataSource="+(dataSource==null?"(null)":
                            dataSource.toString())+")";
    }
}




