/* -*- Mode: java; c-basic-indent: 4; tab-width: 4 -*- */
package freenet.node;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.net.DatagramSocket;
import java.net.SocketException;
import java.util.Vector;
import java.util.Enumeration;
import freenet.support.Checkpointed;
import freenet.Core;
import freenet.support.Logger;

/**
 * A class to autodetect our IP address(es)
 */

class IPAddressDetector implements Checkpointed {
	//private String preferedAddressString = null;
    public IPAddressDetector() {
    	
    }
    

    /**
     * @return our name
     */
    public String getCheckpointName() {
	return "Autodetection of IP addresses";
    }

    /** 
     * @return next scheduling point
     */
    public long nextCheckpoint() {
	return System.currentTimeMillis() + 10000; // We are pretty cheap
    }

    InetAddress lastInetAddress = null;
    InetAddress[] lastAddressList = null;
    long lastDetectedTime = -1;

	/** 
	 * Fetches the currently detected IP address. If not detected yet a detection is forced
	 * @param preferedAddress An address that for some reason is prefered above others. Might be null
	 * @return Detected ip address
	 */
    public InetAddress getAddress(String preferedAddress) {
		return getAddress(0,preferedAddress);
    }
    
	/** 
	 * Fetches the currently detected IP address. If not detected yet a detection is forced
	 * @return Detected ip address
	 */
	public InetAddress getAddress() {
		return getAddress(0,null);
	}
    
    /**
     * Get the IP address
     * @param preferedAddress An address that for some reason is prefered above others. Might be null
	 * @return Detected ip address
     */
    public InetAddress getAddress(long recheckTime,String preferedAddress) {
		if(lastInetAddress == null || 
	   	System.currentTimeMillis() > 
	   	(lastDetectedTime + recheckTime))
	    	checkpoint(preferedAddress);
		return lastInetAddress;
    }
    
	/**
	 * Get the IP address
	 * @return Detected ip address
	 */
	public InetAddress getAddress(long recheckTime) {
		return getAddress(recheckTime,null);
	}
    
	public void checkpoint() {
		checkpoint(null);
	}
    
    /**
     * Execute a checkpoint - detect our internet IP address and log it
     * @param preferedAddress An address that for some reason is prefered above others. Might be null
     */
    protected synchronized void checkpoint(String preferedAddress) {
	boolean logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
	Vector addrs = new Vector();
	boolean old = false;
	
	Enumeration interfaces = null;
	try {
	    interfaces = java.net.NetworkInterface.getNetworkInterfaces();
	} catch (NoClassDefFoundError e) {
	    addrs.add(oldDetect());
	    old = true;
	} catch (SocketException e) {
	    Core.logger.log(this, "SocketException trying to detect NetworkInterfaces", e,
			    Core.logger.ERROR);
	    addrs.add(oldDetect());
	    old = true;
	}
	
	if(!old) {
	    while(interfaces.hasMoreElements()) {
		java.net.NetworkInterface iface = 
		    (java.net.NetworkInterface)(interfaces.nextElement());
		if(logDEBUG)
		    Core.logger.log(this, "Scanning NetworkInterface "+
				    iface.getDisplayName(), Core.logger.DEBUG);
		Enumeration ee = iface.getInetAddresses();
		while(ee.hasMoreElements()) {
			
		    InetAddress addr = (InetAddress)(ee.nextElement());
		    addrs.add(addr);
		    if(logDEBUG)
			Core.logger.log(this, "Adding address "+addr+" from "+
					iface.getDisplayName(), Core.logger.DEBUG);
		}
		if(logDEBUG) Core.logger.log(this, "Finished scanning interface "+
					     iface.getDisplayName(), Logger.DEBUG);
	    }
	    if(logDEBUG) Core.logger.log(this, "Finished scanning interfaces",
					 Logger.DEBUG);
	}
	
	InetAddress preferedInetAddress=null;
	if(preferedAddress == null && lastInetAddress!=null?isInternetAddress(lastInetAddress):true) //If no specific other address is preferred then we prefer to keep our old address
		preferedInetAddress = lastInetAddress;
	else
		try {
			preferedInetAddress = InetAddress.getByName(preferedAddress); //It there was something preferred then convert it to a proper class 
		} catch (UnknownHostException e) {}
	
	InetAddress oldAddress = lastInetAddress;	
	onGetAddresses(addrs,preferedInetAddress);
	lastDetectedTime = System.currentTimeMillis();
	if(oldAddress != null && !lastInetAddress.equals(oldAddress)) {
	    Core.logger.log(this, "Public IP Address changed from "+
			    oldAddress.getHostAddress() +" to "+
			    lastInetAddress.getHostAddress(), 
			    Core.logger.MINOR);
	    Main.newInetAddress(lastInetAddress);
		// We know it changed
	}
    }
    
    protected InetAddress oldDetect() {
	boolean shouldLog = Core.logger.shouldLog(Logger.DEBUG);
	if(shouldLog) Core.logger.log(this, "Running old style detection code", 
				      Logger.DEBUG);
	DatagramSocket ds = null;
	try {
	    try {
		ds = new DatagramSocket();
	    } catch (SocketException e) {
		Core.logger.log(this, "SocketException", e,
				Core.logger.ERROR);
		return null;
	    }
	    
	    // This does not transfer any data
	    // The ip is a.root-servers.net, 42 is DNS
	    try {
		ds.connect(InetAddress.getByName("198.41.0.4"), 42);
	    } catch (UnknownHostException ex) {
		Core.logger.log(this, "UnknownHostException", ex,
				Core.logger.ERROR);
		return null;
	    }
	    return ds.getLocalAddress();
	} finally {
	    if(ds != null) {
		ds.close();
	    }
	}
    }
    
    /** Do something with the list of detected IP addresses.
     * @param v Vector of InetAddresses
     * @param preferedAddress An address that for some reason is prefered above others. Might be null
     */
    protected void onGetAddresses(Vector v,InetAddress preferedInetAddress) {
	if(Core.logger.shouldLog(Logger.DEBUG))
	    Core.logger.log(this, "onGetAddresses found "+v.size()+" potential addresses)", 
			    Logger.DEBUG);
	boolean detectedInetAddress = false;
	if(v.size() == 0) {
	    Core.logger.log(this, "No addresses found!", Core.logger.ERROR);
		lastInetAddress = null;
	} else {
            InetAddress lastNonValidAddress = null;
	    for(int x=0;x<v.size();x++) {
			if(v.elementAt(x) != null) {
				InetAddress i = (InetAddress)(v.elementAt(x));
		    	if(Core.logger.shouldLog(Core.logger.DEBUG))
					Core.logger.log(this, "Address "+x+": "+i,Core.logger.DEBUG);
		    	if(isInternetAddress(i)) { //Do not even consider this address if it isn't globally addressable
					if(Core.logger.shouldLog(Core.logger.DEBUG))
			    		Core.logger.log(this, "Setting default address to "+
						i.getHostAddress(),Core.logger.DEBUG);

					lastInetAddress = i; //Use the last detected valid IP as 'detected' IP
					detectedInetAddress = true;
					if(preferedInetAddress != null && lastInetAddress.equals(preferedInetAddress)){ //Prefer the specified address if it is still available to us. Do not look for more ones
						if(Core.logger.shouldLog(Core.logger.DEBUG))
							Core.logger.log(this, "Detected address is the preferred address, setting final address to "+
											lastInetAddress.getHostAddress(),Core.logger.DEBUG);
						return;
					}
					
			    }else
					lastNonValidAddress = i;
			}
	    }
	    //If we are here we didn't manage to find a valid globally addressable IP. Do the best of the situation, return the last valid non-addressable IP
		//This address will be used by the node if the user has configured localIsOK.
		if(lastInetAddress == null || (!detectedInetAddress))
			lastInetAddress = lastNonValidAddress; 
	}
	// FIXME: add support for multihoming
    }
    
    protected boolean isInternetAddress(InetAddress addr) {
	return freenet.transport.tcpTransport.checkAddress(addr);
    }
}


