package freenet.node;

import java.io.File;
import java.util.Hashtable;
import java.util.Enumeration;
import freenet.*;
import freenet.config.*;
import freenet.support.*;
import freenet.fs.dir.NativeFSDirectory;
import freenet.node.Main;
import freenet.node.states.maintenance.Checkpoint;

import java.util.Iterator;
import java.util.HashSet;
import java.util.HashMap;
import java.util.StringTokenizer;

/**
 *  Checks config files for updates, and if found, applies what it can
 *
 * @author     Pascal
 * @created    January 4, 2003
 */

public class NodeConfigUpdater implements Checkpointed {
    /**  How often (in minutes) to check for config updates */
    private int updateInterval;
    /**  Configuration we have already seen */
    private static Params oldParams;
    /**  New configuration */
    private Params newParams;
    /**  config file(s) to monitor */
    private Hashtable configFiles = new Hashtable();
    /**  ConfigUpdateListeners registered: map of path to HashSet of listeners */
    private static HashMap listeners = new HashMap();
    
    /**
     *  Gets the checkpointName attribute of the NodeConfigUpdater object
     *
     *@return    The checkpointName value
     */
    public String getCheckpointName() {
        return "On-the-fly configuration updater";
    }

    /**
     *  Determines when the next NodeConfigUpdater checkpoint should run
     *
     *@return    Time the next NodeConfigUpdater checkpoint should be run
     */
    public long nextCheckpoint() {
        if (updateInterval == 0 ) return -1;
        return System.currentTimeMillis() + updateInterval*60000;
    }

    /**
     *  Initializes on-the-fly configuration updater functionality
     *
     *@param  updateInterval  How often to check for config updates
     */
    public NodeConfigUpdater(int updateInterval) throws Throwable {
        this.updateInterval = updateInterval;
        oldParams = new Params(Node.config.getOptions());
        if (Main.paramFile != null) {
            oldParams.readParams(Main.paramFile);
            File paramFile = new File(Main.paramFile);
            configFiles.put(paramFile, new Long(paramFile.lastModified()));
        } else for (int i = 0 ; i < Main.defaultRCfiles.length ; i++)
            try {
                oldParams.readParams(Main.defaultRCfiles[i]);
                File paramFile = new File(Main.defaultRCfiles[i]);
                configFiles.put(paramFile, new Long(paramFile.lastModified()));
            } catch (Throwable e) {}
        if (configFiles.isEmpty()) throw new Throwable();
	fireUpdates(oldParams);
    }
    
    /**
     *  Periodically check to see if the configuration file has been updated.
     *  If it has, read in the new config, compare to the old, and update
     *  as much of the active config as it knows how.
     */
    public void checkpoint() {
        boolean updated = false;
        for(Enumeration en = configFiles.keys(); en.hasMoreElements();) {
            File paramfile = (File) en.nextElement();
            long modified = paramfile.lastModified();
            if (modified != ((Long)configFiles.get(paramfile)).longValue()) {
                updated = true;
                configFiles.put(paramfile, new Long(modified));
            }
        }
        if (updated == false) return;
        newParams = new Params(Node.config.getOptions());
        for(Enumeration en = configFiles.keys(); en.hasMoreElements();)
            try { newParams.readParams(((File)en.nextElement()).getName());
            } catch (Throwable e) {}
        
        fireUpdates(newParams);
        
        Option[] newOptions = newParams.getOptions();
        ConfigOptions options = new ConfigOptions();
        for (int i = 0 ; i < newOptions.length ; i++) {
            if ((oldParams.getParam(newOptions[i].name()) == null) && (newParams.getParam(newOptions[i].name()) == null)) continue;
            if (((oldParams.getParam(newOptions[i].name()) == null) ^ (newParams.getParam(newOptions[i].name()) == null)) || !oldParams.getParam(newOptions[i].name()).equalsIgnoreCase(newParams.getParam(newOptions[i].name()))) try {
                options.getClass().getMethod(newOptions[i].name(), null).invoke(options, null);
            } catch (Throwable e) {
                Core.logger.log(NodeConfigUpdater.class, "Option " + newOptions[i].name() + " changed to " + newParams.getParam(newOptions[i].name()) + " but no handler was available.", e, Core.logger.ERROR);
            }
        }
        oldParams = newParams;
    }
    
    public static void addUpdateListener(String path, ConfigUpdateListener listener) {
	if(Core.logger.shouldLog(Logger.DEBUG))
	    Core.logger.log(NodeConfigUpdater.class, "registering path [" + path + 
			    "] listener [" + listener.getClass().getName() + "]", 
			    Logger.DEBUG);
        if (!listeners.containsKey(path))
            listeners.put(path, new HashSet());
        HashSet lsnrs = (HashSet)listeners.get(path);
        lsnrs.add(listener);
        fireInitialUpdate(path, listener);
    }
    public static void removeUpdateListener(ConfigUpdateListener listener) {
        for (Iterator iter = listeners.values().iterator(); iter.hasNext(); ) {
            HashSet listeners = (HashSet)iter.next();
            listeners.remove(listener);
        }
    }
    
    private static void fireInitialUpdate(String path, ConfigUpdateListener listener) {
	boolean logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
	if(logDEBUG) Core.logger.log(NodeConfigUpdater.class, 
				     "Firing initial update for "+path+", "+listener,
				     Logger.DEBUG);
        StringTokenizer tok = new StringTokenizer(path, ".");
        String val = null;
        Params fs = oldParams;
	if(fs == null && logDEBUG)
	    Core.logger.log(NodeConfigUpdater.class, "fs == null !",
			    new Exception("grrr"), Logger.DEBUG);
        while ( (fs != null) && (tok.hasMoreTokens()) ) {
            String tpath = tok.nextToken();
	    if(logDEBUG) Core.logger.log(NodeConfigUpdater.class, "In while(), tok = "+
					 tpath, Logger.DEBUG);
            if (fs.isSet(tpath))
                fs = (Params)fs.getSet(tpath);
            else {
                val = fs.get(tpath);
                break;
            }
        }
	if(logDEBUG) Core.logger.log(NodeConfigUpdater.class, "Out of while()", 
				     Logger.DEBUG);
        if (fs != null) {
	    if(logDEBUG) Core.logger.log(NodeConfigUpdater.class, "fs != null", 
					 Logger.DEBUG);
            if (val == null)
                listener.configPropertyUpdated(path, fs);
            else
                listener.configPropertyUpdated(path, val);
        }
    }
    
    private static void fireUpdates(Params params) {
	boolean logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
	if(logDEBUG) Core.logger.log(NodeConfigUpdater.class, 
				     "Firing configuration updates with params " + 
				     params.toString(), Logger.DEBUG);
        for (Iterator iter = listeners.keySet().iterator(); iter.hasNext();) {
            String path = (String)iter.next();
            HashSet lsnrs = (HashSet)listeners.get(path);
	    if(logDEBUG) Core.logger.log(NodeConfigUpdater.class, 
					 "Firing configuration updates for path " + 
					 path + " w/ " + lsnrs.size() + " listeners", 
					 Logger.DEBUG);
            StringTokenizer tok = new StringTokenizer(path, ".");
            String val = null;
            Params fs = params;
            while ( (fs != null) && (tok.hasMoreTokens()) ) {
                String tpath = tok.nextToken();
                if (fs.isSet(tpath)) 
                    fs = (Params)fs.getSet(tpath);
                else {
                    val = fs.get(tpath);
                    break;
                }
            }
            if (fs != null) {
                if (val == null) {
                    for (Iterator fireIter = lsnrs.iterator(); fireIter.hasNext(); ) {
                        ConfigUpdateListener lsnr = (ConfigUpdateListener)fireIter.next();
                        lsnr.configPropertyUpdated(path, fs);
                    }
                } else {
                    for (Iterator fireIter = lsnrs.iterator(); fireIter.hasNext(); ) {
                        ConfigUpdateListener lsnr = (ConfigUpdateListener)fireIter.next();
                        lsnr.configPropertyUpdated(path, val);
                    }
                }
            }
        }
    }

    public class ConfigOptions {

        /**
         *  Each config option that can be updated on-the-fly needs its own
         *  public method.
         */

        public void configUpdateInterval() {
            int interval = newParams.getInt("configUpdateInterval");
            if (updateInterval == interval) return;
            updateInterval = interval;
            if (interval == 0)
                Core.logger.log(NodeConfigUpdater.class, "Disabled on-the-fly config updater.", Core.logger.NORMAL);
            else Core.logger.log(NodeConfigUpdater.class, "Changed interval to check for configuration updates to " + interval + " minutes.", Core.logger.NORMAL);
        }

        public void logLevel() {
            String logLevel = newParams.getString("logLevel");
            int thresh = Logger.priorityOf(logLevel);
            Core.logger.setThreshold(thresh);
            LoggerHook[] hooks = Core.logger.getHooks();
            for (int i = 0 ; i < hooks.length ; i++)
                hooks[i].setThreshold(thresh);
            if (Main.node.dir instanceof NativeFSDirectory)
                ((NativeFSDirectory)Main.node.dir).logDEBUG = 
		    Core.logger.shouldLog(Logger.DEBUG);
            Core.logger.log(NodeConfigUpdater.class, "Changed logging level to " + 
			    logLevel + ".", Logger.NORMAL);
        }

        public void aggressiveGC() {
            int interval = newParams.getInt("aggressiveGC");
            if (Main.node.aggressiveGC == interval) return;
            if ((Main.node.aggressiveGC <= 0) && (interval != 0)) {
                Main.node.aggressiveGC = interval;
                new Checkpoint(new Main.GarbageCollectionCheckpointed()).schedule(Main.node);
                Core.logger.log(NodeConfigUpdater.class, "Enabled aggressive garbage collection with a " + interval + " second interval.", Core.logger.NORMAL);
            } else {
                Main.node.aggressiveGC = interval;
                if (interval == 0)
                    Core.logger.log(NodeConfigUpdater.class, "Disabled aggressive garbage collection.", Core.logger.NORMAL);
                else Core.logger.log(NodeConfigUpdater.class, "Changed aggressive garbage collection interval to " + interval + " seconds.", Core.logger.NORMAL);
            }
        }
    }
}

