package freenet.node.states.announcing;

import freenet.*;
import freenet.node.*;
import freenet.node.rt.Routing;
import freenet.support.Fields;
import freenet.support.Logger;
import freenet.diagnostics.Diagnostics;

import java.util.Hashtable;

/**
 * This is the envelope state around the sub-chain to each target node.
 */

public class Announcing extends AggregatingState {


    static long startTime = -1;
    /**
     * Places the Announcing state in the nodes state table, where it will
     * live.
     * @param n               The node to announce.
     * @param hopsToLive      HopsToLive to use on each message
     * @param targets         Seed nodes to announce to.
     * @param useRT           Should we fall back on the nodes routing
     *                        table if we cannot announce to the seeds?
     * @param delay           How often should we wake up and check that the
     *                        node has traffic?
     */
    public static void placeAnnouncing(Node n, int hopsToLive, 
                                       NodeReference[] targets,
                                       boolean useRT, int delay) {

	startTime = System.currentTimeMillis();
	
        if (targets == null)
            targets = new NodeReference[0];
        Core.logger.log(Announcing.class, 
                        "Starting announce background task. HTL: " +
                        hopsToLive + " Number of targets: " + targets.length
                        + " Use routing table: " + useRT + " Poll Interval: " 
                        + delay + " Threads: " + Node.announcementThreads +
                        " Attempts: " + Node.announcementAttempts, 
                        Core.logger.MINOR);
        n.schedule(0, new PlaceAnnouncing(n.randSource.nextLong(),
                                          hopsToLive, targets, useRT, delay));
    }
    
    private static class PlaceAnnouncing extends EventMessageObject {
        private boolean useRT;
        private int hopsToLive; 
        private NodeReference[] targets;
        private long delay;
        
        public PlaceAnnouncing(long id, int hopsToLive, 
                               NodeReference[] targets, boolean useRT,
                               long delay) {
            super(id, true);
            this.useRT = useRT;
            this.hopsToLive = hopsToLive;
            this.targets = targets;
            this.delay = delay;
        }

        public State getInitialState() {
            return new Announcing(id, hopsToLive, targets, useRT, delay);
        }

        public String toString() {
            return "Initiate announcement procedure.";
        }
    }

    private static class ScheduleAnnouncing extends EventMessageObject {
        public ScheduleAnnouncing(long id) { 
            super(id, true);
        }

        public String toString() {
            return "Wake announcement procedure if there is no traffic.";
        }
    }

    private static class Announced {
        public Identity peer;
        public int times;
        public int successful;

        public Announced(Identity peer) {
            this.peer = peer;
        }

        public void success() {
            times++;
            successful++;
        }

        public void failed() {
            times++;
        }
    }

    private long delay;
    private int hopsToLive;
    private int origHopsToLive;

    private NodeReference[] targets;
    private int nextTarget;
    private boolean useRT;

    private Hashtable attempted;
    private int totalAttempts;
    private volatile int successes;

    private Announcing(long id, int hopsToLive, NodeReference[] targets,
                       boolean useRT, long delay) {
        super(id, 0);

        this.hopsToLive = hopsToLive;
	this.origHopsToLive = hopsToLive;
        this.targets = targets;
        this.useRT = useRT;
        this.delay = delay;
        nextTarget = 0;

        attempted = new Hashtable();
        totalAttempts = 0;
        successes = 0;
    } 

    public String getName() {
        return "Announcing Chains Aggregate";
    }

    long timeLastAnnounced = -1;

    public State received(Node n, MessageObject mo) throws StateException {
        if (mo instanceof PlaceAnnouncing) {
            n.schedule(delay, new ScheduleAnnouncing(id));
        } else if (mo instanceof Completed) {
            Completed c = (Completed) mo;
            Announced a = (Announced) attempted.get(c.peer);
            if (a == null) {
                a = new Announced(c.peer);
                attempted.put(c.peer, a);
            }
            totalAttempts++;
            if (c.successful) {
                Core.logger.log(this, "Announced node successfully to " +
                                c.peer.fingerprintToString() + " at depth  " +
                                c.htl + ".", Logger.NORMAL);
                a.success();
                successes++;
		int x = hopsToLive++;
		int y = (hopsToLive*5)/4;
		if(y>x) x = y;
		if(x > origHopsToLive) x = origHopsToLive;
		origHopsToLive = x;
	    } else {
		String s = c.reasonString;
                Core.logger.log(this, "Announcement failed to " +
                                c.peer.fingerprintToString() + " at depth  " +
                                c.htl + ((c == null) ? "" : (":"+s)),
				Logger.NORMAL);
		hopsToLive--;
		if(hopsToLive < 2) hopsToLive = 2;
                a.failed();
                if (c.terminal)
                    a.times = Node.announcementAttempts;
		// Only successful announcements reset timeLastAnnounced
            }
        } else if (mo instanceof ScheduleAnnouncing) {
            try {
                double traffic = n.diagnostics.getValue("localQueryTraffic", 
                                                        Diagnostics.HOUR, 
                                                Diagnostics.NUMBER_OF_EVENTS);

                double connections = n.diagnostics.getValue("connectingTime",
							    Diagnostics.HOUR,
							    Diagnostics.NUMBER_OF_EVENTS);

		Core.logger.log(this, "Traffic: "+traffic+", Connections: "+
				connections, Core.logger.DEBUG);

                if (traffic < 1.0 || connections < 1.0 || 
		    (successes > 0 && successes < Node.announcementThreads) ||
		    ( (n.loadStats.globalQueryTraffic()*
		       n.defaultResetProbability > 
		       n.loadStats.localQueryTraffic()) &&
		      (System.currentTimeMillis() - timeLastAnnounced > 
		       3600*1000) &&
		      (System.currentTimeMillis() - startTime > delay*2))) {
		    // FIXME: this only ensures that we have 3 successes _THE FIRST TIME_
		    if(Core.logger.shouldLog(Core.logger.DEBUG))
			Core.logger.log(this, "Announcing", Core.logger.DEBUG);
		    
                    // find three peers.
		    timeLastAnnounced = System.currentTimeMillis();
		    
                    int m = 0;
		    int x = Node.announcementThreads - successes;
		    if(x < 1) x = 1;
                    NodeReference[] now = 
                        new NodeReference[x];
		    
                    for (int i = 0 ; (i < targets.length && 
                                      m < now.length) ; i++) {
                        NodeReference nr = targets[nextTarget];
                        nextTarget = (nextTarget + 1) % targets.length;
			// Very weak reseeding with the nodes we are to announce to in case the node is _really_ fucked up
			if(connections < 1.0)
			    Main.seedRoutingTable(n.rt, new NodeReference[] 
						  {nr}, false);
			
                        Announced a = 
                            (Announced) attempted.get(nr.getIdentity());
                        if (a == null || a.times < Node.announcementAttempts) {
                            now[m] = nr;
                            m++;
                        }
                    }

                    if (m < now.length && useRT) {
                        byte[] b = new byte[8];
                        Core.randSource.nextBytes(b);
                        Routing routes = n.rt.route(new Key(b), true);
                        while (m < now.length) {
                            NodeReference nr = routes.getNextRoute();
                            if (nr == null) {
                                break;
                            } else {
                                for (int i = 0 ; i < m ; i++) {
                                    if (nr.getIdentity().equals(now[i]))
                                        continue;
                                }
				
				if(connections < 1.0)
				    Main.seedRoutingTable(n.rt, 
							  new NodeReference[] 
							  {nr}, false);
				
                                Announced a = 
                                   (Announced) attempted.get(nr.getIdentity());
                                if (a == null || 
                                    a.times < Node.announcementAttempts) {

                                    now[m] = nr;
                                    m++;
                                }
                            }
                        }
                    }

                    Core.logger.log(this, "Found " + m + 
                                    " announcement targets for this node.",
                                    Logger.NORMAL);
                    for (int i = 0 ; i < m ; i++) {
                        SendAnnouncement.makeTry(n, id, n.getPeer(now[i]), 
                                                 hopsToLive);
                    }
                } else {
                    Core.logger.log(this, traffic + 
                                    " requests in the last hour. " + 
                                    "Won't announce.", Core.logger.MINOR);
		    successes = 0;
                }
            } finally {
                n.schedule(delay, new ScheduleAnnouncing(id));
            }
        } else
            super.received(n, mo);

        return this; // announcing always stays!
    }
    
    public int priority() {
        return CRITICAL;
    }
}



