package freenet.node.states.data;

import freenet.*;
import freenet.fs.dir.BufferException;
import freenet.node.*;
import freenet.node.ds.*;
import freenet.support.*;
import freenet.support.io.*;
import java.io.*;


/**
 * Sends data from the store. If the reading from the store fails, then
 * the sent data will be padded until the end of the next part, where
 * CB_RESTARTED will be written as the control byte. When the send is finished
 * (for better or worse) a DataSent message will be returned to the parent
 * chain indicating the state with a control byte.
 *
 * ^^^^^^  This is wrong. No padding.
 *
 * @author oskar 
 */

public class SendData extends DataState {

    private final OutputStream send;
    private final KeyInputStream in;
    private boolean closedSend = false, closedIn = false;
    private final long length, partSize;
    private Exception abortedException = null;
    private volatile int result = -1;
    private boolean silent = false;
    
    
    public SendData(long id, long parent, OutputStream send,
                    KeyInputStream in, long length, long partSize) {
        super(id, parent);
        this.send     = send;
        this.in       = in;
        this.length   = length;
        this.partSize = partSize;
    }

    public final String getName() {
        return "Sending Data";
    }

    /** If sending upstream, you want CB_ABORTED.
      * If sending downstream, you want CB_RESTARTED.
      */
    public final void abort(int cb) {
        silent = true;
        result = cb;
	if(Core.logger.shouldLog(Logger.DEBUG)) {
	    abortedException = new Exception("debug");
	    Core.logger.log(this, "Aborted send for "+
			    Long.toHexString(parent)+" with cb="+
			    Integer.toHexString(cb), abortedException,
			    Logger.DEBUG);
	}
    }
    
    public final int result() {
        return result;
    }
    
    public void finalize() {
	try {
	    if(!closedIn) in.close();
	} catch (IOException e) {};
	try {
	    if(!closedSend) send.close();
	} catch (IOException e) {};
    }

    /** Sheesh!  We're too overworked to even try to write CB_ABORTED.
      */
    public final void lost(Node n) {
        try {
            in.close();
	    closedIn = true;
        }
        catch (IOException e) {
            n.logger.log(this, "I/O error closing KeyInputStream",
                         e, Logger.ERROR);
        }
        try {
            send.close();
	    closedSend = true;
        }
        catch (IOException e) {
            n.logger.log(this, "I/O error closing data send stream",
                         e, Logger.MINOR);
        }
    }

    boolean inWrite = false;
    long moved = 0;
    byte[] buffer = null;
    int m = 0;

    public State received(Node n, MessageObject mo) throws BadStateException {
        if (!(mo instanceof DataStateInitiator))
            throw new BadStateException("expecting DataStateInitiator");

        // if there is an IOException, this says
        // whether it was while writing to the store
        inWrite = false;
        moved = 0;
	m = 0;
	
	boolean logDEBUG = n.logger.shouldLog(Logger.DEBUG);
	
        try {
            buffer = new byte[Core.blockSize];
            while (moved < length) {
                inWrite = false;
                if (result != -1) throw new CancelledIOException();
                int m = in.read(buffer, 0, (int) Math.min(length - moved, buffer.length));
                if (m == -1) {
		    throw new IOException("Stopped short of full transfer");
		}
		inWrite = true;
		if(logDEBUG)
		    n.logger.log(this, "Read "+(moved+m)+" of "+length+
				 " bytes for "+Long.toHexString(parent)+" ("+in+")",
				 Logger.DEBUG);
                send.write(buffer, 0, m);
                moved += m;
		if(logDEBUG)
		    n.logger.log(this, "Moved "+moved+" of "+length+" bytes for "+
				 Long.toHexString(parent)+" ("+in+")",
				 Logger.DEBUG);
            }
            send.close();
	    closedSend = true;
            result = Presentation.CB_OK;
        }
        catch (CancelledIOException e) {
	    if(logDEBUG)
		n.logger.log(this, "Cancelled IO: "+abortedException+" for "+
			     Long.toHexString(parent), abortedException, 
			     Logger.DEBUG);
	}  // result already set
        //catch (BadDataException e) {
        //    result = e.getFailureCode();
        //}
        catch (IOException e) {
            // if it was aborted we can expect the aborter to set the failure
            // code.
            result = (inWrite ? Presentation.CB_SEND_CONN_DIED
                      : in.getFailureCode());
            if (result == -1) // it broke some time between writing and reading
		{
		    if(logDEBUG) n.logger.log(this, "Cache failed between writing "+
					      "and reading for "+Long.toHexString(id),
					      e, Logger.DEBUG);
		    result = Presentation.CB_CACHE_FAILED;
		}
            if (result == Presentation.CB_CACHE_FAILED) {
                Core.logger.log(this, 
                                "Cache failed signalled after exception " +
                                "after " + moved + " of " + length 
                                + " bytes: "+e+" for "+Long.toHexString(id)+
				" ("+Long.toHexString(parent)+".", e ,
				Logger.ERROR);
            }
        } finally {
	    
            n.diagnostics.occurrenceBinomial("sentData", 1, 
                                             result == Presentation.CB_OK ?
                                             1 : 0);
	    
	    if(result != Presentation.CB_OK) {
		if(abortedException != null) {
		    Core.logger.log(this, "Send aborted for "+Long.toHexString(id)+
				    " ("+Long.toHexString(parent)+" - result="+
				    Long.toHexString(result),
				    abortedException, Logger.MINOR);
		    abortedException = null;
		} else {
		    Core.logger.log(this, "Send failed for "+Long.toHexString(id)+
				    " ("+Long.toHexString(parent)+" - result="+
				    Long.toHexString(result), Logger.MINOR);
		}
	    }
	    
	    buffer = null;

            try {
                in.close();
		closedIn = true;
            }
            catch (IOException e) {
                n.logger.log(this, "I/O error closing KeyInputStream",
                             e, Logger.ERROR);
            }

            if (moved < length && !inWrite) {
                try {
                    // pad until end of part
                    long tmpLen = partSize + Key.getControlLength();
                    tmpLen = Math.min(tmpLen - moved % tmpLen, length - moved) - 1;
                    
                    byte[] zeroes = new byte[Core.blockSize];
                    while (tmpLen > 0) {
                        int m = (int) Math.min(tmpLen, zeroes.length);
                        send.write(zeroes, 0, m);
                        tmpLen -= m;
                    }
                    send.write(result == Presentation.CB_ABORTED ? Presentation.CB_ABORTED
                                                                 : Presentation.CB_RESTARTED);
                }
                catch (IOException e) {
                    // this is important b/c it lets the parent chain know that
                    // it shouldn't try to get back in touch with the upstream
                    result = Presentation.CB_SEND_CONN_DIED;
                }
            }
            
            if (result != Presentation.CB_OK) {
                try {
                    send.close();
		    closedSend = true;
                }
                catch (IOException e) {
                    n.logger.log(this, "I/O error closing data send stream",
                                 e, Logger.MINOR);
                }
            }
            
            // we could be exiting with an uncaught exception or something..
            if (result == -1) result = Presentation.CB_SEND_CONN_DIED;
            
            // had to wait around to see if we'd get a CB_SEND_CONN_DIED
            // when padding the write
            if (!silent) n.schedule(new DataSent(this));
        }
        
        return null;
    }
}

