package freenet.node.states.request;

import freenet.*;
import freenet.fs.dir.BufferException;
import freenet.node.*;
import freenet.node.ds.*;
import freenet.message.*;
import freenet.support.io.*;
import freenet.node.states.data.*;
import freenet.support.Logger;
import java.io.*;

/**
 * The state that an insert is in when newly begun, before the DataInsert
 * has been received.
 */
public class InsertPending extends Pending {

    NoInsert ni;
    DataInsert dim;    // cache it for sending upstream
    
    long dimRecvTime,  // when we got the DataInsert
         checkTime;    // when we got the Accepted/InsertReply
    
    boolean approved = false;  // got an InsertReply?
    
    
    /**
     * For new requests.
     */
    public InsertPending(long id, int htl, Key key, Peer orig,
                         FeedbackToken ft, RequestInitiator ri) {
        super(id, htl, key, orig, ft, ri);
    }

    /**
     * For restarts from InsertPending.
     */
    InsertPending(InsertPending ancestor) {
        super(ancestor);
        ni = ancestor.ni;
    }


    public String getName() {
        return "InsertRequest Pending";
    }

    final Request createRequest(FieldSet otherFields, NodeReference ref) {
        return new InsertRequest(id, hopsToLive, searchKey, ref, otherFields);
    }

    
    //=== message handling =====================================================
    
    // these two cause the state to be recreated
    
    public State receivedMessage(Node n, QueryRejected qr) throws StateException {
        try {
            super.receivedQueryRejected(n, qr);
        }
        catch (EndOfRouteException e) {
            return endRoute(n);
        }
        catch (RequestAbortException rae) {
            cancelNoInsert();
            // we are either going to SendingReply or RequestDone with no route found
            return rae.state;
        }
        return new InsertPending(this);
    }

    public State receivedMessage(Node n, RequestInitiator ri) throws StateException {
	if (this.ri == null || ri != this.ri) {
	    throw new BadStateException("Not my request initiator: "+ri+" for "+this);
	}
        if (ni == null) {
            // we give them 7 hop times to produce the DataInsert
            ni = new NoInsert(this);
            n.schedule(Core.hopTime(7), ni);
        }
        try {
            super.receivedRequestInitiator(n, ri);
        }
        catch (EndOfRouteException e) {
            return endRoute(n);
        }
        catch (RequestAbortException rae) {
            cancelNoInsert();
            // this is going to RequestDone with no route found
            return rae.state;
        }
        return new InsertPending(this);
    }

    // the rest don't

    public State receivedMessage(Node n, InsertRequest ir) {
        super.receivedRequest(n, ir);
        return this;
    }

    public State receivedMessage(Node n, QueryRestarted qr) throws StateException {
        try {
            super.receivedQueryRestarted(n, qr);
        }
        catch (RequestAbortException rae) {
            cancelNoInsert();
            // going to RequestDone with SendFailedException
            return rae.state;
        }
        return this;
    }

    public State receivedMessage(Node n, NoInsert ni) throws StateException {
        if (this.ni != ni) {
            throw new BadStateException("Not my NoInsert: "+ni+" for "+this);
        }
        fail(n, "DataInsert never received");
        queryAborted(n);
        return new RequestDone(this);
    }

    public State receivedMessage(Node n, QueryAborted qf) throws StateException {
        if (!fromOrigPeer(qf)) {
            throw new BadStateException("QueryAborted from the wrong peer! for "+
					this);
        }
        cancelNoInsert();
        queryAborted(n, qf);

        return new RequestDone(this);
    }

    public State receivedMessage(Node n, DataInsert dim) throws StateException {
        if (!fromOrigPeer(dim)) {
            throw new BadStateException("DataInsert from the wrong peer! for "+this);
        }
        dimRecvTime = System.currentTimeMillis();

        cancelNoInsert();

        this.dim = dim;
        
        try {
            receivingData = dim.cacheData(n, searchKey);
        }
        catch (KeyCollisionException e) {
            // we've already sent the Accepted so we should try..
            dim.eatData(n);
            // propagate QueryAborted upstream in case they
            // are waiting on DataInsert
            queryAborted(n);
            
            try {
                searchData(n);
            }
            catch (RequestAbortException rae) {
                return rae.state;
            }

            // well damn, it's gone already
            fail(n, "I/O error receiving insert");
            n.logger.log(this, "Failed to find data after key collision for "+this,
                         Logger.NORMAL);
	    
            return new RequestDone(this);
        }
        catch (IOException e) {
            dim.drop(n);
            fail(n, "I/O error receiving insert");
            n.logger.log(this, "Failed to cache insert for " + this,
			 e, Logger.ERROR);
            queryAborted(n);
            return new RequestDone(this);
        }
        
        try {
            if (accepted) {  // got the Accepted, or InsertReply
		if(logDEBUG) n.logger.log(this, "Got DataInsert " + 
					  ((dimRecvTime - checkTime) / 1000)+
					  "s after " + 
					  (approved ? "InsertReply" : "Accepted")+
					  " for "+this, Logger.DEBUG);
		
                relayInsert(n);
                return new TransferInsert(this);
            }
            else {
                KeyInputStream doc = receivingData.getKeyInputStream();
                doc.setParent(id, n.ticker().getMessageHandler(),
                              "Set to TransferInsertPending for "+
			      searchKey);
                return new TransferInsertPending(this, doc);
            }
        }
        catch (RequestAbortException e) {
            // immediate restart, or failure
            return e.state;
        }
        catch (IOException e) {
            receivingData.cancel();
            fail(n, "I/O error receiving insert");
            if (e instanceof BufferException)
                n.logger.log(this, "Failed to cache insert: "+e+" for "+this,
			     Logger.MINOR);
            else
                n.logger.log(this, "Failed to cache insert for "+this,
			     e, Logger.ERROR);
            queryAborted(n);
            return new RequestDone(this);
        }
        finally {
            receivingData.schedule(n);
        }
    }

    public State receivedMessage(Node n, DataReply dr) throws StateException {
        State ret = super.receivedDataReply(n, dr);
        if (this != ret) cancelNoInsert();
        return ret;
    }

    public State receivedMessage(Node n, Accepted a) throws StateException {
        if (!accepted)
            checkTime = System.currentTimeMillis();
        super.receivedAccepted(n, a);
        return this;
    }

    public State receivedMessage(Node n, InsertReply ir) throws StateException {
        if (!fromLastPeer(ir)) {
            throw new BadStateException("InsertReply from wrong peer!");
        }
        if (!approved) {
            if (routedTime > 0) 
                Core.diagnostics.occurrenceContinuous("hopTime",
                                                  (System.currentTimeMillis() -
                                                   routedTime) / hopsToLive);

            checkTime = System.currentTimeMillis();
            cancelRestart();
            accepted = true;  // essentially..
            approved = true;
            routes.routeAccepted();
            try {
                insertReply(n);
            }
            catch (RequestAbortException rae) {
                cancelNoInsert();
                // send failed to iriginator..
                return rae.state;
            }
        }
        return this;
    }


    //=== support methods ======================================================

    final void cancelNoInsert() {
        if (ni != null) {
            ni.cancel();
            ni = null;
        }
    }

    private final State endRoute(Node n) {
        cancelRestart();
	logFailure(n);
        try {
            insertReply(n);
        }
        catch (RequestAbortException rae) {
            cancelNoInsert();
            return rae.state;
        }
        return new AwaitingInsert(this);
    }

    protected State publicEndRoute(Node n) {
	return endRoute(n);
    } // FIXME!
    
    final void queryAborted(Node n) {
	queryAborted(n, new QueryAborted(id));
    }
    
    final void queryAborted(Node n, QueryAborted qf) {
        if (lastPeer != null) {
            try {
                n.makeConnection(lastPeer).sendMessage(qf);
            } catch (CommunicationException e) {
                n.logger.log(this, 
			     "Failed to send QueryAborted upstream for "+
			     this, e, Logger.MINOR);
            }
        }
    }

    final void insertReply(Node n) throws RequestAbortException {
        try {
            ft.insertReply(n,
                Core.storeDataTime(hopsToLive,
                                   searchKey.getExpectedTransmissionLength()));
        }
        catch (CommunicationException e) {
            if (receivingData != null) receivingData.cancel();
            queryAborted(n);
            n.logger.log(this,
			 "Failed to send back InsertReply, dropping for "+
			 this, e, Logger.MINOR);
            throw new RequestAbortException(new RequestDone(this));
        }
    }
    
    /** Schedules the SendData state to relay the data upstream
      * @throws RequestAbortException  if there is a failure
      */
    void relayInsert(Node n) throws RequestAbortException {
        try {
            KeyInputStream doc = receivingData.getKeyInputStream();
            doc.setParent(id, n.ticker().getMessageHandler(),
                          "InsertPending.relayInsert");
            try {
                relayInsert(n, doc);
            }
            catch (RequestAbortException e) {
                doc.close();
                throw e;
            }
        }
        catch (IOException e) {
            receivingData.cancel();
            fail(n, "I/O error receiving insert");
            queryAborted(n);
            if (e instanceof BufferException) {
                n.logger.log(this, "Failed to read data from store: "+e+
			     " for "+this, Logger.MINOR);
            }
            else {
                n.logger.log(this, "Failed to read data from store for "+
			     this, e, Logger.ERROR);
            }
            throw new RequestAbortException(new RequestDone(this));
        }
    }

    void relayInsert(Node n, KeyInputStream doc) throws RequestAbortException {
        OutputStream out;
        try {
	    receivedTime = -2; // receivedTime on the next message will be meaningless
	    if(logDEBUG)
		n.logger.log(this, "Opening connection for "+this, Logger.DEBUG);
	    ConnectionHandler ch = n.makeConnection(lastPeer);
	    if(logDEBUG) n.logger.log(this, "Opened connection for "+this+
				      ", sending "+dim+" on "+ch, Logger.DEBUG);
            out = ch.sendMessage(dim);
	    if(logDEBUG)
		n.logger.log(this, "Sent message for "+this+" on "+ch, Logger.DEBUG);
        }
        catch (CommunicationException e) {
            // restart right away
            scheduleRestart(n, 0);
            throw new RequestAbortException(this);
        }
	if(logDEBUG)
	    n.logger.log(this, "Relaying insert pending "+searchKey+
			 " without RAE for "+this, new Exception("debug"), 
			 Logger.DEBUG);
        sendingData = new SendData(n.randSource.nextLong(), this.id, out, doc,
                                   doc.length(), 
				   doc.getStorables().getPartSize());
        sendingData.schedule(n);
	if(logDEBUG) n.logger.log(this, "Scheduled "+sendingData+" for "+this,
				  Logger.DEBUG);
    }
}



