package freenet.node.states.request;

import freenet.*;
import freenet.fs.dir.BufferException;
import freenet.node.*;
import freenet.node.ds.*;
import freenet.message.*;
import freenet.node.states.data.*;
import freenet.support.Fields;
import freenet.support.Logger;
import freenet.support.io.*;
import java.io.*;
import java.util.Enumeration;

/**
 * This is the abstract superclass for states pertaining to Insert and
 * Data Requests that are being processed.
 */
public abstract class Pending extends RequestState {

    /** the auxiliary state that runs the data receive */
    ReceiveData receivingData;
    
    /** the auxiliary state that runs the data send */
    SendData sendingData;

    /** may need to cache it here */
    DataReceived dataReceived;

    /** may need to cache it here */
    DataSent dataSent;
    
    /** may need to cache it here */
    StoreData storeData;

    /** got Accepted yet? */
    boolean accepted = false;

    /** Time marked when routing so that we can register time taken for NGrouting. DO NOT RESET ON QUERYRESTARTED, as it is part of the node routing metric. */
    long routedTime = -1;
    
    /** Time marked when Accepted got. Not sure why we need this but routedTime used to be the same thing */
    long acceptedTime = -1;
    
    /** Marked which messages that cause a routing, so that we can see how 
        long until the message is actually routed */
    long receivedTime = -1;
    
    /** Marked when we got the DataReply or InsertReply - AFTER the Storables verify */
    long replyTime = -1;
    
    /** Time the message entered Pending
     */
    long gotTime = -1;
    
    long gotRouteTime = -1;
    
    long searchDataRoutingTime = -1;
    
    public String toString() {
	return super.toString()+", routedTime="+routedTime;
    }
    
    /** Pending states may be created from scratch.
      */
    Pending(long id, int htl, Key key, Peer orig, FeedbackToken ft, RequestInitiator ri) {
        super(id, htl, key, orig, ft, ri);
	if(logDEBUG)
	    Core.logger.log(this, "Created new Pending from scratch: "+this,
			    new Exception("debug"), Logger.DEBUG);
    }
    
    /** We don't retain the state variables above
      * in a transition back to Pending (a restart).
      * Except for routedTime. We need that for the
      * ngrouting stats.
      */
    Pending(RequestState ancestor) {
        super(ancestor);
	if(ancestor instanceof Pending) {
	    routedTime = ((Pending)ancestor).routedTime;
	    replyTime = ((Pending)ancestor).replyTime;
	}
	if(logDEBUG)
	    Core.logger.log(this, "Created new Pending: "+this+" from "+
			    ancestor, new Exception("debug"), Logger.DEBUG);
    }
    
    /** State implementation.
      */
    public void lost(Node n) {
        Core.diagnostics.occurrenceCounting("lostRequestState", 1);
        fail(n, "Request state lost due to overflow for "+this);
    }
    
    /** @return  A request Message that could be routed upstream
      *          (note that unrecognized fields are lost)
      */
    abstract Request createRequest(FieldSet otherFields, NodeReference ref);

    
    //=== message handling =====================================================

    void receivedStoreData(Node n, StoreData sd) throws BadStateException {
        if (!fromLastPeer(sd)) {
            throw new BadStateException("StoreData from the wrong peer!");
        }
        // just hold on to it, when the Pending state is ready to finish, it
        // can check for it and go to AwaitingStoreData if it's still null
        storeData = sd;
    }
    
    void receivedQueryRestarted(Node n, QueryRestarted qr) throws BadStateException,
                                                                  RequestAbortException {
	acceptedTime = -1; // this time is no longer meaningful
	receivedTime = -2;

        if (!fromLastPeer(qr)) {
            throw new BadStateException("QueryRestarted from the wrong peer!");
        }
        cancelRestart();
        long timeout = Core.hopTime(hopsToLive);
        relayRestarted(n, timeout);
        scheduleRestart(n, timeout);
    }

    /**
     * Note the EndOfRouteException and RequestAbortException.
     * Must be called by subclass as part of the implementation
     * for receivedMessage(Node n, QueryRejected qr)
     */
    void receivedQueryRejected(Node n, QueryRejected qr) throws BadStateException,
                                                                RequestAbortException,
                                                                EndOfRouteException {
        if (!fromLastPeer(qr)) {
            throw new BadStateException("QueryRejected from the wrong peer!");
        }

	gotTime = System.currentTimeMillis();
	if(receivedTime >= -1) {
	    receivedTime = qr.getReceivedTime(); // measure routing time again
	    if(logDEBUG) n.logger.log(this, "Remeasured receivedTime: "+receivedTime,
				      Logger.DEBUG);
	}
	if (receivedTime < 1000*1000*1000*1000) {
	    if(logDEBUG && receivedTime > -1)
		n.logger.log(this, "qr.getReceivedTime() returned time way "+
			     "before the present ("+receivedTime+") for "+
			     this, Logger.DEBUG);
	    receivedTime = 0;
	} else {
	    if(logDEBUG)
		n.logger.log(this, "QueryRejected gotTime-recievedTime: "+
			     (gotTime-receivedTime)+" for "+this,
			     new Exception("debug"), Logger.DEBUG);
	}
	
	logFailure(n, gotTime);
	
        ++rejected;

        // reduce our HTL like the man sez, but by at least 1
        hopsToLive = (int) Math.min(hopsToLive-1, qr.hopsToLive);
        
	if(logDEBUG)
	    n.logger.log(this, "Rejected count: "+rejected+", current hopsToLive: "+
			 hopsToLive+" for "+this, Logger.DEBUG);
	
        routes.queryRejected(qr.source.isCached(), qr.attenuation);
	long toldRTTime = System.currentTimeMillis();
	if(logDEBUG)
	    n.logger.log(this, "Time to inform RT: "+toldRTTime+" for "+
			 this, Logger.DEBUG);
	
        // FIXME - do we want to do a QueryRestarted if the sanity check fails?
        
        cancelRestart();
        searchData(n);  // check for data in case of
                        // a parallel request succeeding
	long preGotRouteTime = System.currentTimeMillis();
	if(logDEBUG)
	    n.logger.log(this, "Time to cancel restart and search data "+
			 "(unsuccessfully): "+(preGotRouteTime - toldRTTime)+
			 " for "+this, Logger.DEBUG);
	Request newReq = createRequest(qr.otherFields, n.myRef);
	gotRouteTime = System.currentTimeMillis();
	if(logDEBUG)
	    n.logger.log(this, "Got request in "+(gotRouteTime-preGotRouteTime)+
			 " for "+this, Logger.DEBUG);
        sendOn(n, newReq, false);
    }
    
    /**
     * Note the EndOfRouteException and RequestAbortException.
     * Must be called by subclass as part of the implementation
     * for receivedMessage(Node n, *RequestInitiator ri)
     */
    void receivedRequestInitiator(Node n, RequestInitiator ri) throws BadStateException,
                                                                      RequestAbortException,
                                                                      EndOfRouteException {        
        // it must be my own RequestInitiator
        if (this.ri == null || this.ri != ri) {
            throw new BadStateException("Not my request initiator: "+ri);
        }

	gotTime = System.currentTimeMillis();
	logFailure(n, gotTime);
	
	if(receivedTime >= -1) {
	    receivedTime = ri.startTime(); 
	    // message receive time or the 
	    // scheduled restart time.
	}
	if (receivedTime < 1000*1000*1000*1000) {
	    if(logDEBUG)
		n.logger.log(this, "ri.startTime() returned time way before the "+
			     "present for "+this, Logger.DEBUG);
	    receivedTime = 0;
	}
	if (receivedTime > 1000L*1000L*1000L*1000L) {
	    long x = gotTime - receivedTime;
	    n.diagnostics.occurrenceContinuous("preRoutingTime", x);
	    if(x > 500 && logDEBUG)
		Core.logger.log(this, "Got long preRoutingTime ("+x+
				" = "+gotTime+"-"+receivedTime+
				") for "+this, new Exception("debug"),
				Logger.DEBUG);
	}
	long regotTime = System.currentTimeMillis();
	n.diagnostics.occurrenceContinuous("regotTime", regotTime - gotTime);
        // is this the first time?
        if (routes == null) {
	    if(logDEBUG)
		n.logger.log(this, "Starting Pending chain for " + 
			     this, Logger.DEBUG);
            // check for data in cache
            searchData(n);
	    long searchDataTime = System.currentTimeMillis();
	    searchDataRoutingTime = searchDataTime - regotTime;
	    if(searchDataTime - regotTime > 1000)
		n.logger.log(this, "searchDataRoutingTime took "+
			     (searchDataTime - regotTime)+"ms on "+
			     this+"!", Core.logger.MINOR);
	    if (origPeer != null)
	    {
		if(logDEBUG)
		    Core.logger.log(this, "About to route() at "+
				    System.currentTimeMillis()+" for "+
				    this, Logger.DEBUG);
                routes = n.rt.route(searchKey, false);
	    }
	    else
	    {
		// Route to random first hop, copied from NewAnnouncement
		long l = n.randSource.nextLong();
		Key k = new Key(new byte[] {
		    (byte) l, (byte) (l >> 8), (byte) (l >> 16),
		    (byte) (l >> 24), (byte) (l >> 32), (byte) (l >> 40),
		    (byte) (l >> 48), (byte) (l >> 56)
		});
		if(logDEBUG)
		    Core.logger.log(this, "About to route() at "+
				    System.currentTimeMillis()+
				    " for "+this+", random routed as "+k,
				    Logger.DEBUG);
		routes = n.rt.route(k, true);
	    }
	    gotRouteTime = System.currentTimeMillis();
	    n.diagnostics.occurrenceContinuous("getRouteTime", gotRouteTime - searchDataTime);
	    if((gotRouteTime - searchDataTime) > 1000)
		n.logger.log(this, "getting routing object took "+
			     (gotRouteTime - searchDataTime)+
			     " for "+this, Logger.MINOR);
	    if(logDEBUG)
		n.logger.log(this, "Routing: "+routes+" for "+this,
			     Logger.DEBUG);
        } else {
	    
            ++restarted;
            
	    if(logDEBUG) {
		n.logger.log(this, "Restarting Pending chain "+
			     this, new Exception("debug"),
			     Logger.DEBUG);
		n.ticker().getMessageHandler().
		    printChainInfo(id, Core.logStream);
		n.logger.log(this, "Restarting Pending chain "+
			     this+" his was:", 
			     ri.initException, Logger.DEBUG);
	    }
	    
            n.diagnostics.occurrenceBinomial("restartedRequestAccepted",
                                             1, accepted ? 1 : 0);
	    
            
            // decrease contact probability for node that didn't reply
            if (lastPeer != null)
                routes.timedOut();

            // Don't allow indefinite restarts.
            //
            // Note:
            // sendOn will throw an EndOfRouteException below
            // if hopsToLive hits 0.

            if (--hopsToLive > 0) {
                // send QueryRestarted to initiating chain
                relayRestarted(n, Core.hopTime(hopsToLive + 1));
                // +1 for Accepted
            }
	    
	    receivedTime = -2;
	    
            // check for data in cache in case of a parallel request
            searchData(n);
	    gotRouteTime = System.currentTimeMillis();
        }
        
        // note the null; unknown fields are not restored after 
        // restarts since they are not saved. This means that 
        // Requests should actually not carry unknown fields... 
        sendOn(n, createRequest(null, n.myRef), true);
    }

    /**
     * Because method selection on superclasses doesn't work,
     * implementation must call this from a method with the actual
     * message (ie, receivedMessage(Node n, DataRequest dr)
     */
    void receivedRequest(Node n, Request r) {
        // This is a loop, no real problem here.
	if(logDEBUG)
	    n.logger.log(this, "Backtracking", Logger.DEBUG);
        try {
            n.sendMessage(
                new QueryRejected(id, hopsToLive, "Looped request", r.otherFields),
                r.getSource(), 0
            );
        } catch (CommunicationException e) {
            n.logger.log(this, "Failed to send backtrack reply: "+e+" for "+this,
			 Logger.MINOR);
        }
	receivedTime = -2;
        r.drop(n);
    }
    
    void receivedAccepted(Node n, Accepted a) throws BadStateException {
        if (!fromLastPeer(a)) {
            throw new BadStateException("Accepted from the wrong peer!");
        }
        if (!accepted) {
	    if(logDEBUG)
		n.logger.log(this, "Got Accepted in Pending, chain "+
			     this, new Exception("debug"), 
			     Logger.DEBUG);
            scheduleRestart(n, Core.hopTime(hopsToLive));
            acceptedTime = System.currentTimeMillis();
            accepted = true;
            routes.routeAccepted();
        }
    }

    State receivedDataReply(Node n, DataReply dr) throws BadStateException {
        if (!fromLastPeer(dr)) {
            throw new BadStateException("DataReply from wrong peer!");
        }
	if(logDEBUG) {
	    n.logger.log(this, "Got DataReply("+dr+") for "+this,
			 Logger.DEBUG);
	    n.logger.log(this, "Pending receivedDataReply for "+
			 this, new Exception("debug"), Logger.DEBUG);
	}
	
        accepted = true;  // more or less..
	
        routes.routeAccepted();
        cancelRestart();
        
        try {
            receivingData = dr.cacheData(n, searchKey);
	    replyTime = System.currentTimeMillis(); 
	    // replyTime must be set AFTER verifying Storables
	    if(routedTime > 0 && logDEBUG) {
		n.logger.log(this, "NGROUTING: Key "+searchKey+" took "+
			     (replyTime - routedTime)+" for "+hopsToLive,
			     Logger.DEBUG);
		
	    } else {
		if(logDEBUG)
		    n.logger.log(this, "NGROUTING: Key "+searchKey+
				 ": no valid routedTime", Logger.DEBUG);
	    }
            n.ft.remove(searchKey); // remove if in FT.
        } catch (DataNotValidIOException e) {
            dr.drop(n);
            n.logger.log(this,
			 "Got DNV: "+Presentation.getCBdescription(e.getCode())+
			 " for "+this,
			 Logger.MINOR);
            scheduleRestart(n, 0); // schedule restart now
            return this;           // back to pending
        }
        catch (KeyCollisionException e) {
            // oh well, try to go to SendingReply
	    if(logDEBUG)
		n.logger.log(this, "Got KeyCollisionException on "+
			     this, Logger.DEBUG);
            dr.drop(n);
            try {
                searchData(n);
            }
            catch (RequestAbortException rae) {
		if(logDEBUG)
		    n.logger.log(this, "RAE in got KCE on "+this, rae,
				 Logger.DEBUG);
                return rae.state;
            }
            // damn, already gone
            fail(n, "I/O error replying with data");
            n.logger.log(this,
			 "Failed to find data after key collision on "+
			 this+" while caching DataReply",
			 Logger.NORMAL);
            return new RequestDone(this);
        }
        catch (IOException e) {
            dr.drop(n);
            fail(n, "I/O error replying with data");
            n.logger.log(this, "I/O error caching DataReply "+this,
			 e, Logger.ERROR);
            return new RequestDone(this);
        }
        
        try {
            KeyInputStream kin = receivingData.getKeyInputStream();
            kin.setParent(id, n.ticker().getMessageHandler(),
                          "Transfering reply.");

            boolean worked = false;
            try {
                sendingData = sendData(n, kin);
                sendingData.schedule(n);
                worked = true;
            }
            finally {
                if (!worked)
                    kin.close();
            }
        }
        catch (IOException e) {
            // couldn't get the KeyInputStream
            fail(n, "I/O error replying with data");
            if (e instanceof BufferException) {
                n.logger.log(this, "Failed to get KeyInputStream: "+e+
			     " for "+this, Logger.NORMAL);
            }
            else {
                n.logger.log(this, "I/O error getting KeyInputStream for "+
			     this, e, Logger.ERROR);
            }
            return new ReceivingReply(this);
        }
        catch (CommunicationException e) {
            n.logger.log(this, "Error replying to peer: "+e+" for "+this,
			 Logger.MINOR);
            return new ReceivingReply(this);
        }
        finally {
            receivingData.schedule(n);
        }

        return new TransferReply(this);
    }

    
    //=== support methods ======================================================

    private void sendOn(Node n, Request r, boolean isFirst)
	throws EndOfRouteException, RequestAbortException {
	
	loggedResult = false;
	
        if (hopsToLive <= 0)
            throw new EndOfRouteException("Reached last node in route");
        
        Peer addr = null;
        NodeReference nr;
	
        int attemptCount = 0;
	while (true) { // until the send doesn't fail
	    if(logDEBUG)
		n.logger.log(this, "Still trying to route "+searchKey+"("+
			     this+"), attempt "+attemptCount, Logger.DEBUG);
            nr = routes.getNextRoute();
	    if(logDEBUG) n.logger.log(this, "Got next route for "+this,
				      Logger.DEBUG);
	    long stillInSendOnTime = System.currentTimeMillis();
	    if(isFirst && attemptCount == 0) {
		n.diagnostics.occurrenceContinuous("stillInSendOnTime", 
						   stillInSendOnTime - 
						   gotRouteTime);
		if(logDEBUG) n.logger.log(this, "stillInSendOnTime="+
					  (stillInSendOnTime - gotRouteTime)+
					  " for "+this, Logger.DEBUG);
	    }
            if (nr == null) {
		if(logDEBUG)
		    n.logger.log(this, "Ran out of routes for "+this+" after "+
				 attemptCount+" tries.", Logger.DEBUG);
                fail(n, "No route found", r.otherFields);
		if(logDEBUG)
		    n.logger.log(this, "rt exhaused for "+this, 
				 Logger.DEBUG);
                throw new RequestAbortException(new RequestDone(this));
            } else {
		if(logDEBUG)
		    n.logger.log(this, "Got a route for "+this,
				 Logger.DEBUG);
	    }
            addr = n.getPeer(nr);
	    if(logDEBUG) n.logger.log(this, "Got peer "+addr+" for "+this,
				      Logger.DEBUG);
	    long stillStillInSendOnTime = System.currentTimeMillis();
	    if(isFirst && attemptCount == 0) {
		n.diagnostics.occurrenceContinuous("stillStillInSendOnTime", stillStillInSendOnTime - 
						   stillInSendOnTime);
	    }
            if (addr == null) {  // bad node ref
                // or, one we just don't have a transport for?
                //n.logger.log(this,
                //    "Found bad node ref while routing, removing: "+nr,
                //    Logger.MINOR);
                //n.rt.dereference(nr.getIdentity());
		if(logDEBUG)
		    n.logger.log(this, this+": Route address is NULL!", 
				 Logger.DEBUG);
		routes.ignoreRoute();
                continue;
            }
            if (origPeer != null && origPeer.equalsIdent(addr)) {
                // funnily enough, this happens quite often
                // and results in some pretty dumb "loops"
		if(logDEBUG)
		    n.logger.log(this, this+": Route is self",
				 Logger.DEBUG);
		routes.ignoreRoute();
                continue;
            }
	    if(logDEBUG) n.logger.log(this, "Still here after origPeer checks: "+
				      this, Logger.DEBUG);
	    routedTime = System.currentTimeMillis();
	    if(logDEBUG)
		n.logger.log(this, "Forwarding query "+this+" to "+
			     addr+" at "+routedTime, Logger.DEBUG);
            try {
                if (attemptCount == 0 && receivedTime > 0 &&
                    receivedTime < routedTime) { // sanity
		    long rt = routedTime - receivedTime;
                    n.diagnostics.occurrenceContinuous("routingTime", rt);
		    if(rt > 1000 && logDEBUG)
			n.logger.log(this, "routingTime "+rt+" for "+
				     this, new Exception("debug"), 
				     Logger.DEBUG);
		    n.diagnostics.occurrenceContinuous("subRoutingTime",
						       routedTime - 
						       stillStillInSendOnTime);
		    if(searchDataRoutingTime > 0)
			n.diagnostics.occurrenceContinuous("searchDataRoutingTime", 
							   searchDataRoutingTime);
		    receivedTime = -2; // make sure we aren't called twice
                }
                attemptCount++;
		ConnectionHandler ch = 
		    n.makeConnection(addr, n.routeConnectTimeout);
		if(logDEBUG) Core.logger.log(this, "Sending message "+r+" on "+ch+" for "+
				this, Logger.DEBUG);
		ch.sendMessage(r);
		if(logDEBUG) Core.logger.log(this, "Sent message "+r+" on "+ch+" for "+
				this, Logger.DEBUG);
                break;
            } catch (CommunicationException e) {
                ++unreachable;
                // don't care if it's terminal or nonterminal
                // because routing is too time-critical
		if(logDEBUG)
		    n.logger.log(this, "Routing ("+this+") failure to: "
				 +e.peerAddress() + " -- " + e, e, 
				 Logger.DEBUG);
		//e, Logger.DEBUG);
                if (e instanceof AuthenticationFailedException) {
                    routes.routeConnected();
                    routes.authFailed();
                }
                else {
                    routes.connectFailed();
                }
            }
        }

        // Count outbound Requests.
        n.diagnostics.occurrenceBinomial("outboundAggregateRequests", attemptCount, 1);
        if (n.outboundRequests != null) {
            n.outboundRequests.incTotal(addr.getAddress().toString());
        }

        // Keep track of outbound requests for rate limiting.
        n.outboundRequestLimit.inc();

        lastPeer = addr;
        routes.routeConnected();

        scheduleRestart(n, Core.hopTime(1));  // timeout to get the Accepted
    }

    private final void relayRestarted(Node n, long timeout) throws RequestAbortException {
        try {
            ft.restarted(n, timeout);
        }
        catch (CommunicationException e) {
            n.logger.log(this,
                         "Couldn't restart because relaying QueryRestarted failed: "+
			 e+" for "+this, Logger.MINOR);
            throw new RequestAbortException(new RequestDone(this));
        }
    }

    /** Given an existing KeyInputStream, sets up a SendData state to
      * transfer the data back to the requester. 
      */
    SendData sendData(Node n, KeyInputStream doc) throws CommunicationException {
	if(logDEBUG) n.logger.log(this, "Sending data (,"+doc+") for "+this,
				  Logger.DEBUG);
        Storables storables = doc.getStorables();
        OutputStream out = ft.dataFound(n, storables, doc.length());
        // this means the initiator is not interested in seeing the 
        // data (e.g., KeyCollision response in FCP)
        if (out == null) out = new NullOutputStream();
	
        // FIXME: don't waste resources piping data to the NullOutputStream
        
	SendData sd = new SendData(n.randSource.nextLong(), this.id, out,
				   doc, doc.length(), storables.getPartSize());
	if(logDEBUG) n.logger.log(this, "Got SendData("+sd+") for "+this,
				  Logger.DEBUG);
	return sd;
    }
                                                  
    /** Attempts to retrieve the key from the cache and transition to
      * SendingReply.  Will transition to null in the event of an
      * unrecoverable error.  Does nothing if the key is not found.
      */
    void searchData(Node n) throws RequestAbortException {
	long startTime = System.currentTimeMillis();
	long thrownTime = -1;
        KeyInputStream doc = null;
	try {
	    try {
		if(logDEBUG)
		    n.logger.log(this, "Trying to fetch "+this
				 +" at "+startTime, Core.logger.DEBUG);
		doc = n.ds.getData(searchKey);
		long gotDataTime = System.currentTimeMillis();
		if(logDEBUG)
		    n.logger.log(this, "getData took "+(gotDataTime-startTime)+
				 " millis on "+this, Logger.DEBUG);
		if (doc != null) {
		    doc.setParent(id, n.ticker().getMessageHandler(),
				  "Replying with data from store");
		    sendingData = sendData(n, doc);
		    long sendingDataTime = System.currentTimeMillis();
		    if(logDEBUG)
			n.logger.log(this, "sendData() took "+
				     (sendingDataTime-gotDataTime)+
				     " millis on "+this, Logger.DEBUG);
		    sendingData.schedule(n);
		    long scheduledTime = System.currentTimeMillis();
		    if(logDEBUG)
			n.logger.log(this, "schedule took "+
				     (scheduledTime - sendingDataTime)+
				     " millis on "+this, Logger.DEBUG);
		    doc = null;
		    thrownTime = System.currentTimeMillis();
		    if(origPeer != null)
			n.diagnostics.occurrenceContinuous("sendingReplyHTL", 
							   hopsToLive);
		    throw new RequestAbortException(new SendingReply(this));
		}
	    } catch (IOException e) {
		fail(n, "I/O error replying with data");
		n.logger.log(this, "I/O error replying with data on "+this,
			     e, Logger.MINOR);
		thrownTime = System.currentTimeMillis();
		throw new RequestAbortException(new RequestDone(this));
	    } catch (CommunicationException e) {
		n.logger.log(this, "Error replying to peer: "+e+" on "+this, 
			     e, Logger.MINOR);
		thrownTime = System.currentTimeMillis();
		throw new RequestAbortException(new RequestDone(this));
	    } finally {
		if (doc != null) {
		    try {
			doc.close();
		    }
		    catch (IOException e) {
			n.logger.log(this,
				     "Failed to close KeyInputStream after failing "+
				     "on "+this, e, Logger.MINOR);
		    }
		}
	    }
	} catch (RequestAbortException e) {
	    long endTime = System.currentTimeMillis();
	    long length = endTime - startTime;
	    n.diagnostics.occurrenceContinuous("searchFoundDataTime", length);
	    throw e;
	}
	long endTime = System.currentTimeMillis();
	long length = endTime - startTime;
	n.diagnostics.occurrenceContinuous("searchNotFoundDataTime", length);
    }
    
    // Helpers for NGrouting stats
    long endTransferTime = -1;
    boolean loggedResult = false;
    
    public void logSuccess(Node n) {
	if(loggedResult) return;
	loggedResult = true;
	endTransferTime = System.currentTimeMillis();
	
	double transferRate = (((double)receivingData.length()) /
			       ((double)(endTransferTime - replyTime))) * 1000;
	long stdFileSize;
	if(n.dir.countKeys() > 16)
	    stdFileSize = (n.storeSize - n.dir.available()) / n.dir.countKeys();
	else stdFileSize = 100000;
	long expectedTime = (replyTime - routedTime) +
	    (((endTransferTime - replyTime) * stdFileSize) / 
	     receivingData.length());
	if(logDEBUG)
	    n.logger.log(this, "NGROUTING: Key "+searchKey+": search took "+
			 (replyTime-routedTime)+" in "+hopsToLive+", transfer took "+
			 (endTransferTime-replyTime)+" for file length "+
			 receivingData.length()+": transfer rate "+transferRate+
			 " bytes per second, expected total time for file of "+
			 "length "+stdFileSize+": "+expectedTime+"; sent to "+
			 lastPeer, Logger.DEBUG);
    }
    
    public void logFailure(Node n) {
	logFailure(n, System.currentTimeMillis());
    }
    
    public void logFailure(Node n, long endTime) {
	if(routedTime > 0) {
	    if(replyTime > 0) {
		logFailedTransfer(n, endTime);
	    } else {
		logFailedSearch(n, endTime);
	    }
	}
    }
    
    public void logFailedTransfer(Node n) {
	endTransferTime = System.currentTimeMillis();
	logFailure(n, endTransferTime);
    }
    
    public void logFailedTransfer(Node n, long endTime) {
	if(loggedResult) return;
	loggedResult = true;
	if(logDEBUG)
	    n.logger.log(this, "NGROUTING: Key "+searchKey+": search took "+
			    (replyTime-routedTime)+" in "+hopsToLive+
			    ", failed transfer took "+(endTransferTime - replyTime)+
			    "; sent to "+lastPeer, Logger.DEBUG);
	replyTime = -1;
    }
    
    public void logFailedSearch(Node n, long endTime) {
	if(loggedResult) return;
	loggedResult = true;
	if(logDEBUG)
	    n.logger.log(this, "NGROUTING: Key "+searchKey+": search failed in "+
			 (endTime-routedTime)+" for "+hopsToLive+"; sent to "+
			 lastPeer, new Exception("debug"), Logger.DEBUG);
    }
}
