package freenet.node.states.request;

import freenet.*;
import freenet.node.*;
import freenet.message.*;
import freenet.node.states.data.*;
import freenet.node.ds.KeyCollisionException;
import freenet.support.Logger;
import java.io.IOException;

/**
 * State for the terminating node on an insert chain that is only 
 * receiving the data, but should generate and send back a StoreData
 * when it is done.
 */

public class ReceivingInsert extends RequestState {
    
    /** the auxiliary chain that is reading the data into the store */
    ReceiveData receivingData;
    
    ReceivingInsert(InsertPending ancestor) {
        super(ancestor);
    }
    
    ReceivingInsert(AwaitingInsert ancestor) {
        super(ancestor);
        receivingData = ancestor.receivingData;
	// no routing outwards so no ngrouting logging
    }
    
    public final String getName() {
        return "Receiving Insert";
    }
    
    // There's not much we can do if the state is lost while receiving an insert.
    // The MessageHandler will log it. 
    public final void lost(Node n) {
        Core.diagnostics.occurrenceCounting("lostRequestState", 1);
    }
    
    public State receivedMessage(Node n, DataReceived dr) throws BadStateException {
        if (receivingData != dr.source()) {
            throw new BadStateException("Not my DataReceived: "+dr);
        }
        int cb = dr.getCB();
        switch (cb) {
            case Presentation.CB_OK:
                try {
                    receivingData.commit();  // make the key available
                } catch (KeyCollisionException e) {
                    // this is a little bit of a hack.  we jump into a
                    // DataPending state and then handle a restart which
                    // makes us check for the data in the store again
                    n.logger.log(this, "Going to DataPending after key collision",
                                 Logger.MINOR);
                    scheduleRestart(n, 0);
                    return new DataPending(this);
                } catch (IOException e) {
                    fail(n, "Cache failed");
                    n.logger.log(this, "Cache failed on commit", e, Logger.ERROR);
                    return new RequestDone(this);
                }
                n.logger.log(this, "Data received successfully!", Logger.MINOR);
                try {
                    // return the noderef we would have routed to
                    // (there is still a chance of DataSource resetting)
                    // -1 means we don't know the rate for that ref.
		    n.diagnostics.occurrenceCounting("storeDataReceivingInsert", 1);
                    ft.storeData(n, n.rt.route(searchKey,false).
				 getNextRoute(), -1, 0);
		    // We are the terminal node; we cache it
		    if (origPeer != null)
		    {
			if (n.successInsertDistribution != null)
			{
                            n.successInsertDistribution.add(searchKey.getVal());
			}
			if (n.inboundRequests != null)
			{
			    n.inboundRequests.incActive(origPeer.getAddress().toString());
			}
                    }
                }
                catch (CommunicationException e) {
                    n.logger.log(this,
                        "Failed to send back StoreData to peer " + e.peer,
                        e, Logger.MINOR);
                }
                return new RequestDone(this);

            case Presentation.CB_CACHE_FAILED:
                fail(n, "Cache failed");
                // fall through
            default:
                n.logger.log(this,
                    "Failed to receive insert with CB "+Presentation.getCBdescription(cb)
                    +", on chain "+Long.toHexString(id),
                    (cb == Presentation.CB_CACHE_FAILED ? Logger.ERROR : Logger.MINOR));
                return new RequestDone(this);
        }
    }
}

