/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU General Public Licence (GPL) 
  version 2.  See http://www.gnu.org/ for further details of the GPL.
 */

/**
 * This class represents a raw message in the Freenet Protocol
 * expected by Freenet.  It can be created either from
 * an InputStream or manually created from scratch.  It can
 * then by piped out to an OutputStream.  Methods are provided
 * for manipulation of normal fields, however the type of the
 * message and the trailing field should be set by direct
 * manipulation of fields.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <A HREF="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</A>
 **/
package freenet.presentation;
import freenet.*;
import freenet.support.*;
import freenet.support.io.*;
import java.io.*;
import java.util.*;

public final class FNPRawMessage extends RawMessage {
    
    // Constructors

    // FIXME -- what is the TransportHandler for??  -t.c.
    // god knows - oskar (consider it zapped)

    /**
     * Constructs a new RawMessage off an FNP Stream
     * @param i An InputStream of decrypted FNP data
     * @param th  The nodes TransportHandler 
     **/
    public FNPRawMessage(InputStream i) 
        throws InvalidMessageException, EOFException {
	
        //Core.logger.log(this,"Reading message",Logger.DEBUGGING);
	this(new EOFingReadInputStream(i));
    }

    public FNPRawMessage(ReadInputStream ris) 
	throws InvalidMessageException, EOFException{
        //        PushbackInputStream in = new PushbackInputStream(i);
        fs = new FieldSet();

        try {
            // Read message type
            messageType = ris.readToEOF('\n','\r');

            //            System.out.println(messageType);
            trailingFieldName = fs.parseFields(ris,
                                               '\n', // ends a field
                                               '\r', // ignore before end
                                               '=', // delimits field
                                               '.'); // delimits subset
	    if (trailingFieldName==null)
	    	throw new EOFException("incomplete FNP message - hopefully");
            setFields(ris);
     /*   } catch (EOFException e) {
            if (messageType != null) {
                Core.logger.log(this, "Stream died while reading message of type: " + messageType, Logger.ERROR); 
            } else {
                // stream closed without getting a new message
                Core.logger.log(this, "Stream closed", Logger.DEBUGGING);
            }
            throw e;*/
        } catch (IOException e) {
            throw new EOFException("Could not parse message from stream : " + e);
        } catch(Exception e) {
            Core.logger.log(this, "Exception in RawMessage()", Logger.ERROR);
            e.printStackTrace();
        }
    }

    private void setFields(ReadInputStream in) {
        // Read and set the presentation related fields

        // setting KeepAlive
        String cvalue = fs.get("Connection");
        close   = cvalue != null && cvalue.equals("close");
        sustain = cvalue != null && cvalue.equals("sustain");

        // setting DataLength and trailing
        String dlvalue = fs.get("DataLength");
        if (dlvalue == null) {
            trailingFieldLength = 0;
        }
        else {
            trailingFieldLength = Fields.stringToLong(dlvalue);
            fs.remove("DataLength");
        }
        /*
        trailingFieldLength = dlvalue == null ? 0 : Fields.stringToLong(dlvalue);
        if (dlvalue != null) 
            fs.remove("DataLength");
        if (trailingFieldLength != 0) {// we have a trailing
            trailingFieldStream = in;
        } else {
            trailingFieldName = null; // no trailing
        }
        */
    }

    protected FNPRawMessage(String messageType, boolean close, 
                            boolean sustain, FieldSet fs,
                            long trailingLength, String trailingName, 
                            DiscontinueInputStream trailing) {
        super(messageType, close, sustain, fs == null ? new FieldSet() : fs, 
              trailingLength, trailingName, trailing);
    }
    
    // Public Methods

    public void writeMessage(OutputStream out) throws IOException {

        WriteOutputStream writer=new WriteOutputStream(out);
        
        // Output message type
        writer.writeUTF(messageType, '\n');

        // Output tansport options

        if (close)
            fs.add("Connection","close");
        else if (sustain)
            fs.add("Connection","sustain");

        if (trailingFieldLength != 0)
            fs.add("DataLength",Fields.longToString(trailingFieldLength));

        // Output message fields

        fs.writeFields(writer,
                       (trailingFieldName == null ? 
                        "EndMessage" : 
                        trailingFieldName),
                       '\n', // ends a pair
                       '=',  // delimits a pair
                       '.'); // delimits a subset
        // empty writer
        writer.flush();
    }

    public String toString() {
        StringBuffer sb = new StringBuffer(400);
        sb.append(messageType).append("{Close=").append(close);
        sb.append(",Sustain=").append(sustain);
        sb.append(",DataLength=").append(trailingFieldLength);
        sb.append(",").append(fs.toString()).append("}");
        return sb.toString();
    }
}

