package freenet.support;

import java.io.*;
import freenet.support.BinaryTree.Node;

public abstract class AbstractBinaryTree implements BinaryTree {

    public static final void dump(Node n, PrintWriter out) throws IOException {
        out.println(n.toString());
        if (n.hasLeftChild())
            dump(n.getLeftChild(), out);
        if (n.hasRightChild())
            dump(n.getRightChild(), out);
    }
    

    protected static class TreeWalk implements Walk {

        protected final BinaryTree tree;
        protected final boolean ascending;
        
        protected Node node = null;
        
        protected Comparable searchKey;
        protected boolean inclusive;
        
    
        public TreeWalk(BinaryTree tree, boolean ascending) {
            this.tree = tree;
            this.ascending = ascending;
        }

        public TreeWalk(BinaryTree tree, Node node, boolean ascending) {
            this(tree, ascending);
            this.node = node;
        }
    
        public TreeWalk(BinaryTree tree, Comparable searchKey,
                        boolean inclusive, boolean ascending) {
            this(tree, ascending);
            this.searchKey = searchKey;
            this.inclusive = inclusive;
        }

        public Object getNext() {
            if (node == null) {
                if (searchKey == null) {
                    return node = ascending ? tree.treeMin() : tree.treeMax();
                }
                else {
                    return node = ascending ? tree.treeMinConstrained(searchKey, inclusive)
                                            : tree.treeMaxConstrained(searchKey, inclusive);
                }
            }
            else {
                return node = ascending ? tree.treeSuccessor(node)
                                        : tree.treePredecessor(node);
            }
        }
    }
    


    // the only thing not implemented is getObject()
    public static abstract class AbstractNode implements Node {

        protected Node parent, leftChild, rightChild;

        public String toString() {
            StringBuffer sb = new StringBuffer();
            sb.append("node: ");
            sb.append(Integer.toHexString(System.identityHashCode(this)));
            sb.append(" left: ");
            sb.append(leftChild == null
                      ? "null"
                      : Integer.toHexString(System.identityHashCode(leftChild)));
            sb.append(" right: ");
            sb.append(rightChild == null
                      ? "null"
                      : Integer.toHexString(System.identityHashCode(rightChild)));
            return sb.toString();
        }
        
        public final boolean hasParent() {
            return parent != null;
        }
        
        public final Node getParent() {
            return parent;
        }

        public final void setParent(Node n) {
            parent = n;
        }

        public final boolean hasLeftChild() {
            return leftChild != null;
        }
        
        public final Node getLeftChild() {
            return leftChild;
        }
        
        public final void setLeftChild(Node n) {
            leftChild = n;
        }
        
        public final boolean hasRightChild() {
            return rightChild != null;
        }
        
        public final Node getRightChild() {
            return rightChild;
        }
        
        public final void setRightChild(Node n) {
            rightChild = n;
        }
    }


    
    protected abstract Node treeRoot();


    protected static final int compare(Node n1, Node n2) {
        return n1.getObject().compareTo(n2.getObject());
    }

    protected static final int compare(Comparable c, Node n) {
        return c.compareTo(n.getObject());
    }
    
    
    public final Node treeSearch(Comparable searchKey) {
        return treeSearch(treeRoot(), searchKey);
    }
    
    public static final Node treeSearch(Node node, Comparable searchKey) {
        while (node != null) {
            int cmp = compare(searchKey, node);
            if (cmp > 0)
                node = node.getRightChild();
            else if (cmp < 0)
                node = node.getLeftChild();
            else
                break;
        }
        return node;
    }
    

    public final Node treeMatch(Comparable searchKey, int cmpSense) {
        return treeMatch(treeRoot(), searchKey, cmpSense);
    }

    public static final Node treeMatch(Node node, Comparable searchKey, int cmpSense) {

        Node next = node;
        int cmp = 0;
        
        while (next != null) {
            node = next;
            cmp = compare(searchKey, node);
            if (cmp > 0)
                next = node.getRightChild();
            else if (cmp < 0)
                next = node.getLeftChild();
            else
                return node;
        }
        
        if (cmp < 0 && cmpSense < 0) {
            next = treePred(node);
            if (next != null)
                node = next;
        }
        else if (cmp > 0 && cmpSense > 0) {
            next = treeSucc(node);
            if (next != null)
                node = next;
        }
        
        return node;
    }
    


    public final Node treeSuccessor(Node node) {
        if (!node.hasParent() && node != treeRoot())
            return treeMinConstrained(node.getObject(), false);
        else
            return treeSucc(node);
    }

    public static final Node treeSucc(Node node) {
        if (node == null) {
            return null;
        }
        Node next = node.getRightChild();
        if (next != null) {
            return treeMin(next);
        }
        next = node.getParent();
        while (next != null && compare(node, next) > 0) {
            node = next;
            next = node.getParent();
        }
        return next;
    }


    public final Node treePredecessor(Node node) {
        if (!node.hasParent() && node != treeRoot())
            return treeMaxConstrained(node.getObject(), false);
        else
            return treePred(node);
    }

    public static final Node treePred(Node node) {
        if (node == null) {
            return null;
        }
        Node next = node.getLeftChild();
        if (next != null) {
            return treeMax(next);
        }
        next = node.getParent();
        while (next != null && compare(node, next) < 0) {
            node = next;
            next = node.getParent();
        }
        return next;
    }


    public final Node treeMinConstrained(Comparable searchKey, boolean inclusive) {
        return treeMinConstrained(treeRoot(), searchKey, inclusive);
    }

    public static final Node treeMinConstrained(Node node,
                                                Comparable searchKey,
                                                boolean inclusive) {
        if (node == null) {
            return null;
        }
        while (true) {
            
            Node next;            
            
            int cmp = compare(searchKey, node);
            if (cmp > 0)
                next = node.getRightChild();
            else if (cmp < 0)
                next = node.getLeftChild();
            else if (!inclusive)
                next = null;
            else
                return node;

            if (next == null) {
                return cmp >= 0 ? treeSucc(node) : node;
            }
            node = next;
        }
    }

    
    public final Node treeMaxConstrained(Comparable searchKey, boolean inclusive) {
        return treeMaxConstrained(treeRoot(), searchKey, inclusive);
    }

    public static final Node treeMaxConstrained(Node node,
                                                Comparable searchKey,
                                                boolean inclusive) {
        if (node == null) {
            return null;
        }
        while (true) {
    
            Node next;

            int cmp = compare(searchKey, node);
            if (cmp > 0)
                next = node.getRightChild();
            else if (cmp < 0)
                next = node.getLeftChild();
            else if (!inclusive)
                next = null;
            else
                return node;
        
            if (next == null) {
                return cmp <= 0 ? treePred(node) : node;
            }
            node = next;
        }
    }

    
  
    public final Node treeMin() {
        return treeMin(treeRoot());
    }
    
    public static final Node treeMin(Node node) {
        if (node == null) {
            return null;
        }
        Node next;
        while (null != (next = node.getLeftChild())) {
            node = next;
        }
        return node;
    }

    
    
    public final Node treeMax() {
        return treeMax(treeRoot());
    }
  
    public static final Node treeMax(Node node) {
        if (node == null) {
            return null;
        }
        Node next;
        while (null != (next = node.getRightChild())) {
            node = next;
        }
        return node;
    }





    protected static final Node treeInsert(Node node, Node newNode) {
        
        Node next;
        
        while (true) {
            
            int cmp = compare(newNode, node);
            if (cmp > 0)
                next = node.getRightChild();
            else if (cmp < 0)
                next = node.getLeftChild();
            else
                return node;

            if (next == null) {
                newNode.setParent(node);
                if (cmp < 0)
                    node.setLeftChild(newNode);
                else
                    node.setRightChild(newNode);
                return null;
            }

            node = next;
        }
    }

    public Node treeRemove(Comparable searchKey) {
        Node n = treeSearch(searchKey);
        if (n == null)
            return null;
        treeRemove(n);
        return n;
    }



    public Walk treeWalk(boolean ascending) {
        return new TreeWalk(this, ascending);
    }

    public static final Walk treeWalk(BinaryTree tree, boolean ascending) {
        return new TreeWalk(tree, ascending);
    }


    public Walk treeWalk(Node node, boolean ascending) {
        return new TreeWalk(this, node, ascending);
    }

    public static final Walk treeWalk(BinaryTree tree, Node node, boolean ascending) {
        return new TreeWalk(tree, node, ascending);
    }


    public Walk treeWalk(Comparable searchKey, boolean inclusive, boolean ascending) {
        return new TreeWalk(this, searchKey, inclusive, ascending);
    }
    
    public static final Walk treeWalk(BinaryTree tree, Comparable searchKey,
                                      boolean inclusive, boolean ascending) {
        return new TreeWalk(tree, searchKey, inclusive, ascending);
    }
}


