/* -*- Mode: java; c-basic-indent: 4; tab-width: 4 -*- */
package freenet.support.servlet;
import java.io.*;
import java.util.*;
import javax.servlet.http.*;

/**
 * A template of text into which values in a Dictionary may be
 * inserted.  The template is read from an InputStream and
 * everything is pre-processed to make the actual generation
 * of the complete document as rapid as possible.  Values
 * to be replaced in the template should be in the format
 * ##VALUE##.  Behavior is undefined if this syntax is
 * abused (eg. having an odd number of '##' in the template).
 *
 * Code using this class probably won't be thread safe (unless
 * you create a new HtmlTemplate every time you render the page
 * which would be very inefficient) so synchronize accordingly.
 *
 * @author     Ian Clarke
 * @created    August 27, 2002
 */
public class HtmlTemplate implements TemplateElement {
	/**  The default directory in which templates are stured */
	public final static String TEMPLATEDIR = "/freenet/node/http/templates/";
	
	
	/**
	 *  Returns a new template
	 *
	 * @param  template         The name of a template within Freenet's template directory
	 * @return                  The newly created template
	 * @exception  IOException  Description of the Exception
	 */
	public static HtmlTemplate createTemplate(String template)
		throws IOException {
		return new HtmlTemplate(TEMPLATEDIR + template);
	}
	
	
	/**
	 * A list of Fragments, each of which can either be a plain piece
	 * of text, or a variable to be replaced in the template
	 */
	Vector fragments = new Vector();
	
	/**
	 * A hashtable which stores the desired Strings or TemplateElements
	 * for each given variable
	 */
    Hashtable d = new Hashtable();
    
    /**
     * Construct an HtmlTemplate by cloning another one
	 * We keep fragments exactly as is, except that d is deep copied
	 */
	public HtmlTemplate(HtmlTemplate source) {
		fragments = source.fragments;
		d = (Hashtable)(source.d.clone());
	}
	
	/**
	 *Constructor for the HtmlTemplate object
	 *
	 * @param  resource         Description of the Parameter
	 * @exception  IOException  Description of the Exception
	 */
	// TODO: Make this fail more gracefully (ie. replacing the template with text warning
	// of the problem)
	public HtmlTemplate(String resource)
		throws IOException {
		InputStream is = this.getClass().getResourceAsStream(resource);
		readInputStream(is);
		is.close();
	}
	
	
	/**
	 *Constructor for the HtmlTempate object
	 *
	 * @param  template         Description of the Parameter
	 * @exception  IOException  Description of the Exception
	 */
	public HtmlTemplate(InputStream template)
		throws IOException {
		readInputStream(template);
	}
	
	
	/**
	 *Constructor for the HtmlTemplate object
	 *
	 * @param  template         Description of the Parameter
	 * @exception  IOException  Description of the Exception
	 */
	public void readInputStream(InputStream template)
		throws IOException {
		// Parse the template into the fragments vector
		BufferedInputStream bTemplate = new BufferedInputStream(template);
		
		// We use a and b to keep track of the last and second last characters
		// to be read from the stream
		int a = bTemplate.read();
		int b = bTemplate.read();
		// The ele bool is true if and only if we are reading a ##variable## rather than
		// plain text
		boolean ele = false;
		if ((a == -1) || (b == -1)) {
			throw new
				// FIXME: This is ugly, why shouldn't it be less than two characters?
				RuntimeException("Template file cannot be less than two characters in length");
		}
		StringBuffer sb = new StringBuffer(5000);
		
		// Now parse through the string
		while (true) {
			// If the last two characters read were "##" then...
			if ((a == (int) '#') && (b == (int) '#')) {
				String s = new String(sb.toString()); //Detach the value we want (and not a single byte more) from the sb:s underlying char[]
				if (ele) {
					// We just finished reading a variable
					fragments.addElement(new VarFragment(s));
				} else {
					// We just finished reading a string
					fragments.addElement(new StringFragment(s));
				}
				// Clear the StringBuffer
				// FIXME: It would be cleaner to have something similar to a StringBuffer
				// so we could reuse the allocated space - alas StringBuffer doesn't
				// provide this functionality
				sb.setLength(0);
				ele = !(ele);
				// Read next char
				a = bTemplate.read();
				// Handle EOF
				if (a == -1) {
					break;
				}
				b = bTemplate.read();
				if (b == -1) {
					break;
				}
			} else {
				// Add this char to our StringBuffer and read the next one
				sb.append((char) a);
				a = b;
				b = bTemplate.read();
				if (b == -1) {
					break;
				}
			}
		}
		if (ele) {
			fragments.addElement(new VarFragment(sb.toString()));
		} else {
			fragments.addElement(new StringFragment(sb.toString()));
		}
		sb.setLength(0);
		ele = !(ele);
	}
	
	
	/**
	 * Set a variable to a String value
	 *
	 * @param  var  The variable to set
	 * @param  val  The value to set it to
	 */
	public void set(String var, String val) {
		d.put(var, val);
	}
	
	
	/**
	 * Set a variable to the output of a TemplateElement
	 *
	 * @param  var  The variable to set
	 * @param  val  The TemplateElement which will produce the output
	 *            to replace the variable in the template.
	 */
	public void set(String var, TemplateElement val) {
		d.put(var, val);
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  var  Description of the Parameter
	 * @param  val  Description of the Parameter
	 * @param  req  Description of the Parameter
	 */
	public void set(String var, TemplateElement val, HttpServletRequest req) {
		set(var, val);
		d.put(var + "#req", req);
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  pw  Description of the Parameter
	 */
	public void toHtml(PrintWriter pw) {
		toHtml(pw, null);
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  pw   Description of the Parameter
	 * @param  hsr  Description of the Parameter
	 */
	public void toHtml(PrintWriter pw, 
					   javax.servlet.http.HttpServletRequest hsr) {
		for (Enumeration e = fragments.elements(); e.hasMoreElements(); ) {
			((Fragment) e.nextElement()).toHtml(pw, hsr, d);
		}
	}
}

/**
 *  Description of the Interface
 *
 * @author     ian
 * @created    August 27, 2002
 */
interface Fragment {
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  pw   Description of the Parameter
	 * @param  d    Description of the Parameter
	 * @param  hsr  Description of the Parameter
	 */
	public void toHtml(PrintWriter pw, 
					   javax.servlet.http.HttpServletRequest hsr, 
					   Dictionary d);
}

/**
 *  Description of the Class
 *
 * @author     ian
 * @created    August 27, 2002
 */
class StringFragment implements Fragment {
	
	
	String s;
	
	
	/**
	 *Constructor for the StringFragment object
	 *
	 * @param  s  Description of the Parameter
	 */
	public StringFragment(String s) {
		this.s = s;
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  pw   Description of the Parameter
	 * @param  d    Description of the Parameter
	 * @param  hsr  Description of the Parameter
	 */
	public void toHtml(PrintWriter pw, javax.servlet.http.HttpServletRequest hsr, Dictionary d) {
		pw.print(s);
	}
	
	/*
	 *  For testing only
	 *  public String toString()
	 *  {
	 *  return "StringFragment: \""+s+"\"";
	 *  }
	 */
}

/**
 *  Description of the Class
 *
 * @author     ian
 * @created    August 27, 2002
 */
class VarFragment implements Fragment {


	String v;


	/**
	 *Constructor for the VarFragment object
	 *
	 * @param  v  Description of the Parameter
	 */
	public VarFragment(String v) {
		this.v = v;
	}


	/**
	 *  Description of the Method
	 *
	 * @param  pw   Description of the Parameter
	 * @param  d    Description of the Parameter
	 * @param  hsr  Description of the Parameter
	 */
	public void toHtml(PrintWriter pw, javax.servlet.http.HttpServletRequest hsr, Dictionary d) {
		Object o = d.get(v);
		if (o == null) {
			//o = "##" + v + "##";
			// will this cause problems? I don't hope so. :) /Bombe
			o = "";
		}
		if (o instanceof TemplateElement) {
			HttpServletRequest req = (HttpServletRequest) d.get(v + "#req");
			if (req == null) {
				((TemplateElement) o).toHtml(pw);
			} else {
				((TemplateElement) o).toHtml(pw, req);
			}
		} else {
			pw.print((String) o);
		}
	}

}

