package freenet.support.test;

import junit.framework.*;

import freenet.support.Comparable;
import freenet.support.BinaryTree;
import freenet.support.BinaryTree.Node;
import freenet.support.RedBlackTree;
import freenet.support.RedBlackTree.RBNode;
import freenet.support.RedBlackTree.RBNodeImpl;
import freenet.support.Walk;
import freenet.support.sort.*;

import java.util.Hashtable;
import java.util.Enumeration;


/**
 * This is a unit test for freenet.support.RedBlackTree
 * Sequences of values to try are contained in a statically
 * initialized hashtable.  The test may be run by invoking
 * the main method of this class, in which case an additional
 * sequence of values to try may be passed in as the
 * command-line args.
 *
 * NOTE: it is assumed that there are no duplicate values
 *       in the test sequences, that each contains at least
 *       ten elements, and that the values are spaced
 *       at least two units apart!
 * 
 * @author tavin
 */
public class RedBlackTreeTest extends TreeTester {

    public static final void main(String[] args) {
        if (args.length > 0) {
            int[] seq = new int[args.length];
            for (int i=0; i<args.length; ++i) {
                System.out.println(args[i]);
                seq[i] = Integer.parseInt(args[i]);
            }
            sequences.put("__ARGS", seq);
        }
        SimpleTestRunner.main(
             new String[] { RedBlackTreeTest.class.getName() }
        );
    }

    
    public RedBlackTreeTest(String name) {
        super(name);
    }

    /**
     * Constructs a red-black tree filled with values from
     * the named sequence.
     */
    public BinaryTree makeTree() {
        return new RedBlackTree();
    } 

    public Node makeNode(Comparable comp) {
        return new RBNodeImpl(comp);
    }

    public void integrityCheck(String seqStr, Node node) {
        balanceCheck(seqStr, node);
    }

    /**
     * This returns the black-node-depth of the subtree descending
     * from this node, while asserting that the two subtrees of the
     * child nodes have equal black-node-depths (and that there are
     * no adjacent red nodes).
     */
    private int balanceCheck(String seqStr, Node node) {

        RBNode n = (RBNode) node;
        
        int ldepth = (n.hasLeftChild()
                      ? balanceCheck(seqStr, (RBNode) n.getLeftChild()) : 0);
        
        int rdepth = (n.hasRightChild()
                      ? balanceCheck(seqStr, (RBNode) n.getRightChild()) : 0);
        
        assertEquals("Subtrees are balanced at ("+seqStr+"): "+n,
                     ldepth, rdepth);
        
        if (n.getColor() == RedBlackTree.RED) {
            if (n.hasParent()) {
                assertEquals("No adjacent red nodes at ("+seqStr+"): "+n,
                             RedBlackTree.BLACK,
                             ((RBNode) n.getParent()).getColor());
            }
        }
        else ++ldepth;
        
        return ldepth;
    }

    public void testConstruction() {
        super.testConstruction();
    }


    public void testRemoval() {
        super.testRemoval();
    }

    public void testReplacing() {
        super.testReplacing();
    }

    public void testMatching() {
        super.testMatching();
    }

    public void testOrdering() {
        super.testOrdering();
    }

    public void testWalking() {
        super.testWalking();
    }

    public void testWalkingModificationAsc() {
        super.testWalkingModificationAsc();
    }

    public void testWalkingModificationDesc() {
        super.testWalkingModificationDesc();
    }

}



