package freenet.thread;

import freenet.Core;
import freenet.support.Irreversible;
import freenet.support.Logger;

/**
 * A derivative of PThreadFactory that attempts the keep the available
 * threads within a sane ratio of the number of active ones. It does not
 * stop pooling when overflow is reached.
 *
 * @author oskar
 * @author tavin
 */
public final class QThreadFactory implements ThreadFactory, Runnable {

    /** The absolute minimum amount of active threads to be available */
    public static final int MINIMUM_AVAILABLE_ABS = 3;
    /** The minimum ratio of the number of active threads to be available */
    private static final double MINIMUM_AVAILABLE_RATIO = 0.1;
    /** The maximum ratio of the number of active threads to keep available */
    private static final double MAXIMUM_AVAILABLE_RATIO = 1;
    /** The ideal ratio of the number of active threads to keep available */
    private static final double IDEAL_AVAILABLE_RATIO = 
        (2 * MINIMUM_AVAILABLE_RATIO + MAXIMUM_AVAILABLE_RATIO) / 3;

    private final ThreadGroup tg;

    private int active = 0;
    private int available = 0;

    private long threadNumber = 0;

    private QThread headerThread = null;

    private final Object maxLock = new Object();
    private int desiredMax;
    /**
     * @param tg     ThreadGroup all created threads will belong to
     */
    public QThreadFactory(ThreadGroup tg, int desiredMax) {
        this.tg = tg;
        this.desiredMax = desiredMax;
        (new Thread(this, "Thread creation thread.")).start();
    }

    public void run() {
        synchronized (this) {
            while (true) {
                Throwable lastEx = null;
                try {

                    if (available < MINIMUM_AVAILABLE_ABS || 
                        available < active * MINIMUM_AVAILABLE_RATIO) {
                        
                        int desired = 
                            Math.max((int) (active * IDEAL_AVAILABLE_RATIO),
                                     2 * MINIMUM_AVAILABLE_ABS);
                        while (available < desired) {
                            createThread();
                        }
                    } else if (available > (3 * MINIMUM_AVAILABLE_ABS) &&
                               available > active * MAXIMUM_AVAILABLE_RATIO) {
                        
                        int desired = 
                            Math.max((int) (active * IDEAL_AVAILABLE_RATIO),
                                     2 * MINIMUM_AVAILABLE_ABS);
                        
                        
                        while (available > desired) {
                            destroyThread();
                        }
                    }

                    try {
                        wait(1000);
                    } catch (InterruptedException e) {
                    }
                } catch (Throwable e) {
                    if (lastEx == null || !lastEx.getClass().equals(e.getClass()))
                        Core.logger.log(this, "Exception in QThreadFactory. "
                                        + available + " thread available ," 
                                        + active + " running. Top: " + headerThread, 
                                        e, Core.logger.ERROR);
                    lastEx = e;
                    try {
                        wait(20); // avoid runaway loop.
                    } catch (InterruptedException e2) {}
                }
            }
        }
    }

    public final int maximumThreads() {
        synchronized (maxLock) {
            return desiredMax;
        }
    }

    /**
     * @return  the number of currently executing jobs
     */
    public final int activeThreads() {
        return active;
    }

    /**
     * @return  the instantaneous number of idle threads;
     *          may return a negative number if the force option
     *          was used with getThread()
     */
    public final int availableThreads() {
        return available;
    }

    /**
     * @param job       The job to be executed
     */
    public synchronized final Thread getThread(Runnable job) {

        // hopefully this will not happen often.
        if (headerThread == null) {
            createThread();
        }

        QThread thread = headerThread;
        headerThread = headerThread.next;
        thread.next = null;
        thread.job = job;
        ++active;
        --available;
        thread.start();

        awaken();
	return thread;
    }

    /**
     * Creates a thread and adds it to the stack.
     */
    private synchronized final void createThread() {
        QThread newThread = new QThread(++threadNumber);
        newThread.next = headerThread;
        headerThread = newThread;
        available++;
    }

    /**
     * Removes a thread from the stack and signals it to die.
     */
    private synchronized final void destroyThread() {
        QThread dyingThread = headerThread;
        headerThread = headerThread.next;
        dyingThread.die();
        available--;
    }

    /**
     * Returns a thread to the stack when it is finished executing.
     */
    private synchronized final void returnThread(QThread returnThread) {
        returnThread.next = headerThread;
        headerThread = returnThread;
        active--;
        available++;
    }

    private synchronized final void awaken() {
         if ( ( available < MINIMUM_AVAILABLE_ABS) ||
              ( available < active * MINIMUM_AVAILABLE_RATIO) || 
              ( ( available > (3 * MINIMUM_AVAILABLE_ABS)) &&
                ( available > active * MAXIMUM_AVAILABLE_RATIO))) {
             notifyAll();
         }
    }

    private final class QThread extends Thread implements PooledThread {

        private QThread next;
    
        private Runnable job = null;

        private int jobsDone = 0;

        private final Irreversible alive = new Irreversible(true);
    
        public QThread(long num) {
            super(tg, "QThread-"+num);
            super.start();
        }

        public final Runnable job() {
            return job;
        }

        public final synchronized void start() {
            this.notify();
        }

        public final void run() {
            while (alive.state()) {
                synchronized (this) {
                    while (alive.state() && job == null) {
                        try {
                            this.wait(200);
                        }
                        catch (InterruptedException e) {}
                    }
                }
                
                if (job != null) {
                    try {
                        Core.diagnostics.occurrenceCounting("jobsExecuted", 1);
                        job.run();
                    } catch (OutOfMemoryError e) {
                        Core.logger.log(this, "Got out of memory error, " +
                                        "decreasing maximum threads by 5 from ~"+
					desiredMax+".", e, Core.logger.ERROR);
			String status = "dump of interesting objects from Qthread:"+
						"\ntcpConnections " +freenet.transport.tcpConnection.instances+
						"\nFnpLinks " +freenet.session.FnpLink.instances+
						"\nNIOOS " + freenet.support.io.NIOOutputStream.instances+
						"\nNIOIS " + freenet.support.io.NIOInputStream.instances+
						"\nCH " + freenet.ConnectionHandler.instances+
						"\nCHIS " +freenet.ConnectionHandler.CHISinstances+
						"\nCHOS " +freenet.ConnectionHandler.CHOSinstances+
						"\nRIS " +freenet.ConnectionHandler.RISinstances+
						"\nSOS " +freenet.ConnectionHandler.SOSinstances;
			System.err.println(status);
			Core.logger.log(this, status, Logger.ERROR);
                        synchronized(maxLock) {
                            desiredMax = Math.max(desiredMax - 5, 0);
                        }
                    } catch (Throwable e) {
                        Core.logger.log(this, "Unhandled throw in job",
                                        e, Core.logger.ERROR);
                    } finally {
                        jobsDone++;
                        job = null;
                        if (alive.state()) // this should always be true
                            returnThread(this);
                    }
                }
            }

            Core.diagnostics.occurrenceContinuous("jobsPerQThread", jobsDone);
            
            Core.logger.log(this, getName() + " ended. " + available + 
                            "threads available.", Core.logger.DEBUGGING);
        }

        /**
         * You should probably only call this when the QThread is idle.
         */
        protected synchronized final void die() {
            if (alive.state()) {
                alive.change();
                this.notify();
            }
        }
    }
}




