/**
 * jline - Java console input library
 * Copyright (c) 2002,2003 Marc Prud'hommeaux marc@apocalypse.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package jline;

import java.io.*;
import java.util.*;


/** 
 *	<p>
 *  A simple {@link Completor} implementation that handles a pre-defined
 *  list of completion words.
 *  </p>
 *
 *	<p>
 *  Example usage:
 *  </p>
 *  <pre>
 *  myConsoleReader.addCompletor (new SimpleCompletor (new String [] { "now", "yesterday", "tomorrow" }));
 *  </pre>
 *  
 *  @author  <a href="mailto:marc@apocalypse.org">Marc Prud'hommeaux</a>
 */
public class SimpleCompletor
	implements Completor
{
	/** 
	 *  The list of candidates that will be completed.
	 */
	String [] candidateStrings;


	/** 
	 *  Create a new SimpleCompletor with a single possible completion
	 *  values.
	 */
	public SimpleCompletor (String candidateString)
	{
		this (new String [] { candidateString });
	}


	/** 
	 *  Create a new SimpleCompletor with a list of possible completion
	 *  values.
	 */
	public SimpleCompletor (String [] candidateStrings)
	{
		this.candidateStrings = candidateStrings;
	}


	/** 
	 *  Complete candidates using the contents of the specified Reader.
	 */
	public SimpleCompletor (Reader reader)
		throws IOException
	{
		this (getStrings (reader));
	}


	/** 
	 *  Complete candidates using the whitespearated values in
	 *  read from the specified Reader.
	 */
	public SimpleCompletor (InputStream in)
		throws IOException
	{
		this (getStrings (new InputStreamReader (in)));
	}


	private static String [] getStrings (Reader reader)
		throws IOException
	{
		if (!(reader instanceof BufferedReader))
			reader = new BufferedReader (reader);

		List words = new LinkedList ();
		String line;
		while ((line = ((BufferedReader)reader).readLine ()) != null)
		{
			for (StringTokenizer tok = new StringTokenizer (line);
				tok.hasMoreTokens (); words.add (tok.nextToken ()));
		}

		return (String [])words.toArray (new String [words.size ()]);
	}


	public int complete (String buffer, int cursor, List candidates)
	{
		for (int i = 0; i < candidateStrings.length; i++)
		{
			if (buffer == null || buffer.length () == 0 ||
				candidateStrings [i].startsWith (buffer))
			{
				candidates.add (candidateStrings [i]);
			}
		}

		if (candidates.size () == 1)
			candidates.set (0, ((String)candidates.get (0)) + " ");

		// the index of the completion is always from the beginning of
		// the buffer.
		return candidates.size () == 0 ? -1 : 0;
	}


	/** 
	 *  Set the list of candidate Strings.
	 */
	public void setCandidateStrings (String [] candidateStrings)
	{
		this.candidateStrings = candidateStrings;
	}


	/** 
	 *  Returns the list of candidate Strings.
	 */
	public String [] getCandidateStrings ()
	{
		return this.candidateStrings;
	}


	public void addCandidateString (String candidateString)
	{
		List cand = new LinkedList (Arrays.asList (candidateStrings));
		cand.add (candidateString);
		candidateStrings = (String [])cand.toArray (new String [cand.size ()]);
	}
}
