/**
 * jline - Java console input library
 * Copyright (c) 2002,2003 Marc Prud'hommeaux marc@apocalypse.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package jline;

import java.io.*;

import junit.framework.*;


public abstract class JLineTestCase
	extends TestCase
{
	ConsoleReader console;


	public JLineTestCase (String test)
	{
		super (test);
	}


	public void setUp ()
		throws Exception
	{
		super.setUp ();
		console = new ConsoleReader (null, new PrintWriter (
			new OutputStreamWriter (new ByteArrayOutputStream ())));
	}


	public void assertBuffer (String expected, Buffer buffer)
		throws IOException
	{
		assertBuffer (expected, buffer, true);
	}


	public void assertBuffer (String expected, Buffer buffer, boolean clear)
		throws IOException
	{
		// clear current buffer, if any
		if (clear)
		{
			console.finishBuffer ();
			console.getHistory ().clear ();
		}

		console.setInput (new ByteArrayInputStream (buffer.getBytes ()));

		// run it through the reader
		while (console.readLine (null) != null);

		assertEquals (expected, console.getCursorBuffer ().toString ());
	}


	class Buffer
	{
		private final ByteArrayOutputStream bout = new ByteArrayOutputStream ();

		public Buffer ()
		{
		}


		public Buffer (String str)
		{
			append (str);
		}


		public byte [] getBytes ()
		{
			return bout.toByteArray ();
		}


		public Buffer op (short operation)
		{
			return append (console.getKeyForAction (operation));
		}


		public Buffer ctrlA ()
		{
			return append (console.getKeyForAction (
				ConsoleReader.MOVE_TO_BEG));
		}


		public Buffer ctrlU ()
		{
			return append (console.getKeyForAction (
				ConsoleReader.KILL_LINE_PREV));
		}


		public Buffer tab ()
		{
			return append (console.getKeyForAction (
				ConsoleReader.COMPLETE));
		}


		public Buffer back ()
		{
			return append (console.getKeyForAction (
				ConsoleReader.DELETE_PREV_CHAR));
		}


		public Buffer left ()
		{
			return append (ConsoleReader.ARROW_START)
				.append (ConsoleReader.ARROW_PREFIX)
				.append (ConsoleReader.ARROW_LEFT);
		}


		public Buffer right ()
		{
			return append (ConsoleReader.ARROW_START)
				.append (ConsoleReader.ARROW_PREFIX)
				.append (ConsoleReader.ARROW_RIGHT);
		}


		public Buffer up ()
		{
			return append (ConsoleReader.ARROW_START)
				.append (ConsoleReader.ARROW_PREFIX)
				.append (ConsoleReader.ARROW_UP);
		}


		public Buffer down ()
		{
			return append (ConsoleReader.ARROW_START)
				.append (ConsoleReader.ARROW_PREFIX)
				.append (ConsoleReader.ARROW_DOWN);
		}


		public Buffer append (String str)
		{
			byte [] bytes = str.getBytes ();

			for (int i = 0; i < bytes.length; i++)
			{
				append (bytes [i]);
			}

			return this;
		}


		public Buffer append (int i)
		{
			return append ((byte)i);
		}


		public Buffer append (byte b)
		{
			bout.write (b);

			return this;
		}
	}
}
