<?php
/*
Copyright (©) 2003-2013 Teus Benschop.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/


class Database_Modifications
{
  private static $instance;
  private $db;
  private function __construct() {
    $this->db = Database_SQLite::connect ("modifications");
  }
  public static function getInstance()
  {
    if (empty (self::$instance)) {
      self::$instance = new Database_Modifications ();
    }
    return self::$instance;
  }


  public function create ()
  {
$sql = <<<'EOD'
CREATE TABLE IF NOT EXISTS teams (
  bible text,
  book integer,
  chapter integer,
  data text
);
EOD;
    Database_SQLite::exec ($this->db, $sql);

$sql = <<<'EOD'
CREATE TABLE IF NOT EXISTS users (
  username text,
  timestamp integer,
  bible text,
  book integer,
  chapter integer,
  oldid integer,
  oldtext text,
  newid integer,
  newtext text
);
EOD;
    Database_SQLite::exec ($this->db, $sql);

$sql = <<<'EOD'
CREATE TABLE IF NOT EXISTS notifications (
  timestamp integer,
  username text,
  category text,
  bible integer,
  book integer,
  chapter integer,
  verse integer,
  oldtext text,
  modification text,
  newtext text
);
EOD;
    Database_SQLite::exec ($this->db, $sql);
  }


  public function trim ()
  {
    // Change notifications expire after 90 days.
    $time = time () - 7776000; 
    $query = "DELETE FROM notifications WHERE timestamp < $time;";
    Database_SQLite::exec ($this->db, $query);
  }


  public function optimize () 
  {
    Database_SQLite::exec ($this->db, "REINDEX teams;");
    Database_SQLite::exec ($this->db, "VACUUM teams;");
    Database_SQLite::exec ($this->db, "REINDEX users;");
    Database_SQLite::exec ($this->db, "VACUUM users;");
    Database_SQLite::exec ($this->db, "REINDEX notifications;");
    Database_SQLite::exec ($this->db, "VACUUM notifications;");
  }


  // Code dealing with the "teams" table.
  

  // Returns true if diff data exists for the chapter.
  // Else it returns false.
  public function teamDiffExists ($bible, $book, $chapter)
  {
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $query = "SELECT count(*) FROM teams WHERE bible = '$bible' AND book = $book AND chapter = $chapter;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $exists = (boolean) $row [0];
      return $exists;
    }
    return false;
  }


  // Stores diff data for a "bible" (string) and $book (int) and $chapter (int).
  public function storeTeamDiff ($bible, $book, $chapter)
  {
    // Return when the diff exists.
    if ($this->teamDiffExists ($bible, $book, $chapter)) {
      return;
    }

    // Retrieve current chapter USFM data.
    $database_bibles = Database_Bibles::getInstance ();
    $data = $database_bibles->getChapter ($bible, $book, $chapter);

    // Store.
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $query = "INSERT INTO teams (bible, book, chapter, data) VALUES ('$bible', $book, $chapter, '$data');";
    Database_SQLite::exec ($this->db, $query);
  }


  // Gets the diff data as a string.
  public function getTeamDiff ($bible, $book, $chapter)
  {
    if (is_numeric ($bible)) {
      $database_bibles = Database_Bibles::getInstance ();
      $bible = $database_bibles->getName ($bible);
    }
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $query = "SELECT data FROM teams WHERE bible = '$bible' AND book = $book AND chapter = $chapter;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row [0];
    }
    return "";
  }


  // Stores diff data for all chapters in a "bible" (string) and $book (int).
  public function storeTeamDiffBook ($bible, $book, $transaction = true)
  {
    $database_bibles = Database_Bibles::getInstance ();
    $chapters = $database_bibles->getChapters ($bible, $book);
    if ($transaction) Database_SQLite::exec ($this->db, "BEGIN;");
    foreach ($chapters as $chapter) {
      $this->storeTeamDiff ($bible, $book, $chapter);
    }
    if ($transaction) Database_SQLite::exec ($this->db, "COMMIT;");
  }


  // Stores diff data for all books in a "bible" (string).
  public function storeTeamDiffBible ($bible)
  {
    $database_bibles = Database_Bibles::getInstance ();
    $books = $database_bibles->getBooks ($bible);
    Database_SQLite::exec ($this->db, "BEGIN;");
    foreach ($books as $book) {
      $this->storeTeamDiffBook ($bible, $book, false);
    }
    Database_SQLite::exec ($this->db, "COMMIT;");
  }


  // Deletes the diffs for a whole Bible.
  public function deleteTeamDiffBible ($bible)
  {
    if (is_numeric ($bible)) {
      $database_bibles = Database_Bibles::getInstance ();
      $bible = $database_bibles->getName ($bible);
    }
    $bible = Database_SQLiteInjection::no ($bible);
    $query = "DELETE FROM teams WHERE bible = '$bible';";
    Database_SQLite::exec ($this->db, $query);
  }


  // Deletes the diffs for one chapter of a Bible.
  public function deleteTeamDiffChapter ($bible, $book, $chapter)
  {
    if (is_numeric ($bible)) {
      $database_bibles = Database_Bibles::getInstance ();
      $bible = $database_bibles->getName ($bible);
    }
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $query = "DELETE FROM teams WHERE bible = '$bible' AND book = $book AND chapter = $chapter;";
    Database_SQLite::exec ($this->db, $query);
  }


  // Returns an array with the available chapters that have diff data in a $book in a Bible.
  public function getTeamDiffChapters ($bible, $book)
  {
    if (is_numeric ($bible)) {
      $database_bibles = Database_Bibles::getInstance ();
      $bible = $database_bibles->getName ($bible);
    }
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapters = array ();
    $query = "SELECT DISTINCT chapter FROM teams WHERE bible = '$bible' AND book = $book ORDER BY chapter ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $chapters [] = $row [0];
    }
    return $chapters;
  }


  // Returns an array with the available books that have diff data in a Bible.
  // The $bible can be a name, or an identifier. This is because the $bible identifier may no longer exist.
  public function getTeamDiffBooks ($bible)
  {
    if (is_numeric ($bible)) {
      $database_bibles = Database_Bibles::getInstance ();
      $bible = $database_bibles->getName ($bible);
    }
    $bible = Database_SQLiteInjection::no ($bible);
    $books = array ();
    $query = "SELECT DISTINCT book FROM teams WHERE bible = '$bible' ORDER BY book ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $books[] = $row[0];
    }
    return $books;
  }


  // Returns an array with the available Bibles that have diff data.
  public function getTeamDiffBibles ()
  {
    $bibles = array ();
    $query = "SELECT DISTINCT bible FROM teams ORDER BY bible ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $bibles [] = $row [0];
    }
    return $bibles;
  }


  // Truncates all team modifications.
  public function truncateTeams ()
  {
    $query = "DELETE FROM teams;";
    Database_SQLite::exec ($this->db, $query);
  }


  // Code dealing with the "users" table.


  public function recordUserSave ($username, $bible, $book, $chapter, $oldID, $oldText, $newID, $newText)
  {
    $username = Database_SQLiteInjection::no ($username);
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $oldID = Database_SQLiteInjection::no ($oldID);
    $oldText = Database_SQLiteInjection::no ($oldText);
    $newID = Database_SQLiteInjection::no ($newID);
    $newText = Database_SQLiteInjection::no ($newText);
    $time = time ();
    $query = "INSERT INTO users VALUES ('$username', $time, '$bible', $book, $chapter, $oldID, '$oldText', '$newID', '$newText');";
    Database_SQLite::exec ($this->db, $query);
  }


  public function clearUserUser ($username)
  {
    $username = Database_SQLiteInjection::no ($username);
    $query = "DELETE FROM users WHERE username = '$username';";
    Database_SQLite::exec ($this->db, $query);
  }


  public function getUserUsernames ()
  {
    $usernames = array ();
    $query = "SELECT DISTINCT username FROM users;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $usernames [] = $row [0];
    }
    return $usernames;
  }


  public function getUserBibles ($username)
  {
    $username = Database_SQLiteInjection::no ($username);
    $bibles = array ();
    $query = "SELECT DISTINCT bible FROM users WHERE username = '$username';";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $bibles [] = $row [0];
    }
    return $bibles;
  }


  public function getUserBooks ($username, $bible)
  {
    $username = Database_SQLiteInjection::no ($username);
    $bible = Database_SQLiteInjection::no ($bible);
    $books = array ();
    $query = "SELECT DISTINCT book FROM users WHERE username = '$username' AND bible = '$bible' ORDER BY book ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $books [] = $row [0];
    }
    return $books;
  }


  public function getUserChapters ($username, $bible, $book)
  {
    $username = Database_SQLiteInjection::no ($username);
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapters = array ();
    $query = "SELECT DISTINCT chapter FROM users WHERE username = '$username' AND bible = '$bible' AND book = $book ORDER BY chapter ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $chapters [] = $row [0];
    }
    return $chapters;
  }


  public function getUserIdentifiers ($username, $bible, $book, $chapter)
  {
    $username = Database_SQLiteInjection::no ($username);
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $ids = array ();
    $query = "SELECT oldid, newid FROM users WHERE username = '$username' AND bible = '$bible' AND book = $book AND chapter = $chapter ORDER BY oldid ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      unset ($row[0]);
      unset ($row[1]);
      $ids [] = $row;
    }
    return $ids;
  }


  public function getUserChapter ($username, $bible, $book, $chapter, $newID)
  {
    $username = Database_SQLiteInjection::no ($username);
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $newID = Database_SQLiteInjection::no ($newID);
    $query = "SELECT oldtext, newtext FROM users WHERE username = '$username' AND bible = '$bible' AND book = $book AND chapter = $chapter AND newid = $newID;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      unset ($row[0]);
      unset ($row[1]);
      return $row;
    }
    return array ();
  }


  public function getUserTimestamp ($username, $bible, $book, $chapter, $newID)
  {
    $username = Database_SQLiteInjection::no ($username);
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $newID = Database_SQLiteInjection::no ($newID);
    $query = "SELECT timestamp FROM users WHERE username = '$username' AND bible = '$bible' AND book = $book AND chapter = $chapter AND newid = $newID;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row [0];
    }
    return time ();
  }


  // Code dealing with the "notifications" table.
 

  public function recordNotification ($users, $category, $bible, $book, $chapter, $verse, $oldtext, $modification, $newtext)
  {
    $bible = Database_SQLiteInjection::no ($bible);
    $category = Database_SQLiteInjection::no ($category);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    if ($verse == "") $verse = 0;
    $verse = Database_SQLiteInjection::no ($verse);
    $oldtext = Database_SQLiteInjection::no ($oldtext);
    $modification = Database_SQLiteInjection::no ($modification);
    $newtext = Database_SQLiteInjection::no ($newtext);
    // Normally this function is called just after midnight.
    // It would then put the current time on changes made the day before.
    // Make a correction for that by subtracting 6 hours.
    $timestamp = time () - 21600;
    foreach ($users as $user) {
      $user = Database_SQLiteInjection::no ($user);
      $query = "INSERT INTO notifications VALUES ($timestamp, '$user', '$category', $bible, $book, $chapter, $verse, '$oldtext', '$modification', '$newtext');";
      Database_SQLite::exec ($this->db, $query);
    }
  }


  public function getNotificationIdentifiers ($username = "")
  {
    $ids = array ();
    $query = "SELECT rowid FROM notifications ";
    if ($username != "") {
      $username = Database_SQLiteInjection::no ($username);
      $query .= " WHERE username = '$username' ";
    }
    // Sort on reference, so that related change notifications are near each other.
    $query .= " ORDER BY book ASC, chapter ASC, verse ASC, rowid ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $ids [] = $row [0];
    }
    return $ids;
  }


  // This gets the identifiers of the personal change proposals.
  // For easier comparison, it also gets the identifiers of the changes
  // in the verses that have personal change proposals.
  public function getNotificationPersonalIdentifiers ($username, $category)
  {
    $username = Database_SQLiteInjection::no ($username);

    // Get all personal change proposals.
    $personalIDs = array ();
    $query = "SELECT rowid FROM notifications WHERE username = '$username' AND category = '$category' ORDER BY book ASC, chapter ASC, verse ASC, rowid ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $personalIDs [] = $row [0];
    }

    $allIDs = array ();

    // Go through each of the personal change proposals.
    foreach ($personalIDs as $personalID) {
      // Add the personal change proposal to the results.
      $allIDs [] = $personalID;
      // Get the Bible and passage for this change proposal.
      $bibleID = $this->getNotificationBible ($personalID);
      $passage = $this->getNotificationPassage ($personalID);
      $book = $passage ['book'];
      $chapter = $passage ['chapter'];
      $verse = $passage ['verse'];
      // Look for change proposals for this Bible and passage.
      $query = "SELECT rowid FROM notifications WHERE username = '$username' AND bible = $bibleID AND book = $book AND chapter = $chapter AND verse = $verse ORDER BY rowid ASC;";
      $result = Database_SQLite::query ($this->db, $query);
      foreach ($result as $row) {
        $id = $row [0];
        // Add the identifier if it's not yet in.
        if (!in_array ($id, $allIDs)) {
          $allIDs [] = $id;
        }
      }
    }

    return $allIDs;
  }


  // This gets the identifiers of the team's changes.
  public function getNotificationTeamIdentifiers ($username, $category)
  {
    $username = Database_SQLiteInjection::no ($username);
    $ids = array ();
    $query = "SELECT rowid FROM notifications WHERE username = '$username' AND category = '$category' ORDER BY book ASC, chapter ASC, verse ASC, rowid ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $ids [] = $row [0];
    }
    return $ids;
  }


  public function deleteNotification ($id)
  {
    $id = Database_SQLiteInjection::no ($id);
    $query = "DELETE FROM notifications WHERE rowid = $id;";
    Database_SQLite::exec ($this->db, $query);
  }


  public function getNotificationTimeStamp ($id)
  {
    $id = Database_SQLiteInjection::no ($id);
    $query = "SELECT timestamp FROM notifications WHERE rowid = $id;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row [0];
    }
    return time ();
  }


  public function getNotificationCategory ($id)
  {
    $id = Database_SQLiteInjection::no ($id);
    $query = "SELECT category FROM notifications WHERE rowid = $id;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row [0];
    }
    return NULL;
  }


  public function getNotificationBible ($id)
  {
    $id = Database_SQLiteInjection::no ($id);
    $query = "SELECT bible FROM notifications WHERE rowid = $id;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row [0];
    }
    return 0;
  }


  public function getNotificationPassage ($id)
  {
    $id = Database_SQLiteInjection::no ($id);
    $query = "SELECT book, chapter, verse FROM notifications WHERE rowid = $id;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      unset ($row[0]);
      unset ($row[1]);
      unset ($row[2]);
      return $row;
    }
    return NULL;
  }


  public function getNotificationOldText ($id)
  {
    $id = Database_SQLiteInjection::no ($id);
    $query = "SELECT oldtext FROM notifications WHERE rowid = $id;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row [0];
    }
    return "";
  }


  public function getNotificationModification ($id)
  {
    $id = Database_SQLiteInjection::no ($id);
    $query = "SELECT modification FROM notifications WHERE rowid = $id;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row [0];
    }
    return NULL;
  }


  public function getNotificationNewText ($id)
  {
    $id = Database_SQLiteInjection::no ($id);
    $query = "SELECT newtext FROM notifications WHERE rowid = $id;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row [0];
    }
    return "";
  }


  public function clearNotificationsUser ($username)
  {
    $username = Database_SQLiteInjection::no ($username);
    $query = "DELETE FROM notifications WHERE username = '$username';";
    Database_SQLite::exec ($this->db, $query);
  }


  // This function deletes personal change proposals and their matching change notifications.
  public function clearNotificationMatches ($username, $personal, $team)
  {
    // Clean input.
    $personal = Database_SQLiteInjection::no ($personal);
    $team = Database_SQLiteInjection::no ($team);

    // Select all identifiers of the personal change proposals.
    $query = "SELECT rowid FROM notifications WHERE username = '$username' AND category = '$personal';";

    $personals = array ();
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $personals [] = $row [0];
    }
    
    // Matches to be deleted.
    $deletes = array ();

    // Go through each of the personal change proposals.
    foreach ($personals as $personalID) {
      $bible = $this->getNotificationBible ($personalID);
      $passage = $this->getNotificationPassage ($personalID);
      $book = $passage ['book'];
      $chapter = $passage ['chapter'];
      $verse = $passage ['verse'];
      $modification = $this->getNotificationModification ($personalID);
      $modification = Database_SQLiteInjection::no ($modification);
      // Get all matching identifiers from the team's change notifications.
      $query = "SELECT rowid FROM notifications WHERE username = '$username' AND category = '$team' AND bible = $bible AND book = $book AND chapter = $chapter AND verse = $verse AND modification = '$modification';";
      $teamMatches = array ();
      $result = Database_SQLite::query ($this->db, $query);
      foreach ($result as $row) {
        $teamMatches [] = $row [0];
      }
      // There should be exactly one candidate for the matches to be removed.
      // If there are none, it means that the personal change didn't make it to the team's text.
      // If there are two or more matching changes, then one could have undone the other, so should not be automatically removed.
      if (count ($teamMatches) == 1) {
        foreach ($teamMatches as $teamMatches) {
          // Check there are only two change notifications for this user / Bible / book / chapter / verse.
          // If there are more, we can't be sure that the personal change proposal was not overwritten somehow.
          $passageMatches = array ();
          $query = "SELECT rowid FROM notifications WHERE username = '$username' AND bible = $bible AND book = $book AND chapter = $chapter AND verse = $verse;";
          $result = Database_SQLite::query ($this->db, $query);
          foreach ($result as $row) {
            $passageMatches [] = $row [0];
          }
          if (count ($passageMatches) == 2) {
            // Store the personal change proposal to be deleted.
            // Store the matching change notification to be deleted also.
            foreach ($passageMatches as $passageMatche) {
              $deletes [] = $passageMatche;
            }
          }
        }
      }
    }

    // Delete all stored identifiers to be deleted.
    foreach ($deletes as $delete) {
      $query = "DELETE FROM notifications WHERE rowid = $delete;";
      Database_SQLite::exec ($this->db, $query);
    }
  }


  public function debug ()
  {
    $result = Database_SQLite::query ($this->db, "SELECT * from notifications;");
    foreach ($result as $row) {
      var_dump ($row);
    }
  }


}


?>
