package Bibulus::LaTeX;

# For copyright and license please see the final part of this file.

use strict;
use warnings;
use Carp;
use Bibulus;

our @ISA = qw(Bibulus);

# Debugging is a package global:
my $DEBUG = 0;
sub setdebug {
  $DEBUG = 1;
  print STDERR 'This is Bibulus::LaTeX $Id: LaTeX.pm,v 1.10 2003/09/09 20:20:59 twid Exp $', "\n";
  Bibulus->setdebug;
}

# The following translates an ISO language code into a babel language
# name.  Often there is more than one possible name, in which case
# I have chosen one more or less at random.
#
# Problem: There are many languages that can be typeset with LaTeX but
# that aren't handled by the babel system -- what should be done in
# these cases?
my %iso2babel =
  (
   'af' => 'afrikaans',
   'bg' => 'bulgarian',
   'br' => 'breton',
   'ca' => 'catalan',
   'cs' => 'czech',
   'cy' => 'welsh',
   'da' => 'danish',
   'de' => 'ngermanb',
   'de_DE' => 'ngermanb',
   'de_AT' => 'naustrian',
   'el' => 'greek',
   'en' => 'english',
   'en_CA' => 'canadian',
   'en_UK' => 'british',
   'en_US' => 'american',
   'eo' => 'esperanto',
   'es' => 'spanish',
   'et' => 'estonian',
   'eu' => 'basque',
   'fi' => 'finnish',
   'fr' => 'frenchb',
   'fr_CA' => 'canadien',
   'ga' => 'irish',
   'gd' => 'scottish',
   'gl' => 'galician',
   'he' => 'hebrew',
   'hr' => 'croatian',
   'hu' => 'hungarian',
   'hu' => 'magyar',
   'id' => 'bahasa',
   'is' => 'icelandic',
   'it' => 'italian',
   'la' => 'latin',
   'nl' => 'dutch',
   'nn' => 'nynorsk',
   'nb' => 'norsk',
   'pl' => 'polish',
   'pt' => 'portuguese',
   'pt_BR' => 'brazil',
   'ro' => 'romanian',
   'ru' => 'russian',
   'ru' => 'russianb',
   'sk' => 'slovak',
   'sl' => 'slovene',
   'smi' => 'samin',
   'sr' => 'serbian',
   'sv' => 'swedish',
   'tr' => 'turkish',
   'uk' => 'ukraineb',
   'wen' => 'usorbian',	# or 'lsorbian',
  );

# The following translates babel language names into ISO language codes.
my %babel2iso =
  (
   UKenglish => 'en_UK',
   acadian => 'fr_CA',
   afrikaans => 'af',
   american => 'en_US',
   austrian => 'de_AT',
   bahasa => 'id',
   basque => 'eu',
   brazil => 'pt_BR',
   brazilian => 'pt_BR',
   breton => 'br',
   british => 'en_UK',
   bulgarian => 'bg',
   canadian => 'en_CA',
   canadien => 'fr_CA',
   catalan => 'ca',
   croatian => 'hr',
   czech => 'cs',
   danish => 'da',
   dutch => 'nl',
   english => 'en',
   esperanto => 'eo',
   estonian => 'et',
   finnish => 'fi',
   francais => 'fr',
   french => 'fr',
   frenchb => 'fr',
   galician => 'gl',
   german => 'de',
   germanb => 'de',
   greek => 'el',
   hebrew => 'he',
   hungarian => 'hu',
   icelandic => 'is',
   irish => 'ga',
   italian => 'it',
   latin => 'la',
   lsorbian => 'wen', # ISO code for Sorbian, Lower variant desig. unknown
   magyar => 'hu',
   naustrian => 'de_AT',
   ngerman => 'de',
   ngermanb => 'de',
   norsk => 'nb', # Bokmaal
   nynorsk => 'nn',
   polish => 'pl',
   portuges => 'pt',
   portuguese => 'pt',
   romanian => 'ro',
   russian => 'ru',
   russianb => 'ru',
   samin => 'smi', # this really is North Sami, ISO code unknown
   scottish => 'gd',
   serbian => 'sr',
   slovak => 'sk',
   slovene => 'sl',
   spanish => 'es',
   swedish => 'sv',
   turkish => 'tr',
   ukraineb => 'uk',
   ukrainian => 'uk',
   usorbian => 'wen', # ISO code for Sorbian, Upper variant desig. unknown
   welsh => 'cy',
  );

# Parse the .aux file
sub procaux {
  my $self = shift;
  my ($filename) = @_;
  $filename =~ s/^(.*)(\.aux)$/$1/i;
  print "The top-level-auxiliary file: $filename.aux\n" if $self->{VERBOSE};
  open AUX, "<$filename.aux" or croak "Cannot open $filename.aux";
  $self->{FILENAME} = $filename;

  $self->{STYLE} = undef;
  $self->{CITE} = [];
  $self->{CITATION} = [];

  while (<AUX>) {
    if (/^\\citation\{\*\}$/) {
      $self->citeall;

    } elsif (/^\\citation\{(.*)\}$/) {
      $self->cite($1);

    } elsif (/^\\bibstyle\{(.*)\}$/) {
      defined($self->{STYLE}) and carp "Duplicate \\bibstyle";
      $self->{STYLE} = $1;
      print "The style file: $1.bst\n" if $self->{VERBOSE};
      # The various existing BibTeX styles should be defined here.
      # (At some point, it might more sense to put all of this into
      # an external file, but at present it's OK just to do it here.
      if ($1 eq 'plain') {
	$self->{STYLE} = {
			  cite => 'numerical', # undef might be better
#			  titlefont => "emph",
			  givenbefore => 1,
			  # More definitions are needed, of course!
			 };
      } else {
	croak "Unknown style '$1'";
      }

    } elsif (/^\\bibdata\{(.*)\}$/) {
      exists($self->{DATA}) and croak "Duplicate \\bibdata";
      $self->{DATA} = [split(/\s*,\s*/, $1)];
      print STDERR "bibdata: ", join('; ', @{$self->{DATA}}), "\n" if $DEBUG;

    } elsif (/^\\b\@bulus\ ?\{(.*)\}$/) {
      defined($self->{STYLE}) and carp "Duplicate style definitions (perhaps two \\bibulus commands,\nor both \\bibulus and \\bibstyle)";
      my $x = $1;
      my @x = split(/\ *\,\ */, $x);
      my %x;
      foreach $x (@x) {
	if ($x =~ /(.*)\=(.*)/) {
	  $x{$1} = $2;
	} else {
	  $x{$x} = 1;
	}
      }
      $self->{STYLE} = {%x};
      $DEBUG and print STDERR "bibulus:\n  ", join("\n  ",
				     map {
				       "$_: $x{$_}"
				     } keys %x),
				       "\n";

    } elsif (/^\\bibcite\{(.*?)\}\{(.*)\}$/) {
      push @{$self->{CITE}}, [$1, $2];
      print STDERR "bibcite: <$1, $2>\n" if $DEBUG;

    } else {
      print STDERR "Ignoring aux line $_" if $DEBUG;
    }
  }
  close AUX;

  croak "No \\bibdata" unless $self->{DATA};
  croak "No style defined with either \bibulus or \bibliographystyle"
     unless $self->{STYLE};

  foreach my $f (@{$self->{DATA}}) {
    unless (-e $f) {
      $f .= '.xml';
      unless (-e $f) {
	croak "$f not found";
      }
    }
    $self->load($f);
  }
}

# Now we override Bibulus built-ins to output LaTeX

sub bibliography_start {
  my $self = shift;
  my $w = $self->widest_label;
  return "\\begin{thebibliography}{$w}\n";
}

# find the widest label (in characters)
# [It would be great with a better algorithm which takes into account
# that certain letters are much wider than others.]
sub widest_label {
  my $self = shift;
  my $max = '';
  foreach my $i (@{$self->{EL}}) {
    my $l = $i->first_child('label')->text;
    length($l) > length($max) and $max = $l;
  }

  return $max;
}

sub bibliography_end {
  return "\\end{thebibliography}\n";
}

sub item_start {
  my $self = shift;
  my ($id, $label) = @_;
  my $t;
  $t = "\\bibitem";
  defined($label) and $t .= "[$label]";
  $t .= "{$id}\n";
  return $t;
}

sub item_end {
  return "\n";
}

sub newblock {
  my $self = shift;
  return "\n\\newblock ";
}

# return text in given language
sub language {
  my $self = shift;
  my ($lang, $t) = @_;
  my $r;
  if ($lang and defined($iso2babel{$lang})) {
    # also add a test for the bibliography language (perhaps no need
    # to switch)
    $r = "\\foreignlanguage{$iso2babel{$lang}}{$t}";
  } else {
    warn "Unknown language code $lang" if $lang;
    $r = $t;
  }

  return $r;
}

sub emph {
  my $self = shift;
  my ($t) = @_;
  return "\\emph{$t}";
}

sub bold {
  my $self = shift;
  my ($t) = @_;
  return "\\textbf{$t}";
}

sub smallcaps {
  my $self = shift;
  my ($t) = @_;
  return "\\textsc{$t}";
}

# Should this be done using a LaTeX command instead?
sub fullcaps {
  my $self = shift;
  my ($t) = @_;
  return uc($t);
}

# this is very English -- other languages use other quotes
sub singlequote {
  my $self = shift;
  my ($t) = @_;
  return "`$t'";
}

# again: this is very English -- other languages use other quotes
sub doublequote {
  my $self = shift;
  my ($t) = @_;
  return "``$t''";
}

sub nobreakspace {
  return '~';
}


1;
__END__

=head1 NAME

Bibulus::LaTeX - Perl extension for interfacing between Bibulus and LaTeX

=head1 SYNOPSIS

  use Bibulus::LaTeX;
  my $bib = new Bibulus::LaTeX;
  $bib->procaux($filename);
  print $bib->getbib;

=head1 DESCRIPTION

A module to let Bibulus work together with LaTeX.  See the examples
in the doc directory.

=head2 EXPORT

None.

=head1 BUGS

BibTeX converts some spaces to nonbreakable ones,
and because one can insert LaTeX commands into the
bibliographic database, one can prevent hyphens
from becoming linebreaks.  Bibulus at the moment
does nothing about this.

=head1 SEE ALSO

The main Bibulus module.

The homepage at present is
E<lt>http://www.nongnu.org/bibulus/E<gt>.

=head1 AUTHOR

Thomas M. Widmann, E<lt>twid@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2003 by Thomas M. Widmann

This module is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

=cut
