/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include "frame.h"

@implementation Frame

-(Frame *)init
{
    [super init];
    
    self->thickness = 1;
    self->radius = 0.003;

    return self;
}

-(float)thickness
{
    return self->thickness;
}

-(float)radius
{
    return self->radius;
}

-(void) get: (lua_State *)L
{
    const char *k;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "thickness")) {
	lua_pushnumber (L, self->thickness);
    } else if (!strcmp(k, "radius")) {
	lua_pushnumber (L, self->radius);
    } else {
	[super get: L];
    }
}

-(void) set: (lua_State *)L
{
    const char *k;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "thickness")) {
	self->thickness = lua_tonumber (L, 3);
    } else if (!strcmp(k, "radius")) {
	self->radius = lua_tonumber (L, 3);
    } else {
	[super set: L];
    }
}

-(void) transform: (lua_State *)L
{
    [super transform: L];

    if([self children]) {
	self->allocated[0] = [[self children] width];
	self->allocated[1] = [[self children] height];
    } else {
	self->allocated[0] = 0;
	self->allocated[1] = 0;
    }

    self->allocated[0] = self->allocated[0] > self->requested[0] ?
	                 self->allocated[0] : self->requested[0];
    self->allocated[1] = self->allocated[1] > self->requested[1] ?
	                 self->allocated[1] : self->requested[1];
}

-(void) cleanup: (lua_State *)L
{
    GLfloat w, h, rho, *r, *R;

    r = [self translation];
    R = [self rotation];
 
    glMatrixMode (GL_MODELVIEW);
    glPushMatrix();
    glMultMatrixf ((GLfloat[16]) {R[0], R[3], R[6], 0,
		                  R[1], R[4], R[7], 0,
		                  R[2], R[5], R[8], 0,
		                  r[0], r[1], r[2], 1});

    rho = self->radius;
    w = self->allocated[0] / 2;
    h = self->allocated[1] / 2;

    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glLineWidth ([self thickness]);

    glEnable(GL_LINE_SMOOTH);
    glEnable(GL_BLEND);

    glDepthMask (GL_FALSE);

    glColor3fv([self color]);

    glBegin(GL_LINE_STRIP);
    glVertex2f(-w + rho, -h);
    glVertex2f(w - rho, -h);

    glVertex2f(w, -h + rho);
    glVertex2f(w, h - rho);
    
    glVertex2f(w - rho, h);
    glVertex2f(-w + rho, h);

    glVertex2f(-w, h - rho);
    glVertex2f(-w, -h + rho);
    
    glVertex2f(-w + rho, -h);
    glEnd();

    glColor4fv ([self color]);
    
    glBegin(GL_POLYGON);
    glVertex2f(-w + rho, -h);
    glVertex2f(w - rho, -h);

    glVertex2f(w, -h + rho);
    glVertex2f(w, h - rho);
    
    glVertex2f(w - rho, h);
    glVertex2f(-w + rho, h);

    glVertex2f(-w, h - rho);
    glVertex2f(-w, -h + rho);
    
    glVertex2f(-w + rho, -h);
    glEnd();

    glDepthMask (GL_TRUE);

    glDisable(GL_BLEND);
    glDisable(GL_LINE_SMOOTH);
    
    glMatrixMode (GL_MODELVIEW);
    glPopMatrix();
    
    [super cleanup: L];
}

@end
