/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <time.h>
#include <unistd.h>
#include <getopt.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <ode/ode.h>

#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include <AL/al.h>
#include <AL/alc.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include <X11/Xatom.h>

#include "node.h"

int simulator_index (lua_State *L);
int simulator_newindex (lua_State *L);

static id head;

static Display *display;
static Window window;
static ALCdevice *device;

static int iterate = 1;
static int hide, reference, frames, grabkeyboard, grabpointer;
static float canvas[3];
static char *title;

static int server, port, pages, mime, block;

static int test (lua_State *L)
{
    return 0;
}

static int panic(lua_State *L)
{
    printf("Hmmm, I just got this unprotected error from "
	   "the Lua interpreter:\n\"%s\"\n"
           "No point in prolonging this dreary existance I suppose...\n",
           lua_tostring(L, -1));

    abort ();
    
    return 0;
}

static Bool wait(Display *d, XEvent *e, XPointer arg) {
    return (e->type == MapNotify) && (e->xmap.window == (Window)arg);
}

static int head_index(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    
    [N get: L];

    return 1;
}

static int head_newindex(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);

    [N set: L];

    return 0;
}

static int graph_index (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "iterate")) {
	lua_pushboolean(L, iterate);
    } else {
	lua_rawgeti (L, LUA_REGISTRYINDEX, reference);
	lua_replace (L, 1);
	lua_gettable(L, 1);
    }

    return 1;
}

static int network_index (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);
    
    if (!strcmp(k, "port")) {
	lua_pushnumber(L, port);
    } else if (!strcmp(k, "pages")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, pages);
    } else if (!strcmp(k, "block")) {
	lua_pushboolean(L, block);
    } else if (!strcmp(k, "mime")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, mime);
    }

    return 1;
}

static int network_newindex (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);
    
    if (!strcmp(k, "port")) {
	if (port > 0) {
	    close (server);
	}
	
	port = lua_tonumber (L, 3);

	if (port > 0) {
	    struct sockaddr_in address;
	    struct sigaction ignore;
	    sigset_t empty;
	    int bound, listening, yes = 1;
    
	    server = socket(AF_INET, SOCK_STREAM, 0);

	    fcntl(server, F_SETFL, block ? 0 : O_NONBLOCK);
	    setsockopt(server, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof(int));

	    sigemptyset(&empty);
	    
	    ignore.sa_handler = SIG_IGN;
	    ignore.sa_mask = empty;
	    ignore.sa_flags = 0;

	    sigaction(SIGPIPE, &ignore, NULL);
    
	    address.sin_family = AF_INET;
	    address.sin_port = htons(port);
	    address.sin_addr.s_addr = INADDR_ANY;
    
	    bound = bind(server, (struct sockaddr *)&address, sizeof (address));
	    listening = listen(server, 10);

	    if (server < 0 || bound < 0 || listening < 0) {
		printf ("Could not setup port %d for http requests.\n", port);

		close(server);
		port = 0;
	    } else {
		printf ("Listening for http requests on port %d.\n", port);
	    }
    	}
    } else if (!strcmp(k, "block")) {
	block = lua_toboolean(L, 3);

	fcntl(server, F_SETFL, block ? 0 : O_NONBLOCK);
    } else if (!strcmp(k, "pages")) {
	luaL_unref (L, LUA_REGISTRYINDEX, pages);
	pages = luaL_ref (L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "mime")) {
	luaL_unref (L, LUA_REGISTRYINDEX, mime);
	mime = luaL_ref (L, LUA_REGISTRYINDEX);
    }

    return 0;
}

static int graph_newindex (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);
    
    if (!strcmp(k, "iterate")) {
	iterate = lua_toboolean (L, 3);
    } else {
	lua_rawgeti (L, LUA_REGISTRYINDEX, reference);
	lua_replace (L, 1);
	lua_settable(L, 1);
    }

    return 0;
}

static int graphics_index (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "window")) {
	XWindowAttributes attributes;
	
	XGetWindowAttributes (display, window, &attributes);

	lua_newtable (L);
	lua_pushnumber (L, attributes.width);
	lua_rawseti (L, -2, 1);
	lua_pushnumber (L, attributes.height);
	lua_rawseti (L, -2, 2);
    } else if (!strcmp(k, "title")) {
	lua_pushstring (L, title);
    } else if (!strcmp(k, "canvas")) {
	lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, canvas[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!strcmp(k, "perspective") ||
	!strcmp(k, "orthographic")) {
	lua_gettable (L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "frames")) {
	lua_pushnumber (L, frames);
    } else if (!strcmp(k, "pointer")) {
	Window root, child;
	int x, y, a, b;
	unsigned int mask;

	XQueryPointer (display, window, &root, &child,
		       &a, &b, &x, &y, &mask);
	
	lua_newtable (L);
	lua_pushnumber (L, x);
	lua_rawseti (L, -2, 1);
	lua_pushnumber (L, y);
	lua_rawseti (L, -2, 2);
    } else if (!strcmp(k, "hide")) {
	lua_pushboolean (L, hide);
    } else if (!strcmp(k, "grabkeyboard")) {
	lua_pushboolean (L, grabkeyboard);
    } else if (!strcmp(k, "grabpointer")) {
	lua_pushboolean (L, grabpointer);
    } else if (!strcmp(k, "colorbuffer")) {
        char *pixels;
        int v[4];

	glGetIntegerv (GL_VIEWPORT, v);
	pixels = malloc(v[2] * v[3] * 3);

	glReadBuffer (GL_BACK);
	glReadPixels(v[0], v[1], v[2], v[3], GL_RGB, GL_UNSIGNED_BYTE, pixels);

        lua_pushlstring(L, pixels, v[2] * v[3] * 3);

        free(pixels);
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int graphics_newindex (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);
    
    if (!strcmp(k, "window")) {
	XSizeHints *hints;
	int w[2];
	
	if (!lua_isnil (L, -1)) {
	    Screen *screen;
	    Atom state, normal, fullscreen;
	    
	    for (i = 0 ; i < 2 ; i += 1) {
		lua_rawgeti (L, 3, i + 1);
		w[i] = lua_tonumber(L, -1);
		lua_pop (L, 1);
	    }

	    screen = DefaultScreenOfDisplay (display);
	    state = XInternAtom (display, "_NET_WM_STATE", False);
	    normal = XInternAtom (display, "_NET_WM_STATE_NORMAL", False);
	    fullscreen = XInternAtom (display,
				      "_NET_WM_STATE_FULLSCREEN",
				      False);

	    hints = XAllocSizeHints();
	    hints->flags = PMinSize | PMaxSize;
	    hints->min_width = w[0];
	    hints->min_height = w[1];
	    hints->max_width = w[0];
	    hints->max_height = w[1];

	    XSetWMNormalHints (display, window, hints);
	    XFree (hints);

	    XResizeWindow(display, window, w[0], w[1]);

	    if (w[0] == WidthOfScreen(screen) &&
		w[1] == HeightOfScreen(screen)) {
		XChangeProperty (display, window,
				 state, XA_ATOM, 32,
				 PropModeReplace,
				 (unsigned char *)&fullscreen, 1);
	    } else {
		XChangeProperty (display, window,
				 state, XA_ATOM, 32,
				 PropModeReplace,
				 (unsigned char *)&normal, 1);
	    }
	}
	
	lua_settable (L, LUA_REGISTRYINDEX);
    } else if (!strcmp(k, "title")) {
	Atom name;

	title =(char *) lua_tostring (L, 3);
	name = XInternAtom (display, "WM_NAME", False);

	XChangeProperty (display, window,
			 name, XA_STRING, 8,
			 PropModeReplace,
			 (unsigned char *)title, strlen (title));
    } else if (!strcmp(k, "perspective")) {
	GLdouble p[6];

	for (i = 0 ; i < 6 ; i += 1) {
	    lua_rawgeti (L, 3, i + 1);
	    p[i] = lua_tonumber(L, -1);
	    lua_pop (L, 1);
	}

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
 	glFrustum (p[0], p[1], p[2], p[3], p[4], p[5]);
	
	lua_settable (L, LUA_REGISTRYINDEX);

	lua_pushnil (L);
	lua_setfield (L, LUA_REGISTRYINDEX, "orthographic");
    } else if (!strcmp(k, "orthographic")) {
	GLdouble p[4];

	for (i = 0 ; i < 4 ; i += 1) {
	    lua_rawgeti (L, 3, i + 1);
	    p[i] = lua_tonumber(L, -1);
	    lua_pop (L, 1);
	}

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
 	gluOrtho2D (p[0], p[1], p[2], p[3]);
	
	lua_settable (L, LUA_REGISTRYINDEX);

	lua_pushnil (L);
	lua_setfield (L, LUA_REGISTRYINDEX, "perspective");
    } else if (!strcmp(k, "canvas")) {
	for (i = 0 ; i < 3 ; i += 1) {
	    lua_rawgeti (L, 3, i + 1);
	    canvas[i] = lua_tonumber(L, -1);
	    lua_pop (L, 1);
	}

 	glClearColor (canvas[0], canvas[1], canvas[2], 0);
    } else if (!strcmp(k, "hide")) {
	XEvent event;

	hide = lua_toboolean(L, 3);

	if (hide) {
	    XUnmapWindow (display, window);
	} else {
	    XMapWindow (display, window);
	    XIfEvent(display, &event, wait, (XPointer)window);

	    if (grabkeyboard) {
		XGrabKeyboard (display, window, True,
			       GrabModeAsync, GrabModeAsync,
			       CurrentTime);
	    }

	    if (grabpointer) {
		XGrabPointer (display, window, True, 0,
			      GrabModeAsync, GrabModeAsync,
			      window, None, CurrentTime);
	    }
	}
    } else if (!strcmp(k, "grabkeyboard")) {
	grabkeyboard = lua_toboolean(L, 3);

	if(window && grabkeyboard) {
	    XGrabKeyboard (display, window, True,
			   GrabModeAsync, GrabModeAsync,
			   CurrentTime);
	} else {
	    XUngrabKeyboard (display, CurrentTime);
	}
    } else if (!strcmp(k, "grabpointer")) {
	grabpointer = lua_toboolean(L, 3);

	if(window && grabpointer) {
	    XGrabPointer (display, window, True, 0,
			  GrabModeAsync, GrabModeAsync,
			  window, None, CurrentTime);
	} else {
	    XUngrabPointer (display, CurrentTime);
	}
    } else {
	lua_rawset (L, 1);
    }
    
    return 0;
}

int main(int argc, char **argv)
{
    lua_State *L;

    GLXContext graphics;
    XVisualInfo *info;
    Colormap colormap;
    XSetWindowAttributes attributes;

    ALCcontext *audio;
    ALenum error;

    char *script = "scripts/configuration.lua";
    int option;
    
    int configuration[] = {
	GLX_RGBA,
	GLX_DOUBLEBUFFER,
	GLX_RED_SIZE, 1,
	GLX_GREEN_SIZE, 1,
	GLX_BLUE_SIZE, 1,
	GLX_ALPHA_SIZE, 0,
	GLX_DEPTH_SIZE, 12,
	GLX_STENCIL_SIZE, 1,
	None
    };

    int parameters[] = {0, 0, 0};

    static struct option options[] = {
	{"option", 1, 0, 'O'},
	{"red", 1, 0, 'r'},
	{"green", 0, 0, 'g'},
	{"blue", 1, 0, 'b'},
	{"alpha", 1, 0, 'a'},
	{"depth", 1, 0, 'z'},
	{"stencil", 1, 0, 's'},
	{"frequency", 1, 0, 'f'},
	{"interval", 1, 0, 'i'},
	{"configuration", 1, 0, 'c'},
	{"help", 0, 0, 'h'},
	{0, 0, 0, 0}
    };
    
    /* Create the Lua state. */

    L = luaL_newstate();
    
    lua_atpanic(L, panic);

    luaL_openlibs(L);
    lua_settop(L, 0);

    lua_pushcfunction(L, test);
    lua_setfield(L, LUA_GLOBALSINDEX, "test");

    /* Modify the Lua path and cpath. */

    lua_getglobal (L, "package");
    lua_getfield (L, -1, "path");
    lua_pushstring(L, ";./?.lua;./?.lc;./scripts/?.lua;./scripts/?.lc");
    lua_concat (L, 2);
    lua_setfield (L, -2, "path");
    lua_pop (L, 1);

    lua_getglobal (L, "package");
    lua_getfield (L, -1, "cpath");
    lua_pushstring(L, "./?.so;./scripts/?.so");
    lua_concat (L, 2);
    lua_setfield (L, -2, "cpath");
    lua_pop (L, 1);
    
    /* Create the options table. */

    lua_newtable (L);
    
    /* Parse the command line. */
    
    while ((option = getopt_long (argc, argv,
				  "O:r:g:b:a:z:s:f:i:c:h",
				  options, 0)) != -1) {
	if (option == 'O') {
	    lua_pushstring (L, optarg);
	    lua_pushboolean (L, 1);
	    lua_settable(L, -3);
	} else if (option == 'r') {
	    configuration[3] = atoi(optarg);
	} else if (option == 'g') {
	    configuration[5] = atoi(optarg);
	} else if (option == 'b') {
	    configuration[7] = atoi(optarg);
	} else if (option == 'a') {
	    configuration[9] = atoi(optarg);
	} else if (option == 'z') {
	    configuration[11] = atoi(optarg);
	} else if (option == 's') {
	    configuration[13] = atoi(optarg);
	} else if (option == 'f') {
	    if (parameters[0] == 0) {
		parameters[0] = ALC_FREQUENCY;
		parameters[1] = atoi(optarg);
	    } else {
		parameters[2] = ALC_FREQUENCY;
		parameters[3] = atoi(optarg);
	    }
	} else if (option == 'i') {
	    if (parameters[0] == 0) {
		parameters[0] = ALC_REFRESH;
		parameters[1] = atoi(optarg);
	    } else {
		parameters[2] = ALC_REFRESH;
		parameters[3] = atoi(optarg);
	    }
	    
	} else if (option == 'c') {
	    script = optarg;
	} else if (option == 'h') {
	    printf ("Usage: %s [OPTION...]\n\n"
		    "Options:\n"
		    "  -h, --help                        "
		    "Display this help message.\n\n"
		    "General options:\n"
                    "  -c FILE, --configuration=FILE     "
		    "Specify the configuration script.\n"
		    "  -O OPTION, --option=OPTION        "
		    "Set the option OPTION.  (Consult the\n"
		    "                                    "
		    "documentation for a list of options\n"
		    "                                    "
		    "and their meaning.)\n\n"
		    "Audio related options:\n"
                    "  -f FREQ, --frequency=FREQ         "
		    "Specify the audio sampling rate in Hz.\n"
                    "  -i INT, --interval=INT            "
		    "Specify the audio refresh interval in Hz.\n\n"
		    "Video related options:\n"
                    "  -r BITS, --red=BITS               "
		    "Set the red color component depth.\n"
                    "  -g BITS, --green=BITS             "
		    "Set the green color component depth.\n"
                    "  -b BITS, --blue=BITS              "
		    "Set the blue color component depth.\n"
                    "  -a BITS, --alpha=BITS             "
		    "Set the alpha color component depth.\n"
                    "  -s BITS, --stencil=BITS           "
		    "Set the stencil buffer depth.\n"
                    "  -z BITS, --depth=BITS             "
		    "Set the Z-buffer depth.\n", argv[0]);

	    exit(1);
	} else {
	    exit(1);
	}
    }

    lua_setglobal (L, "options");

    /* Create the head node. */

    head = [[Node alloc] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = head;
    
    lua_newtable (L);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)head_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)head_newindex);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    reference = luaL_ref (L, LUA_REGISTRYINDEX);
    [head toggle: L];
    
    /* Create the network table. */
    
    lua_newtable (L);    
    lua_newtable (L);

    lua_pushcfunction (L, network_index);
    lua_setfield (L, -2, "__index");

    lua_pushcfunction (L, network_newindex);
    lua_setfield (L, -2, "__newindex");

    lua_setmetatable (L, -2);
    
    lua_setglobal (L, "network");
    
    lua_newtable (L);
    pages = luaL_ref (L, LUA_REGISTRYINDEX);
    
    lua_newtable (L);
    mime = luaL_ref (L, LUA_REGISTRYINDEX);

    /* Create the graph table. */
    
    lua_newtable (L);    
    lua_newtable (L);

    lua_pushcfunction (L, graph_index);
    lua_setfield (L, -2, "__index");

    lua_pushcfunction (L, graph_newindex);
    lua_setfield (L, -2, "__newindex");

    lua_setmetatable (L, -2);
    lua_setglobal (L, "graph");

    /* Create the graphics table. */
    
    lua_newtable (L);    
    lua_newtable (L);

    lua_pushcfunction (L, graphics_index);
    lua_setfield (L, -2, "__index");

    lua_pushcfunction (L, graphics_newindex);
    lua_setfield (L, -2, "__newindex");

    lua_setmetatable (L, -2);
    lua_setglobal (L, "graphics");

    /* Create the simulator table. */
    
    lua_newtable (L);    
    lua_newtable (L);
    
    lua_pushcfunction (L, simulator_index);
    lua_setfield (L, -2, "__index");

    lua_pushcfunction (L, simulator_newindex);
    lua_setfield (L, -2, "__newindex");

    lua_setmetatable (L, -2);
    lua_setglobal (L, "simulator");

    /* Create the rendering context and
       associated visual. */

    display = XOpenDisplay(0);
    info = glXChooseVisual(display, DefaultScreen(display), configuration);

    if (!info) {
	fprintf (stderr, "Could not find a suitable visual.\n");
	exit(1);
    }
    
    graphics = glXCreateContext(display, info, 0, GL_TRUE);

    if (!graphics) {
	fprintf (stderr, "Could not create a rendering context.\n");
	exit(1);
    }

    /* Create the window. */

    colormap = XCreateColormap(display, RootWindow(display, info->screen),
			       info->visual, AllocNone);
    
    attributes.colormap = colormap;
    attributes.border_pixel = 0;
    attributes.event_mask = StructureNotifyMask |
	ButtonPressMask | ButtonReleaseMask |
	KeyPressMask | KeyReleaseMask |
	PointerMotionMask;

    window = XCreateWindow(display,
			   RootWindow(display, info->screen),
			   0, 0, 640, 480,
			   0, info->depth, InputOutput, info->visual,
			   CWBorderPixel | CWColormap | CWEventMask,
			   &attributes);
    
    glXMakeCurrent(display, window, graphics);
    
    /* Print useful debug information. */
    
    {
	int r, g, b, a, z, s;

	glXGetConfig (display, info, GLX_RED_SIZE, &r);
	glXGetConfig (display, info, GLX_GREEN_SIZE, &g);
	glXGetConfig (display, info, GLX_BLUE_SIZE, &b);
	glXGetConfig (display, info, GLX_ALPHA_SIZE, &a);
	glXGetConfig (display, info, GLX_DEPTH_SIZE, &z);
	glXGetConfig (display, info, GLX_STENCIL_SIZE, &s);

	printf ("Graphics renderer is '%s %s'.\n",
		glGetString(GL_RENDERER),
		glGetString(GL_VERSION));

	printf ("Framebuffer configuration is "
		"[%d, %d, %d, %d, %d, %d].\n", r, g, b, a, z, s);

	printf ("Rendering context is%sdirect.\n",
		glXIsDirect (display, graphics) == True ? " " : " *not* ");
    }

    XFree(info);
    
    /* Create an openal context. */

    alGetError();

#if 0
    device = alcOpenDevice (NULL);
    error = alGetError();

    if (error) {
	fprintf (stderr, "Could not open the audio device.\n");
    }
    
    audio = alcCreateContext (device, parameters);
    error = alGetError();

    if (error) {
	fprintf (stderr, "Could not create the audio context.\n");
    }

    alcMakeContextCurrent(audio);
    alcProcessContext(audio);
    
    printf ("Audio renderer is '%s %s'.\n",
	    alGetString(AL_RENDERER),
	    alGetString(AL_VERSION));
#endif

    printf ("We're up and running!\nReading the configuration script.\n");
    
    /* Run the main configuration script. */
    
    if(!luaL_loadfile(L, script)) {
	lua_call (L, 0, 0);
    } else {
	lua_error (L);
    }

    printf ("Done configuring, entering the main loop.\n");
    
    /* Enter the main loop. */
    
    do {
	FILE *stream;
	char line[1024], *request, *uri, *version;
	int client;

	/* Check for http requests. */

	client = accept(server, NULL, NULL);

	if (client > 0) {
	    int i, j, n;

	    fcntl(client, F_SETFL, O_NONBLOCK);
	    
	    stream = fdopen (client, "w+");
	    setbuf(stream, NULL);

	    /* Read a whole line and break it up
	       into request, uri and protocol. */
	    
	    while (!fgets (line, 1024, stream));
	    puts (line);
	    
	    request = strtok (line, " \t");
	    uri = strtok (NULL, " \t");
	    version = strtok (NULL, " \t");

	    /* Unescape the uri. */
	    
	    for (i = 0, j = 0;
		 (uri[i] = uri[j]) != '\0';
		 n = sscanf (&uri[j], "%%%2hhx", &uri[i]),
	         j += 2 * n + 1,
		 i += 1);

	    if (!request || !uri || !version) {
		fputs ("HTTP/1.0 400 Bad Request\r\n"
		       "Server: Billiards/0.1\r\n"
		       "Connection: close\r\n"
		       "Content-Type: text/html\r\n"
		       "Content-Length: 0\r\n"
		       "\r\n", stream);
	    } else {
		if (!strcmp (request, "GET")) {
		    int h_0;
		    char *page, *query;

		    h_0 = lua_gettop(L);

		    /* Break up the uri into a page/query pair. */
		    
		    page = strtok (uri, "?");
		    query = strtok (NULL, "");

		    /* Retrieve the MIME type and content for the page. */
		    
		    lua_rawgeti (L, LUA_REGISTRYINDEX, mime);

		    if (lua_istable (L, -1)) {
			lua_pushstring (L, page);
			lua_gettable (L, -2);
			lua_replace (L, -2);
		    }
		    
		    lua_rawgeti (L, LUA_REGISTRYINDEX, pages);

		    if (lua_istable (L, -1)) {
			lua_pushstring (L, page);
			lua_gettable (L, -2);
			lua_replace (L, -2);
		    }
		    
		    if (lua_isfunction (L, -1)) {
			char *key, *value;

			/* If this isn't a static page, parse the query
			   arguments then call the relevant function to
			   create the content. */
			
			lua_pushstring (L, page);
			lua_newtable (L);

			if (query) {
			    for (i = 0 ; query[i] !='\0' ; i += 1) {
				query[i] = query[i] == '+' ? ' ' : query[i];
			    }
			    
			    for (key = strtok (query, "="),
				 value = strtok (NULL, "&");
				 key;
				 key = strtok (NULL, "="),
				 value = strtok (NULL, "&")) {
				lua_pushstring (L, key);
				lua_pushstring (L, value);
				lua_settable (L, -3);
			    }
			}

			if (lua_pcall (L, 2, 1, 0)) {
			    fprintf (stream,
				     "HTTP/1.0 500 Server Error\r\n"
				     "Server: Billiards/0.1\r\n"
				     "Connection: close\r\n"
				     "Content-Type: text/html\r\n"
				     "Content-Length: %zd\r\n"
				     "\r\n"
				     "<html><body>\r\n"
				     "<h2>Server Error</h2>\r\n"
				     "An error occured while running the "
				     "script for the requested page.  The "
				     "interpreter said:\r\n"
				     "<blockquote>%s</blockquote>\r\n"
				     "</body></html>\r\n",
				     lua_strlen (L, -1) + 133, 
				     lua_tostring (L, -1));
			}
		    }

		    /* Return the content of the requested page. */
		    
		    if (lua_isstring (L, -1)) {
			fprintf (stream,
				 "HTTP/1.0 200 OK\r\n"
				 "Server: Billiards/0.1\r\n"
				 "Connection: close\r\n"
				 "Content-Type: %s\r\n"
				 "Content-Length: %zd\r\n"
				 "\r\n",
				 lua_tostring (L, -2),
				 lua_strlen (L, -1));
			
			fwrite (lua_tostring (L, -1),
				1, lua_strlen (L, -1),
				stream);
		    } else {
			fputs ("HTTP/1.0 204 No Content\r\n"
			       "Server: Billiards/0.1\r\n"
			       "Connection: close\r\n"
			       "Content-Length: 0\r\n"
			       "\r\n", stream);
		    }

		    lua_settop(L, h_0);
		} else {
		    fputs ("HTTP/1.0 501 Not Implemented\r\n"
			   "Server: Billiards/0.1\r\n"
			   "Connection: close\r\n"
			   "Content-Type: text/html\r\n"
			   "Content-Length: 103\r\n"
			   "\r\n"
			   "<html><body>\r\n"
			   "<h2>Not Implemented</h2>\r\n"
			   "The request is not implemeted by this server.\r\n"
			   "</body></html>\r\n", stream);
		}
	    }

	    fclose (stream);
	}	
	
	/* Check for window system events and
	   fire the relevant bindings. */
	
	while (XPending(display)) {
	    XEvent event;

	    XNextEvent (display, &event);

	    if(event.type == ConfigureNotify) {
		glViewport (0, 0,
			    event.xconfigure.width,
			    event.xconfigure.height);
	    } else {
		int i, h_0, h_1, h_2;
	    
		h_0 = lua_gettop (L);
    
		lua_getglobal (L, "graphics");

		h_1 = lua_gettop (L) + 1;

		if (event.type == ButtonPress ||
		    event.type == ButtonRelease) {
		    if (event.type == ButtonPress) {
			lua_getfield (L, -1, "buttonpress");
		    } else {
			lua_getfield (L, -1, "buttonrelease");
		    }
		
		    lua_pushnumber (L, event.xbutton.button);
		} else if (event.type == KeyPress ||
			   event.type == KeyRelease) {
		    KeySym sym;
		    char *s;
		    int i;

		    XLookupString (&event.xkey, NULL, 0, &sym, NULL);

		    if (event.type == KeyPress) {
			lua_getfield (L, -1, "keypress");
		    } else {
			lua_getfield (L, -1, "keyrelease");
		    }
		
		    s = strdup(XKeysymToString (sym));

		    /* Convert all strings to lowercase to
		       make things more consistent. */
		    
		    for (i = 0 ; i < strlen (s) ; i += 1) {
			s[i] = tolower(s[i]);
		    }

		    lua_pushstring (L, s);
		    free(s);
		} else if (event.type == MotionNotify) {
		    lua_getfield (L, -1, "motion");
		    lua_pushnumber (L, event.xmotion.x);
		    lua_pushnumber (L, event.xmotion.y);
		} else {
		    lua_pushnil (L);
		}

		h_2 = lua_gettop (L);
	    
		if (lua_isfunction (L, h_1)) {
		    lua_call (L, h_2 - h_1, 0);
		} else if (lua_istable (L, h_1)) {
		    lua_pushnil(L);
		    while (lua_next(L, h_1) != 0) {
			for (i = 0 ; i < h_2 - h_1 ; i += 1) {
			    lua_pushvalue(L, h_1 + i + 1);
			}

			lua_call (L, h_2 - h_1, 0);
		    }
		}

		lua_settop (L, h_0);
	    }
	}

	glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT |
	        GL_STENCIL_BUFFER_BIT);
	
	/* Traverse the scene. */

	[head transform: L];
	[head prepare: L];
	[head traverse: L];
	[head cleanup: L];
    
	glXSwapBuffers(display, window);
	frames += 1;
    } while (iterate);

    lua_close (L);

    puts ("Adieu!");
    return 0;
}
