/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>

#include "network.h"

static int server, port, pages, mime, block, length;
static char *line;

static ssize_t xread(int fd, void *buffer, size_t count)
{
    int n;

    do {
        n = read(fd, buffer, count);
    } while (n < 0 && (errno == EAGAIN || errno == EWOULDBLOCK));

    return n;
}

static ssize_t xwrite(int fd, const void *buffer, size_t count)
{
    int n, m;

    for (m = 0, n = 0 ; m < count && n >=0 ; m += n > 0 ? n : 0) {
        n = write(fd, buffer + m, count - m);

        if (n < 0 && (errno == EAGAIN || errno == EWOULDBLOCK)) {
	    n = 0;
	}
    }

    return m;
}

static int execute (lua_State *L)
{
    if (luaL_loadstring(L, lua_tostring (L, 1)) || lua_pcall(L, 0, 1, 0)) {
/* 	puts (lua_tostring (L, -1)); */
	lua_pushliteral (L, "");
    }

    return 1;
}

static int node_index(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object get];

    return 1;
}

static int node_newindex(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    [object set];

    return 0;
}

static int network_index (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);
    
    if (!xstrcmp(k, "port")) {
	lua_pushnumber(L, port);
    } else if (!xstrcmp(k, "pages")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, pages);
    } else if (!xstrcmp(k, "block")) {
	lua_pushboolean(L, block);
    } else if (!xstrcmp(k, "mime")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, mime);
    }

    return 1;
}

static int network_newindex (lua_State *L)
{
    const char *k;
    int flags;

    k = lua_tostring(L, 2);
    
    if (!xstrcmp(k, "port")) {
	if (port > 0) {
	    close (server);
	}
	
	port = lua_tonumber (L, 3);

	if (port > 0) {
	    struct sockaddr_in address;
	    struct sigaction ignore;
	    int bound, listening, yes = 1;
    
	    server = socket(AF_INET, SOCK_STREAM, 0);

            flags = fcntl(server, F_GETFL, 0);
	    fcntl(server, F_SETFL, flags | (block ? 0 : O_NONBLOCK));
	    setsockopt(server, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof(int));

	    ignore.sa_handler = SIG_IGN;
            ignore.sa_flags = 0;
	    ignore.sa_restorer = NULL;

            sigemptyset(&ignore.sa_mask);
	    sigaction(SIGPIPE, &ignore, NULL);
    
	    address.sin_family = AF_INET;
	    address.sin_port = htons(port);
	    address.sin_addr.s_addr = INADDR_ANY;
    
	    bound = bind(server, (struct sockaddr *)&address, sizeof (address));
	    listening = listen(server, 10);

	    if (server < 0 || bound < 0 || listening < 0) {
		printf ("Could not setup port %d for http requests.\n", port);

		close(server);
		port = 0;
	    } else {
		printf ("Listening for http requests on port %d.\n", port);
	    }
    	}
    } else if (!xstrcmp(k, "block")) {
	block = lua_toboolean(L, 3);

        flags = fcntl(server, F_GETFL, 0);
	fcntl(server, F_SETFL, flags | (block ? 0 : O_NONBLOCK));
    } else if (!xstrcmp(k, "pages")) {
	luaL_unref (L, LUA_REGISTRYINDEX, pages);
	pages = luaL_ref (L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "mime")) {
	luaL_unref (L, LUA_REGISTRYINDEX, mime);
	mime = luaL_ref (L, LUA_REGISTRYINDEX);
    }

    return 0;
}

@implementation Network

-(void) traverse
{
    char *request, *uri, *version;
    int client, flags;

    /* Check for http requests. */

    client = accept(server, NULL, NULL);

    if (client > 0) {
	int i, j, n;

        flags = fcntl(server, F_GETFL, 0);
	fcntl(client, F_SETFL, flags | (block ? 0 : O_NONBLOCK));

	memset (line, 0, length);
	
	for (i = 0 ; !line || line[i - 1] != '\n' ; i += xread(client, &line[i], 1)) {
	    if (i >= length) {
		length += 1024;
		line = realloc (line, length);
	    }
	}

	line[i] = '\0';
		
	/* Read a whole line and break it up
	   into request, uri and protocol. */
		
	printf ("received: %s", line);
		    
	request = strtok (line, " \t");
	uri = strtok (NULL, " \t");
	version = strtok (NULL, " \t");

	/* Unescape the uri. */
	
	for (i = 0, j = 0;
	     (uri[i] = uri[j]) != '\0';
	     n = sscanf (&uri[j], "%%%2hhx", &uri[i]),
		 j += 2 * n + 1,
		 i += 1);

	if (!request || !uri || !version) {
	    const char *s = "HTTP/1.0 400 Bad Request\r\n"
		"Server: Billiards/0.1\r\n"
		"Connection: close\r\n"
		"Content-Type: text/html\r\n"
		"Content-Length: 0\r\n"
		"\r\n";
	    xwrite(client, s, strlen(s));
	} else {
	    if (!xstrcmp (request, "GET")) {
		int h_0;
		char *page, *query;

		h_0 = lua_gettop (_L);

		/* Break up the uri into a page/query pair. */
			    
		page = strtok (uri, "?");
		query = strtok (NULL, "");

		/* Retrieve the MIME type and content for the page. */
			    
		lua_rawgeti (_L, LUA_REGISTRYINDEX, mime);

		if (lua_istable (_L, -1)) {
		    lua_pushstring (_L, page);
		    lua_gettable (_L, -2);
		    lua_replace (_L, -2);
		}
			    
		lua_rawgeti (_L, LUA_REGISTRYINDEX, pages);

		if (lua_istable (_L, -1)) {
		    lua_pushstring (_L, page);
		    lua_gettable (_L, -2);
		    lua_replace (_L, -2);
		}

		/* Parse the query if any and gather all
		   fields inside a global table. */
			
		if (query) {
		    int i;
		    char *field, *value;
			    
		    lua_newtable (_L);
			    
		    for (i = 0 ; query[i] !='\0' ; i += 1) {
			query[i] = query[i] == '+' ? ' ' : query[i];
		    }
				    
		    for (field = strtok (query, "="),
			     value = strtok (NULL, "&");
			 field;
			 field = strtok (NULL, "="),
			     value = strtok (NULL, "&")) {
			lua_pushstring (_L, field);
			lua_pushstring (_L, value);
			lua_settable (_L, -3);
		    }

		    lua_setglobal (_L, "query");
		} else {
		    lua_pushnil (_L);
		    lua_setglobal (_L, "query");
		}

		/* Return the content of the requested
		   page after preprocessing it via string.gsub. */
			    
		if (lua_isstring (_L, -1)) {
		    lua_getglobal (_L, "string");
		    lua_getfield (_L, -1, "gsub");
		    lua_insert (_L, -3);
		    lua_pop (_L, 1);
		    lua_pushliteral (_L, "<%?lua(.-)%?>");
		    lua_pushcfunction (_L, execute);
		    lua_call (_L, 3, 1);
			    
		    lua_pushfstring (_L,
				     "HTTP/1.0 200 OK\r\n"
				     "Server: Billiards/0.1\r\n"
				     "Connection: close\r\n"
				     "Content-Type: %s\r\n"
				     "Content-Length: %d\r\n"
				     "\r\n",
				     lua_tostring (_L, -2),
				     lua_objlen (_L, -1));

		    xwrite (client,
			    lua_tostring (_L, -1),
			    lua_objlen (_L, -1));
		    xwrite (client,
			    lua_tostring (_L, -2),
			    lua_objlen (_L, -2));
		} else {
		    const char *s ="HTTP/1.0 404 Not Found\r\n"
			"Server: Billiards/0.1\r\n"
			"Connection: close\r\n"
			"Content-Type: text/html\r\n"
			"Content-Length: 90\r\n"
			"\r\n"
			"<html><body>\r\n"
			"<h2>Not Found</h2>\r\n"
			"The requested page could not be found.\r\n"
			"</body></html>\r\n";

		    xwrite(client, s, strlen(s));
		}

		lua_settop (_L, h_0);
	    } else {
		const char *s = "HTTP/1.0 501 Not Implemented\r\n"
		    "Server: Billiards/0.1\r\n"
		    "Connection: close\r\n"
		    "Content-Type: text/html\r\n"
		    "Content-Length: 103\r\n"
		    "\r\n"
		    "<html><body>\r\n"
		    "<h2>Not Implemented</h2>\r\n"
		    "The request is not implemeted by this server.\r\n"
		    "</body></html>\r\n";

		xwrite(client, s, strlen(s));
	    }
	}

	close (client);
    }	

    [super traverse];
}

@end

int luaopen_network (lua_State *L)
{
    id object;
    
    lua_newtable (_L);
    pages = luaL_ref (_L, LUA_REGISTRYINDEX);
    
    lua_newtable (_L);
    mime = luaL_ref (_L, LUA_REGISTRYINDEX);

    /* Create the network table. */
    
    lua_newtable (_L);    
    lua_newtable (_L);

    lua_pushcfunction (_L, network_index);
    lua_setfield (_L, -2, "__index");

    lua_pushcfunction (_L, network_newindex);
    lua_setfield (_L, -2, "__newindex");

    lua_setmetatable (_L, -2);
    
    lua_setglobal (_L, "network");

    /* Create and link a network node. */

    object = [[Network alloc] init];

    lua_getglobal (L, "graph");
    lua_pushlightuserdata (L, object);

    *(id *)lua_newuserdata(L, sizeof(id)) = object;

    lua_newtable (L);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)node_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)node_newindex);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    lua_settable (L, -3);
    lua_pop (L, 1);
    
    return 0;
}
