-- Copyright (C) 2008 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "dynamics"
require "shapes"
require "physics"
require "widgets"
require "switches"
require "meshes"
require "textures"
require (options.toon and "toon" or "shading")

widgets.regular = widgets.face "Sans-20"
widgets.small = widgets.face "Sans-14"

meshes.butt = meshes.static(resources.dofile "scripts/meshes/butt.lc")
meshes.tip = meshes.static(resources.dofile "scripts/meshes/tip.lc")

local theta, psi, alpha, beta = math.rad(125), 0, 0, 0

local function adjustcue(azimuth, elevation, horizontal, vertical)
   local c, d, r, l
   local n, t, t_0, psi_0
   local h, w, h, h_0, rcosc
   local gamma

   theta = tonumber(azimuth) or theta
   psi = tonumber(elevation) or psi
   alpha, beta = tonumber(horizontal) or alpha, tonumber(vertical) or beta

   if not bodies.observer.isaiming then
      return
   end

   -- Make sure we're aiming at the ball.

   gamma = math.length {alpha, beta, 0}
   if math.abs(gamma) >= 0.9 * billiards.ballradius then
      alpha = 0.9 * alpha / gamma * billiards.ballradius
      beta = 0.9 * beta / gamma * billiards.ballradius
   end

   -- Setup the cue rig.

   bodies.cue.spin = {0, 0, 0}
   bodies.cue.velocity = {0, 0, 0}
   bodies.cue.orientation = transforms.relue (90 - math.deg(psi),
					      0,
					      math.deg(theta) + 90)

   c = transforms.fromnode (bodies.cue, {alpha, beta,
					 0.3 * billiards.cuelength +
					 billiards.ballradius + 0.02})

   bodies.cue.position = math.add(bodies.cueball.position, c)

   joints.arm.axis = transforms.fromnode (bodies.cue, {0, 0, 1})
   joints.arm.bodies = {nil, bodies.cue}

   -- Correct the elevation to clear the rails if necessary.

   w = billiards.tablewidth
   h = billiards.tableheight
   h_0 = billiards.cushionheight

   n = joints.arm.axis
   d = transforms.fromnode (bodies.cue, {alpha, beta, 0})
   c = transforms.translate(bodies.cueball.position, d)

   t = {
      (-c[1] - 0.5 * w) / n[1],
      (-c[1] + 0.5 * w) / n[1], 
      (-c[2] - 0.5 * h) / n[2],
      (-c[2] + 0.5 * h) / n[2]
   }

   for i = 1, 4 do
      if t[i] < 0 then
	 t[i] = math.huge
      end
   end

   t_0 = math.min (t[1], t[2], t[3], t[4], billiards.cuelength)
   psi_0 = math.atan2(h_0 + 0.02 - c[3], t_0)

   -- Take neighbouring balls into account as well.

   r = billiards.ballradius
   l = billiards.cuelength
   
   for _, ball in ipairs(bodies.balls) do
      c = ball.position
      d = physics.pointfrombody (bodies.cue,
				 {0, 0, -0.3 * billiards.cuelength})

      t_0 = n[1] * (c[1] - d[1]) + n[2] * (c[2] - d[2])
      s_0 = -n[1] * (c[2] - d[2]) + n[2] * (c[1] - d[1])

      if t_0 >= 0 and t_0 < l then
	 h = math.sqrt (r^2 - s_0^2 + 4e-4) + r - d[3]
	 psi_0 = math.max (psi_0, math.atan2(h, t_0))
      end
   end

   if psi_0 > psi then
      adjustcue(theta, psi_0, alpha, beta)
   else
      callhooks(billiards.adjusting)
   end
end

bodies.cue = bodies.capsule {
   length = 0.6 * billiards.cuelength,
   radius = billiards.cueradius,
   mass =  {billiards.cuemass,
	    {0, 0, 0},
	    {billiards.cueinertia, 0, 0,
	     0, billiards.cueinertia, 0,
	     0, 0, 9.13e-6}},

   butt = switches.button {
      position = {0, 0, -0.36},

--       prepare = function (self)
--          bodies.cue.butt.position = {0, 0, -0.36 + bodies.observer.speed}
--       end,

      surface = options.toon and toon.cel {
	 color = {1.0000, 0.9605, 0.6179},
	 mesh = meshes.butt()
      } or shading.anisotropic {
	 diffuse = textures.clamped(resources.dofile "scripts/imagery/diffuse/cue.lc"),
	 specular = {0.3, 0.3, 0.3},
	 parameter = 64,
      
	 mesh = meshes.butt()
      },

      selected = frames.event {
	 motion = function (self, button, x, y, dx, dy)
	    if button == bindings.strike then
	       joints.arm.motor = {billiards.stroke * dy,
				   billiards.cueforce}
	    elseif button == bindings.elevate then
	       self.parent.parent.elevation = math.clamp(bodies.cue.elevation -
							 0.005 * dy,
						         0, math.pi / 2) 
	    end
	 end,

	 outline = shapes.halo {
	    color = {0.7, 0.6, 0.1},
	    width = 4.0,
	    geometry = meshes.butt(),
	 },
  
	 annotation = widgets.annotation {
	    color = {1.0, 0.8, 0.2}, 
	    opacity = 0.8,
	    thickness = 1.5,
	    padding = {0.01, 0.01},
	    radius = 0.07,
	    angle = 40,

	    table = widgets.row {
	       widgets.column {
		  widgets.small {
		     size = {0.09, 0},
		     align = {-1, 0},
		     text = "Elevation"},
	       },

	       widgets.column {
		  widgets.small {},

		  prepare = function (self)
		     self[1].text = string.format("% d\176", math.deg(bodies.cue.elevation))
		  end
	       }
	    }
	 }
      },
   },

   tip = switches.button {
      position = {0, 0, -0.41},

--       prepare = function (self)
-- 	 bodies.cue.tip.position = {0, 0, -0.41 + bodies.observer.speed}
--       end,

      surface = options.toon and toon.cel {
	 color = {0.9065, 0.9212, 0.9197},
	 mesh = meshes.tip()
      } or shading.anisotropic {
	 diffuse = {0.9065, 0.9212, 0.9197},
	 specular = {0.3, 0.3, 0.3},
	 parameter = 64,
      
	 geometry = meshes.tip()
      },

      selected = frames.event {
	 motion = function (self, button, x, y, dx, dy)
	    if button == bindings.offset then
	       self.parent.parent.sidespin = bodies.cue.sidespin + 0.0005 * dx
	       self.parent.parent.follow = bodies.cue.follow - 0.0005 * dy
	    end
	 end,

	 outline = shapes.halo {
	    color = {0.7, 0.6, 0.1},
	    width = 4.0,
	    geometry = meshes.tip(),
	 },

	 annotation = widgets.annotation {
	    color = {1.0, 0.8, 0.2}, 
	    opacity = 0.8,
	    thickness = 1.5,
	    padding = {0.01, 0.01},
	    radius = 0.07,
	    angle = 40,

	    table = widgets.row {
	       widgets.column {
		  widgets.small {
		     size = {0.09, 0},
		     align = {-1, 0},
		     text = "Sidespin"
		  },
		  
		  widgets.small {
		     size = {0.09, 0},
		     align = {-1, 0},
		     text = "Follow"
		  },
	       },

	       widgets.column {
		  widgets.small {},
		  widgets.small {},

		  prepare = function (self)
	             self[1].text = string.format("% d%%",
						  bodies.cue.sidespin /
						  billiards.ballradius * 100)
		     self[2].text = string.format("% d%%",
						  bodies.cue.follow /
					          billiards.ballradius * 100)
		  end
	       }
	    }
	 }
      }
   },

   iscue = true,

   prepare = function()
		-- This makes sure the cue doesn't trail off
		-- when the user stops moving the mouse.

  		joints.arm.motor = {0, billiards.cueforce}
	     end
}

joints.arm = joints.slider {
   motor = {0, billiards.cueforce},
   stops = {{-1, 0.1}, {0, 0}, 0}
}

local meta = getmetatable (bodies.cue)
local oldindex = meta.__index
local oldnewindex = meta.__newindex

meta.__index = function (self, key)
		  if key == "azimuth" then
		     return theta
		  elseif key == "elevation" then
		     return psi
		  elseif key == "sidespin" then
		     return alpha
		  elseif key == "follow" then
		     return beta
		  elseif key == "speed" then
		     return joints.arm.state[2]
		  else
		     return oldindex (self, key)
		  end
	       end

meta.__newindex = function (self, key, value)
		     if key == "azimuth" then
			adjustcue (value, psi, alpha, beta)
		     elseif key == "elevation" then
			adjustcue (theta, value, alpha, beta)
		     elseif key == "sidespin" then
			adjustcue (theta, psi, value, beta)			
		     elseif key == "follow" then
			adjustcue (theta, psi, alpha, value)			
		     else
			oldnewindex(self, key, value)
		     end
		  end

meta.__tostring = function(self)
		     return "cue stick"
		  end
    
billiards.aiming.cue = function ()
			  bodies.cue.parent = bodies.cueball.parent
		       end

billiards.looking.cue = function ()
			   bodies.cue.parent = nil
-- 			   joints.arm.stops = {{-19.5, -19.5}, 
-- 					      physics.spring (2000, 350),
-- 					      0.1}
			end

billiards.waiting.cue = function ()
			   bodies.cue.parent = nil  
			end

billiards.cuecollision.letgo = function ()
			-- Release the grip on the cue to
			-- allow it to rebound of the ball

			joints.arm.bodies = {nil, nil}
			joints.arm.motor = {0, 0}
		     end
