-- Copyright (C) 2008 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "frames"
require "transforms"
require "switches"
require "dynamics"
require "physics"
require "morejoints"
require "moremath"

local x, y, rho, theta, phi = 0, 0, 2, 0, math.rad (70)
local islooking, iswaiting, isaiming = false, true, false, false

local function adjusteye (radius, azimuth, elevation, horizontal, vertical)
   theta, phi = azimuth, math.clamp (tonumber(elevation) or phi, 0, math.pi / 2)
   rho = math.clamp (tonumber(radius) or rho, 0, 10)
   x = math.clamp (tonumber(horizontal) or x, -5, 5)
   y = math.clamp (tonumber(vertical) or y, -5, 5)

   joints.neck.preload = rho
   joints.waist.preload = {phi, 0, theta}
      
   joints.feet.axis = math.normalize {x - bodies.observer.legs.position[1],
				      y - bodies.observer.legs.position[2],
				      0}

   joints.feet.preload = math.length {x - bodies.observer.legs.position[1],
				      y - bodies.observer.legs.position[2],
				      0}
   
--   print (theta)

   callhooks(billiards.reorienting)
end

-- The bodies.

bodies.observer = bodies.system {
   step = function (self)
	     if theta >= math.pi and
		joints.waist.state[3] < 0 then
		adjusteye (rho, theta - 2 * math.pi, phi, x, y)
	     elseif theta <= -math.pi and
		joints.waist.state[3] > 0 then
		adjusteye (rho, theta + 2 * math.pi, phi, x, y)
	     end
	     -- 		print (math.deg(joints.waist.state[3]),
	     -- 		       math.deg(joints.waist.stops[3][1][1]), 
	     -- 		       math.length (self.velocity))

	     if physics.joined (bodies.cueball, bodies.cue) then
		self.striking = true
	     elseif self.striking then
		callhooks (billiards.striking)
		
		self.striking = false
	     end
	  end,

    eye = bodies.point {
      position = {0, 0, 2 * billiards.ballradius},
      orientation = transforms.euler (0, 10, 0),

      mass = physics.spheremass (1, 0.1),

      transform = function (self)
		     graph.position,
		     graph.orientation = transforms.lookthrough (self)
		  end
   },

   torso = bodies.point {
      position = {0, 0, billiards.ballradius},
      mass = physics.spheremass (1, 0.1),
   },

   legs = bodies.point {
      position = {0, 0, 0},
      mass = physics.spheremass (1, 0.1),
   },

   controls = switches.button {
      selected = frames.event {
	 motion = function (self, button, x, y, dx, dy)
            if button == bindings.rotate then
	       local c = bodies.observer.isaiming and
		         billiards.finetune or
		         billiards.angular

	       bodies.observer.azimuth = bodies.observer.azimuth - c * dx
	       bodies.observer.elevation = bodies.observer.elevation - c * dy
	    elseif button == bindings.zoom then
	       if bodies.observer.isaiming then
		  bodies.observer.radius = math.clamp(bodies.observer.radius - billiards.linear * dy, billiards.ballradius + 0.05, 0.5)
	       else
		  bodies.observer.radius = math.clamp(bodies.observer.radius - billiards.linear * dy, billiards.ballradius + 1e-2, 5)
	       end
	    end
	 end,
      }
   }
}

-- The joints.

joints.waist = springs.euler {
   anchor = {0, 0, billiards.ballradius},
   axes = {{0, 1, 0}, {0, 0, 1}, {0, 0, 1}},
   stiffness = 3000,
   damping = 1000,
   bodies = {bodies.observer.torso, bodies.observer.legs}
}

joints.neck = springs.linear {
   stiffness = 3000,
   damping = 1000,
   bodies = {bodies.observer.torso, bodies.observer.eye}
}

joints.feet = springs.linear {
   stiffness = 3000,
   damping = 1000,
   bodies = {bodies.observer.legs, nil}
}

local observermeta = getmetatable(bodies.observer)
local observerindex = observermeta.__index
local observernewindex = observermeta.__newindex

observermeta.__index = function(table, key)
   if key == "azimuth" then
      return theta
   elseif key == "elevation" then
      return phi
   elseif key == "radius" then
      return rho
   elseif key == "longitude" then
      return x
   elseif key == "latitude" then
      return y
   elseif key == "islooking" then
      return islooking
   elseif key == "isaiming" then
      return isaiming
   elseif key == "iswaiting" then
      return iswaiting
   elseif key == "speed" then
      return math.length (bodies.observer.eye.velocity)
   else
      return observerindex(table, key)
   end
end

observermeta.__newindex = function(table, key, value)
   if key == "radius" then
      adjusteye (value, theta, phi, x, y)
      callhooks(billiards.zooming)
   elseif key == "azimuth" then
      adjusteye (rho, value, phi, x, y)
      callhooks(billiards.turning)
   elseif key == "elevation" then
      adjusteye (rho, theta, value, x, y)
      callhooks(billiards.turning)
   elseif key == "longitude" then
      adjusteye (rho, theta, phi, value, y)
      callhooks(billiards.panning)
   elseif key == "latitude" then
      adjusteye (rho, theta, phi, x, value)
      callhooks(billiards.panning)
   elseif key == "islooking" and value == true then
      islooking = true
      isaiming = false
      iswaiting = false
      isfinished = false
      
      callhooks(billiards.looking)
   elseif key == "isaiming" and value == true then
      islooking = false
      isaiming = true
      iswaiting = false
      isfinished = false
      
      callhooks(billiards.aiming)
   elseif key == "iswaiting" and value == true then
      islooking = false
      isaiming = false
      iswaiting = true
      isfinished = false
      
      callhooks(billiards.waiting)
   else
      observernewindex(table, key, value)
   end
end
