/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "face.h"
#include "string.h"

@implementation String

-(String *)initFromFace: (id) object
{
    [super init];
    
    self->face = object;
    
    self->texels[0] = 0;
    self->texels[1] = 0;

    self->allocated[0] = 0;
    self->allocated[1] = 0;

    glGenTextures(1, &self->texture);

    return self;
}

-(void) free
{
    glDeleteTextures(1, &self->texture);

    [super free];
}

-(void)transform
{
    GLint v[4];

    glGetIntegerv(GL_VIEWPORT, v);

    self->minimum[0] = (float)self->texels[0] / v[2];
    self->minimum[1] = (float)self->texels[1] / v[3];

    self->allocated[0] = self->minimum[0] > self->requested[0] ?
	                 self->minimum[0] : self->requested[0];
    self->allocated[1] = self->minimum[1] > self->requested[1] ?
	                 self->minimum[1] : self->requested[1];

    if (self->align[0] < 0) {
	self->position[0] -= 0.5 * self->allocated[0] - self->padding[0];
    } else if (self->align[0] > 0) {
	self->position[0] += 0.5 * self->allocated[0] - self->padding[0] -
	    self->minimum[0];
    } else {
	self->position[0] -= 0.5 * self->minimum[0];
    }

    if (self->align[1] < 0) {
	self->position[1] -= 0.5 * self->allocated[1] - self->padding[1];
    } else if (self->align[1] > 0) {
	self->position[1] += 0.5 * self->allocated[1] - self->padding[1] -
		      self->minimum[1];
    } else {
	self->position[1] -= 0.5 * self->minimum[1];
    }

    [super transform];
}

-(void) traverse
{
    glMatrixMode (GL_MODELVIEW);
    glPushMatrix();
    glMultMatrixf ([self homogenous]);

    glColor4fv (self->color);
    
    glUseProgramObjectARB(0);

    glActiveTexture(GL_TEXTURE0);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, self->texture);

    glDepthMask (GL_FALSE);

    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_BLEND);

    glBegin(GL_QUADS);
    glTexCoord2f(0, 0);
    glVertex2f(0, 0);
    glTexCoord2f(1, 0);
    glVertex2f(self->minimum[0], 0);
    glTexCoord2f(1, 1);
    glVertex2f(self->minimum[0], self->minimum[1]);
    glTexCoord2f(0, 1);
    glVertex2f(0, self->minimum[1]);
    glEnd();

    glDepthMask (GL_TRUE);

    glDisable(GL_BLEND);
    glDisable(GL_TEXTURE_2D);
    
    glMatrixMode (GL_MODELVIEW);
    glPopMatrix();

    [super traverse];
}
    
-(void) get
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "text")) {
	lua_rawgeti (_L, LUA_REGISTRYINDEX, self->reference);
    } else {
	[super get];
    }
}

-(void) set
{    
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "text")) {
	int viewport[4];

	[self->face layoutString: lua_tostring (_L, 3)
	             intoTexture: self->texture];

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, self->texture);
	
	glGetTexLevelParameteriv(GL_TEXTURE_2D,
				 0,
				 GL_TEXTURE_WIDTH,
				 &self->texels[0]);
	
	glGetTexLevelParameteriv(GL_TEXTURE_2D,
				 0,
				 GL_TEXTURE_HEIGHT,
				 &self->texels[1]);
	
	glGetIntegerv(GL_VIEWPORT, viewport);

	self->allocated[0] = (float)self->texels[0] / viewport[2];
	self->allocated[1] = (float)self->texels[1] / viewport[3];

	self->reference = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else {
	[super set];
    }
}

@end


