%{

/* This file is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option) */
/* any later version. */

/* This file is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with GNU Emacs; see the file COPYING.  If not, write to */
/* the Free Software Foundation, Inc., 59 Temple Place - Suite 330, */
/* Boston, MA 02111-1307, USA. */

/* Copyright (C) 2004 California Digital Corporation */
/* $Id: layoutgrm.y,v 1.7 2004/04/29 22:51:19 summerisle Exp $ */

#include "layoutlex.h"
#include "object.h"

#ifdef HAVE_ERROR_H
#include <error.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>

static void yyerror (char const *s);

#define YYLEX_PARAM scanner
#define YYPARSE_PARAM scanner

static void*
xalloc (size_t size)
{
  void* p;

  p = malloc (size);
  if (!p) error (1, errno, "xalloc");
  return p;
}

%}

%token BANKS
%token CHECKSUMS
%token ENTRIES
%token REGIONS
%token ENUMS
%token <ident> ID
%token ADDRESS_PORT
%token DATA_PORT
%token LENGTH
%token <num> NUM
%token BANK
%token START
%token TYPE
%token SINGLE_BYTE
%token LOW_FIRST
%token HIGH_FIRST
%token ENUM_ID
%token CHECKSUM
%token RESERVED
%token NUMERIC
%token ENUM
%token <str> STRING
%token VENDOR
%token COMPATIBLE
%token VERSION
%token DATE

%type <values> enum_attribs
%type <enums> enums
%type <banks> bank_attribs
%type <banks> banks
%type <regions> region_attribs
%type <regions> regions
%type <entrytype> entry_type
%type <checksumtype> checksum_type
%type <checksums> checksums
%type <checksums> checksum_attribs
%type <entries> entries
%type <entries> entry_attribs
%type <backpatches> ids
%type <layoutp> sections
%type <info>  compatibles
%type <info> info

%union {
  long num;
  ident_t* ident;
  enumeration_t* enums;
  enum_value_t* values;
  bank_t* banks;
  region_t* regions;
  entry_type_t entrytype;
  checksum_type_t checksumtype;
  checksum_t* checksums;
  cmentry_t* entries;
  backpatch_t* backpatches;
  layout_t* layoutp;
  char* str;
  bios_info_t* info;
}

%pure-parser

%%

layout :
'{' sections '}'
{
  (*((input_state_t**)scanner))->result_layout = $2;
}
;

sections :
  empty
{
  layout_t* l;

  l = xalloc (sizeof (layout_t));
  l->compat = NULL;
  l->banks = NULL;
  l->checksums = NULL;
  l->entries = NULL;
  l->regions = NULL;
  l->enums = NULL;
  $$ = l;
}
| sections COMPATIBLE '=' '{' compatibles '}' ';'
{
  layout_t* l;

  l = $1;
  l->compat = $5;
  $$ = l;
}
| sections BANKS '=' '{' banks '}' ';'
{
  layout_t* l;

  l = $1;
  l->banks = $5;
  $$ = l;
}
| sections CHECKSUMS '=' '{' checksums '}' ';'
{
  layout_t* l;

  l = $1;
  l->checksums = $5;
  $$ = l;
}
| sections ENTRIES '=' '{' entries '}' ';'
{
  layout_t* l;

  l = $1;
  l->entries = $5;
  $$ = l;
}
| sections REGIONS '=' '{' regions '}' ';'
{
  layout_t* l;

  l = $1;
  l->regions = $5;
  $$ = l;
}
| sections ENUMS '=' '{' enums '}' ';'
{
  layout_t* l;

  l = $1;
  l->enums = $5;
  $$ = l;
}
;

compatibles :
  empty
{
  $$ = NULL;
}
| compatibles ID '=' '{' info '}' ';'
{
  bios_info_t* c;

  c = $5;
  c->next = $1;
  c->name = $2;
  $$ = c;
}
;

info :
  empty
{
  bios_info_t* bi;

  bi = xalloc (sizeof(bios_info_t));
  bi->vendor = NULL;
  bi->version = NULL;
  bi->date = NULL;
  $$ = bi;
}
| info VENDOR '=' STRING ';'
{
  bios_info_t* bi;

  bi = $1;
  bi->vendor = $4;
  $$ = bi;
}
| info VERSION '=' STRING ';'
{
  bios_info_t* bi;

  bi = $1;
  bi->version = $4;
  $$ = bi;
}
| info DATE '=' STRING ';'
{
  bios_info_t* bi;

  bi = $1;
  bi->date = $4;
  $$ = bi;
}
;

checksums :
  empty
{
  $$ = NULL;
}
| checksums ID '=' '{' checksum_attribs '}' ';'
{
  checksum_t* c;

  c = $5;
  c->next = $1;
  c->name = $2;
  $$ = c;
}
;

checksum_attribs :
  empty
{
  checksum_t* c;

  c = xalloc (sizeof (checksum_t));
  c->next = NULL;
  c->bp_bank.filled = 0;
  c->bp_bank.u.name = NULL;
  c->type = ct_none;
  c->start = -1;
  $$ = c;
}
| checksum_attribs BANK '=' ID ';'
{
  checksum_t* c;

  c = $1;
  c->bp_bank.u.name = $4;
  $$ = c;
}
| checksum_attribs START '=' NUM ';'
{
  checksum_t* c;

  c = $1;
  c->start = $4;
  $$ = c;
}
| checksum_attribs TYPE '=' checksum_type ';'
{
  checksum_t* c;

  c = $1;
  c->type = $4;
  $$ = c;
}
;

checksum_type :
  SINGLE_BYTE
{
  $$ = single_byte;
}
| LOW_FIRST
{
  $$ = low_first;
}
| HIGH_FIRST
{
  $$ = high_first;
}
;

entries :
  empty
{
  $$ = NULL;
}
| entries ID '=' '{' entry_attribs '}' ';'
{
  cmentry_t* e;

  e = $5;
  e->next = $1;
  e->name = $2;
  $$ = e;
}
;
 
entry_attribs :
  empty
{
  cmentry_t* e;

  e = xalloc (sizeof (cmentry_t));
  e->bp_bank.filled = 0;
  e->bp_bank.u.name = NULL;
  e->start_bit = e->bit_length = -1;
  e->type = et_none;
  e->bp_enum.filled = 0;
  e->bp_enum.u.name = NULL;
  e->pbp_checksums = NULL;
  e->next = NULL;
  $$ = e;
}
| entry_attribs BANK '=' ID ';'
{
  cmentry_t* e;

  e = $1;
  e->bp_bank.u.name = $4;
  $$ = e;
}
| entry_attribs START '=' NUM ';'
{
  cmentry_t* e;

  e = $1;
  e->start_bit = $4;
  $$ = e;
}
| entry_attribs LENGTH '=' NUM ';'
{
  cmentry_t* e;

  e = $1;
  e->bit_length = $4;
  $$ = e;
}
| entry_attribs TYPE '=' entry_type ';'
{
  cmentry_t* e;

  e = $1;
  e->type = $4;
  $$ = e;
}
| entry_attribs ENUM_ID '=' ID ';'
{
  cmentry_t* e;

  e = $1;
  e->bp_enum.u.name = $4;
  $$ = e;
}
| entry_attribs CHECKSUM '=' ID ';'
{
  cmentry_t* e;
  backpatch_t* pbp;
  
  e = $1;
  pbp = xalloc (sizeof (backpatch_t));
  pbp->filled = 0;
  pbp->u.name = $4;
  pbp->next = NULL;
  e->pbp_checksums = pbp;
  $$ = e;
}
| entry_attribs CHECKSUMS '=' '(' ids ')' ';'
{
  cmentry_t* e;

  e = $1;
  e->pbp_checksums = $5;
  $$ = e;
}
;

ids :
  ID
{
  backpatch_t* pbp;

  pbp = xalloc (sizeof (backpatch_t));
  pbp->filled = 0;
  pbp->u.name = $1;
  pbp->next = NULL;
  $$ = pbp;
}
| ids ',' ID
{
  backpatch_t* pbp;

  pbp = xalloc (sizeof (backpatch_t));
  pbp->filled = 0;
  pbp->u.name = $3;
  pbp->next = $1;
  $$ = pbp;
}
;

entry_type :
  RESERVED
{
  $$ = reserved;
}
| NUMERIC
{
  $$ = numeric;
}
| ENUM
{
  $$ = enumtype;
}
;

regions :
  empty
{
  $$ = NULL;
}
| regions ID '=' '{' region_attribs '}' ';'
{
  region_t* r;

  r = $5;
  r->next = $1;
  r->name = $2;
  $$ = r;
}
;

region_attribs :
  empty
{
  region_t* r;

  r = xalloc (sizeof (region_t));
  r->bp_bank.filled = 0;
  r->bp_bank.u.name = 0;
  r->start_byte = r->byte_length = -1;
  r->next = NULL;
  $$ = r;
}
| region_attribs BANK '=' ID ';'
{
  region_t* r;

  r = $1;
  r->bp_bank.u.name = $4;
  $$ = r;
}
| region_attribs START '=' NUM ';'
{
  region_t* r;

  r = $1;
  r->start_byte = $4;
  $$ = r;
}
| region_attribs LENGTH '=' NUM ';'
{
  region_t* r;

  r = $1;
  r->byte_length = $4;
  $$ = r;
}
;

enums :
  empty
{
  $$ = NULL;
}
| enums ID '=' '{' enum_attribs '}' ';'
{
  enumeration_t* e;

  e = xalloc (sizeof (enumeration_t));
  e->next = $1;
  e->name = $2;
  e->values = $5;
  $$ = e;
}
;

enum_attribs :
  empty
{
  $$ = NULL;
}
| enum_attribs ID '=' NUM ';'
{
  enum_value_t* v;
  
  v = xalloc (sizeof (enum_value_t));
  v->next = $1;
  v->name = $2;
  v->value = $4;
  $$ = v;
}
;

banks :
  empty
{
  $$ = NULL;
}
| banks ID '=' '{' bank_attribs '}' ';'
{
  bank_t* b;

  b = $5;
  b->next = $1;
  b->name = $2;
  $$ = b;
}
;

bank_attribs :
  empty
{
  bank_t* b;

  b = xalloc (sizeof (bank_t));
  b->address_port = b->data_port = 0xffff;
  b->length = -1;
  b->next = NULL;
  $$ = b;
}
| bank_attribs ADDRESS_PORT '=' NUM ';'
{
  bank_t* b;

  b = $1;
  b->address_port = $4;
  $$ = b;
}
| bank_attribs DATA_PORT '=' NUM ';'
{
  bank_t* b;

  b = $1;
  b->data_port = $4;
  $$ = b;
}
| bank_attribs LENGTH '=' NUM ';'
{
  bank_t* b;

  b = $1;
  b->length = $4;
  $$ = b;
}
;

empty :                         /* empty */
;

%%

void
yyerror (char const *s)
{
  fprintf (stderr, "%s\n", s);
}
