%{

/* This file is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option) */
/* any later version. */

/* This file is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with GNU Emacs; see the file COPYING.  If not, write to */
/* the Free Software Foundation, Inc., 59 Temple Place - Suite 330, */
/* Boston, MA 02111-1307, USA. */

/* Copyright (C) 2004 California Digital Corporation */
/* $Id: layoutlex.l,v 1.11 2004/04/29 22:51:19 summerisle Exp $ */

#include "object.h"
#include "layoutgrm.h"
#include "layoutlex.h"
#include "escape.h"

#ifdef HAVE_ERROR_H
#include <error.h> 
#endif

#include <search.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

struct kw
{
  char* name;
  int token;
};

typedef struct kw kw_t;

/* must be kept sorted! */ 
static kw_t kw_tbl[] =
  {
    { "address_port", ADDRESS_PORT },
    { "bank", BANK },
    { "banks", BANKS },
    { "checksum", CHECKSUM },
    { "checksums", CHECKSUMS },
    { "compatible", COMPATIBLE },
    { "data_port", DATA_PORT },
    { "date", DATE },
    { "entries", ENTRIES },
    { "enum", ENUM },
    { "enum_id", ENUM_ID },
    { "enums", ENUMS },
    { "high_first", HIGH_FIRST },
    { "length", LENGTH },
    { "low_first", LOW_FIRST },
    { "numeric", NUMERIC },
    { "regions", REGIONS },
    { "reserved", RESERVED },
    { "single_byte", SINGLE_BYTE },
    { "start", START },
    { "type", TYPE },
    { "vendor", VENDOR },
    { "version", VERSION },
  };
/* must NOT be 0-terminated! */ 
    
static int compare_kws (const void*, const void*);

#define YY_USER_ACTION do \
{ \
  input_state_t* state; \
  state = (input_state_t*)(yyextra); \
  state->column += strlen (yytext); \
} \
while (0);

#define YY_DECL int yylex (YYSTYPE* yylval, void* yyscanner)

%}

%option reentrant
%option noyywrap

%%

\"([^"]|\\\")*\" {
  char* p;
  input_state_t* state;

  state = (input_state_t*)(yyextra);
  p = malloc (yyleng - 1);
  if (p == NULL) error (1, errno, "malloc");
  if (unescape_array (p, yytext + 1, yyleng - 2) == NULL)
    error_at_line (1, 0, state->fname, state->line,
                   "bad escape near column %d", state->column);
  yylval->str = p;
  return STRING;
}

[[:alpha:]][-_[:alnum:]]* {
  kw_t kw_bait;
  kw_t* pkw;
  input_state_t* state;

  state = (input_state_t*)(yyextra);
  kw_bait.name = yytext;
  pkw = bsearch (&kw_bait, kw_tbl, sizeof(kw_tbl)/sizeof(kw_t), sizeof(kw_t), compare_kws);
  if (pkw) return pkw->token;
  else
    {
      yylval->ident = maybe_insert_ident (&state->tree, yytext,
                                          state->line, state->column);
      return ID;
    }
}

([[:digit:]]+|"0x"[[:xdigit:]]+) {
  yylval->num = strtol (yytext, NULL, 0);
  return NUM;
}

("{"|"}"|"="|";"|"("|")"|",") {
  return *(yytext);
}

"\n" {
  input_state_t* state;

  state = (input_state_t*)(yyextra);
  state->column = 0;
  state->line++;
}

"#".*

(" "|"\t"|"\v"|"\f")+

%%

static int
compare_kws (const void* vp1, const void* vp2)
{
  return strcmp (((kw_t*)vp1)->name, ((kw_t*)vp2)->name);
}

/* parse_file - read a configuration file and create all corresponding data structures */
/* fname = name of file to parse, "-" for standard input */
/* return : pointer to layout data structure */

layout_t*
parse_file (const char* fname)
{
  FILE* fp;
  yyscan_t scanner;
  input_state_t state;
  extern int yyparse (void*);

  if (strcmp (fname, "-") == 0) fp = stdin;
  else
    {
      fp = fopen (fname, "r");
      if (!fp) error (1, errno, "parse_file: %s", fname);
    }
  yylex_init (&scanner);
  yyset_in (fp, scanner);
  state.line = 1;
  state.column = 0;
  state.tree = NULL;
  state.fname = fname;
  yyset_extra (&state, scanner);
  if (yyparse (scanner) != 0)
    error_at_line (1, 0, fname, state.line, "syntax error near column %d", state.column);
  yylex_destroy (scanner);
  if (fp != stdin) fclose (fp);
  state.result_layout->tree = state.tree;
  return state.result_layout;
}
