/* This file is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option) */
/* any later version. */

/* This file is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with GNU Emacs; see the file COPYING.  If not, write to */
/* the Free Software Foundation, Inc., 59 Temple Place - Suite 330, */
/* Boston, MA 02111-1307, USA. */

/* Copyright (C) 2004 California Digital Corporation */
/* $Id: smbios.c,v 1.5 2004/04/29 22:51:20 summerisle Exp $ */


#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <string.h>
#include <limits.h>
#include <inttypes.h>

#ifdef HAVE_ERROR_H
#include <error.h>
#endif

#include <errno.h>

#define SMBIOS_ENTRY_ANCHOR_OFFSET      0x10
#define SMBIOS_AREA_START               0x000f0000
#define SMBIOS_AREA_END                 0x000fffff
#define SMBIOS_AREA_LEN                 ((SMBIOS_AREA_END - SMBIOS_AREA_START) + 1)
#define SMBIOS_AREA_ALIGN               16
#define SMBIOS_ENTRY_TLEN_OFFSET        0x16
#define SMBIOS_ENTRY_PTR_OFFSET 	0x18

#define SMBIOS_BIOS_INFO_SIG            0
#define SMBIOS_BIOS_INFO_LENGTH_OFFSET  1
#define SMBIOS_BIOS_INFO_VENDOR_OFFSET  4
#define SMBIOS_BIOS_INFO_VERSION_OFFSET 5
#define SMBIOS_BIOS_INFO_DATE_OFFSET    8

/* SMBIOS Reference Specification: map area between 000f0000 and
   000fffff.  The IPMI Entry Structure begins on a 16-byte boundary,
   with a 4 byte "_SM_" signature.  */

/* is_ipmi_entry
   ARGUMENTS:
   sigp = points to start of purported SMBIOS entry structure
   RETURNS:
   0 = not really a SMBIOS entry structure
   1 = yes, a real SMBIOS entry structure */ 

static int
is_smbios_entry (uint8_t* sigp)
{
  static const char smbios_entry_sig[4] = { '_', 'S', 'M', '_' };
  static const char smbios_entry_anchor[5] = { '_', 'D', 'M', 'I', '_' };

  if (memcmp (sigp, smbios_entry_sig, sizeof (smbios_entry_sig)) != 0)
    return 0;

  if (memcmp (sigp + SMBIOS_ENTRY_ANCHOR_OFFSET, smbios_entry_anchor,
              sizeof (smbios_entry_anchor)) != 0)
    return 0;
    
  return 1;

}

/* map_physmem
   ARGUMENTS:
   physaddr = physical address to access
   len = length of area to access
   startp = place to store pointer to unmap (caller responsible for unmapping)
   totallen = length of area to unmap
   RETURNS:
   pointer to area of physical memory at physmem */

static uint8_t*
map_physmem (uint32_t physaddr, size_t len, void** startp, size_t* totallen)
{
  uint32_t startaddr;
  uint32_t pad;
  int mem_fd;

  mem_fd = open ("/dev/mem", O_RDONLY);
  if (mem_fd < 0) error (1, errno, "map_physmem: /dev/mem");
  pad = physaddr % sysconf (_SC_PAGESIZE);
  startaddr = physaddr - pad;
  *totallen = len + pad;
  *startp = mmap (NULL, *totallen, PROT_READ, MAP_PRIVATE, mem_fd, startaddr);
  close (mem_fd);

  if (*startp == MAP_FAILED) error (1, errno, "map_physmem: /dev/mem");
  return ((uint8_t*)(*startp) + pad);
}

static inline char*
xstrdup (char* cp)
{
  void* p;

  p = strdup (cp);
  if (!p) error (1, errno, "xstrdup");
  return p;
}

/* copy_bios_info_strings_internal - read type 0 SMBIOS structure for id strings */
/* p_bios_info = address of pointer to SMBIOS structure, advanced to next structure */
/* scan_only = if nonzero, only advance strcuture pointer, don't read strings */
/* vec = vector of identification strings: vendor, version, date */

static void
copy_bios_info_strings_internal (uint8_t** p_bios_info, int scan_only, char** vec)
{
  unsigned length;
  unsigned vendor_string_no;
  unsigned version_string_no;
  unsigned date_string_no;
  unsigned ui;
  uint8_t* p;
  uint8_t* bios_info;

  bios_info = *p_bios_info;
  length = bios_info[SMBIOS_BIOS_INFO_LENGTH_OFFSET];
  if (!scan_only)
    {
      vendor_string_no = bios_info[SMBIOS_BIOS_INFO_VENDOR_OFFSET];
      version_string_no = bios_info[SMBIOS_BIOS_INFO_VERSION_OFFSET];
      date_string_no = bios_info[SMBIOS_BIOS_INFO_DATE_OFFSET];
    }

  p = bios_info + length;
  ui = 1;
  while (*p != 0)
    {
      if (!scan_only && ui == vendor_string_no)
        vec[0] = xstrdup (p);
      else if (!scan_only && ui == version_string_no)
        vec[1] = xstrdup (p);
      else if (!scan_only && ui == date_string_no)
        vec[2] = xstrdup (p);
      p = p + strlen (p) + 1;
      ui = ui + 1;
    }
  *p_bios_info = p + 1;
}

/* copy_bios_info_strings - probe SMBIOS for id strings */
/* vec = vector of identification strings: vendor, version, date */

void
copy_bios_info_strings (char** vec)
{
  void* map_entry;
  size_t map_entry_len;
  uint8_t* pmem_entry; 
  uint8_t* sigp;

  vec[0] = vec[1] = vec[2] = NULL;
  pmem_entry = map_physmem (SMBIOS_AREA_START, SMBIOS_AREA_LEN, &map_entry, &map_entry_len);
  for (sigp = pmem_entry; sigp - pmem_entry < SMBIOS_AREA_LEN; sigp += SMBIOS_AREA_ALIGN)
    {
      if (is_smbios_entry (sigp))
        {
          uint16_t s_table_len;
          uint8_t* pmem_table;
          void* map_table;
          size_t map_table_len;
          uint8_t* bios_info_p;

          s_table_len = *(uint16_t*)(sigp + SMBIOS_ENTRY_TLEN_OFFSET);
          pmem_table = map_physmem (*(uint32_t*)(sigp + SMBIOS_ENTRY_PTR_OFFSET), s_table_len,
                                    &map_table, &map_table_len);
          bios_info_p = pmem_table;
          while (bios_info_p - pmem_table < s_table_len)
            {
              int scan_only;

              scan_only = (*bios_info_p != SMBIOS_BIOS_INFO_SIG);
              copy_bios_info_strings_internal (&bios_info_p, scan_only, vec);
              if (!scan_only) break;
            }
          munmap (map_table, map_table_len);
        }
    }
  munmap (map_entry, map_entry_len);
}
