/**
 *
 * Copyright (C) 2004 Brett Hall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <config.h>

#include <string.h>
#include <stdlib.h>

#include <panel-applet.h>
#include <gtk/gtk.h>
#include <gnome.h>
#include <gconf/gconf-client.h>
#include <librsvg/rsvg.h>
#include <libgnomevfs/gnome-vfs.h>

/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*forward declarations*/
const gchar* bloglines_uri =
   "http://rpc.bloglines.com/update?user=bretthall@speedymail.org&ver=1";

const gint millis_per_minute = 60000;
const gint bloglines_buffer_length = 30;
const gint bl_timeout_len = 5000;
const gint bl_browser_timeout_len = 10000;
static guint bloglines_timer_id = 0;

typedef enum 
{
   BL_NO_NEWS, BL_HAVE_NEWS, BL_CHECKING, BL_ERROR
} BLState;

static BLState bl_state = BL_NO_NEWS;
static guint bl_num_new_items = 0;

static gboolean
start_bloglines_ping(gpointer data);

static void
launch_browser();

/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*Icons*/

static GdkPixbuf* no_news_icon = NULL;
static GdkPixbuf* have_news_icon = NULL;
static GdkPixbuf* checking_icon = NULL;
static GdkPixbuf* error_icon = NULL;

static const gchar* no_news_icon_file = "bloglines_applet/no_news.svg";
static const gchar* have_news_icon_file = "bloglines_applet/have_news.svg";
static const gchar* checking_icon_file = "bloglines_applet/checking.svg";
static const gchar* error_icon_file = "bloglines_applet/error.svg";
static const gchar* bl_icon_file = "bloglines_applet/bloglines.svg";

static gint icon_size = 0;

static void
free_icons()
{
   if(no_news_icon)
      g_object_unref(no_news_icon);
   if(have_news_icon)
      g_object_unref(have_news_icon);
   if(checking_icon)
      g_object_unref(checking_icon);
   if(error_icon)
      g_object_unref(error_icon);
   
}

static void
load_icons(PanelApplet* applet)
{
   gint size = (gint)panel_applet_get_size(applet);
   
   if(size != icon_size)
   {
      if(no_news_icon != NULL)
         free_icons();
      
      /*figure out full path to files*/
      gchar* full_no_news = gnome_program_locate_file(NULL,
                                                      GNOME_FILE_DOMAIN_PIXMAP,
                                                      no_news_icon_file,
                                                      TRUE,
                                                      NULL);
      g_assert(full_no_news != NULL);
      gchar* full_have_news = gnome_program_locate_file(NULL,
                                                        GNOME_FILE_DOMAIN_PIXMAP,
                                                        have_news_icon_file,
                                                        TRUE,
                                                        NULL);
      g_assert(full_have_news != NULL);
      gchar* full_checking = gnome_program_locate_file(NULL,
                                                       GNOME_FILE_DOMAIN_PIXMAP,
                                                       checking_icon_file,
                                                       TRUE,
                                                       NULL);
      g_assert(full_checking != NULL);
      gchar* full_error = gnome_program_locate_file(NULL,
                                                    GNOME_FILE_DOMAIN_PIXMAP,
                                                    error_icon_file,
                                                    TRUE,
                                                    NULL);
      g_assert(full_error != NULL);
      
      /*load and scale icon to panel size*/
      no_news_icon = rsvg_pixbuf_from_file_at_size(full_no_news, size, size, NULL);
      have_news_icon = rsvg_pixbuf_from_file_at_size(full_have_news, size, size, NULL);
      checking_icon = rsvg_pixbuf_from_file_at_size(full_checking, size, size, NULL);
      error_icon = rsvg_pixbuf_from_file_at_size(full_error, size, size, NULL);
      
      g_free(full_no_news);
      g_free(full_have_news);
      g_free(full_checking);
      g_free(full_error);

      icon_size = size;
   }
}

/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*Applet options*/
static gchar* account = NULL;  /*bloglines account email address*/
static guint ping_interval = 600; /*interval between blogline pings in
                                   * seconds*/
static const gchar* gconf_app_dir = "/apps/bloglines_notifier";
static const gchar* account_key = "/apps/bloglines_notifier/account";
static const gchar* interval_key = "/apps/bloglines_notifier/interval";

/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*Applet instance struct and list of instances*/
typedef struct _Instance Instance;
struct _Instance
{
   PanelApplet* applet;
   GtkWidget* ebox;
   GtkWidget* image;
   GtkTooltips* tooltips;

   GtkWidget* about;
   GtkWidget* prefs;
};

GSList* instances = NULL;

static void
instance_destroy(GtkWidget* widget, gpointer data)
{
   instances = g_slist_remove(instances, data);

   Instance* instance = data;
   
   if(instance->about)
   {      
      gtk_widget_destroy(instance->about);
      instance->about = NULL;
   }
   if(instance->prefs)
   {
      gtk_widget_destroy(instance->prefs);
      instance->prefs = NULL;
   }
   
   g_free(data);
}

/*double click handler for event box*/
static gboolean
on_button_press(GtkWidget* box,
				GdkEventButton* event,
				gpointer data)
{
   //do we have double click with the left button?
   if(event->button == 1 && event->type == GDK_2BUTTON_PRESS)
   {
      launch_browser();
      
	  return TRUE;
   }
  
   return FALSE;
}

static Instance*
instance_create(PanelApplet* applet)
{
   Instance* instance = g_new0(Instance, 1);

   instance->applet = applet;
   
   instance->ebox = gtk_event_box_new();
   gtk_container_add(GTK_CONTAINER(applet), instance->ebox);

   g_signal_connect(G_OBJECT(instance->ebox), 
                    "button_press_event",
                    G_CALLBACK(on_button_press),
                    NULL);
   g_signal_connect(G_OBJECT(instance->ebox),
                    "destroy",
                    G_CALLBACK(instance_destroy),
                    instance);

   instance->image = gtk_image_new_from_file(no_news_icon_file);
   gtk_container_add(GTK_CONTAINER(instance->ebox), instance->image);

   instance->tooltips = gtk_tooltips_new();

   instances = g_slist_append(instances, instance);

   return instance;
}

static void
instance_set_icon(Instance* instance, GdkPixbuf* icon)
{
   gtk_image_set_from_pixbuf(GTK_IMAGE(instance->image), icon);
}

static void
instance_set_tooltip(Instance* instance, const gchar* tip)
{
   gtk_tooltips_set_tip(instance->tooltips,
                        instance->ebox,
                        tip,
                        NULL);
}

static void
set_state(BLState state, guint num_items)
{
   GdkPixbuf* icon = NULL;
   gchar* tip = NULL;
   
   switch(state)
   {
   case BL_NO_NEWS:
      icon = no_news_icon;
      tip = _("No new items");
      break;
      
   case BL_HAVE_NEWS:
      icon = have_news_icon;
      tip = g_strdup_printf(_("%d new items"), num_items);
      break;
      
   case BL_CHECKING:
      icon = checking_icon;
      tip = _("Checking for new items");
      break;
      
   case BL_ERROR:
      icon = error_icon;
      tip = _("Error communicating with bloglines server");
      break;
   };
   
   g_slist_foreach(instances, (GFunc)instance_set_icon, icon);
   g_slist_foreach(instances, (GFunc)instance_set_tooltip, tip);

   if(num_items > 0)
      g_free(tip);

   bl_state = state;
   bl_num_new_items = num_items;
}

/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*Bloglines update handlers*/
static GnomeVFSAsyncHandle* ping_handle = NULL; 

static void
stop_bloglines_ping()
{
   if(ping_handle)
   {
      gnome_vfs_async_cancel(ping_handle);

      set_state(BL_ERROR, 0);

      ping_handle = NULL;
   }
}

static gboolean
bloglines_ping_timeout(gpointer data)
{
   stop_bloglines_ping();
   
   return FALSE;
}


static void
on_bloglines_read_done(GnomeVFSAsyncHandle* handle,
                       GnomeVFSResult result,
                       gpointer buffer,
                       GnomeVFSFileSize num_req,
                       GnomeVFSFileSize num_read,
                       guint timeout_id)
{
   g_assert(ping_handle == handle);
   
   g_source_remove(timeout_id);
   
   /*make sure we got data*/
   if(result != GNOME_VFS_OK)
   {
      set_state(BL_ERROR, 0);
      g_free(buffer);
      ping_handle = NULL;
      return;
   }

   /*parse the response*/
   gchar* msg = buffer + 1;
   gchar** fields = g_strsplit(msg, "|", 0);
   gchar* num_str = fields[0];
   gint num_msgs = atoi(num_str);
   g_strfreev(fields);
   g_free(buffer);

   /*set the icon and tooltip*/
   if(num_msgs > 0)
   {
      set_state(BL_HAVE_NEWS, num_msgs);
   }
   else
   {
      set_state(BL_NO_NEWS, 0);
   }

   ping_handle = NULL;
}

static void
on_bloglines_open(GnomeVFSAsyncHandle* handle,
                  GnomeVFSResult result,
                  guint timeout_id)
{
   g_assert(ping_handle == handle);
   g_source_remove(timeout_id);
   
   /*make sure we have a connection*/
   if(result != GNOME_VFS_OK)
   {
      set_state(BL_ERROR, 0);
      ping_handle = NULL;
      return;
   }

   /*start the read*/
   gchar* buffer = g_new(gchar, bloglines_buffer_length);
   timeout_id = g_timeout_add(bl_timeout_len, (GSourceFunc)bloglines_ping_timeout, handle);
   gnome_vfs_async_read(handle, buffer, bloglines_buffer_length,
                        (GnomeVFSAsyncReadCallback)on_bloglines_read_done,
                        (gpointer)timeout_id);
}

/*data must be non-null if the timeout is to keep firing*/
static gboolean
start_bloglines_ping(gpointer data)
{
   /*is there already a ping running?*/
   if(ping_handle)
      return;
   
   /*open the connection*/
   guint timeout_id = g_timeout_add(bl_timeout_len, (GSourceFunc)bloglines_ping_timeout, NULL);
   gnome_vfs_async_open(&ping_handle, bloglines_uri, GNOME_VFS_OPEN_READ,
                        GNOME_VFS_PRIORITY_DEFAULT,
                        (GnomeVFSAsyncOpenCallback)on_bloglines_open,
                        (gpointer)timeout_id);
   set_state(BL_CHECKING, 0);

   return data != NULL;
}


/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*Options update handlers (attached to GConf signals)*/
static void
update_options(GConfClient* client,
               guint cnxn_id,
               GConfEntry* entry,
               gpointer data)
{
   if(entry->value == NULL)
      return;
   
   if(g_ascii_strcasecmp(entry->key, account_key) == 0)
   {
      if(account != NULL)
      {
         g_free(account);
         account = NULL;
      }

      account = g_strdup(gconf_value_get_string(entry->value));
      start_bloglines_ping(NULL);
      if(bloglines_timer_id != 0)
         g_source_remove(bloglines_timer_id);
      if(ping_interval > 0)
         bloglines_timer_id = g_timeout_add(ping_interval*millis_per_minute,
                                            start_bloglines_ping,
                                            (gpointer)1);
   }
   else if(g_ascii_strcasecmp(entry->key, interval_key) == 0)
   {
      ping_interval = gconf_value_get_int(entry->value);
      start_bloglines_ping(NULL);
      if(bloglines_timer_id != 0)
         g_source_remove(bloglines_timer_id);
      if(ping_interval > 0)
         bloglines_timer_id = g_timeout_add(ping_interval*millis_per_minute,
                                            start_bloglines_ping,
                                            (gpointer)1);
   }   
}

/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*context menu defintion*/
static const char context_menu[] = 
   "<popup name=\"button3\">\n"
   "   <menuitem name=\"Check Item\" verb=\"DoCheck\" _label=\"Check Now\"\n"
   "             pixtype=\"stock\" pixname=\"gtk-refresh\"/>\n"
   "   <menuitem name=\"Browser Item\" verb=\"DoBrowser\" _label=\"Launch Browser\"\n"
   "             pixtype=\"stock\" pixname=\"gtk-home\"/>\n"   
   "   <separator/>\n"
   "   <menuitem name=\"Pref Item\" verb=\"DoPrefs\" _label=\"Preferences...\"\n"
   "             pixtype=\"stock\" pixname=\"gtk-properties\"/>\n"
   "   <menuitem name=\"Help Item\" verb=\"DoHelp\" _label=\"_Help\"\n"
   " 		 pixtype=\"stock\" pixname=\"gtk-help\"/>\n"
   "   <menuitem name=\"About Item\" verb=\"DoAbout\" _label=\"About ...\"\n"
   "             pixtype=\"stock\" pixname=\"gnome-stock-about\"/>\n"
   "</popup>\n";

/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*Bloglines checking*/

static void
do_blogline_check(BonoboUIComponent *uic, Instance* instance, const gchar *verbname)
{
   start_bloglines_ping(NULL);
}

static void
do_cancel_check(BonoboUIComponent *uic, Instance* instance, const gchar *verbname)
{
   stop_bloglines_ping();
}

/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*Launch Browser*/
static void
launch_browser()
{
   GError* error = NULL;
   if(!gnome_url_show("http://www.bloglines.com/myblogs", &error))
      printf("Error launching browser");
   g_timeout_add(bl_browser_timeout_len, start_bloglines_ping, NULL);
}


static void
do_broswer_launch(BonoboUIComponent *uic, Instance* instance, const gchar *verbname)
{
   launch_browser();
}

/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*Preferences Dialog*/
typedef struct _PrefsDlg PrefsDlg;
struct _PrefsDlg
{
   GtkEntry* account;
   GtkSpinButton* interval;
   guint notify_id;
   Instance* instance;
};

static gboolean
on_account_entry_gain_focus(GtkEntry* account_entry,
                            GdkEventFocus* evt,
                            gpointer user_data)
{
   gtk_editable_select_region(GTK_EDITABLE(account_entry), 0, -1);
   
   return FALSE;
}
static gboolean
on_ping_entry_gain_focus(GtkSpinButton* ping_entry,
                         GdkEventFocus* evt,
                         gpointer user_data)
{
   gtk_editable_select_region(GTK_EDITABLE(ping_entry), 0, -1);

   return FALSE;
}

static gboolean
on_account_entry_lose_focus(GtkEntry* account_entry,
                            GdkEventFocus* evt,
                            gpointer user_data)
{
   gconf_client_set_string(gconf_client_get_default(),
                           account_key,
                           gtk_entry_get_text(account_entry),
                           NULL);

   return FALSE;
}
static gboolean
on_ping_entry_lose_focus(GtkSpinButton* ping_entry,
                         GdkEventFocus* evt,
                         gpointer user_data)
{
   /*convert value to int*/
   gint value = gtk_spin_button_get_value_as_int(ping_entry);
   
   /*save value*/
   gconf_client_set_int(gconf_client_get_default(),
                        interval_key,
                        value,
                        NULL);

   return FALSE;
}

static void
update_dlg_options(GConfClient* client,
                   guint cnxn_id,
                   GConfEntry* entry,
                   PrefsDlg* dlg)
{
   if(entry->value == NULL)
      return;
   
   if(g_ascii_strcasecmp(entry->key, account_key) == 0)
   {
      gtk_entry_set_text(dlg->account, gconf_value_get_string(entry->value));
   }
   else if(g_ascii_strcasecmp(entry->key, interval_key) == 0)
   {
      gtk_spin_button_set_value(dlg->interval, (gdouble)gconf_value_get_int(entry->value));
   }   
}

static void
on_prefs_destroy(GtkWidget* widget, gpointer data)
{
   PrefsDlg* prefs = data;
   gconf_client_notify_remove(gconf_client_get_default(), prefs->notify_id);
   
   g_free(data);
}

static void
on_prefs_response(GtkDialog *dialog, gint id, PrefsDlg* prefs)
{
   if(id == GTK_RESPONSE_HELP)
   {
      gnome_help_display("bloglines_applet.xml", "bloglines-prefs", NULL);
   }
   else
   {
      /*make sure our values get saved*/
      on_account_entry_lose_focus(prefs->account, NULL, NULL);
      on_ping_entry_lose_focus(prefs->interval, NULL, NULL);

      /*Destroy the dialog*/
      gtk_widget_destroy(GTK_WIDGET(dialog));
      prefs->instance->prefs = 0;
   }
}

static void 
display_prefs_dlg(BonoboUIComponent *uic, Instance* instance, const gchar *verbname)
{
   /*do we have an existing dialog?*/
   if(instance->prefs)
   {
      gtk_window_set_screen(GTK_WINDOW (instance->prefs),
                            gtk_widget_get_screen(GTK_WIDGET(instance->applet)));
      
      gtk_window_present(GTK_WINDOW(instance->prefs));
      return;
   }
   
   /*create the dialog*/
   GtkDialog* dlg = GTK_DIALOG(gtk_dialog_new_with_buttons(
                                  _("Bloglines Notifier Preferences"),
                                  NULL,
                                  GTK_DIALOG_NO_SEPARATOR|
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_STOCK_CLOSE, 
                                  GTK_RESPONSE_CLOSE,
                                  GTK_STOCK_HELP, 
                                  GTK_RESPONSE_HELP,                                  
                                  NULL));
   GtkBox* vbox = GTK_BOX(dlg->vbox);
   gtk_box_set_spacing(vbox, 10);

   /*Create title*/
   gchar* title = g_strconcat ("<span weight=\"bold\">", _("Options"), "</span>", NULL);
   GtkWidget* label = gtk_label_new (title);
   gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
   gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
   gtk_box_pack_start (vbox, label, FALSE, FALSE, 0);
   g_free (title);

   /*create the options*/
   GtkBox* account_box = GTK_BOX(gtk_hbox_new(FALSE, 5));
   GtkWidget* account_lbl = gtk_label_new_with_mnemonic(_("_Account:"));   
   gtk_box_pack_start(account_box, GTK_WIDGET(account_lbl), FALSE, FALSE, 0);
   GtkEntry* account_entry = GTK_ENTRY(gtk_entry_new());
   gtk_entry_set_text(account_entry, account);
   gtk_entry_set_width_chars(account_entry, 40);
   gtk_label_set_mnemonic_widget(GTK_LABEL(account_lbl), GTK_WIDGET(account_entry));
   gtk_box_pack_start(account_box, GTK_WIDGET(account_entry), TRUE, TRUE, 0);
   gtk_box_pack_start(vbox, GTK_WIDGET(account_box), FALSE, FALSE, 0);
   
   GtkBox* ping_box = GTK_BOX(gtk_hbox_new(FALSE, 5));
   GtkWidget* ping_lbl = gtk_label_new_with_mnemonic(_("Check for _new items every:"));   
   gtk_box_pack_start(ping_box, GTK_WIDGET(ping_lbl), FALSE, FALSE, 0);
   GtkAdjustment* ping_adj = GTK_ADJUSTMENT(gtk_adjustment_new(ping_interval, 0.0, 10000.0, 1.0, 10.0, 10.0));
   GtkSpinButton* ping_entry = GTK_SPIN_BUTTON(gtk_spin_button_new(ping_adj, 1.0, 0));
   gtk_spin_button_set_numeric(ping_entry, TRUE);
   gtk_entry_set_width_chars(GTK_ENTRY(account_entry), 5);
   gtk_label_set_mnemonic_widget(GTK_LABEL(ping_lbl), GTK_WIDGET(ping_entry));
   gtk_box_pack_start(ping_box, GTK_WIDGET(ping_entry), FALSE, FALSE, 0);
   GtkWidget* min_lbl = gtk_label_new(_("minutes"));   
   gtk_box_pack_start(ping_box, GTK_WIDGET(min_lbl), FALSE, FALSE, 0);
   gtk_box_pack_start(vbox, GTK_WIDGET(ping_box), FALSE, FALSE, 0);

   /*create PrefsDlg object and hook up signals*/
   PrefsDlg* prefs = g_new0(PrefsDlg, 1);
   prefs->account = account_entry;
   prefs->interval = ping_entry;
   prefs->instance = instance;
   g_signal_connect(G_OBJECT(dlg),
                    "destroy",
                    G_CALLBACK(on_prefs_destroy),
                    prefs);
   g_signal_connect(G_OBJECT(dlg),
                    "response",
                    G_CALLBACK(on_prefs_response),
                    prefs);
   g_signal_connect(G_OBJECT(account_entry),
                    "focus-in-event",
                    G_CALLBACK(on_account_entry_gain_focus),
                    NULL);
   g_signal_connect(G_OBJECT(ping_entry),
                    "focus-in-event",
                    G_CALLBACK(on_ping_entry_gain_focus),
                    NULL);
   g_signal_connect(G_OBJECT(account_entry),
                    "focus-out-event",
                    G_CALLBACK(on_account_entry_lose_focus),
                    NULL);
   g_signal_connect(G_OBJECT(ping_entry),
                    "focus-out-event",
                    G_CALLBACK(on_ping_entry_lose_focus),
                    NULL);
   
   prefs->notify_id = gconf_client_notify_add(gconf_client_get_default(),
                                              gconf_app_dir,
                                              (GConfClientNotifyFunc)update_dlg_options,
                                              prefs,
                                              NULL,
                                              NULL);

   /*set up tool-tips*/
   GtkTooltips* tips = GTK_TOOLTIPS(gtk_tooltips_new());
   gtk_tooltips_set_tip(tips,
                        GTK_WIDGET(account_entry),
                        _("The email address used when signing up for a bloglines account"),
                        NULL);
   gtk_tooltips_set_tip(tips,
                        GTK_WIDGET(ping_entry),
                        _("The number of minutes between checks of the bloglines account"),
                        NULL);

   /*finish setting up window and show it*/
   instance->prefs = GTK_WIDGET(dlg);
   gtk_window_set_wmclass(GTK_WINDOW(instance->prefs),
                          "bloglines_notifier",
                          "Bloglines_notifier");
   
   gtk_window_set_screen(GTK_WINDOW(instance->prefs),
                         gtk_widget_get_screen(GTK_WIDGET(instance->applet)));
   
   g_signal_connect(G_OBJECT(instance->prefs), "destroy",
                    G_CALLBACK(gtk_widget_destroyed), &instance->prefs );
   
   gtk_widget_show_all(instance->prefs);   
}

/*!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*About Dialog*/
static void
display_help(BonoboUIComponent *uic, Instance* instance, const gchar *verbname)
{
   GError* error = 0;
   gboolean ok = gnome_help_display("bloglines_applet.xml", NULL, &error);
}

static void 
display_about_dlg(BonoboUIComponent *uic, Instance* instance, const gchar *verbname)
{
   /*load the pixmap*/
   gchar* icon_file = gnome_program_locate_file(NULL,
                                                GNOME_FILE_DOMAIN_PIXMAP,
                                                bl_icon_file,
                                                TRUE,
                                                NULL);
   g_assert(icon_file != NULL);
   GdkPixbuf* icon = rsvg_pixbuf_from_file(icon_file, NULL);
   g_free(icon_file);
   icon_file = NULL;
   
   const gchar* authors[] = {"Brett Hall <bretthall@fastmail.fm>", NULL};

   if(instance->about)
   {
      gtk_window_set_screen(GTK_WINDOW (instance->about),
                            gtk_widget_get_screen(GTK_WIDGET(instance->applet)));
      
      gtk_window_present(GTK_WINDOW(instance->about));
      return;
   }

   instance->about = gnome_about_new("Bloglines Notifier",
                                     VERSION,
                                     "Copyright 2004 Brett Hall",
                                     "Checks a given bloglines account for new items",
                                     authors,
                                     NULL,
                                     NULL,
                                     icon);

   gtk_window_set_wmclass(GTK_WINDOW(instance->about),
                          "bloglines_notifier",
                          "Bloglines_notifier");
   
   gtk_window_set_screen(GTK_WINDOW(instance->about),
                         gtk_widget_get_screen(GTK_WIDGET(instance->applet)));
   
/*    gnome_window_icon_set_from_file(GTK_WINDOW(instance->about), */
/*                                     GNOME_ICONDIR"/gnome-mailcheck.png"); */
   
   g_signal_connect(G_OBJECT(instance->about), "destroy",
                    G_CALLBACK(gtk_widget_destroyed), &instance->about );
   
   gtk_widget_show(instance->about);
}


static const BonoboUIVerb context_menu_verbs[] = 
{
   BONOBO_UI_UNSAFE_VERB("DoCheck", do_blogline_check),
   BONOBO_UI_UNSAFE_VERB("DoBrowser", do_broswer_launch),
   BONOBO_UI_UNSAFE_VERB("DoPrefs", display_prefs_dlg),
   BONOBO_UI_UNSAFE_VERB("DoHelp", display_help),
   BONOBO_UI_UNSAFE_VERB("DoAbout", display_about_dlg),
   BONOBO_UI_VERB_END
};

static void
on_panel_size_change(PanelApplet* applet, gint size, Instance instance)
{
   load_icons(applet);
   set_state(bl_state, bl_num_new_items);
}

static gboolean
init_applet(PanelApplet* applet, const gchar* iid, gpointer data)
{

   if(strcmp(iid, "OAFIID:BloglinesApplet") != 0)
      return FALSE;

   /*load icons*/
   load_icons(applet);

   /*get options*/
   gconf_client_add_dir(gconf_client_get_default(),
                        gconf_app_dir,
                        GCONF_CLIENT_PRELOAD_ONELEVEL,
                        NULL);
   account = gconf_client_get_string(gconf_client_get_default(),
                                     account_key,
                                     NULL);
   ping_interval = gconf_client_get_int(gconf_client_get_default(),
                                        interval_key,
                                        NULL);
   gconf_client_notify_add(gconf_client_get_default(),
                           gconf_app_dir,
                           update_options,
                           NULL, NULL, NULL);

   /*create panel gui*/
   Instance* instance = instance_create(applet);
   panel_applet_setup_menu(applet, context_menu, context_menu_verbs, instance);

   g_signal_connect(G_OBJECT(applet),
                    "change_size",
                    G_CALLBACK(on_panel_size_change),
                    instance);

   /*do an initial check*/
   start_bloglines_ping(NULL);
   if(bloglines_timer_id == 0 && ping_interval > 0)
      bloglines_timer_id = g_timeout_add(ping_interval*millis_per_minute,
                                         start_bloglines_ping,
                                         NULL);
   
   gtk_widget_show_all(GTK_WIDGET(applet));

   return TRUE;
}


PANEL_APPLET_BONOBO_FACTORY("OAFIID:BloglinesApplet_Factory",
							PANEL_TYPE_APPLET,
							"bloglines_applet",
							"0",
							init_applet,
							NULL);

