/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <cmath>
#include <limits>
#include <iostream>
#include <sstream>
#include "basicCalculator.h" // to be commented out later
#include "evaluator.h"
#include "evalError.h"
#include "Utils.h"
#include "Faddeeva.h"

namespace bmEval
{
  namespace
  {
    const cValue Zero (0.0, 0.0);
    const cValue One (1.0, 0.0);
    const cValue ImagUnit (0.0, 1.0);
    const rValue LogTwo (0.69314718056);
    const rValue LogTen (2.302585092994);
    const rValue OneHundred (100.0);
    const rValue EPS (std::numeric_limits<rValue>::min());
    const rValue MATH_PI (3.1415926535898);
    const rValue VSV (1.0e-4);
    const rValue VGV (1.0e+2);
    const cValue NotANumber (std::numeric_limits<rValue>::quiet_NaN(), 
			       std::numeric_limits<rValue>::quiet_NaN());

    #include "functionList.h"
  }

  std::map<mathToken::tokenType, basicCalculator::mathFunction> basicCalculator::createFnTable ()
  {
    std::map<mathToken::tokenType, basicCalculator::mathFunction> table;

    table [mathToken::OP_POS]   = _pos;
    table [mathToken::OP_NEG]   = _neg;
    table [mathToken::OP_NOT]   = _not;
    table [mathToken::OP_DEL]   = _pos;
    table [mathToken::FN_RE]    = _re;
    table [mathToken::FN_IM]    = _im;
    table [mathToken::FN_ARG]   = _arg;
    table [mathToken::FN_ABS]   = _abs;
    table [mathToken::FN_CONJ]  = _conj;
    table [mathToken::FN_EXP]   = _exp;
    table [mathToken::FN_SQRT]  = _sqrt;
    table [mathToken::FN_CBRT]  = _cbrt;
    table [mathToken::FN_LOG]   = _log;
    table [mathToken::FN_LOG2]  = _log2;
    table [mathToken::FN_LOG10] = _log10;
    table [mathToken::FN_SIN]   = _sin;
    table [mathToken::FN_COS]   = _cos;
    table [mathToken::FN_TAN]   = _tan;   
    table [mathToken::FN_ASIN]  = _asin;  
    table [mathToken::FN_ACOS]  = _acos;    
    table [mathToken::FN_ATAN]  = _atan;  
    table [mathToken::FN_SINH]  = _sinh;  
    table [mathToken::FN_COSH]  = _cosh;  
    table [mathToken::FN_TANH]  = _tanh;  
    table [mathToken::FN_ASINH] = _asinh; 
    table [mathToken::FN_ACOSH] = _acosh; 
    table [mathToken::FN_ATANH] = _atanh; 
    table [mathToken::FN_ERF]   = _erf;
    table [mathToken::FN_ERFC]  = _erfc; 
    table [mathToken::FN_GAMMA] = _gamma; 
    table [mathToken::FN_FLOOR] = _floor;
    table [mathToken::FN_CEIL]  = _ceil;
    table [mathToken::FN_ROUND] = _round;
    table [mathToken::FN_FIX]   = _fix;
    table [mathToken::FN_FRAC]  = _frac;
    table [mathToken::FN_STEP]  = _step;
    table [mathToken::FN_OSTEP] = _ostep; 
    table [mathToken::FN_SIGN]  = _sign;
    table [mathToken::FN_X01CC] = _X01cc;
    table [mathToken::FN_X01OO] = _X01oo;
    table [mathToken::FN_X01CO] = _X01co;
    table [mathToken::FN_X01OC] = _X01oc;
    table [mathToken::FN_DISP]  = _disp;

    return table;
  }

  std::map<mathToken::tokenType, basicCalculator::mathFunction> basicCalculator::sm_fnTable = createFnTable ();

  std::map<mathToken::tokenType, basicCalculator::binaryOperator> basicCalculator::createOpTable ()
  {
    std::map<mathToken::tokenType, basicCalculator::binaryOperator> table;

    table[mathToken::OP_AND] = _and;
    table[mathToken::OP_OR ] = _or;
    table[mathToken::OP_XOR] = _xor;
    table[mathToken::OP_LT ] = _lt;
    table[mathToken::OP_GT ] = _gt;
    table[mathToken::OP_LE ] = _le;
    table[mathToken::OP_GE ] = _ge;
    table[mathToken::OP_EQ ] = _eq;
    table[mathToken::OP_NE ] = _ne;
    table[mathToken::OP_ADD] = _add;
    table[mathToken::OP_SUB] = _sub;
    table[mathToken::OP_MUL] = _mul;
    table[mathToken::OP_DIV] = _div;
    table[mathToken::OP_PERCENT] = _percent;
    table[mathToken::OP_MOD]     = _mod;
    table[mathToken::OP_IDIV]    = _idiv;
    table[mathToken::OP_POW] = _pow;
    table[mathToken::OP_MAX] = _max;
    table[mathToken::OP_MIN] = _min;
    table[mathToken::OP_DEF] = _nop;
    table[mathToken::OP_STO] = _nop;
    table[mathToken::OP_ADD_STO] = _add;
    table[mathToken::OP_SUB_STO] = _sub;
    table[mathToken::OP_MUL_STO] = _mul;
    table[mathToken::OP_DIV_STO] = _div;
    table[mathToken::OP_PERCENT_STO] = _percent;
    table[mathToken::OP_MOD_STO] = _mod;
    table[mathToken::OP_IDIV_STO]= _idiv;
    table[mathToken::OP_POW_STO] = _pow;
    table[mathToken::OP_MAX_STO] = _max;
    table[mathToken::OP_MIN_STO] = _min;
    return table;
  }

  std::map<mathToken::tokenType, basicCalculator::binaryOperator> basicCalculator::sm_opTable = createOpTable ();

  void basicCalculator::evaluateBinaryOperator (const mathToken& operation, const mathToken& arg1, const mathToken& arg2, mathToken& result)
  {
    // The evaluation stack should always contain only numbers and variables.
    // Since arguments are coming from there, each of them should be a number
    // or a variable.
    if (arg1.Type() != mathToken::NUMBER && arg1.Type() != mathToken::VARIABLE)
      {
	std::string hint = arg1.Id() + " is not a number nor a variable";                  

	throw evalError ("", "Unexpected token", hint.c_str(), arg1.startPosition());      
      }
    else if (arg2.Type() != mathToken::NUMBER && arg2.Type() != mathToken::VARIABLE)
      {
	std::string hint = arg2.Id() + " is not a number nor a variable";                  
      
	throw evalError ("", "Unexpected token", hint.c_str(), arg2.startPosition());      
      }
    else
      {
	cValue arg1Value, arg2Value;
	bool isReadOnly;
      
	if (arg1.Type() == mathToken::VARIABLE)
	  {
	    if (!m_rVTable.isVariableDefined (arg1.Id(), arg1Value, isReadOnly))
	      {
		std::string hint = arg1.Id() + " is neither a variable nor a function"; 
              
		throw evalError ("", "Undefined variable", hint.c_str(), arg1.startPosition());
	      }
	  }
	else // ARG1 is a number
	  {
	    arg1Value = arg1.Value();
	  }
	if (arg2.Type() == mathToken::VARIABLE)
	  {
	    if (!m_rVTable.isVariableDefined (arg2.Id(), arg2Value, isReadOnly))
	      {
		std::string hint = arg2.Id() + " is neither a variable nor a function"; 
              
		throw evalError ("", "Undefined variable", hint.c_str(), arg2.startPosition());
	      }
	  }
	else // ARG2 is a number
	  {
	    arg2Value = arg2.Value();
	  }
	if (arg1Value == NotANumber || arg2Value == NotANumber)
	  {
	    // std::cerr << "Tried application of " << operation.Id() << " to NaN" << std::endl;
	    result = mathToken (operation.startPosition(), mathToken::NUMBER, "", NotANumber);
	  }
	else
	  {
	    std::map<mathToken::tokenType, basicCalculator::binaryOperator>::const_iterator constIt = sm_opTable.find (operation.Type());

	    if (constIt != sm_opTable.end())
	      {
		try
		  {
		    result = constIt->second(arg1Value, arg2Value, operation.startPosition());
		    if ( !m_isComplexArithmeticAllowed && Utils::isNotZero(result.Value().imag()) == true )
		      {
			throw evalError ("", "Out of domain", 
					 "The result of the operation has non-zero imaginary part", 
					 operation.startPosition());
		      }
		  }
		catch (const evalError& e)
		  {
		    m_lastError = e;
		    result = mathToken (operation.startPosition(), mathToken::NUMBER, "", NotANumber);
		  }
	      }
	    else
	      {
		std::string hint = operation.Id() + " is not a known operator";
          
		throw evalError ("", "Unknown operator", hint.c_str(), operation.startPosition());      
	      } // end of: OPERATION is unknown
	  } // end of: the values of the arguments are valid numbers
      } // end of: every argument is a number or a variable 
  }

  void basicCalculator::evaluateAssignmentOperator (const mathToken& operation, const mathToken& arg1, const mathToken& arg2, mathToken& result)
  {
    // The evaluation stack should always contain only numbers and variables.
    // Since arguments are coming from there, each of them should be a number
    // or a variable.
    // In case of an assignment operator, the first argument should be
    // a variable.
    if (arg1.Type() != mathToken::VARIABLE)
      {
	std::string hint = "Left-hand side of " + operation.Id() + " should always be a variable identifier"; 

	throw evalError ("", "Illegal left-hand side for the operator", hint.c_str(), 
			 operation.startPosition());
      }
    else if (arg2.Type() != mathToken::NUMBER && arg2.Type() != mathToken::VARIABLE)
      {
	std::string hint = arg2.Id() + " is not a number nor a variable";                  
      
	throw evalError ("", "Unexpected token", hint.c_str(), arg2.startPosition());      
      }
    else
      {
	cValue arg1Value, arg2Value;
	bool isReadOnly;

	// ARG1 is always a variable
	if (operation.Type() != mathToken::OP_STO && 
	    operation.Type() != mathToken::OP_DEF && 
	    !m_rVTable.isVariableDefined (arg1.Id(), arg1Value, isReadOnly))
	  {
	    std::string hint = arg1.Id() + " has not been defined yet"; 
	    
	    throw evalError ("", "Undefined variable", hint.c_str(), arg1.startPosition());
	  }
	if (arg2.Type() == mathToken::VARIABLE)
	  {
	    if (!m_rVTable.isVariableDefined (arg2.Id(), arg2Value, isReadOnly))
	      {
		std::string hint = arg2.Id() + " is neither a variable nor a function"; 
		
		throw evalError ("", "Undefined variable", hint.c_str(), arg2.startPosition());
	      }
	  }
	else // ARG2 is a number
	  {
	    arg2Value = arg2.Value();
	  }
	if (arg1Value == NotANumber || arg2Value == NotANumber)
	  {
	    // std::cerr << "Tried application of " << operation.Id() << " to NaN" << std::endl;
	    result = mathToken (operation.startPosition(), mathToken::NUMBER, "", NotANumber);
	  }
	else
	  {
	    std::map<mathToken::tokenType, basicCalculator::binaryOperator>::const_iterator constIt = sm_opTable.find (operation.Type());

	    if (constIt != sm_opTable.end())
	      {
		try
		  {
		    result = constIt->second(arg1Value, arg2Value, operation.startPosition());
		    if ( !m_isComplexArithmeticAllowed && Utils::isNotZero(result.Value().imag()) == true )
		      {
			throw evalError ("", "Out of domain", 
					 "The result of the operation has non-zero imaginary part", 
					 operation.startPosition());
		      }
		  }
		catch (const evalError& e)
		  {
		    m_lastError = e;
		    result = mathToken (operation.startPosition(), mathToken::NUMBER, "", NotANumber);
		  }
		if ( !m_rVTable.setVariable (arg1.Id(), result.Value(), operation.Type() == mathToken::OP_DEF) )
		  {
		    std::string hint = arg1.Id() + " is a constant and cannot be modified";
                      
		    throw evalError ("", "Failed assignment", hint.c_str(), 
				     operation.startPosition());
		  }
	      }
	    else
	      {
		std::string hint = operation.Id() + " is not a known operator";
          
		throw evalError ("", "Unknown operator", hint.c_str(), operation.startPosition());      
	      } // end of: OPERATION is unknown
	  } // end of: the values of the arguments are valid numbers
      } // end of: every argument is a number or a variable 
  }

  void basicCalculator::evaluateFunction (const mathToken& function, const mathToken& arg, mathToken& result)
  {
    // The evaluation stack should always contain only numbers and variables.
    // Since ARG is coming from there, it should be a number or a variable.
    if (arg.Type() != mathToken::NUMBER && arg.Type() != mathToken::VARIABLE)
      {
	std::string hint = arg.Id() + " is neither a number nor a variable";                  
      
	throw evalError ("", "Unexpected token", hint.c_str(), arg.startPosition());      
      }
    else
      {
	bool isReadOnly;
	cValue argValue;

	if (arg.Type() == mathToken::VARIABLE)
	  {
	    if (!m_rVTable.isVariableDefined (arg.Id(), argValue, isReadOnly))
	      {
		std::string hint = arg.Id() + " is neither a variable nor a function"; 
		
		throw evalError ("", "Undefined variable", hint.c_str(), arg.startPosition());
	      }
	  }
	else // ARG is a number
	  {
	    argValue = arg.Value();
	  }
	if (argValue == NotANumber)
	  {
	    // std::cerr << "Tried application of " << function.Id() << " to NaN" << std::endl;
	    result = mathToken (function.startPosition(), mathToken::NUMBER, "", NotANumber);
	  }
	else
	  {
	    std::map<mathToken::tokenType, basicCalculator::mathFunction>::const_iterator constIt = sm_fnTable.find (function.Type());

	    if (constIt != sm_fnTable.end())
	      {
		try
		  {
		    result = constIt->second(argValue, function.startPosition());
		    if ( !m_isComplexArithmeticAllowed && Utils::isNotZero(result.Value().imag()) == true )
		      {
			throw evalError ("", "Out of domain", 
					 "The result of the function evaluation has non-zero imaginary part",
					 function.startPosition());
		      }
		  } // end of try block
		catch (const evalError& e)
		  {
		    m_lastError = e;
		    result = mathToken (function.startPosition(), mathToken::NUMBER, "", NotANumber);
		  }
		if ( function.Type() == mathToken::OP_DEL )
		  {
		    if ( arg.Type() != mathToken::VARIABLE )
		      {
			std::string hint = function.Id() + " must be applied to a variable";
			  
			throw evalError ("", "Wrong argument for operator", hint.c_str(), 
					 function.startPosition());
		      }
		    else
		      {
			m_rVTable.eraseVariable (arg.Id());
		      } // end of: ARG is a variable
		  } // end of: the function is a delete operator
	      }
	    else
	      {
		std::string hint = function.Id() + " is not a known function/operator";                 
          
		throw evalError ("", "Unknown function/operator", hint.c_str(), function.startPosition());      
	      } // end of: FUNCTION is unknown
	  } // end of: the value of the argument is a valid number
      } // end of: the argument is a number or a variable
  }
} // end of namespace bmEval
