/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <cmath>
#include <limits>
#include <iostream>
#include <sstream>
#include "evaluator.h"

namespace bmEval
{
  void evaluator::compileExpression ()
  {
    // An error thrown by parser::tokenize() should be handled at higher level
    std::vector<mathToken> tokenSequence = m_parser.tokenize (m_expr);

#ifdef _DEBUG_
    for (size_t i = 0; i < tokenSequence.size(); i++)
      {
	std::cerr << "@@ Token n. " << i << " --- " ;
	std::cerr << tokenSequence[i] << std::endl;
      }
#endif // _DEBUG_

    // An error thrown by translator::translate() should be handled at higher level
    m_compiledExpr = m_translator.translate (tokenSequence);

#ifdef _DEBUG_
    std::queue<mathToken> outputQueue (m_compiledExpr);
  
    std::cerr << "@@ After applying the shunting yard:" << std::endl;
    for (size_t j = 0; outputQueue.empty() == false; j++)
      {
	std::cerr << "@@ Token n. " << j << " +++ " ;
	std::cerr << outputQueue.front() << std::endl;
	outputQueue.pop();
      }
#endif // _DEBUG_
  }

  cValue evaluator::evaluateCompiledExpression ()
  {
    if ( m_compiledExpr.empty() == true )
      {
	throw evalError ("", "Empty expression", "Provide a valid mathematical expression", 0);
      }
    else
      {
	const std::string nullString = std::string("");
	std::queue <mathToken> copyOfCompiledExpr (m_compiledExpr);
	mathToken currentToken, resultToken;
	mathToken::tokenType typeOfCurrentToken = mathToken::UNDEFINED;
	cValue value;

	do
	  {
	    m_calculator.eraseLastError();
	    currentToken = copyOfCompiledExpr.front();
	    typeOfCurrentToken = currentToken.Type();
	    if (typeOfCurrentToken == mathToken::VARIABLE ||  typeOfCurrentToken == mathToken::NUMBER)
	      {
		m_evaluationStack.push (currentToken);
	      } // end of: the current token is a number or a variable
	    else if ( currentToken.isBinaryOperator() == true ||
		      currentToken.isAssignmentOperator() == true )
	      {
		if ( m_evaluationStack.size() < 2 )
		  {
		    purgeEvaluationStack();

		    std::string hint = std::string("Please, provide argument(s) for operator ") + currentToken.Id();                  

		    throw evalError ("", "Missing argument(s) for operator", hint.c_str(), currentToken.startPosition());
		  }
		else
		  {
		    mathToken arg2 (m_evaluationStack.top());
		    m_evaluationStack.pop();
		    mathToken arg1 (m_evaluationStack.top());
		    m_evaluationStack.pop();
                    try
                      {
		        if (currentToken.isBinaryOperator() == true)
		          {
			    m_calculator.evaluateBinaryOperator (currentToken, arg1, arg2, resultToken);
		          }
		        else
		          {
			    m_calculator.evaluateAssignmentOperator (currentToken, arg1, arg2, resultToken);
			  }
		      }
		    catch (const evalError& e)
                      {
                        purgeEvaluationStack();
                        throw;
		      }
		    if ( m_calculator.getLastError().what() != nullString )
		      {
			m_listOfComputationalErrors.push_back (m_calculator.getLastError());
		      }
		    m_evaluationStack.push(resultToken);
		  }
	      } // end of: the current token is a binary operator
	    else if ( currentToken.isUnaryOperator()  == true )
	      {
		if ( m_evaluationStack.empty() == true )
		  {
		    std::string hint = std::string("Please, provide argument for operator ") + currentToken.Id();                  
                      
		    throw evalError ("", "Missing argument for operator", hint.c_str(), currentToken.startPosition());
		  }
		else
		  {
		    mathToken arg (m_evaluationStack.top());
		    m_evaluationStack.pop();
		    try
		      {
		        m_calculator.evaluateFunction (currentToken, arg, resultToken);
                      }
                    catch (const evalError& e)
                      {
                        purgeEvaluationStack();
                        throw;
                      }
		    if ( m_calculator.getLastError().what() != nullString )
		      {
			m_listOfComputationalErrors.push_back (m_calculator.getLastError());
		      }
		    m_evaluationStack.push(resultToken);
		  }
	      } // end of: the current token is a unary operator
	    else if ( currentToken.isFunction() == true )
	      {
		if ( m_evaluationStack.empty() == true )
		  {
		    std::string hint = std::string("Please, provide argument for function ") + currentToken.Id();                  

		    throw evalError ("", "Missing argument for function", hint.c_str(), currentToken.startPosition());
		  }                  
		else
		  {
		    mathToken arg (m_evaluationStack.top());
		    m_evaluationStack.pop();
		    try
		      {
		        m_calculator.evaluateFunction (currentToken, arg, resultToken);
                      }
                    catch (const evalError& e)
                      {
                         purgeEvaluationStack();
                         throw;
                      }
		    if ( m_calculator.getLastError().what() != nullString )
		      {
			m_listOfComputationalErrors.push_back (m_calculator.getLastError());
		      }
		    m_evaluationStack.push(resultToken);
		  }
	      } // end of: the current token is a function
	    else
	      {
		purgeEvaluationStack();

		std::string hint = std::string("Please, correct this: ") + currentToken.Id();                  

		throw evalError ("", "Found invalid token", hint.c_str(), currentToken.startPosition());
	      } // end of: the current token is not a number/variable/function/operator
	    copyOfCompiledExpr.pop();
	  } while ( !copyOfCompiledExpr.empty() );
	// Whenever we arrive here, the copy of the compiled expression is empty, and
	// the evaluation stack is NOT empty
	if ( m_evaluationStack.size() != 1 )
	  {
	    m_evaluationStack.pop();
	    mathToken unexpectedToken = m_evaluationStack.top();

	    purgeEvaluationStack();          
	    throw evalError ("", "Error occurred at evaluation time: missing operator or function",
			     "Please, provide the necessary function(s)/operator(s)",
			     unexpectedToken.startPosition());
	  } // end of: the evaluation stack has more than one element
	else
	  {
	    // the evaluation stack has exactly one element
	    cValue value;
	    bool isReadOnly;
	    mathToken topElement = m_evaluationStack.top();

	    m_evaluationStack.pop();
	    // The evaluation stack is empty now
	    if (topElement.Type() == mathToken::VARIABLE)
	      {
		if (!m_vTable.isVariableDefined (topElement.Id(), value, isReadOnly))
		  {
		    std::string hint = topElement.Id() + " is neither a variable nor a function"; 
		    
		    throw evalError ("", "Undefined variable", hint.c_str(), topElement.startPosition());
		  }              
	      }
	    else // the element on the top of the stack is a number
	      {
		value = topElement.Value();
	      }
	    return value;
	  } // end of: the evaluation stack has exactly one element
      } // end of: the compiled expression is not empty
  }
} // end of namespace bmEval
