/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "mathToken.h"
#include "Utils.h"

namespace bmEval
{
  const char* mathToken::multiplicationSign = "*";

  std::vector<mathToken::association> mathToken::createOpLookupTable()
  {
    std::vector <mathToken::association> lookupTable;

    lookupTable.push_back(association("!", OP_NOT));
#ifndef _BUILD_WITH_LESS_FUNCTIONALITY_ 
    lookupTable.push_back(association("~", OP_DEL));
#endif // _BUILD_WITH_LESS_FUNCTIONALITY_ 
    lookupTable.push_back(association("&&", OP_AND));    
    lookupTable.push_back(association("||", OP_OR));	    
    lookupTable.push_back(association("^^", OP_XOR));    
    lookupTable.push_back(association("<",  OP_LT));	    
    lookupTable.push_back(association(">",  OP_GT));	    
    lookupTable.push_back(association("<=", OP_LE));	    
    lookupTable.push_back(association(">=", OP_GE));	    
    lookupTable.push_back(association("==", OP_EQ));	    
    lookupTable.push_back(association("!=", OP_NE));	    
    lookupTable.push_back(association("+",  OP_ADD));    
    lookupTable.push_back(association("-",  OP_SUB));    
    lookupTable.push_back(association(multiplicationSign,  OP_MUL));    
    lookupTable.push_back(association("/",  OP_DIV));    
    lookupTable.push_back(association("%",  OP_MOD));    
    lookupTable.push_back(association("//", OP_IDIV));   
    lookupTable.push_back(association("%%",  OP_PERCENT));
    lookupTable.push_back(association("^",  OP_POW));    
    lookupTable.push_back(association("<>", OP_MAX));    
    lookupTable.push_back(association("><", OP_MIN));    
#ifndef _BUILD_WITH_LESS_FUNCTIONALITY_
    lookupTable.push_back(association(":=", OP_DEF));    
    lookupTable.push_back(association("=",  OP_STO));    
    lookupTable.push_back(association("+=", OP_ADD_STO));
    lookupTable.push_back(association("-=", OP_SUB_STO));
    lookupTable.push_back(association("*=", OP_MUL_STO));
    lookupTable.push_back(association("/=", OP_DIV_STO));
    lookupTable.push_back(association("%=", OP_MOD_STO));
    lookupTable.push_back(association("//=", OP_IDIV_STO));
    lookupTable.push_back(association("%%=", OP_PERCENT_STO));
    lookupTable.push_back(association("^=", OP_POW_STO));
    lookupTable.push_back(association("<>=", OP_MAX_STO));
    lookupTable.push_back(association("><=", OP_MIN_STO));  
#endif // _BUILD_WITH_LESS_FUNCTIONALITY_ is not defined
    std::sort (lookupTable.begin(), lookupTable.end());

    return lookupTable;
  }

  std::vector <mathToken::association> mathToken::sm_opLookupTable = mathToken::createOpLookupTable();

  std::vector <mathToken::association> mathToken::createFnLookupTable()
  {
    std::vector<mathToken::association> lookupTable;

    lookupTable.push_back(association("re",    FN_RE));
    lookupTable.push_back(association("im",    FN_IM));
    lookupTable.push_back(association("arg",   FN_ARG));
    lookupTable.push_back(association("abs",   FN_ABS));
    lookupTable.push_back(association("conj",  FN_CONJ));
    lookupTable.push_back(association("exp",   FN_EXP));
    lookupTable.push_back(association("sqrt",  FN_SQRT));
    lookupTable.push_back(association("cbrt",  FN_CBRT));
    lookupTable.push_back(association("log",   FN_LOG));
    lookupTable.push_back(association("log2",  FN_LOG2));
    lookupTable.push_back(association("log10", FN_LOG10));
    lookupTable.push_back(association("sin",   FN_SIN));
    lookupTable.push_back(association("cos",   FN_COS));
    lookupTable.push_back(association("tan",   FN_TAN));
    lookupTable.push_back(association("asin",  FN_ASIN));
    lookupTable.push_back(association("acos",  FN_ACOS));  
    lookupTable.push_back(association("atan",  FN_ATAN));
    lookupTable.push_back(association("sinh",  FN_SINH));
    lookupTable.push_back(association("cosh",  FN_COSH));
    lookupTable.push_back(association("tanh",  FN_TANH));
    lookupTable.push_back(association("asinh", FN_ASINH));
    lookupTable.push_back(association("acosh", FN_ACOSH));
    lookupTable.push_back(association("atanh", FN_ATANH));
    lookupTable.push_back(association("erf",   FN_ERF));
    lookupTable.push_back(association("erfc",  FN_ERFC));
    lookupTable.push_back(association("gamma", FN_GAMMA));
    lookupTable.push_back(association("floor", FN_FLOOR));
    lookupTable.push_back(association("ceil",  FN_CEIL));
    lookupTable.push_back(association("round", FN_ROUND));
    lookupTable.push_back(association("fix",   FN_FIX));
    lookupTable.push_back(association("frac",  FN_FRAC));
    lookupTable.push_back(association("step",  FN_STEP));
    lookupTable.push_back(association("ostep", FN_OSTEP)); 
    lookupTable.push_back(association("sign",  FN_SIGN));
    lookupTable.push_back(association("X01cc", FN_X01CC));
    lookupTable.push_back(association("X01oo", FN_X01OO));
    lookupTable.push_back(association("X01co", FN_X01CO));
    lookupTable.push_back(association("X01oc", FN_X01OC));
#ifndef _BUILD_WITH_LESS_FUNCTIONALITY_
    lookupTable.push_back(association("disp",  FN_DISP));
#endif // _BUILD_WITH_LESS_FUNCTIONALITY_ is not defined 
    std::sort (lookupTable.begin(), lookupTable.end());

    return lookupTable;
  }

  std::vector<mathToken::association> mathToken::sm_fnLookupTable = mathToken::createFnLookupTable();


  void mathToken::extractTokenFrom (const std::string& str, bool isComplexInputAllowed, 
				    size_t& startPosition, tokenType& type, 
				    std::string& id, cValue& value, 
				    size_t& afterEndToken)
  {
    // Ignore initial spaces and set startPosition to the one of the
    // first non-space character
    size_t newStartPosition = Utils::posOfFirstNonSpace (str, startPosition);
    if (newStartPosition == std::string::npos)
      {
	// No token left
	type = END_TOKEN;
	// We can leave ID and VALUE to their default values
	afterEndToken = startPosition;
	return;
      }
    else
      {
	startPosition = newStartPosition;
	std::string substr = str.substr(startPosition);

	// str[startPosition] is neither a space nor the null character
	if (str[startPosition] == oParenthesis)
	  {
	    type = OPEN_PARENTHESIS;
	    id = std::string (1, oParenthesis);
	    // VALUE can be left at its default value
	    afterEndToken = startPosition + 1;
	  }
	else if (str[startPosition] == cParenthesis)
	  {
	    type = CLOSED_PARENTHESIS;
	    id = std::string (1, cParenthesis);
	    // VALUE can be left at its default value
	    afterEndToken = startPosition + 1;
	  }
	else if ( (Utils::doesStartWithSymbols (substr, afterEndToken)) )
	  {
	    id = substr.substr(0, afterEndToken);
	    type = lookForKeyInOperatorsTable (id);
	    afterEndToken += startPosition;
	    // VALUE can be left at its default value
	  }
	else 
	  {
	    int errorCode;
	    if ( (isComplexInputAllowed) )
	      {
		value = Utils::str2c (substr, afterEndToken, errorCode);
	      }
	    else
	      {
		value = Utils::str2r (substr, afterEndToken, errorCode);
	      }
	    if (errorCode == Utils::OVERFLOW_OCCURRED)
	      {
		id = substr.substr(0, afterEndToken);
		type = INVALID_TOKEN;
		afterEndToken += startPosition;
		// VALUE has been already set
	      }
	    else if (errorCode == Utils::CONVERSION_DONE)
	      {
		id = substr.substr(0, afterEndToken);
		type = NUMBER;
		afterEndToken += startPosition;
		// VALUE has been already set
	      }
	    else
	      {
		// errorCode == Utils::NOT_A_VALUE
		if ( (Utils::doesStartWithId (substr, afterEndToken)) )
		  {
		    id = substr.substr(0, afterEndToken);
		    type = lookForKeyInFunctionsTable (id);
		    if (type == INVALID_TOKEN)
		      {
			// If ID has not been found in the lookup table for functions,
			// then assume for now that it is a variable name
			type = VARIABLE;
		      }
		    afterEndToken += startPosition;
		    // VALUE can be left at its default value
		  }
		else
		  {
		    afterEndToken = Utils::posOfFirstSpace (substr);
		    id = substr.substr(0, afterEndToken);
		    type = INVALID_TOKEN;
		    // Since the token has not been recognized
		    // we stop parsing by setting
		    afterEndToken = startPosition;
		    // VALUE can be left at its default value
		  } // end of: token is NOT a variable/function
	      } // end of: token is NOT a number
	  } // end of: token is NEITHER a parenthesis NOR an operator 
      } // end of: SOME token left
  }

  std::ostream& operator<< (std::ostream& os, const mathToken& token)
  {
    os << "position: " << token.m_startPosition << ", id: ("; 
    if (token.m_type == mathToken::UNDEFINED)
      {
	os << ") type: *UNDEFINED*";
      }
    else if (token.m_type == mathToken::NUMBER)
      {
	os << ") type: number, value "  << token.m_value;
      }
    else if (token.m_type == mathToken::VARIABLE)
      {
	os << token.m_id << ") type: variable, value "  << token.m_value;
      }
    else if (token.m_type == mathToken::OPEN_PARENTHESIS)
      {
	os << ") type: open parenthesis";
      }
    else if (token.m_type == mathToken::CLOSED_PARENTHESIS)
      {
	os << ") type: closed parenthesis";
      }
    else if ( (token.isUnaryOperator()) )
      {
	os << token.m_id << ") type: unary operator (" 
	   << token.m_type << ')';
      }
    else if ( (token.isBinaryOperator()) )
      {
	os << token.m_id << ") type: binary operator (" 
	   << token.m_type << ')';
      }
    else if ( (token.isAssignmentOperator()) )
      {
	os << token.m_id << ") type: assignment operator (" 
	   << token.m_type << ')';
      }
    else if ( (token.isFunction()) )
      {
	os << token.m_id << ") type: function (" 
	   << token.m_type << ')';
      }
    else if (token.m_type == mathToken::INVALID_TOKEN)
      {
	os << token.m_id << ") type: *INVALID TOKEN*";
      }  
    else if (token.m_type == mathToken::END_TOKEN)
      {
	os << ") type: *END*";
      }
    else
      {
	os << token.m_id << ") type: =UNKNOWN=";
      }
    return os;
  }
} // end of namespace bmEval
