/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "parser.h"
#include "evalError.h"
#include "Utils.h"

namespace bmEval
{
  std::vector<mathToken> parser::tokenize (const std::string& str) const
  {
    std::vector<mathToken> tokenSequence;
    size_t startWith = 0;
    size_t afterEndToken;
    mathToken::tokenType typeOfLastReadToken;

    do
      {
	tokenSequence.push_back (mathToken(str, m_isComplexInputAllowed, afterEndToken, startWith));
	typeOfLastReadToken = tokenSequence.back().Type();
	startWith = afterEndToken;
	if ( (m_isImplicitMultiplicationAllowed) && 
	     typeOfLastReadToken == mathToken::NUMBER &&
	     Utils::doesIdOrOpenParenthesisFollow (str, startWith) == true )
	  {
	    tokenSequence.push_back ( mathToken (startWith, mathToken::OP_MUL, mathToken::multiplicationSign) );
	  }
      } while (typeOfLastReadToken != mathToken::INVALID_TOKEN && 
	       typeOfLastReadToken != mathToken::END_TOKEN);
    if (typeOfLastReadToken == mathToken::INVALID_TOKEN)
      {
	std::string hint = std::string("Please, correct this: ") + tokenSequence.back().Id();
      
	throw evalError ("", "Found invalid token", hint.c_str(), tokenSequence.back().startPosition());
      }
    // If we arrive here, then the TOKENSEQUENCE ends with an END_TOKEN.
    postprocess (tokenSequence);
    checkArrangement (tokenSequence);
    return tokenSequence;
  }

  void parser::postprocess (std::vector<mathToken>& tokenSequence) const
  {
    bool checkTriggered = false;
    mathToken::tokenType typeOfCurrentToken = mathToken::UNDEFINED;

    // This function checks if the given TOKENSEQUENCE contains
    // or not unary minuses/pluses. Whenever a minus sign (OP_SUB)
    // is discovered to be a unary minus it is redefined as OP_NEG.
    // Whenever a plus sign (OP_ADD) is discovered to be a
    // unary plus it is redefined as OP_POS. 
    // A plus/minus sign is considered to be unary if one of
    // the following situations takes place:
    // 1. The sign is found at the start of the 
    //    token sequence, or
    // 2. The token preceding the sign is either
    //    an operator or an open parenthesis.

    for (size_t i = 0; i < tokenSequence.size(); i++)
      {
	typeOfCurrentToken = tokenSequence[i].Type();
	if (typeOfCurrentToken == mathToken::OP_SUB ||
	    typeOfCurrentToken == mathToken::OP_ADD)
	  {
	    if(i == 0)
	      {
		checkTriggered = true;
	      }
	    else
	      {
		checkTriggered = 
		  (tokenSequence[i-1].Type()== mathToken::OPEN_PARENTHESIS ||
		   tokenSequence[i-1].isOperator()== true);
	      }
	    if( (checkTriggered) )
	      {
		if ( typeOfCurrentToken == mathToken::OP_SUB )
		  {
		    tokenSequence[i].Type(mathToken::OP_NEG);
		  }
		else
		  {
		    tokenSequence[i].Type(mathToken::OP_POS);
		  }
	      } // end of: the check has been triggered
	  } // end of: the current token is either OP_SUB or OP_ADD
      } // for cycle ends here
  }

  // The following functions checks if the tokens forming
  // the TOKENSEQUENCE are properly arranged. If this is not the
  // case, then it throws an error that specifies which token
  // has been found in an unexpected position.
  // This function has been introduced since the shunting yard
  // algorithm used in the translator can catch arrangement errors
  // but cannot always guess their exact position in the
  // token sequence. 
  void parser::checkArrangement (std::vector<mathToken>& tokenSequence) const
  {
    if ( tokenSequence.size() > 0 )
      {
	mathToken* pCurrentToken = &tokenSequence[0];

	if ( pCurrentToken->isBinaryOperator() ||
	     pCurrentToken->isAssignmentOperator() )
	  {
	    std::string hint = std::string("Please, provide argument for operator ") + pCurrentToken->Id();                  
          
	    throw evalError ("", "Missing left-side argument for operator", hint.c_str(), pCurrentToken->startPosition());
	  }
	else if ( pCurrentToken->Type() == mathToken::CLOSED_PARENTHESIS )
	  {
	    throw evalError ("", "Closed parethesis found at the beginning of the expression", 
			     "Please, remove it or add what is missing before it", 
			     pCurrentToken->startPosition());
	  }
	else
	  {
	    // The first token is neither a binary/assignment operator
	    // nor a closed parenthesis. We can proceed with the check.
	    if ( tokenSequence.size() > 1 )
	      {
		// Remark: A non empty sequence ends always with
		// and END_TOKEN.
		for (size_t i = 0; i < tokenSequence.size() - 1; i++)
		  {
		    // check if the (i+1)-th token may follow the i-th one.
		    // If not, throws an error.
		    checkArrangement (tokenSequence[i], tokenSequence[i+1]);
		  }
	      }
	  } // end of: the first token is neither a binary/assignment operator
	// nor a closed parenthesis
      } // end of: the given token sequence is not empty
  }

  // Check if, within a mathematical expression, SECONDTOKEN may follow FIRSTTOKEN.
  // If not, throws an error with a clear explanation of what is wrong.
  // Preconditions: 1. Implicit multiplication has been already taken into account,
  //                   i.e. no implicit multiplication is allowed when
  //                   coming to this function.
  //                2. postprocess() has been already called.
  void parser::checkArrangement (const mathToken& firstToken, const mathToken& secondToken) const
  {
    // Remark: Here we accept that a number can be followed by
    //         an assignment operator, since the check that
    //         the left operand of the assignment is a variable
    //         is done by the evaluator class. The same applies if the first
    //         token is a closed parenthesis.
    //  
    //         We also accept that the delete operator (~) is followed
    //         by a number, since the check that the operand is a variable
    //         is done by the evaluator class.
    if ( firstToken.Type() == mathToken::NUMBER ||
	 firstToken.Type() == mathToken::VARIABLE ||
	 firstToken.Type() == mathToken::CLOSED_PARENTHESIS )
      {
	if ( secondToken.Type() == mathToken::NUMBER ||
	     secondToken.Type() == mathToken::VARIABLE || 
	     secondToken.Type() == mathToken::OPEN_PARENTHESIS || 
	     (secondToken.isFunction()) ||
	     (secondToken.isUnaryOperator()) )
	  {
	    throw evalError ("", "Missing operator immediately before token",
			     "Please, provide the necessary operator",
			     secondToken.startPosition());
	  }
      }
    else if ( firstToken.Type() == mathToken::OPEN_PARENTHESIS )
      {
	if ( secondToken.Type() == mathToken::END_TOKEN )
	  {
	    throw evalError ("", "Found parenthesis mismatch", "Remove or add a parenthesis where needed", 
			     firstToken.startPosition());
	  }
	if ( secondToken.Type() == mathToken::CLOSED_PARENTHESIS )
	  {
	    throw evalError ("", "Empty subexpression found", "Has something been forgotten?", 
			     firstToken.startPosition());
	  }
	if ( (secondToken.isBinaryOperator() || secondToken.isAssignmentOperator()) )
	  {
	    std::string hint = std::string("Please, provide argument for operator ") + secondToken.Id();                  
              
	    throw evalError ("", "Missing argument for operator", hint.c_str(), secondToken.startPosition());	  
	  }
      }
    else if ( (firstToken.isOperator()) )
      {
	if ( secondToken.Type() == mathToken::END_TOKEN ||
	     secondToken.Type() == mathToken::CLOSED_PARENTHESIS ||
	     secondToken.isBinaryOperator() || 
	     secondToken.isAssignmentOperator() )
	  {
	    std::string hint = std::string("Please, provide argument for operator ") + firstToken.Id();                  
              
	    throw evalError ("", "Missing argument for operator", hint.c_str(), firstToken.startPosition());	  
	  }
      }
    else if ( (firstToken.isFunction()) )
      {
	if ( secondToken.Type() != mathToken::OPEN_PARENTHESIS )
	  {
	    std::string hint = std::string("Please, put a parenthesis after the function ") + firstToken.Id();                  
              
	    throw evalError ("", "Missing open parenthesis after the function", hint.c_str(), firstToken.startPosition());	  	  
	  }
      }
    else
      {
	// Function execution should never come here
	std::string hint = firstToken.Id() + " cannot stay here";                  
      
	throw evalError ("", "Unexpected token", hint.c_str(), firstToken.startPosition());	  
      }
  }
} // end of namespace bmEval
