/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <iomanip>
#include <cstdlib>
#include "variablesTable.h"

using namespace std;
using namespace bmEval;

const rValue EPS = 1.0e-15;

bool valuesDiffer (cValue z1, cValue z2)
{
  rValue d1 = fabs(z1.real() - z2.real());
  rValue d2 = fabs(z1.imag() - z2.imag());
  return (d1 >= EPS || d2 >= EPS);
}

void printErrorAndExit (const string& message)
{
  cerr << message << endl;
  exit (EXIT_FAILURE);
}

void testSetVariable (variablesTable& vTable, cValue E, cValue Pi, cValue z, cValue Iz)
{
  if ( !vTable.setVariable (variableDefinition("E", E, true)) )
    {
      printErrorAndExit("*** Unexpected event: Could not define constant E");
    }
  if ( !vTable.setVariable (variableDefinition("Pi", Pi, true)) )
    {
      printErrorAndExit("*** Unexpected event: Could not define constant Pi");
    }
  if ( !vTable.setVariable (variableDefinition("z")) )
    {
      printErrorAndExit("*** Unexpected event: Could not define variable z");
    }
  if ( !vTable.setVariable (variableDefinition("z", z, false)) )
    {
      printErrorAndExit("*** Unexpected event: Could not modify variable z");
    }
  if ( !vTable.setVariable (variableDefinition("Iz", cValue(0,0))) )
    {
      printErrorAndExit("*** Unexpected event: Could not define variable Iz");
    }
  if ( !vTable.setVariable (variableDefinition("Iz", Iz, true)) )
    {
      printErrorAndExit("*** Unexpected event: Could not modify variable Iz");
    }
  if ( (vTable.setVariable (variableDefinition("Pi", z, true))) )
    {
      printErrorAndExit("*** Unexpected event: Could modify constant PI");
    }
  if ( (vTable.setVariable (variableDefinition("Pi"))) )
    {
      printErrorAndExit("*** Unexpected event: Could modify constant PI");
    }
}

void testSetVariables (variablesTable& vTable, vector<variableDefinition> definitions)
{
  cValue E, z, Z, w, W;
  bool E_isReadOnly, z_isReadOnly, Z_isReadOnly, w_isReadOnly, W_isReadOnly;
  
  vTable.setVariables (definitions);
  if (!vTable.isVariableDefined ("E", E, E_isReadOnly))
    {
      printErrorAndExit("*** Unexpected event: constant E is not defined");
    }
  else if (!E_isReadOnly)
    {
      printErrorAndExit("*** Unexpected event: E is not a constant");
    }
  else if ( valuesDiffer (E, cValue(exp(1), 0)) == true )
    {
      printErrorAndExit("*** Unexpected value for E");
    }

  if (!vTable.isVariableDefined ("z", z, z_isReadOnly))
    {
      printErrorAndExit("*** Unexpected event: variable z is not defined");
    }
  else if ((z_isReadOnly))
    {
      printErrorAndExit("*** Unexpected event: z is a constant");
    }
  else if ( valuesDiffer (z, cValue(3, 4)) == true )
    {
      printErrorAndExit("*** Unexpected value for z");
    }  

  if (!vTable.isVariableDefined ("Z", Z, Z_isReadOnly))
    {
      printErrorAndExit("*** Unexpected event: constant Z is not defined");
    }
  else if (!Z_isReadOnly)
    {
      printErrorAndExit("*** Unexpected event: Z is not a constant");
    }
  else if ( valuesDiffer (Z, cValue(3, 4)) == true )
    {
      printErrorAndExit("*** Unexpected value for Z");
    }    

  if (!vTable.isVariableDefined ("w", w, w_isReadOnly))
    {
      printErrorAndExit("*** Unexpected event: variable w is not defined");
    }
  else if ((w_isReadOnly))
    {
      printErrorAndExit("*** Unexpected event: w is a constant");
    }
  else if ( valuesDiffer (w, cValue(3, 4)) == true )
    {
      printErrorAndExit("*** Unexpected value for w");
    }    

  if (!vTable.isVariableDefined ("W", W, W_isReadOnly))
    {
      printErrorAndExit("*** Unexpected event: constant W is not defined");
    }
  else if (!W_isReadOnly)
    {
      printErrorAndExit("*** Unexpected event: W is not a constant");
    }
  else if ( valuesDiffer (W, cValue(0.6, -0.8)) == true )
    {
      printErrorAndExit("*** Unexpected value for W");
    }      
}

void testSetVariables_EraseVariables (variablesTable& vTable, const std::string& prefix)
{
  vector<storedValue> definitions;
  string id, msg("*** Unexpected event: variable ");
  cValue z;
  bool isReadOnly;
  size_t nn = vTable.size();
  
  definitions.push_back (storedValue(cValue( 1, -1), false));
  definitions.push_back (storedValue(cValue( 2,  2), false));
  definitions.push_back (storedValue(cValue(-3,  3), false));
  definitions.push_back (storedValue(cValue(-4, -4), false));
  definitions.push_back (storedValue(cValue( 0,  1), true));

  vTable.setVariables ("@", definitions);
  if (vTable.size() != nn)
    {
      printErrorAndExit ("*** Unexpected events: variables with @ prefix actually defined!");
    }
  for (size_t ii = 0; ii < definitions.size(); ii++)
    {
      id = string("@") + Utils::n2str (ii);
      if ( ( vTable.isVariableDefined (id, z, isReadOnly)) )
	{
	  printErrorAndExit(msg + id + " actually defined!");	  
	}
    }
  
  vTable.setVariables (prefix, definitions);
  if (vTable.size() != definitions.size() + nn)
    {
      printErrorAndExit ("*** Unexpected events: the number of defined variables is not the expected one");
    }
  for (size_t ii = 0; ii < definitions.size(); ii++)
    {
      id = prefix + Utils::n2str (ii);
      if (!vTable.isVariableDefined (id, z, isReadOnly))
	{
	  printErrorAndExit(msg + id + " is not defined");	  
	}
    }

  if ( definitions.size() != vTable.eraseVariables (prefix, match::PREFIX) )
    {
      printErrorAndExit ("*** Unexpected events: the number of erased variables is not the expected one");
    }
  for (size_t ii = 0; ii < definitions.size(); ii++)
    {
      id = prefix + Utils::n2str (ii);
      if ( ( vTable.isVariableDefined (id, z, isReadOnly)) )
	{
	  printErrorAndExit(msg + id + " is still defined");	  
	}
    }
}

void testEraseVariable_isVariableDefined_size_clear (variablesTable& vTable)
{
  cValue E, z;
  bool E_isReadOnly, z_isReadOnly;
  size_t initialSize = vTable.size();

  if (!vTable.isVariableDefined ("E", E, E_isReadOnly))
    {
      printErrorAndExit("*** Unexpected event: constant E is not defined");
    }
  else if (!E_isReadOnly)
    {
      printErrorAndExit("*** Unexpected event: E is not a constant");
    }
  else if ( valuesDiffer (E, cValue(exp(1), 0)) == true )
    {
      printErrorAndExit("*** Unexpected value for E");
    }

  if (!vTable.isVariableDefined ("z", z, z_isReadOnly))
    {
      printErrorAndExit("*** Unexpected event: variable z is not defined");
    }
  else if ((z_isReadOnly))
    {
      printErrorAndExit("*** Unexpected event: z is a constant");
    }
  else if ( valuesDiffer (z, cValue(3, 4)) == true )
    {
      printErrorAndExit("*** Unexpected value for z");
    }  
  
  if ( vTable.eraseVariable("E") != 1 )
    {
      printErrorAndExit ("*** Unexpected events: could not erase variable E");
    }
  else if ( vTable.size() != initialSize - 1 )
    {
      printErrorAndExit ("*** Unexpected events: the list of variables has not the expected size");
    }
  else if (vTable.isVariableDefined ("E", E, E_isReadOnly) == true)
    {
      printErrorAndExit("*** Unexpected event: constant E is still defined after being erased");
    }

  if ( vTable.eraseVariable("z") != 1 )
    {
      printErrorAndExit ("*** Unexpected events: could not erase variable z");
    }
  else if ( vTable.size() != initialSize - 2 )
    {
      printErrorAndExit ("*** Unexpected events: the list of variables has not the expected size");
    }
  else if (vTable.isVariableDefined ("z", z, z_isReadOnly) == true)
    {
      printErrorAndExit("*** Unexpected event: variable z is still defined after being erased");
    }
  vTable.clear ();
  if ( vTable.size() != 0 )
    {
      printErrorAndExit ("*** Unexpected events: the list of variables has a non zero size after being cleared");
    }
}

void checkIdentity (const vector<variablesTable::entry>& ee, const string& prefix, const vector<storedValue>& vv)
{
  for (vector<storedValue>::size_type idx = 0; idx < vv.size(); idx++)
    {
      vector<variablesTable::entry>::size_type jdx;      
      string id = prefix + Utils::n2str (idx);
      for (jdx = 0; jdx < ee.size(); jdx++)
	{
	  if (ee[jdx].m_id == id)
	    {
	      if (valuesDiffer (ee[jdx].m_value, vv[idx].getValue()) == true ||
		  ee[jdx].m_isReadOnly != vv[idx].isReadOnly())
		{
		  printErrorAndExit ("*** Unexpected event: found variable with unexpected value");
		}
	      break;
	    }
	}
      if (jdx == ee.size())
	{
	  printErrorAndExit ("*** Unexpected event: variable not found");
	}
    }
}

void testGetVariables_ListVariables (variablesTable& vTable)
{
  vector<storedValue> real_value;
  vector<storedValue> imag_value;
  vector<storedValue> zQ0Qz;    

  real_value.push_back (storedValue(cValue(1, 0), false));
  real_value.push_back (storedValue(cValue(2, 0), true));
  real_value.push_back (storedValue(cValue(3, 0), false));
  real_value.push_back (storedValue(cValue(4, 0), true));
  real_value.push_back (storedValue(cValue(5, 0), false));
  real_value.push_back (storedValue(cValue(6, 0), true));
  vTable.setVariables ("real_value", real_value);
  imag_value.push_back (storedValue(cValue(0,1), false));
  imag_value.push_back (storedValue(cValue(0,2), true));
  imag_value.push_back (storedValue(cValue(0,3), false));
  imag_value.push_back (storedValue(cValue(0,4), true));
  imag_value.push_back (storedValue(cValue(0,5), false));
  imag_value.push_back (storedValue(cValue(0,6), true));
  vTable.setVariables ("imag_value", imag_value);
  zQ0Qz.push_back (storedValue(cValue(1,+1), false));
  zQ0Qz.push_back (storedValue(cValue(2,-2), true));
  zQ0Qz.push_back (storedValue(cValue(3,+3), false));
  zQ0Qz.push_back (storedValue(cValue(4,-4), true));
  zQ0Qz.push_back (storedValue(cValue(5,+5), false));
  zQ0Qz.push_back (storedValue(cValue(6,-6), true));
  vTable.setVariables ("zQ0Qz", zQ0Qz);  

  cout << "All variables:" << endl;
  vTable.listVariables(cout, "");
  cout << "\nVariables whose ID starts with real_value:" << endl;  
  vTable.listVariables(cout, "real_value");
  cout << "\nVariables whose ID starts with imag_value:" << endl;    
  vTable.listVariables(cout, "imag_value");
  cout << "\nVariables whose ID starts with zQ0Qz:" << endl;      
  vTable.listVariables(cout, "zQ0Qz", match::PREFIX);    
  cout << "\nVariables whose ID starts with x: **NONE**" << endl;      
  vTable.listVariables(cout, "x", match::PREFIX);
  cout << "\nVariables whose ID ends with 1:" << endl;  
  vTable.listVariables(cout, "1", match::SUFFIX);
  cout << "\nVariables whose ID ends with 2:" << endl;    
  vTable.listVariables(cout, "2", match::SUFFIX);
  cout << "\nVariables whose ID ends with 6: ** NONE**" << endl;    
  vTable.listVariables(cout, "6", match::SUFFIX);
  cout << "\nVariables whose ID ends with _value4:" << endl;    
  vTable.listVariables(cout, "_value4", match::SUFFIX);
  cout << "\nVariables whose ID ends with _imag5: ** NONE **" << endl;      
  vTable.listVariables(cout, "_imag5", match::SUFFIX);

  vector<variablesTable::entry> allVariables = vTable.getVariables("");
  if (allVariables.size() != real_value.size() + imag_value.size() + zQ0Qz.size())
    {
      printErrorAndExit ("*** Unexpected event: the total number of variables is not the expected one");
    }
  vector<variablesTable::entry> realVariables = vTable.getVariables("real_value");  
  checkIdentity (realVariables, "real_value", real_value);
  vector<variablesTable::entry> imagVariables = vTable.getVariables("imag_value");  
  checkIdentity (imagVariables, "imag_value", imag_value);
  vector<variablesTable::entry> zQ0Qz_variables = vTable.getVariables("zQ0Qz", match::PREFIX);  
  checkIdentity (zQ0Qz_variables, "zQ0Qz", zQ0Qz);
  vector<variablesTable::entry> xVariables = vTable.getVariables("x", match::PREFIX);
  if (xVariables.size() != 0)
    {
      printErrorAndExit ("*** Unexpected event: found variable whose ID starts with x");
    }

  vector<variablesTable::entry> endingWith1      = vTable.getVariables("1", match::SUFFIX);
  if (endingWith1.size() != 3)
    {
      printErrorAndExit ("*** Unexpected event: the number of variables whose ID ends with 1 is not three");
    }
  else if (endingWith1[0].m_id != "imag_value1" ||
	   valuesDiffer(endingWith1[0].m_value, cValue(0, 2)) == true ||
	   endingWith1[0].m_isReadOnly != true)
    {
      printErrorAndExit ("*** Found variable with unexpected ID/value");
    }
  else if (endingWith1[1].m_id != "real_value1" ||
	   valuesDiffer(endingWith1[1].m_value, cValue(2, 0)) == true ||
	   endingWith1[1].m_isReadOnly != true)
    {
      printErrorAndExit ("*** Found variable with unexpected ID/value");
    }
  else if (endingWith1[2].m_id != "zQ0Qz1" ||
	   valuesDiffer(endingWith1[2].m_value, cValue(2, -2)) == true ||
	   endingWith1[2].m_isReadOnly != true)
    {
      printErrorAndExit ("*** Found variable with unexpected ID/value");
    }
  
  vector<variablesTable::entry> endingWith2      = vTable.getVariables("2", match::SUFFIX);
  if (endingWith2.size() != 3)
    {
      printErrorAndExit ("*** Unexpected event: the number of variables whose ID ends with 2 is not three");
    }
  else if (endingWith2[0].m_id != "imag_value2" ||
  	   valuesDiffer(endingWith2[0].m_value, cValue(0, 3)) == true ||
  	   endingWith2[0].m_isReadOnly != false)
    {
      printErrorAndExit ("*** Found variable with unexpected ID/value");
    }
  else if (endingWith2[1].m_id != "real_value2" ||
  	   valuesDiffer(endingWith2[1].m_value, cValue(3, 0)) == true ||
  	   endingWith2[1].m_isReadOnly != false)
    {
      printErrorAndExit ("*** Found variable with unexpected ID/value");
    }
  else if (endingWith2[2].m_id != "zQ0Qz2" ||
  	   valuesDiffer(endingWith2[2].m_value, cValue(3, 3)) == true ||
  	   endingWith2[2].m_isReadOnly != false)
    {
      printErrorAndExit ("*** Found variable with unexpected ID/value");
    }
  
  vector<variablesTable::entry> endingWith6      = vTable.getVariables("6", match::SUFFIX);
  if (endingWith6.size() != 0)
    {
      printErrorAndExit ("*** Unexpected event: There are variables whose ID ends with 6!");
    }

  vector<variablesTable::entry> endingWith_value4 = vTable.getVariables("_value4", match::SUFFIX);
  if (endingWith_value4.size() != 2)
    {
      printErrorAndExit ("*** Unexpected event: the number of variables whose ID ends with _value4 is not two");
    }
  else if (endingWith_value4[0].m_id != "imag_value4" ||
  	   valuesDiffer(endingWith_value4[0].m_value, cValue(0, 5)) == true ||
  	   endingWith_value4[0].m_isReadOnly != false)
    {
      printErrorAndExit ("*** Found variable with unexpected ID/value");
    }
  else if (endingWith_value4[1].m_id != "real_value4" ||
  	   valuesDiffer(endingWith_value4[1].m_value, cValue(5, 0)) == true ||
  	   endingWith_value4[1].m_isReadOnly != false)
    {
      printErrorAndExit ("*** Found variable with unexpected ID/value");
    }
  
  vector<variablesTable::entry> endingWith_imag5 = vTable.getVariables("_imag5", match::SUFFIX);
  if (endingWith_imag5.size() != 0)
    {
      printErrorAndExit ("*** Unexpected event: There are variables whose ID ends with _imag5!");
    }
}

int main (void)
{
  variablesTable vTable;
  cValue E (exp(1), 0);
  cValue Pi (M_PI, 0);
  cValue z (3, 4);
  cValue Iz (0.6, -0.8);

  if (vTable.size() != 0)
    {
      printErrorAndExit ("*** Unexpected events: the number of defined variables is not zero");
    }  
  try
    {
      variableDefinition def("1wrong", z, false);
    }
  catch (const runtime_error& e)
    {
      cout << "+++ Exception successfully caught:\n+++ " << e.what() << '\n' << endl;
    }

  cout << "[1] Test of variableTables::setVariable(...):" << endl;
  testSetVariable (vTable, E, Pi, z, Iz);
  cout << "    OK\n" << endl;

  vector<variableDefinition> definitions;
  storedValue w, W;

  // The first two redefinitions will be ignored
  definitions.push_back (variableDefinition ("E", z, true));
  definitions.push_back (variableDefinition ("E", z, false));

  definitions.push_back (variableDefinition ("Z", storedValue (Iz, false)));
  definitions.push_back (variableDefinition ("Z", z, true));
  // The following request will be ignored
  definitions.push_back (variableDefinition ("Z", E, false));

  w.setTo (z, false);
  W.setValueTo (Iz);
  W.setReadOnlyFlagTo (true);
  
  definitions.push_back (variableDefinition("w", w));  
  definitions.push_back (variableDefinition("W", W));
  cout << "[2] Test of variableTables::setVariables(vector<variableDefinition>):" << endl;  
  testSetVariables (vTable, definitions);
  cout << "    OK\n" << endl;

  cout << "[3] Test of variableTables::setVariables(const string&, const vector<storedValue>&)," << endl;
  cout << "    Test of variableTables::eraseVariables(const string&, matchType::match::PREFIX):" << endl;    
  testSetVariables_EraseVariables (vTable, "x");
  testSetVariables_EraseVariables (vTable, "_");
  testSetVariables_EraseVariables (vTable, "_X");
  testSetVariables_EraseVariables (vTable, "__xx");  
  testSetVariables_EraseVariables (vTable, "xx12");
  cout << "    OK\n" << endl;

  cout << "[4] Test of variableTables::eraseVariable (const std::string&)," << endl;
  cout << "    Test of variableTables::isVariableDefined (const std::string&, cValue&, bool&)," << endl;
  cout << "    Test of variableTables::clear()," << endl;
  cout << "    Test of variableTables::size():" << endl;  
  testEraseVariable_isVariableDefined_size_clear (vTable);
  cout << "    OK\n" << endl;

  cout << "[5] Test of variableTables::getVariables (const std::string&, matchType::match::PREFIX)," << endl;
  cout << "    Test of variableTables::listVariables (std::ostream&, const std::string&, matchType::match::PREFIX):" << endl;  
  testGetVariables_ListVariables (vTable);
  cout << "    OK\n" << endl;
  
  return EXIT_SUCCESS;
}

