/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <cstdlib>
#include <iostream>
#include <iomanip>
#include <string>
#include "evaluator.h"
#include "Utils.h"

using namespace std;
using namespace bmEval;

void print_help (const char* programName)
{
  cout << "\nUSAGE:   " << programName << " [-h|-M|-r]" << endl;
  cout << "\nA simple evaluator of mathematical expressions." << endl;
  cout << "\nOptions:" << endl;
  cout << "-h: Display this help message" << endl;
  cout << "-M: Do not allow for implicit multiplication" << endl;
  cout << "-r: Disable complex arithmetic\n" << endl;
}

void setOptions (int argc, char* argv[], 
                 bool& allowForImplicitMultiplication,
                 bool& allowForComplexArithmetic)
{
  allowForImplicitMultiplication = true;
  allowForComplexArithmetic = true;

  for (int i = 1; i < argc; i++)
    {
      if ( *argv[i] != '-' )
        {
          cerr << "*** Wrong option: " << argv[i] << endl;
          print_help (argv[0]);
          exit (1);
        }
      switch (argv[i][1])
        {
        case 'h':
          print_help (argv[0]);
          exit (0);
          break;
        case 'M':
          allowForImplicitMultiplication = false;
          break;
        case 'r':
          allowForComplexArithmetic = false;
          break;
        default:
          cerr << "*** Wrong option: " << argv[i] << endl;
          print_help (argv[0]);
          exit(1);
          break;
        }
    }
}

void printIntro ()
{
  cout << "USAGE: type a mathematical expression after the prompt sign (>>)\n"
       << "       and press <Enter> to evaluate it." << endl;
  cout << "       Press <Ctrl> and 'C' together to leave the program.\n" << endl;
}

int main (int argc, char* argv[])
{
  const char startOfComment = '#';
  evaluator theEvaluator;
  bool allowForImplicitMultiplication, allowForComplexArithmetic;
  std::string inputLine;
  cValue result;
  size_t idx;
  unsigned long lineno = 0;
  
  setOptions (argc, argv, 
              allowForImplicitMultiplication, 
              allowForComplexArithmetic);
  theEvaluator.allowForImplicitMultiplication (allowForImplicitMultiplication);
  theEvaluator.allowForComplexArithmetic (allowForComplexArithmetic);
  theEvaluator.internalVartable().setVariable (variableDefinition("e", cValue (2.718281828459, 0.0), true));
  theEvaluator.internalVartable().setVariable (variableDefinition("pi", cValue (3.141592653590, 0.0), true));
  theEvaluator.internalVartable().setVariable (variableDefinition("z", cValue ( 2.0, -4.0)));

  if ( theEvaluator.internalVartable().setVariable (variableDefinition("e", cValue (1.0, 0.0))) == true )
    {
      cerr << "Fatal error: It should *not* be possible \n"
           << "             to change the value of a\n"
           << "             read-only variable" << endl;
    }
  if ( theEvaluator.internalVartable().setVariable (variableDefinition("z", cValue (3.0, -4.0))) == false )
    {
      cerr << "Fatal error: It should be possible \n"
           << "             to change the value of a\n"
           << "             read-write variable" << endl;
    }  
  if ( theEvaluator.internalVartable().setVariable (variableDefinition("z", cValue (3.0, 4.0), true)) == false )
    {
      cerr << "Fatal error: It should be possible \n"
           << "             to change the value of a\n"
           << "             read-write variable" << endl;
    }  
  printIntro();
  cout.precision (12);
  do
    {
      cout << "  [" << ++lineno << ']' << endl;
      cout << "   0....:....1....:....2....:....3....:....4....:....5....:....6....:....7....:....8" << endl;
      cout << ">> ";
      getline (cin, inputLine);
      cout << "\n.. " << inputLine << endl;
      idx = Utils::posOfFirstNonSpace (inputLine);
      if (idx == string::npos)
	{
	  // The input line contains only spaces.
	  // For our purposes we can then set
	  idx = 0;
	}
      if ( inputLine[idx] != startOfComment )
	{
	  try
	    {
	      result = theEvaluator.evaluate (inputLine.c_str());
	      if ((allowForComplexArithmetic))
		{
		  // Output the result as complex number (in a convenient format)
		  cout << "Result = ";
		  if ( result.real() != 0)
		    {
		      cout << result.real();
		      if (result.imag() != 0)
			{
			  cout << showpos << result.imag() << 'i' << noshowpos;
			}
		    }
		  else // result.real() == 0
		    {
		      if (result.imag() != 0)
			{		      
			  cout << result.imag() << 'i';
			}
		      else
			{
			  cout << '0';
			}
		    } // end of result.real() == 0
		  cout << endl;
		}
	      else
		{
		  cout << "Result = " << result.real() << endl; 
		}
	      if (theEvaluator.listOfComputationalErrors().size() != 0)
		{
		  vector<evalError> listOfComputationalErrors = theEvaluator.listOfComputationalErrors(); 
		  
		  cout << "+++ Caught computational error(s):" << endl;
		  for (size_t i = 0; i < listOfComputationalErrors.size(); i++)
		    cout << listOfComputationalErrors[i] << endl;
		}
	    }
	  catch (const evalError& e)
	    {
	      cout << e << endl;
	    }
	} // end of: the input Line is not a comment
      cout << '\n';
    } while ( cin.eof() == false && 
	      cin.fail() == false && 
	      cin.bad() == false);
  cout << "\n--- List of currently defined variables ---" << endl;
  theEvaluator.internalVartable().listVariables (cout, "");
  return 0;
}
