/*
 * BMovieReviewer Copyright (C) 2009 Michael J. Beer
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */

package data;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.Iterator;

import tools.AppLogger;
import tools.Utils;
import data.formats.XMLBogen;
import data.wrappers.IntWrapper;
import data.wrappers.QualifiedStringList;
import data.wrappers.StringWrapper;
import data.wrappers.TextWrapper;

public class Bogen implements Cloneable {

    private static final long serialVersionUID = 1L;

    public static final String PUNKT_KEIN = "0";

    public static final String PUNKT_EIN = "1";

    public static final String PUNKT_ZWEI = "2";

    public static final String PUNKT_DREI = "3";

    public static final String PUNKT_VIER = "4";

    public static final String PUNKT_FUENF = "5";

    public static final int MAX_PUNKTE = 5;

    public static final int ANMERKUNGEN_MAX_LEN = 120;

    public static final String[] kategorien = { "Unterhaltungswert", "Pornofaktor", "Gewaltdarstellung", "Gewaltverherrlichung", "Niveau",
            "Sexismus", "Professionalität", "Realismus" };

    public static final String[] texKategorien = { "bmUnterhaltungswert", "bmPornofaktor", "bmGewaltdarstellung", "bmGewaltverherrlichung",
            "bmNiveau", "bmSexismus", "bmProfessionalitaet", "bmRealismus" };

    public static final String texMakeBMovieBogen = "bmMakeBMovieBogen";

    public static final String[] TEXTFELDER = { "titel", "originaltitel", "land", "jahr", "genre", "bemerkungen", "technisch", "inhalt",
            "wissenschaft", "bild", "handlung", "rss", "fsk" };

    // Interna, werden von formats.XMLBogen benoetigt
    public static int erstesTextfeld = 5;
    public static int laengsteKategorie = 0;
    public static int laengsterLinkTyp = 0;

    public final static int I_TITEL = 0;

    public final static int I_ORIGINALTITEL = 1;

    public final static int I_LAND = 2;

    public final static int I_JAHR = 3;

    public final static int I_GENRE = 4;

    public final static int I_FSK = 12;

    public final static int I_BEMERKUNGEN = 5;

    public final static int I_TECHNISCH = 6;

    public final static int I_INHALT = 7;

    public final static int I_WISSENSCHAFT = 8;

    public final static int I_BILD = 9;

    public final static int I_HANDLUNG = 10;

    public final static int I_RSS = 11;

    public final static int I_MAX_INDEX = 13;
    public final static int I_MAX_DOCUMENT = 12;

    public final static int I_UNTERHALTUNGSWERT = 0;
    public final static int I_PORNOFAKTOR = 1;
    public final static int I_GEWALTDARSTELLUNG = 2;
    public final static int I_GEWALTVERHERRLICHUNG = 3;
    public final static int I_NIVEAU = 4;
    public final static int I_SEXISMUS = 5;
    public final static int I_PROFESSIONALITAET = 6;
    public final static int I_REALISMUS = 7;
    
    public final static String EMPTY_STRING = ".";

    public final static String[] FSK_TYPES = { "unbekannt", "0", "6", "12", "16", "18", "Keine Freigabe", "Indiziert", "Beschlagnahmt" };

    protected int[] punkte;

    // protected String[] anmerkungen;

    protected StringWrapper[] anmerkungen;

    protected StringWrapper[] texte;

    // protected ZitateList zitate;

    protected QualifiedStringList links, zitate;

    protected StringWrapper cover;

    protected BufferedImage coverImage;

    /**
     * Gibt an, unter welchem Namen die Datei gespeichert werden soll (ohne
     * Endung) Falls null, wird der Titel des Filmes genommen
     */
    public String fileName = null;

    /**
     * Gibt den aktuellen Suchpfad an
     */
    protected String filePath = null;

    public Bogen() {
        punkte = new int[kategorien.length];
        anmerkungen = new StringWrapper[punkte.length];
        for (int i = 0; i < punkte.length; i++) {
            punkte[i] = 0;
            anmerkungen[i] = new StringWrapper("", 25);
        }

        texte = new StringWrapper[I_MAX_INDEX];

        texte[I_TITEL] = new StringWrapper("unbekannt");
        texte[I_ORIGINALTITEL] = new StringWrapper("unbekannt");
        texte[I_LAND] = new StringWrapper("unbekannt");
        texte[I_GENRE] = new StringWrapper("unbekannt");
        texte[I_JAHR] = new IntWrapper(0, 4);
        texte[I_FSK] = new StringWrapper("unbekannt");
        texte[I_TECHNISCH] = new TextWrapper("Dem Publikum ist nichts aufgefallen");
        texte[I_INHALT] = new TextWrapper("Dem Publikum ist nichts aufgefallen");
        texte[I_WISSENSCHAFT] = new TextWrapper("Dem Publikum ist nichts aufgefallen");
        texte[I_BILD] = new TextWrapper("Dem Publikum ist keines aufgefallen");
        texte[I_HANDLUNG] = new TextWrapper("Dem Publikum ist keine aufgefallen");
        texte[I_BEMERKUNGEN] = new TextWrapper("Dem Publikum erscheint nichts bemerkenswert");
        texte[I_RSS] = new TextWrapper("Nichts Neues");
        zitate = new QualifiedStringList("Zitate");
        links = new QualifiedStringList("Link");
        cover = new StringWrapper(Bogen.EMPTY_STRING);
    }

    public void printTex(PrintStream out) {

        // Latex - Header ausgeben
        out.print("\\documentclass [11pt]{article}");
        out.println();
        out.print("\\usepackage[utf8]{inputenc}");
        out.println();
        out.print("\\usepackage[ngerman]{babel}");
        out.println();
        out.print("\\usepackage {a4wide}");
        out.println();
        out.print("\\usepackage{bmovie}");
        out.println();
        out.print("\\begin{document}");
        out.println();

        for (int index = 0; index < punkte.length; index++) {
            if (punkte[index] > 0) {
                out.print("\\" + texKategorien[index]);
                if (this.anmerkungen[index] != null) {
                    out.print("[" + anmerkungen[index] + "]");
                }
                out.print("{" + punkte[index] + "}");
                out.println();
            }
        }

        
        out.println("\\bmTechnisch{" + textToTex(getText(I_TECHNISCH)) + "}");
        out.println("\\bmWissenschaft{" + textToTex(getText(I_WISSENSCHAFT)) + "}");
        out.println("\\bmInhalt{" + textToTex(getText(I_INHALT)) + "}");
        out.println("\\bmBild{" + textToTex(getText(I_BILD)) + "}");
        out.println("\\bmHandlung{" + textToTex(getText(I_HANDLUNG)) + "}");
        out.println("\\bmBemerkungen{" + textToTex(getText(I_BEMERKUNGEN)) + "}");

        if (!zitate.isEmpty()) {
            out.println("\\bmZitate{");
            Iterator<QualifiedString> it = zitate.iterator();
            while (it.hasNext()) {
                String zitat = it.next().getText();
                // Anfuehrungszeichen konvertieren
                int oeffnend = 0;
                for (int i = 0; i < zitat.length(); i++) {
                    if (zitat.charAt(i) == '\"') {
                        System.out.print("\" entdeckt");
                        switch (oeffnend) {
                        case 0:
                            System.out.println("  oeffnendes \"");
                            zitat = zitat.replaceFirst("\"", "``");
                            break;
                        case 1:
                            zitat = zitat.replaceFirst("\"", "\'\'");
                            break;
                        default:
                            throw new RuntimeException();
                        }
                        oeffnend = 1 - oeffnend;
                    }
                }
                System.out.println(zitat);
                out.println(zitat);
                if (it.hasNext()) {
                    out.println("\\\\");
                }
            }
            out.println("}");
        }

        out.println("\\" + texMakeBMovieBogen + "{" + getText(I_TITEL) + "}{" + getText(I_LAND) + "}{" + getText(I_ORIGINALTITEL) + "}{"
                + getText(I_JAHR) + "}{" + getText(I_FSK) + "}{" + getText(I_GENRE) + "}");
        out.print("\\end{document}");
        out.println();
    }

    
    protected String textToTex(String text) {
        if(text == null) {
            throw new IllegalArgumentException();
        }
        String result = text;
        result = result.replaceAll("<br>", "\\\\\\\\ ").replaceAll("<li>", "\\$\\\\bullet\\$");
        return result;
    }
    
    /**
     * Gibt den Bogen als XML-Datei aus
     * 
     * @param out
     *            PrintStream, in den die Ausgabe geschrieben werden soll
     */
    public void printXML(PrintStream out) {
        XMLBogen.printXML(out, this);
    }

    static {

        for (int index = 0; index < kategorien.length; index++) {
            if (kategorien[index].length() > kategorien[laengsteKategorie].length()) {
                laengsteKategorie = index;
            }
        }

        for (int index = 0; index < Link.TYPES.length; index++) {
            if (Link.TYPES[index].length() > Link.TYPES[laengsterLinkTyp].length()) {
                laengsterLinkTyp = index;
            }
        }
    }
    
    public void writeImage(OutputStream out) throws IOException{
        if(out == null) {
            throw new IllegalArgumentException();
        }
        if(getCoverImage() == null) {
            throw new NullPointerException();
        }
        Globals globs = Globals.getInstance();
        int quality = 85;
        //public static void writeImage(BufferedImage image, OutputStream out, int quality, String mime)
        try{
            quality = Integer.parseInt(globs.getProperty("jpeg.quality"));
        }catch(NumberFormatException e){
            AppLogger.warning("jpegquality besitzt nichtnumerischen Wert");
        }
        try{
            Utils.writeImage(getCoverImage(),out, quality, globs.getProperty("jpeg.mime"));
        }catch(IOException e){
            AppLogger.throwing("Bogen", "writeImage(OutputStream", e);
        }
    }

    public static int getLaengsteKategorie() {
        return laengsteKategorie;
    }

    public static int getLaengsterLinkTyp() {
        return laengsterLinkTyp;
    }

    public String getAnmerkung(int index) throws IllegalArgumentException {
        if (index >= anmerkungen.length) {
            throw new IllegalArgumentException();
        }
        return anmerkungen[index].getText();
    }

    public void setAnmerkungen(int index, String anmerkung) throws IllegalArgumentException {
        if (index >= anmerkungen.length) {
            throw new IllegalArgumentException();
        }
        this.anmerkungen[index].setText(anmerkung);
    }

    public int getPunkt(int index) throws IllegalArgumentException {
        if (index >= punkte.length) {
            throw new IllegalArgumentException();
        }
        return punkte[index];
    }

    public void setPunkt(int index, int punkt) throws IllegalArgumentException {
        if (index >= punkte.length || punkt > MAX_PUNKTE || punkt < 0) {
            throw new IllegalArgumentException();
        }

        this.punkte[index] = punkt;
    }

    public void setText(int i, String s) {
        if (i < 0 || i > I_MAX_INDEX) {
            throw new IllegalArgumentException();
        }
        this.texte[i].setText(s);
    }

    public String getText(int i) {
        if (i < 0 || i > I_MAX_INDEX) {
            throw new IllegalArgumentException();
        }
        return this.texte[i].getText();
    }

    public StringWrapper getTextWrapper(int i) {
        if (i < 0 || i > I_MAX_INDEX) {
            throw new IllegalArgumentException();
        }
        return texte[i];
    }

    public StringWrapper[] getTextWrappers() {
        return texte;
    }

    public void setTitel(String titel) {
        if (titel == null) {
            throw new IllegalArgumentException();
        }
        this.setText(I_TITEL, titel);
    }

    public String getTitel() {
        return getText(I_TITEL);
    }

    public void setFSK(int fsk) {
        if (fsk < 0 || fsk >= Bogen.FSK_TYPES.length) {
            throw new IllegalArgumentException();
        }
        texte[I_FSK].setText(Bogen.FSK_TYPES[fsk]);
    }

    public String getFSK() {
        return texte[I_FSK].getText();
    }

    public String getGenre() {
        return getText(I_GENRE);
    }

    public void setGenre(String genre) {
        if (genre == null) {
            throw new IllegalArgumentException();
        }
        this.setText(I_GENRE, genre);
    }

    public int getJahr() {
        return ((IntWrapper) texte[I_JAHR]).getInt();
    }

    public void setJahr(int jahr) {
        ((IntWrapper) texte[I_JAHR]).setInt(jahr);
    }

    public void setJahr(String s) {
        setText(I_JAHR, s);
    }

    public String getLand() {
        return getText(I_LAND);
    }

    public void setLand(String land) {
        setText(I_LAND, land);
    }

    public String getOriginalTitel() {
        return getText(I_ORIGINALTITEL);
    }

    public void setOriginalTitel(String originalTitel) {
        setText(I_ORIGINALTITEL, originalTitel);
    }

    public StringWrapper getTitelWrapper() {
        return texte[I_TITEL];
    }

    public StringWrapper getOriginalTitelWrapper() {
        return texte[I_ORIGINALTITEL];
    }

    public StringWrapper getLandWrapper() {
        return texte[I_LAND];
    }

    public StringWrapper getGenreWrapper() {
        return texte[I_GENRE];
    }

    public IntWrapper getJahrWrapper() {
        return (IntWrapper) texte[I_JAHR];
    }

    public StringWrapper getTechnischWrapper() {
        return texte[I_TECHNISCH];
    }

    public StringWrapper getInhaltWrapper() {
        return texte[I_INHALT];
    }

    public StringWrapper getWissenschaftWrapper() {
        return texte[I_WISSENSCHAFT];
    }

    public StringWrapper getBildWrapper() {
        return texte[I_BILD];
    }

    public StringWrapper getHandlungWrapper() {
        return texte[I_HANDLUNG];
    }

    public StringWrapper getBemerkungenWrapper() {
        return texte[I_BEMERKUNGEN];
    }

    public IntWrapper getFSKWrapper() {
        return (IntWrapper) texte[I_FSK];
    }

    public StringWrapper getAnmerkungWrapper(int i) {
        if (i < 0 || i >= kategorien.length) {
            throw new IllegalArgumentException();
        }
        return anmerkungen[i];
    }

    public QualifiedStringList getZitate() {
        return zitate;
    }

    public QualifiedStringList getLinks() {
        return links;
    }

    public StringWrapper getCover() {
        return cover;
    }
    
    public BufferedImage getCoverImage() {
        return coverImage;
    }
    
    public void setCoverImage(BufferedImage image) {
        if(image == null) {
            throw new IllegalArgumentException();
        }
        this.coverImage = image;
    }

    /**
     * Setzt den Dateinamen neu, ersetzt Leerzeichen durch Unterstriche und
     * entfernt Endung
     * 
     * @param name
     */
    public synchronized void setFileName(String name) {
        fileName = name.trim();
        int nameEnd = fileName.lastIndexOf('.');
        if (nameEnd > 0) {
            fileName = fileName.substring(0, nameEnd).replaceAll("\\W", "_");
            ;
        }
    }

    /**
     * Liefert den zu benutzenden Dateinamen OHNE Endung Ist entweder der Name,
     * der beim letzten "Speichern unter" angegeben wurde, oder der Filmtitel
     * 
     * @return den zu nutzenden Dateinamen
     */
    public synchronized String getFileName() {
        return (fileName == null) ? getText(Bogen.I_TITEL).toLowerCase().replaceAll("\\W", "_") : fileName;
    }

    /**
     * Setzt den Dateipfad neu, mit abschlieszendem Separator
     * 
     * @param name
     */
    public synchronized void setFilePath(String name) {
        int pathEnd = name.lastIndexOf(File.separatorChar);
        if (pathEnd > 0) {
            filePath = name.substring(0, pathEnd + 1);
        }
    }
    
    /**
     * Liefert den aktuellen Pfad Ist entweder der Pfad zur Datei, der beim
     * letzten "Speichern unter" angegeben wurde, oder das aktuelle Verzeichnis
     * 
     * @return den zu nutzenden Dateinamen
     */
    public String getFilePath() {
        return (filePath == null) ? Globals.getInstance().getProperty("basedirectory") : filePath;
    }

    public String toString() {
        String str = getText(I_TITEL) + "    " + getText(I_LAND) + "   " + getText(I_ORIGINALTITEL) + "   " + getText(I_JAHR) + "   "
                + getText(I_FSK) + "   " + getText(I_GENRE) + "\n";

        for (int index = 0; index < punkte.length; index++) {
            if (punkte[index] > 0) {
                if (this.anmerkungen[index] != null) {
                    str = str + "[" + anmerkungen[index] + "]  ";
                }
                str = str + punkte[index] + "\n";
            }
        }
        str = str + getText(I_TECHNISCH) + "\n" + getText(I_WISSENSCHAFT) + "\n" + getText(I_INHALT) + "\n" + getText(I_BILD) + "\n"
                + getText(I_HANDLUNG) + "\n" + getText(I_BEMERKUNGEN) + "\n";

        str = str + "Zitate:\n";
        Iterator<QualifiedString> it = zitate.iterator();
        while (it.hasNext()) {
            str = str + it.next() + "\n";
        }

        return str;
    }

    public synchronized Bogen clone() {
        Bogen cp = new Bogen();

        // Punktwertungen kopieren
        for (int i = 0; i < kategorien.length; i++) {
            cp.setPunkt(i, this.getPunkt(i));
            cp.setAnmerkungen(i, this.getAnmerkung(i));
        }

        for (int i = 0; i < I_MAX_INDEX; i++) {
            cp.setText(i, this.getText(i));
        }

        cp.setFileName(this.getFileName());
        cp.setFilePath(this.getFilePath());
        cp.getCover().setText(this.getCover().getText());
        BufferedImage image = getCoverImage();
        if(image != null) {
            cp.setCoverImage(image.getSubimage(0, 0, image.getWidth(), image.getHeight()));
        }
        for (QualifiedString l : this.getLinks()) {
            cp.getLinks().add(l.clone());
        }

        for (QualifiedString l : this.getZitate()) {
            cp.getZitate().add(l.clone());
        }

        return cp;
    }    
}
