/*
 * BMovieReviewer Copyright (C) 2009 Michael J. Beer
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package data.formats;


import java.io.InputStreamReader;
import java.io.PrintStream;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.XMLEvent;

import tools.XML;
import data.Bogen;
import data.Globals;
import data.Link;
import data.QualifiedString;
import data.Zitat;
import data.wrappers.QualifiedStringList;

public class XMLBogen {

	/**
	 * Formatversion
	 */
	public static final String VERSION = "1.0.0";
	public static final String ZITAT = "zitat";
	public static final String ZITAT_TYP = "wertung";

	public static final String ZITATE = "zitate";

	public static final String LINK = "link";

	public static final String LINKS = "links";

	public static final String LINKS_TYP = "typ";

	// public static final String PUNKT = "punkt";
	public static final String PUNKTWERTUNGEN = "punktwertungen";

	public static final String TEXTFELDER = "textfelder";

	public static final String BMOVIE = "bmovie";

	public static final String PUNKT_VAL = "val";

	public static final String PUNKT_ANMERKUNG = "note";

	public static final String COVER = "cover";

	protected XMLEventReader reader;

	protected XML xmlHelper = null;
	protected XMLEventWriter writer = null;

	protected XMLEventFactory eventFactory;
	protected Bogen bogen;
	protected PrintStream out;

	// ///////////////////////////////////////////////////////////////
	// public Methoden

	public static Bogen readXML(InputStreamReader in, Bogen bogen) {

		XMLInputFactory inputFactory = XMLInputFactory.newInstance();
		try {
			XMLEventReader eventReader = inputFactory.createXMLEventReader(in);
			XMLBogen xml = new XMLBogen(eventReader, bogen);
			return xml.read();

		} catch (XMLStreamException e) {
			System.err.println("XMLBogen::readXML: Fehler mit XMLStream: "
					+ e.toString());
		}
		return null;
	}

	
	public static void printXML(PrintStream out, Bogen bogen) {
		XMLBogen xml;

		try {
			xml = new XMLBogen( XML.getXMLEventWriter(out),	bogen);
			xml.printBMovie();
		} catch (XMLStreamException e) {
			System.err
					.println("Beim Vorbereiten zum Schreiben des XML-Dokuments trat Fehler auf!");
			return;
		}
		out.flush();
	}

	
	public static void printDTD(PrintStream out) {
		(new XMLBogen(out)).dtdToStream();
	}
	// ///////////////////////////////////////////////////////////
	// Konstruktoren

	
	/** 
	 * Erstellt ein XML-Objekt zum drucken der DTD
	 */
	protected XMLBogen(PrintStream out) {
		if(out == null) {
			throw new IllegalArgumentException();
		}
		this.out = out;
	}
	
	/** 
	 * Erstellt ein XML-Objekt zum Lesen aus Stream
	 */
	protected XMLBogen(XMLEventReader reader, Bogen bogen) {
		if (reader == null) {
			throw new IllegalArgumentException();
		}
		if (bogen == null) {
			this.bogen = new Bogen();
		} else {
			this.bogen = bogen;
		}
		this.reader = reader;
		this.xmlHelper = new XML(reader);
	}
	
	/** 
	 * Erstellt ein XML-Objekt zum Schreiben in Stream
	 */
	protected XMLBogen(XMLEventWriter writer, Bogen bogen) {
		if (writer == null) {
			throw new IllegalArgumentException();
		}
		if (bogen == null) {
			this.bogen = new Bogen();
		} else {
			this.bogen = bogen;
		}
		this.writer = writer;
		this.eventFactory = XMLEventFactory.newInstance();
		this.xmlHelper = new XML(writer, eventFactory);
	}

	// ///////////////////////////////////////////////////////////
	// Schreiben von XML

//	protected void printOpeningTag(String tag) throws XMLStreamException {
//		writer.add(eventFactory.createStartElement(new QName(tag), null, null));
//	}
//
//	protected void printClosingTag(String tag) throws XMLStreamException {
//		writer.add(eventFactory.createEndElement(new QName(tag), null));
//	}

	protected void printPunktWertung(int i) throws XMLStreamException {
		if (i >= Bogen.kategorien.length) {
			throw new IllegalArgumentException();
		}
		QName tag = new QName(Bogen.kategorien[i]);
		Set<Attribute> attr = new HashSet<Attribute>();
		attr.add(eventFactory.createAttribute(PUNKT_VAL, Integer.toString(bogen
				.getPunkt(i))));
		attr.add(eventFactory.createAttribute(PUNKT_ANMERKUNG, bogen
				.getAnmerkung(i)));
		writer.add(eventFactory.createStartElement(tag, attr.iterator(), null));
		writer.add(eventFactory.createEndElement(tag, null));
	}

	protected void printTextFeld(int index) throws XMLStreamException {
		xmlHelper.writeText(Bogen.TEXTFELDER[index], bogen.getText(index));
	}


	protected void printZitat(QualifiedString zitat) throws XMLStreamException {
		QName tag = new QName(ZITAT);
		Set<Attribute> attr = new HashSet<Attribute>();
		attr.add(eventFactory.createAttribute(ZITAT_TYP, Zitat.TYPES[zitat
				.getTyp()]));
		writer.add(eventFactory.createStartElement(tag, attr.iterator(), null));
		writer.add(eventFactory.createCharacters(zitat.getText().replaceAll("\"", "\\\"")));
		writer.add(eventFactory.createEndElement(tag, null));
	}

	protected void printLink(QualifiedString link) throws XMLStreamException {
		QName tag = new QName(LINK);
		Set<Attribute> attr = new HashSet<Attribute>();
		attr.add(eventFactory.createAttribute(LINKS_TYP, Link.TYPES[link
				.getTyp()]));
		writer.add(eventFactory.createStartElement(tag, attr.iterator(), null));
		writer.add(eventFactory.createCharacters(link.getText()));
		writer.add(eventFactory.createEndElement(tag, null));
	}
	
	
	protected void printPunktWertungen() throws XMLStreamException {
		xmlHelper.printOpeningTag(PUNKTWERTUNGEN);
		for (int index = 0; index < Bogen.kategorien.length; index++) {
			printPunktWertung(index);
		}
		xmlHelper.printClosingTag(PUNKTWERTUNGEN);
	}

	protected void printTextfelder() throws XMLStreamException {
		xmlHelper.printOpeningTag(TEXTFELDER);
		for (int index = 0; index < Bogen.I_MAX_INDEX; index++) {
			printTextFeld(index);
		}
		xmlHelper.printClosingTag(TEXTFELDER);
	}

	protected void printZitate() throws XMLStreamException {
		xmlHelper.printOpeningTag(ZITATE);
		Iterator<QualifiedString> it = (Iterator<QualifiedString>)bogen.getZitate().iterator();
		while (it.hasNext()) {
			QualifiedString zitat = it.next();
			printZitat(zitat);
		}
		xmlHelper.printClosingTag(ZITATE);
	}

	protected void printLinks() throws XMLStreamException {
		xmlHelper.printOpeningTag(LINKS);
		Iterator<QualifiedString> it = (Iterator<QualifiedString>) bogen.getLinks().iterator();
		while (it.hasNext()) {
			printLink(it.next());
		}
		xmlHelper.printClosingTag(LINKS);
	}

	protected void printCover() throws XMLStreamException {
		xmlHelper.printOpeningTag(COVER);
		writer.add(eventFactory.createCharacters(bogen.getCover().getText()));
		xmlHelper.printClosingTag(COVER);
	}

	protected void printBMovie() throws XMLStreamException {
		QName tag = new QName(BMOVIE);
		// Schreibe Header
		writer.add(eventFactory.createStartDocument(Globals.getInstance().getProperty("encoding")));
		//writer.add(eventFactory.createDTD("<!DOCTYPE BMovie PUBLIC \"Bmovie-" + VERSION + ".dtd\" \"bmovie.dtd\">"));		
		writer.add(eventFactory.createStartElement(tag, null, null));
		printPunktWertungen();
		printTextfelder();
		printZitate();
		printLinks();
		//if(bogen.getCoverImage() != null) {
		printCover();
		//}
		writer.add(eventFactory.createEndElement(tag, null));
		writer.add(eventFactory.createEndDocument());
		writer.flush();
		writer.close();
	}

	// /////////////////////////////////////////////////////////////////////////////////////////
	// XML Lesen...

	@SuppressWarnings("unchecked")
	protected void setzePunktWertung(XMLEvent ev, int index) {
		Iterator<Attribute> attributes = (Iterator<Attribute>) ev
				.asStartElement().getAttributes();
		while (attributes.hasNext()) {
			Attribute attribute = attributes.next();
			if (attribute.getName().toString().equals(PUNKT_VAL)) {
				bogen.setPunkt(index, Integer.parseInt(attribute.getValue()));
			} else if (attribute.getName().toString().equals(PUNKT_ANMERKUNG)) {
				bogen.setAnmerkungen(index, attribute.getValue());
			}
		}

	}

	protected void readPunktwertungen() throws XMLStreamException {
		String tag = "";
		XMLEvent event = null;
		int index = -1;

		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				tag = event.asStartElement().getName().getLocalPart();
				for (int i = 0; i < Bogen.kategorien.length; i++) {
					if (tag.equals(Bogen.kategorien[i])) {
						index = i;
					}
				}
				if (index > -1) {
					setzePunktWertung(event, index);
				} else {
					throw new XMLStreamException("   Zeile " + tag
							+ " nicht wohlgeformt");
				}
			} else if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (tag.equals(PUNKTWERTUNGEN)) {
					return;
				} else {
					if (index > -1 && tag.equals(Bogen.kategorien[index])) {
						// nichts zu tun, tag wohlgeformt
						index = -1;
					} else {
						throw new XMLStreamException("    " + tag
								+ " unbekannt");
					}
				}
			}
		}
	}

	protected void readTextfelder() throws XMLStreamException {
		String tag = "", str;
		XMLEvent event = null;
		int index = -1;

		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				tag = event.asStartElement().getName().getLocalPart();
				for (int i = 0; i < Bogen.TEXTFELDER.length; i++) {
					if (tag.equals(Bogen.TEXTFELDER[i])) {
						index = i;
					}
				}
				if (index > -1) {
					str = xmlHelper.readText(Bogen.TEXTFELDER[index]);
				bogen.setText(index, str);
				} else {
					throw new XMLStreamException("   Zeile " + tag
							+ " nicht wohlgeformt");
				}
			} else if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (tag.equals(TEXTFELDER)) {
					return;
				} else if (index > -1 && tag.equals(Bogen.TEXTFELDER[index])) {
					// nichts zu tun, tag wohlgeformt
					index = -1;
				} else {
					throw new XMLStreamException("   " + tag + " unbekannt");
				}
			}
		}
	}

	
	@SuppressWarnings("unchecked")
	protected void readQualifiedStrings(String tagName, String typName, String[] types, QualifiedStringList list, 
	        String endTagName) 
	throws XMLStreamException {
		String tag = "", str;
		XMLEvent event = null;
		String typ = "";
		int index = 0;

		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				tag = event.asStartElement().getName().getLocalPart();
				if (tag.equals(tagName)) {
					Iterator<Attribute> attributes = (Iterator<Attribute>) event
							.asStartElement().getAttributes();
					typ = types[0];
					while (attributes.hasNext()) {
						Attribute attribute = attributes.next();
						if (attribute.getName().toString().equals(typName)) {
							typ = attribute.getValue();
						}
					}
					str = xmlHelper.readText(tagName);
					for (index = 0; index < types.length; index++) {
						if (types[index].equals(typ)) {
							list.add(new QualifiedString(index, str, types));
						}
					}
				} else {
					throw new XMLStreamException("   Zeile " + tag
							+ " nicht wohlgeformt");
				}
			} else if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (tag.equals(endTagName)) {
					return;
				} else {
					throw new XMLStreamException("   " + tag + " unbekannt");
				}
			}
		}
	}

	
	
	protected void readZitate() throws XMLStreamException{
		readQualifiedStrings(ZITAT, ZITAT_TYP, Zitat.TYPES, bogen.getZitate(), ZITATE);
	}
	
	protected void readLinks() throws XMLStreamException{
		readQualifiedStrings(LINK, LINKS_TYP, Link.TYPES, bogen.getLinks(), LINKS);
	}

	protected void readBMovie() throws XMLStreamException {
		String tag;
		XMLEvent event = null;

		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				tag = event.asStartElement().getName().getLocalPart();

				if (tag.equals(ZITATE)) {
					readZitate();
				}
				if (tag.equals(PUNKTWERTUNGEN)) {
					readPunktwertungen();
				}
				if (tag.equals(TEXTFELDER)) {
					readTextfelder();
				}
				if (tag.equals(LINKS)) {
					readLinks();
				}
				if (tag.equals(COVER)) {				    
					bogen.getCover().setText(xmlHelper.readText(COVER));
					// Und das Cover einlesen
					
				}
			} else if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (tag.equals(BMOVIE)) {
					return;
				}
			}
		}
	}

	protected Bogen read() throws XMLStreamException {
		XMLEvent event = null;

		// System.out.println(this.bogen);
		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()
					&& event.asStartElement().getName().getLocalPart().equals(
							BMOVIE)) {
				readBMovie();
				if (reader.hasNext()) {
					event = reader.nextEvent();
					if (event.isEndDocument()) {
					}
				}
				return this.bogen;
			}
		}
		return null;
	}
	
	
	////////////////////////////////////////////////////////
	// Drucken der DTD
	
	protected void printEntity(String entity, String cont) {
		if(entity == null) {
			throw new IllegalArgumentException();
		}
		out.print("<!ENTITY " + entity);
		if(cont != null) {
			out.print(" (" + cont + ")");
		}
		out.println(" >");
	}

	protected void printAtt(String tag){
		out.println("<!ATTLIST " + tag);
	}
	
	protected void printAttribute(String name, String type, String def) {
		out.println("   " + name + " " + type + " " + def);
	}
	
	protected void printAllgDTD() {
		for(int i = 0; i < Bogen.erstesTextfeld; i++) {
			printEntity(Bogen.TEXTFELDER[i], "#PCDATA");
		}
	}
	
	protected void printPunktwertungenDTD() {
		String kateg = "";
		if(Bogen.kategorien.length > 0) {
			kateg = Bogen.kategorien[0];
			for(int i = 1; i < Bogen.kategorien.length; i++) {
				kateg += "," + Bogen.kategorien[i];
			}
		}	
		
		printEntity(PUNKTWERTUNGEN, kateg);
		
		for(int i = 0; i < Bogen.kategorien.length; i++) {
			printEntity(Bogen.kategorien[i], "EMPTY");
			printAtt(Bogen.kategorien[i]);
			printAttribute(PUNKT_VAL, "CDATA", "\"0\"");
			printAttribute(PUNKT_ANMERKUNG, "CDATA", "\"\"");
			out.println(">");
			out.println();
		}
		
		out.println();
	}

	protected void printTextfelderDTD() {
		String felder = "";
		if(Bogen.TEXTFELDER.length > Bogen.erstesTextfeld) {
			felder = Bogen.TEXTFELDER[Bogen.erstesTextfeld];
			for(int i = Bogen.erstesTextfeld + 1; i < Bogen.TEXTFELDER.length; i++) {
				felder += "," + Bogen.TEXTFELDER[i];
			}
		}
		
		printEntity(PUNKTWERTUNGEN, felder);
		for(int i = Bogen.erstesTextfeld; i < Bogen.TEXTFELDER.length; i++) {
			printEntity(Bogen.TEXTFELDER[i], "#PCDATA");
		}
		
	}
	
	protected void printZitateDTD() {
		printEntity(ZITATE, ZITAT + '*');
		printEntity(ZITAT, "#PCDATA");
	}
	
	protected void printLinksDTD() {
		printEntity(LINKS, LINK + '*');
		printEntity(LINK, "#PCDATA");
		printAtt(LINK);
		printAttribute(LINKS_TYP, "CDATA", "#REQUIRED");
		out.println(" >");
	}
	
	protected void printCoverDTD() {
		printEntity(COVER, "#PCDATA");
	}
	
	protected void dtdToStream() {
		String allg = "";
		for(int i = 0; i < Bogen.erstesTextfeld; i++) {
			allg += Bogen.TEXTFELDER[i] + ",";
		}
		out.println("<!-- BMovie  version " + VERSION + " DTD -->");
		
		printEntity("bmovie", allg + PUNKTWERTUNGEN + "," + TEXTFELDER + "," + ZITATE + "," + LINKS + "," + COVER);
		printAllgDTD();
		printPunktwertungenDTD();
		printTextfelderDTD();
		printZitateDTD();
		printLinksDTD();
		printCoverDTD();
	}
}
