/*
 * BMovieReviewer Copyright (C) 2009, 2010 Michael J. Beer
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package gui;

import gui.dialogs.AboutDialog;
import gui.dialogs.DisplayDialog;
import gui.dialogs.LoginDialog;
import gui.dialogs.LoginOptionDialog;
import gui.dialogs.PreferencesDialog;
import gui.dialogs.ThreadDialog;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;

import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.KeyStroke;
import javax.swing.filechooser.FileNameExtensionFilter;

import threads.ExportPDFThread;
import threads.ImportPDFThread;
import threads.SendThread;
import threads.ThreadRegistry;
import tools.AppLogger;
import data.Bogen;
import data.Globals;
import data.HTTPPost;
import data.formats.HDBogen;

public class HauptMenue extends JMenuBar {

    private static final long serialVersionUID = 1L;

    protected Gui gui;

    protected static class LadenListener implements ActionListener {

        protected Gui gui;

        public LadenListener(Gui gui) {
            if (gui == null) {
                throw new IllegalArgumentException();
            }
            this.gui = gui;
        }

        /**
         * oeffnet einen FileChooser-Dialog und liefert gewaehlte Datei zurueck
         * 
         * @param filter
         * @return
         */
        protected File getFile(FileNameExtensionFilter filter) {
            return Utils.getFile(this.gui, gui.getBogen().getFilePath(), filter);
        }

        public void actionPerformed(ActionEvent e) {
            File file;

            file = getFile(new FileNameExtensionFilter("XML-Dateien", "xml", "XML"));
            if (file == null) {
                return;
            }

            String msg;
            Bogen bogen = new Bogen();
            try {           
                msg = HDBogen.load(file, bogen);
                this.gui.setStatus(msg);
                AppLogger.info(msg);
                gui.setBogen(bogen); // um die Listener zu benachrichtigen
                gui.getBogen().setFileName(file.getName());
                gui.getBogen().setFilePath(file.getPath());
            }catch(HDBogen.CoverException ex) {
                AppLogger.throwing("LadenListener", "actionPerformed", ex);
                gui.setBogen(bogen); // um die Listener zu benachrichtigen
                gui.getBogen().setFileName(file.getName());
                gui.getBogen().setFilePath(file.getPath());
                msg = ex.getMessage();
                JOptionPane.showMessageDialog(this.gui, msg);
            } catch (IOException ex) {
                AppLogger.throwing("LadenListener", "actionPerformed", ex);
                JOptionPane.showMessageDialog(this.gui, ex.getMessage());
                msg = ex.getMessage();
            } 
            this.gui.setStatus(msg);            
        }
    }

    protected static class ExportTexListener implements ActionListener {

        HauptMenue menue;

        public ExportTexListener(HauptMenue menue) {
            if (menue == null) {
                throw new IllegalArgumentException();
            }
            this.menue = menue;
        }

        public void actionPerformed(ActionEvent e) {
            FileOutputStream fileOut = null;
            PrintStream out = null;
            try {
                // Kopiere die sty mit, da ansonsten die Tex wertlos ist...
                tools.Utils.checkForSTY(Globals.getInstance().getProperty("basedirectory"));
                fileOut = new FileOutputStream(menue.gui.getBogen().getFilePath() + menue.gui.getBogen().getFileName() + ".tex", false);
                out = new PrintStream(fileOut, true, "UTF-8");
                if (out != null) {
                    this.menue.gui.getBogen().printTex(out);
                    out.close();
                    menue.gui.setStatus(menue.gui.getBogen().getFilePath() + menue.gui.getBogen().getFileName() + ".tex" + " gespeichert.");
                }
            } catch (Exception ex) {
                AppLogger.throwing("ExportTexListener", "actionPerformed", ex);
                menue.gui.setStatus(menue.gui.getBogen().getFilePath() + menue.gui.getBogen().getFileName() + ".tex"
                        + " konnte nicht geschrieben werden.");
            }
        }
    }

    protected static class ExportBogenListener extends LadenListener {

        protected boolean choice = false;

        public ExportBogenListener(Gui gui) {
            super(gui);
        }

        public ExportBogenListener(Gui gui, boolean choice) {
            this(gui);
            this.choice = choice;
        }

        public void actionPerformed(ActionEvent e) {
            File file;
            String name;

            if (choice) {
                FileNameExtensionFilter filter = new FileNameExtensionFilter("XML-Dateien", "xml", "XML");

                file = getFile(filter);
                if (file != null) {
                    // Und die Pfadaenderung speichern...
                    this.gui.getBogen().setFileName(file.getName());
                    this.gui.getBogen().setFilePath(file.getPath());
                } else {
                    return;
                }

            } else {
                name = gui.getBogen().getFilePath() + gui.getBogen().getFileName() + ".xml";
                file = new File(name);
            }
            try {
                this.gui.setStatus(HDBogen.save(gui.getBogen()));
            } catch (IOException ex) {
                AppLogger.throwing("ExportBogenListener", "actionPerformed", ex);
                this.gui.setStatus("Fehler beim Speichern von " + this.gui.getBogen().getFilePath() + this.gui.getBogen().getFileName()
                        + ".xml");
                JOptionPane.showMessageDialog(gui, "Fehler beim Speichern von " + this.gui.getBogen().getFilePath()
                        + this.gui.getBogen().getFileName() + ".xml", "Ein/Ausgabefehler", JOptionPane.ERROR_MESSAGE);
            }
        }
    }

    protected static class ImportPDFListener extends LadenListener {

        public ImportPDFListener(Gui gui) {
            super(gui);
        }

        public void actionPerformed(ActionEvent e) {
            File file;

            file = getFile(new FileNameExtensionFilter("PDF-Dateien", "pdf", "PDF"));
            if (file == null) {
                return;
            }

            Runnable importThread = new ImportPDFThread(file, gui);
            new Thread(importThread).start();

        }
    }

    protected static class ExportPDFListener extends LadenListener {

        public ExportPDFListener(Gui gui) {
            super(gui);
        }

        public void actionPerformed(ActionEvent e) {

            AppLogger.info("Schreibe nach " + gui.getBogen().getFileName() + ".pdf");
            gui.setStatus("Daten werden exportiert...");
            Runnable export = new ExportPDFThread(gui, gui.getBogen().getFilePath(), gui.getBogen().getFileName());
            (new Thread(export)).start();
        }
    }

    public static class BeendenListener implements ActionListener {

        protected Gui gui;

        public BeendenListener(Gui gui) {
            if (gui == null) {
                throw new IllegalArgumentException();
            }
            this.gui = gui;
        }

        public void actionPerformed(ActionEvent e) {
            // Gui das Signal zum Terminieren senden...
            gui.terminate();
        }

    }

    public static class SendenListener extends LadenListener {
        public SendenListener(Gui gui) {
            super(gui);
        }

        public SendenListener(Gui gui, boolean withOptions) {
            this(gui);
            this.withOptions = withOptions;
        }

        public void actionPerformed(ActionEvent e) {
            LoginDialog dlgLogin;
            boolean c = true, x = true, t = true, q = true, r = true;

            if (withOptions) {
                dlgLogin = new LoginOptionDialog(gui, options);
            } else {
                dlgLogin = new LoginDialog(gui);
            }

            if (dlgLogin.showLoginDialog() == JFileChooser.APPROVE_OPTION) {
                String name = dlgLogin.getLogin();
                String password = dlgLogin.getPassword();
                if (name == null || password == null) {
                    AppLogger.warning("Nutzername und Passwort zur Anmeldung benoetigt");
                    return;
                }

                SendThread send;
                if (withOptions) {
                    LoginOptionDialog dlg = (LoginOptionDialog) dlgLogin;
                    c = dlg.isOptionChosen(0);
                    x = dlg.isOptionChosen(1);
                    t = dlg.isOptionChosen(2);
                    q = dlg.isOptionChosen(3);
                    r = dlg.isOptionChosen(4);
                }

                send = new SendThread(gui, gui.bogen, new HTTPPost(Globals.getInstance().getProperty("server.protocoll") + "://"
                        + Globals.getInstance().getProperty("server.name") + "/" + Globals.getInstance().getProperty("server.dir"), name,
                        password), c, x, t, q, r);

                new Thread(send).start();
            }
        }

        protected boolean withOptions = false;
        public final static String[] options = { "Sende Titelbild", "Sende XML-Datei", "Sende Tex-Datei", "Registriere Zitate",
                "Registriere Film" };
    }

    public static class ThreadListener implements ActionListener {

        public ThreadListener(Gui gui) {
            if (gui == null) {
                throw new IllegalArgumentException();
            }
            this.gui = gui;
        }

        public void actionPerformed(ActionEvent e) {
            ThreadDialog dialog = new ThreadDialog(gui);
            dialog.showDialog(this.gui, ThreadRegistry.getInstance());
        }

        Gui gui = null;
    }

    public static class GlobalsListener implements ActionListener {

        public GlobalsListener(Gui gui, DisplayDialog<Globals> dialog) {
            if (gui == null || dialog == null) {
                throw new IllegalArgumentException();
            }
            this.gui = gui;
            this.dialog = dialog;
        }

        public void actionPerformed(ActionEvent e) {
            dialog.showDialog(gui, Globals.getInstance());
        }

        Gui gui = null;
        DisplayDialog<Globals> dialog = null;
    }

    public static class PreviewListener implements ActionListener {

        public PreviewListener(Gui gui, DisplayDialog<BufferedImage> dialog) {
            if (gui == null || dialog == null) {
                throw new IllegalArgumentException();
            }
            this.gui = gui;
            this.dialog = dialog;
        }

        public void actionPerformed(ActionEvent e) {
            dialog.showDialog(gui, gui.getBogen().getCoverImage());
        }

        Gui gui = null;
        DisplayDialog<BufferedImage> dialog = null;
    }

    public static class AboutListener extends PreviewListener {

        public AboutListener(Gui gui) {
            super(gui, new AboutDialog(gui, Globals.logo));
        }

        public void actionPerformed(ActionEvent e) {
            dialog.showDialog(gui, null);
        }
    }

    public static class SendenBerichtListener implements ActionListener {

        protected Gui gui = null;

        public SendenBerichtListener(Gui gui) {
            if (gui == null) {
                throw new IllegalArgumentException();
            }
            this.gui = gui;
        }

        @Override
        public void actionPerformed(ActionEvent arg0) {
            JOptionPane
                    .showMessageDialog(this.gui, Globals.getInstance().getSendReport(), "Serverantwort", JOptionPane.INFORMATION_MESSAGE);
        }

    }

    protected JMenuItem erzeugeEintrag(String bez, ActionListener aL, int mn) {
        // Eintrag anlegen
        JMenuItem mi = new JMenuItem(bez);
        if (mn != 0)
            mi.setMnemonic(mn);
        else
            mi = new JMenuItem(bez);

        // actionlistener setzen
        mi.addActionListener(aL);

        return mi;
    }

    /**
     * der einzige Konstruktor, wird benutzt, um das ganze menu herzustellen
     */
    public HauptMenue(Gui gui) {
        super();
        int noEntries = 0;
        Globals globs = Globals.getInstance();

        JMenuItem mi;

        if (gui == null) {
            throw new IllegalArgumentException();
        }
        this.gui = gui;

        JMenu daten = new JMenu("Daten");
        daten.setMnemonic(KeyEvent.VK_D);
        // Bei einem Applet waere Laden/Speichern wegen den Sicherheitspolicies
        // nur
        // umstaendlich zu handhaben, daher bei Applet/Servlet die betreffenden
        // Menuepunkte gar nicht
        // erst erzeugen
        if ((Globals.getAppType() & Globals.APPLET) != Globals.APPLET) {
            mi = erzeugeEintrag("Laden", new LadenListener(gui), KeyEvent.VK_L);
            mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F, InputEvent.CTRL_MASK));
            daten.add(mi);

            mi = erzeugeEintrag("Speichern", new ExportBogenListener(this.gui), KeyEvent.VK_S);
            mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.CTRL_MASK));
            daten.add(mi);

            mi = erzeugeEintrag("Speichern unter", new ExportBogenListener(this.gui, true), KeyEvent.VK_U);
            daten.add(mi);

            noEntries = 3;

            if (!globs.getProperty("texcommand").equals("disabled") && (Globals.getAppType() & Globals.MIT_PDF) == Globals.MIT_PDF) { // Mit
                // Import
                // von
                // PDF
                mi = erzeugeEintrag("Importiere von PDF", new ImportPDFListener(gui), KeyEvent.VK_P);
                daten.add(mi);
                mi = erzeugeEintrag("Exportiere nach PDF", new ExportPDFListener(gui), KeyEvent.VK_D);
                daten.add(mi);
                daten.insertSeparator(3);
                daten.insertSeparator(6);
                noEntries += 4;
            } else {
                Globals.initialStatus = "Texkommando : " + globs.getProperty("texcommand")
                        + " nicht gefunden - PDF-Unterstützung deaktiviert";
            }
            mi = erzeugeEintrag("Exportiere nach LaTex", new ExportTexListener(this), KeyEvent.VK_L);
            daten.add(mi);
            noEntries++;
        }

        if ((Globals.getAppType() & Globals.MIT_SENDEN) == Globals.MIT_SENDEN) {
            daten.insertSeparator(++noEntries);
            mi = erzeugeEintrag("Sende an Server", new SendenListener(gui, true), KeyEvent.VK_T);
            mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T, InputEvent.CTRL_MASK));
            daten.add(mi);
            mi = erzeugeEintrag("Sende Alle Daten an Server", new SendenListener(gui, false), KeyEvent.VK_T);
            daten.add(mi);
            noEntries++;
        }
        daten.insertSeparator(++noEntries);
        mi = erzeugeEintrag("Beenden", new BeendenListener(gui), KeyEvent.VK_B);
        mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q, InputEvent.CTRL_MASK));
        daten.add(mi);

        add(daten);

        JMenu ansicht = new JMenu("Ansicht");
        ansicht.setMnemonic(KeyEvent.VK_A);
        ansicht.add(erzeugeEintrag("Bericht", new SendenBerichtListener(gui), KeyEvent.VK_B));
        ansicht.add(erzeugeEintrag("Einstellungen", new GlobalsListener(gui, new PreferencesDialog(this.gui)), KeyEvent.VK_T));
        ansicht.add(erzeugeEintrag("Threads", new ThreadListener(gui), KeyEvent.VK_T));
        add(ansicht);

        JMenuItem beenden = new JMenuItem("Über");
        beenden.setMnemonic(KeyEvent.VK_B);
        beenden.addActionListener(new AboutListener(gui));
        add(beenden);

    }

}
