/*
 * BMovieReviewer Copyright (C) 2009, 2010 Michael J. Beer
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package tools;

import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.URL;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;

import javax.imageio.ImageIO;

import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGEncodeParam;
import com.sun.image.codec.jpeg.JPEGImageDecoder;
import com.sun.image.codec.jpeg.JPEGImageEncoder;

import data.Globals;
import data.formats.XMLConfig;

public class Utils {
	public static void copy(File s, File d) throws IOException,
			IllegalArgumentException {
		if (s == null || d == null) {
			throw new IllegalArgumentException();
		}
		if( !s.exists() || !s.isFile()) {
		    throw new IOException();
		}
		// Auf Gleichheit testen - dann nat. nicht kopieren...
		if (s.equals(d)) {
			return;
		}

		// falls schon existent, wird die Datei geloescht
		if (d.exists()) {
			d.delete();
		}

		// sonst kopieren
		FileChannel in = null, out = null;
		try {
			in = new FileInputStream(s).getChannel();
			out = new FileOutputStream(d).getChannel();

			long size = in.size();
			MappedByteBuffer buf = in.map(FileChannel.MapMode.READ_ONLY, 0,
					size);

			out.write(buf);

		} finally {
			if (in != null)
				in.close();
			if (out != null)
				out.close();
		}
	}

	public static void scaleAndCopy(File s, File d, int scaledWidth)
			throws IOException, IllegalArgumentException {
		if (s == null || d == null || !s.exists() || !s.isFile()) {
			throw new IllegalArgumentException();
		}
		// Auf Gleichheit testen - dann nat. nicht kopieren...
		if (s.equals(d)) {
			return;
		}

		int scaledHeight = 0, width = 0;
		int quality = 50;

		// Bild laden und skalieren
		FileInputStream in = new FileInputStream(s);
		JPEGImageDecoder decoder = JPEGCodec.createJPEGDecoder(in);
		FileOutputStream out = null;
		BufferedImage image = decoder.decodeAsBufferedImage();
		in.close();		
		if(image == null) {
			throw new IOException("Datei " + s.toString() + " konnte nicht geladen werden");
		}
		
		// Seitenverhaeltnis ermitteln
		scaledHeight = image.getHeight();
		width = image.getWidth();
		if (scaledHeight == 0 || width == 0) { // leeres Bild
			return;
		}
		scaledHeight *= scaledWidth;
		scaledHeight /= width;

		BufferedImage scaledImage = new BufferedImage(scaledWidth,
				scaledHeight, BufferedImage.TYPE_INT_RGB);
		Graphics2D graphics2D = scaledImage.createGraphics();
		graphics2D.setRenderingHint(RenderingHints.KEY_INTERPOLATION,
				RenderingHints.VALUE_INTERPOLATION_BILINEAR);
		graphics2D.drawImage(image, 0, 0, scaledWidth, scaledHeight, null);

		
		// falls schon existent, wird die Datei geloescht
		if (d.exists()) {
			d.delete();
		}
		
		// Und Bild in neue Datei schreiben
		out = new FileOutputStream(d);
		if(out == null) {
			throw new IOException("Outputstream konnte nicht erzeugt werden");
		}
		System.out.println(out.toString());
		JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);
		if(encoder == null) {
			throw new IOException("JPegEncoder konnte nicht erzeugt werden");
		}
        JPEGEncodeParam param = encoder.
          getDefaultJPEGEncodeParam(scaledImage);
        
        try	{
        	quality = Integer.parseInt(Globals.getInstance().getProperty("jpeg.quality"));
        	if(quality > 100 || quality < 0) {
        		throw new NumberFormatException();
        	}
        }catch(NumberFormatException e){
        	AppLogger.severe("Property \'jpeg.quality\' enthaelt nichtnumerischen Wert oder Wert ausserhalb des yulaessigen Bereiches (0 bis 100)");
        	quality = 85;
        }
        quality = Math.max(0, Math.min(quality, 100));
        param.setQuality((float)quality / 100.0f, false);
        encoder.setJPEGEncodeParam(param);
        encoder.encode(scaledImage);
        try{
        	ImageIO.write(scaledImage, Globals.getInstance().getProperty("jpeg.mime"), out); 
        }catch(IOException e) {
        	if(d.exists()) {
        		d.delete();
        	}
        	throw new IOException("Datei konnte nicht geschrieben werden");
        }finally{
        	out.close();
        }
	}
	
	
	
    /**
     * laedt ein neues Titelbild
     * 
     * @param imageURL
     *            URL zum Bild
     * @param scaledWidth
     *            Breite, auf die Bild skaliert werden soll, 0 heiszt keine Skalierung
     */
    public static BufferedImage loadImage(URL imageURL, int scaledWidth) throws IOException {
        if (imageURL == null) {
            throw new IllegalArgumentException();
        }
        BufferedImage loadedImage;
        int scaledHeight = 0, width = 0;

        // Bild laden und skalieren
        InputStream in = imageURL.openStream();
        JPEGImageDecoder decoder = JPEGCodec.createJPEGDecoder(in);
        BufferedImage image = decoder.decodeAsBufferedImage();
        in.close();
        if (image == null) {
            throw new IOException("Bild " + imageURL.toString() + " konnte nicht geladen werden");
        }

        // Seitenverhaeltnis ermitteln
        scaledHeight = image.getHeight();
        width = image.getWidth();
        if (scaledHeight == 0 || width == 0) { // leeres Bild
            AppLogger.info("loadImage: " + imageURL.toString() + " Bild wird nicht skaliert");
            return null;
        }
        scaledHeight *= scaledWidth;
        scaledHeight /= width;

        loadedImage = new BufferedImage(scaledWidth, scaledHeight, BufferedImage.TYPE_INT_RGB);
        Graphics2D graphics2D = loadedImage.createGraphics();
        graphics2D.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
        graphics2D.drawImage(image, 0, 0, scaledWidth, scaledHeight, null);
        AppLogger.info("loadImage: geladen");
        return loadedImage;
    }
    
    
    
    /**
     * Writes File as jpeg to outputstream
     * @param image Bild 
     * @param out   Outputstream, in den geschrieben werden soll
     * @param quality Qualitaet des kompr. Bildes
     * @param mime Mimetyp, der verwandt werden soll
     * @throws IOException
     */
    public static void writeImage(BufferedImage image, OutputStream out, int quality, String mime) 
    throws IOException {
        if(out == null || mime == null) {
            throw new IllegalArgumentException();
        }
 
        if(out == null) {
            throw new IOException("Outputstream konnte nicht erzeugt werden");
        }
        JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);
        if(encoder == null) {
            throw new IOException("JPegEncoder konnte nicht erzeugt werden");
        }
        JPEGEncodeParam param = encoder.
          getDefaultJPEGEncodeParam(image);
        
        try {
           if(quality > 100 || quality < 0) {
                throw new NumberFormatException();
            }
        }catch(NumberFormatException e){
            AppLogger.severe("Bildqualitaet auf unzulaessigen Wert gestellt");
            quality = 85;
        }
        quality = Math.max(0, Math.min(quality, 100));
        param.setQuality((float)quality / 100.0f, false);
        encoder.setJPEGEncodeParam(param);
        encoder.encode(image);
        ImageIO.write(image, mime, out); 
    }

    
	public static void writeImage(BufferedImage image, File file, int quality, String mime) throws IOException{
	    if(image == null || file == null) {
	        throw new IllegalArgumentException();
	    }
	    if(file.exists()) {
	        file.delete();
	    }
	    FileOutputStream out = new FileOutputStream(file);
	    writeImage(image, out, quality, mime);
	    out.close();
	}
	
	
	public static void checkForSTY(String dir) throws IOException{
		File dest = new File(dir + "bmovie.sty");
		AppLogger.info("Suche " + dest.toString());
		System.out.println(Globals.getInstance().getProperty("basedirectory") + File.separatorChar +
		Globals.getInstance().getProperty("datadirectory") + File.separatorChar + "bmovie.sty");
		if(!dest.exists()) {
			copy(new File(Globals.getInstance().getProperty("basedirectory") + File.separatorChar +
					Globals.getInstance().getProperty("datadirectory") + File.separatorChar + "bmovie.sty"), 
					dest);
		}
	}
	
	
	public static boolean isTexProvided() {
	    boolean provided = false;
	    
	    try{
	        Process p = Runtime.getRuntime().exec(Globals.getInstance().getProperty("texcommand") + " --version");
	        p.waitFor();
	        
	        if(p.exitValue() == 0) {
	            provided = true;
	        }
	    }catch(Exception e){
	        // Fehler => tex deaktivieren
	    }
	    
	    return provided;
	}
	
	/**
	 * Laedt die Konfigurationsdatei 
	 * @param argv
	 */
	public static void loadConfig() {
		String name = Globals.getInstance().getProperty("configfile");
		File cfg = new File(name);
		if(cfg.exists() && cfg.isFile()) { // Configdatei wurde gefunden
			try{
			XMLConfig.loadFromStream(new InputStreamReader( 
					new FileInputStream(cfg), Globals.getInstance().getProperty("encoding")));
			}catch(Exception e) {
				System.err.println(e);
			}
			AppLogger.config(Globals.getInstance().toString());
			System.out.println(Globals.getInstance().toString());
		}
	}
	

	public static void setBaseDir() {
		Globals.getInstance().setProperty("basedirectory", System.getProperty("user.dir"));
		System.out.println("BaseDir: " + System.getProperty("user.dir"));
	}
	
	/**
	 * Liefert den reinen Verzeichnispfadanteil eines Dateipfades zurueck
	 * @param url
	 * @return
	 */
	public static String getDirPath(String url) {
		int pathEnd = url.lastIndexOf(File.separatorChar);
		if (pathEnd > 0) {
			return url.substring(0, pathEnd + 1);
		}
		return "";
	}
	
	
	/**
     * Liefert den reinen Dateiname eines Dateipfades zurueck
     * @param url
     * @return
     */
    public static String getFileName(String url) {
        int pathEnd = url.lastIndexOf(File.separatorChar);
        if (pathEnd > 0) {
            return url.substring(pathEnd + 1, url.length());
        }
        return "";
    }
    }
