/*
 * BMovieReviewer Copyright (C) 2009 Michael J. Beer
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package tools;

import java.io.PrintStream;
import java.lang.reflect.Constructor;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.XMLEvent;


/**
 * Hilfsmethoden zum Lesen und Schreiben von XML-Dateien
 * @author mibeer
 *
 */
public class XML {

	protected XMLEventReader reader;
	protected XMLEventWriter writer;
	protected XMLEventFactory eventFactory;

	// ////////////////////////////////////////////
	// Konstruktoren

	public XMLEventReader getReader() {
		return reader;
	}

	public XMLEventWriter getWriter() {
		return writer;
	}

	public XMLEventFactory getEventFactory() {
		return eventFactory;
	}

	/**
	 * Erstellt Hilfskobjekt zum Lesen aus Stream
	 */
	public XML(XMLEventReader reader) {
		if (reader == null) {
			throw new IllegalArgumentException();
		}

		this.reader = reader;
	}

	/**
	 * Erstellt Hilfsobjekt zum Schreiben in Stream
	 * 
	 * @param writer
	 */
	public XML(XMLEventWriter writer, XMLEventFactory xmlEventFactory) {
		if (writer == null || xmlEventFactory == null) {
			throw new IllegalArgumentException();
		}

		this.writer = writer;
		this.eventFactory = xmlEventFactory;
	}

	public String readText(String closingTag) throws XMLStreamException {
		String str = "", tag = "";
		XMLEvent event = null;
		if (reader == null) {
			throw new NullPointerException();
		}

		while (reader.hasNext() && !tag.equals(closingTag)) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				throw new XMLStreamException("Textfeld " + closingTag
						+ " darf keine Untertags enthalten");
			}
			if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (!tag.equals(closingTag)) {
					throw new XMLStreamException("Textfeld " + closingTag
							+ " nicht wohlgeformt");
				}
			}
			if (event.isCharacters()) {
				String np = (event.asCharacters().getData());
				str = str + np.replaceAll("[\r\n]", " ").trim();
			}
		}

		return str;
	}

	@SuppressWarnings("unchecked")
	public static XMLEventWriter getXMLEventWriter(PrintStream out)
			throws XMLStreamException {
		XMLEventWriter writer = null;
		XMLOutputFactory xof = XMLOutputFactory.newInstance();

		try { // es wird versucht, eine formatierte Ausgabe durchzufuehren
			Class<XMLEventWriter> indentingWriter = (Class<XMLEventWriter>) Class
					.forName("javanet.staxutils.IndentingXMLEventWriter");
			Constructor<XMLEventWriter> constr = indentingWriter
					.getConstructor(XMLEventWriter.class);
			writer = constr.newInstance(xof.createXMLEventWriter(out, "UTF-8"));
		} catch (Exception e) {
			writer = xof.createXMLEventWriter(out, "UTF-8");
			System.err
					.println("stax-utils konnte nicht geladen werden - XML-Ausgabe nicht formatiert: "
							+ e.toString());
		}
		return writer;
	}

	

	public void printOpeningTag(String tag) throws XMLStreamException {
		writer.add(eventFactory.createStartElement(new QName(tag), null, null));
	}

	public void printClosingTag(String tag) throws XMLStreamException {
		writer.add(eventFactory.createEndElement(new QName(tag), null));
	}
	
	
	public void writeText(String tag, String value) throws XMLStreamException{
		printOpeningTag(tag);
		writer.add(eventFactory.createCharacters(value));
		printClosingTag(tag);
	}
	
	
}
