(* Copyright (C) 2014-2015 Daniel Wyckoff *)

(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)



Require Import SetUtilities.
Require Import LogicUtilities.
Require Export BoolAlgBasics.
Require Import FieldsOfSets.
Require Import InfiniteOperations.
Require Import TypeUtilities.
Require Import Arith.
Require Import FunctionProperties.
Require Import Equality.
Require Import FunctionalExtensionality.
Require Import FiniteMaps.
Require Import DecidableDec.
Require Import Equality.
Require Import ListUtilities.  
Require Import Description.

Section ListOperations.
Variable B:Bool_Alg.
Let Bt := bt B.
 

Fixpoint times_list {B0:Bool_Alg} (l:list (bt B0)) : (bt B0) := 
  match l with 
  | nil => 1
  | cons a l' => a * (times_list l')
  end.

Fixpoint plus_list {B0:Bool_Alg} (l:list (bt B0)) : (bt B0) := 
  match l with 
  | nil => 0
  | cons a l' => a + (plus_list l')
  end.

Lemma plus_list_app : forall (l1 l2:list Bt), 
  plus_list (l1 ++ l2) = plus_list l1 + plus_list l2.
intros l1 l2.
induction l1 as [|a l1' h1].
simpl. rewrite comm_sum. rewrite zero_sum. reflexivity.
(* a::l1'*)
simpl.
rewrite h1.
apply assoc_sum.
Qed.

Lemma times_list_app : forall (l1 l2:list Bt), 
  times_list (l1 ++ l2) = times_list l1 * times_list l2.
intros l1 l2.
induction l1 as [|a l1' h1].
simpl. rewrite comm_prod. rewrite one_prod. reflexivity.
(* a::l1'*)
simpl.
rewrite h1.
apply assoc_prod.
Qed.


(* maybe turn Bt into a general Boolean algebraic type *)
Lemma dist_list_sing_plus : forall (l:list Bt) (x:Bt), 
  x * (plus_list l) = plus_list (map (fun y:Bt => (x*y)) l).
intros l x.
induction l as [|a l h1].
(* nil *)
simpl.  apply zero_prod.
simpl.
rewrite dist_sum.
rewrite h1.
reflexivity.
Qed.

Lemma dist_list_sing_times : forall (l:list Bt) (x:Bt), 
  x + (times_list l) = times_list (map (fun y:Bt => (x+y)) l).
intros l x.
induction l as [|a l h1].
(* nil *)
simpl.  apply one_sum.
simpl.
rewrite dist_prod.
rewrite h1.
reflexivity.
Qed.

Lemma dist_list_2_plus : forall (l1 l2:list Bt), 
  (plus_list l1) * (plus_list l2) = 
  plus_list (map (fun p:Bt*Bt => (fst p * snd p)) (list_prod l1 l2)).
intros l1 l2.
induction l1 as [|a l1' h1]; destruct l2 as [|b l2']; simpl.
(* nil nil *)
apply zero_prod.
(* nil b::l2' *)
rewrite comm_prod.
apply zero_prod.
(* a::l1' nil *)
simpl in h1.
rewrite <- h1.
do 2 rewrite zero_prod.
reflexivity.
(* a::l1' b::l2'*)
simpl in h1.
rewrite comm_prod at 1.
rewrite dist_sum.
rewrite comm_prod in h1. 
rewrite h1.
rewrite comm_prod.
rewrite dist_sum.
rewrite map_app.

assert (h3:a*plus_list l2' = 
  plus_list (map (fun p:Bt * Bt => fst p * snd p) (map (fun y:Bt => (a, y)) l2'))).
  rewrite dist_list_sing_plus.
  assert (h4:map (fun y:Bt => a * y) l2' = 
    map (fun p : Bt * Bt => fst p * snd p) (map (fun y:Bt => (a, y)) l2')).
    rewrite map_map. simpl. reflexivity.  
  rewrite h4.
  reflexivity.
rewrite h3.
rewrite plus_list_app.
rewrite <- assoc_sum.
reflexivity.
Qed.


Lemma dist_list_2_times : forall (l1 l2:list Bt), 
  (times_list l1) + (times_list l2) = 
  times_list (map (fun p:Bt*Bt => (fst p + snd p)) (list_prod l1 l2)).
intros l1 l2.
induction l1 as [|a l1' h1]; destruct l2 as [|b l2']; simpl.
(* nil nil *)
apply one_sum.
(* nil b::l2' *)
rewrite comm_sum.
apply one_sum.
(* a::l1' nil *)
simpl in h1.
rewrite <- h1.
do 2 rewrite one_sum.
reflexivity.
(* a::l1' b::l2'*)
simpl in h1.
rewrite comm_sum at 1.
rewrite dist_prod.
rewrite comm_sum in h1.
rewrite h1.
rewrite comm_sum.
rewrite dist_prod.
rewrite map_app.

assert (h3:a+times_list l2' = 
  times_list (map (fun p:Bt * Bt => fst p + snd p) (map (fun y:Bt => (a, y)) l2'))).
  rewrite dist_list_sing_times.
  assert (h4:map (fun y:Bt => a + y) l2' = 
    map (fun p : Bt * Bt => fst p + snd p) (map (fun y:Bt => (a, y)) l2')).
    rewrite map_map. simpl. reflexivity.  
  rewrite h4.
  reflexivity.
rewrite h3.
rewrite times_list_app.
rewrite <- assoc_prod.
reflexivity.
Qed.

Definition plus_times_list_of_lists (l:list (list Bt)) : Bt :=
  plus_list (map (fun l':(list Bt) => times_list l') l).

Definition times_plus_list_of_lists (l:list (list Bt)) : Bt :=
  times_list (map (fun l':(list Bt) => plus_list l') l).

Definition times_fun {T:Type} (p:T->Bt) (l:list T) :=
  times_list (map p l).

Definition plus_fun {T:Type} (p:T->Bt) (l:list T) :=
  plus_list (map p l).

Definition plus_times_fun1 {T U:Type} (li:list T) (lj:list U)
           (p:T->U->Bt) :=
  plus_fun (fun i:T => times_fun (p i) lj) li.

Definition plus_times_fun2 {T U:Type} (li:list T) (lj:list U) 
           (p:T->U->Bt) :=
  plus_fun (fun j:U => times_fun (fun i:T => (p i j)) li) lj.

Definition times_plus_fun1 {T U:Type} (li:list T) (lj:list U)
           (p:T->U->Bt) :=
  times_fun (fun i:T => plus_fun (p i) lj) li.

Definition times_plus_fun2 {T U:Type} (li:list T) (lj:list U) 
           (p:T->U->Bt) :=
  times_fun (fun j:U => plus_fun (fun i:T => (p i j)) li) lj.

Definition plus_times_all_funs1 {T U:Type} (li:list T) (lj:list U)
           (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Bt) (def:U): Bt.
pose (list_power li lj) as lp.
pose (map_sig lp) as lp'.
pose (fun l:{x:list (T*U) | In x lp} => 
        times_fun (fun i:T => 
                     (p i (((in_list_power_fpl _ _ (proj1_sig l) pfi (list_power_no_dup _ _ _ pfi pfj (proj2_sig l)) (proj2_sig l)), def) l-> i))) li) as f.
refine (plus_fun f lp').
Defined.


Definition plus_times_all_funs2 {T U:Type} (li:list T) (lj:list U)
           (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Bt) (def:T): Bt.
pose (list_power lj li) as lp.
pose (map_sig lp) as lp'.
pose (fun l:{x:list (U*T) | In x lp} => 
        times_fun (fun j:U => 
                     (p (((in_list_power_fpl _ _ (proj1_sig l) pfj (list_power_no_dup _ _ _ pfj pfi (proj2_sig l)) (proj2_sig l)), def) l-> j) j)) lj) as f.
refine (plus_fun f lp').
Defined.


Definition times_plus_all_funs1 {T U:Type} (li:list T) (lj:list U)
           (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Bt) (def:U): Bt.
pose (list_power li lj) as lp.
pose (map_sig lp) as lp'.
pose (fun l:{x:list (T*U) | In x lp} => 
        plus_fun (fun i:T => 
                     (p i (((in_list_power_fpl _ _ (proj1_sig l) pfi (list_power_no_dup _ _ _ pfi pfj (proj2_sig l)) (proj2_sig l)), def) l-> i))) li) as f.
refine (times_fun f lp').
Defined.

Definition times_plus_all_funs2 {T U:Type} (li:list T) (lj:list U)
           (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Bt) (def:T): Bt.
pose (list_power lj li) as lp.
pose (map_sig lp) as lp'.
pose (fun l:{x:list (U*T) | In x lp} => 
        plus_fun (fun j:U => 
                     (p (((in_list_power_fpl _ _ (proj1_sig l) pfj (list_power_no_dup _ _ _ pfj pfi (proj2_sig l)) (proj2_sig l)), def) l-> j) j)) lj) as f.
refine (times_fun f lp').
Defined.                                          






(* This is the complete distributive rule for finite sets, as 
 quoted in (8.3), but mine is more general since it allows you to 
 take the meet of the joins of lists of arbitrary length
 (not fixed as in the book) *)

(* Do a cmopatible version of this using functional notation, i.e. p:I->J->Bt,
  where [I] and [J] are sets, and [p] is a finite map *) 

Lemma complete_dist_list_times : forall (l:list (list Bt)), 
  times_plus_list_of_lists l = plus_times_list_of_lists (list_of_lists_seqs l).
intro l.
induction l as [|al l' h1].
(* nil *)
unfold plus_times_list_of_lists. simpl. 
rewrite zero_sum. reflexivity.
(* al::l'*)
simpl.
unfold times_plus_list_of_lists.
simpl.
unfold times_plus_list_of_lists in h1.
rewrite h1.
unfold plus_times_list_of_lists.
rewrite dist_list_2_plus.
rewrite map_map. simpl.
induction al as [|a t h2].
  (*nil*)
  simpl. reflexivity.
  (*a::t*)
  simpl.
  rewrite map_map.
  rewrite map_app.
  rewrite plus_list_app.
  rewrite h2.
  rewrite map_app.
  rewrite plus_list_app.
  do 2 rewrite map_map.
  simpl.
  reflexivity.
Qed.


Lemma complete_dist_list_plus : forall (l:list (list Bt)), 
  plus_times_list_of_lists l = times_plus_list_of_lists (list_of_lists_seqs l).
intro l.
induction l as [|al l' h1].
(* nil *)
unfold times_plus_list_of_lists. simpl. 
unfold plus_times_list_of_lists. simpl.
rewrite comm_prod.
rewrite zero_prod. reflexivity.
(* al::l'*)
simpl.
unfold plus_times_list_of_lists.
simpl.
unfold plus_times_list_of_lists in h1.
rewrite h1.
unfold times_plus_list_of_lists.
rewrite dist_list_2_times.
rewrite map_map. simpl.
induction al as [|a t h2].
  (*nil*)
  simpl. reflexivity.
  (*a::t*)
  simpl.
  rewrite map_map.
  rewrite map_app.
  rewrite times_list_app.
  rewrite h2.
  rewrite map_app.
  rewrite times_list_app.
  do 2 rewrite map_map.
  simpl.
  reflexivity.
Qed.


Lemma inf_union : forall (U V:Ensemble Bt) (u v:Bt), inf U u -> inf V v ->
  (inf (Union U V) (u * v)).
intros U V u v h1 h2.
red in h1. destruct h1 as [h1l h1r].
red in h2. destruct h2 as [h2l h2r].
red.
split.
(* lower bound *)
red. red in h1l. red in h2l.
intros s h3.
pose proof (Union_inv _ _ _ _ h3) as h4.
  destruct h4 as [h4l | h4r].
  (* h4l *)
  (* maybe turn into a separate lemma *)
  assert (h5:le (u * v) u).
    red. rewrite eq_ord. rewrite comm_prod. rewrite assoc_prod.
    rewrite idem_prod. reflexivity.
  specialize (h1l _ h4l). apply (trans_le _ _ _ h5 h1l).
  assert (h5:le (u * v) v).
    red. rewrite eq_ord. rewrite <- assoc_prod. rewrite idem_prod. 
    reflexivity. 
  specialize (h2l _ h4r). apply (trans_le _ _ _ h5 h2l).
(* greatest lower bound *)
intros b h3.
red in h3.
assert (h4:lb U b).
  red.
  intros s h5.
  assert (Ensembles.In (Union U V) s). auto with sets.
  apply h3; assumption.
assert (h5:lb V b).
  red.
  intros s h6.
  assert (Ensembles.In (Union U V) s). auto with sets.
  apply h3; assumption.
specialize (h1r b h4).
specialize (h2r b h5).
pose proof (mono_prod _ _ _ _ h1r h2r) as h6.
rewrite idem_prod in h6.
assumption.
Qed.

Lemma sup_union : forall (U V:Ensemble Bt) (u v:Bt), sup U u -> sup V v ->
  (sup (Union U V) (u + v)).
intros U V u v h1 h2.
red in h1. destruct h1 as [h1l h1r].
red in h2. destruct h2 as [h2l h2r].
red.
split.
(* upper bound *)
red. red in h1l. red in h2l.
intros s h3.
pose proof (Union_inv _ _ _ _ h3) as h4.
  destruct h4 as [h4l | h4r].
  (* h4l *)
   assert (h5:le u (u + v)).
    red. rewrite assoc_sum. rewrite idem_sum. reflexivity. 
  specialize (h1l _ h4l). apply (trans_le _ _ _ h1l h5).
  assert (h5:le v (u + v)).
    red. rewrite comm_sum.  rewrite <- assoc_sum. rewrite idem_sum. reflexivity. 
  specialize (h2l _ h4r). apply (trans_le _ _ _ h2l h5).
(* least upper bound *)
intros b h3.
red in h3.
assert (h4:ub U b).
  red.
  intros s h5.
  assert (Ensembles.In (Union U V) s). auto with sets.
  apply h3; assumption.
assert (h5:ub V b).
  red.
  intros s h6.
  assert (Ensembles.In (Union U V) s). auto with sets.
  apply h3; assumption.
specialize (h1r b h4).
specialize (h2r b h5).
pose proof (mono_sum _ _ _ _ h1r h2r) as h6.
rewrite idem_sum in h6.
assumption.
Qed.


Lemma decompose_inf : forall (S X:Ensemble Bt) (s x x':Bt), Included X S -> inf S s ->
  inf (Setminus S X) x' -> inf X x -> inf (Couple x x') s.
intros S X s x x' h1 h2 h3 h4.
red.
red in h2. destruct h2 as [h2l h2r].
red in h3. destruct h3 as [h3l h3r].
red in h4. destruct h4 as [h4l h4r].
split.
(* lower bound *)
red.
intros b h5.
pose proof (Couple_inv _ _ _ _ h5) as h6.
  destruct h6 as [h6l | h6r].
  (* b = x *)
  subst.
  red in h2l.
  assert (h6: forall y:Bt, Ensembles.In X y -> Ensembles.In S y). auto with sets.
  assert (h7: forall y:Bt, Ensembles.In X y -> le s y).  
    intros y h8. specialize (h6 _ h8).
    apply h2l; assumption.
  assert (h8: lb X s). red. assumption.
  apply h4r; assumption.
  (* b = x'*)
  subst.
  assert (h6:Included (Setminus S X) S). 
    red. intros b h7. unfold Setminus in h7. red in h7. 
    tauto.
  red in h3l.
  assert (h7: forall y:Bt, Ensembles.In (Setminus S X) y -> Ensembles.In S y). auto with sets.
  assert (h8: forall y:Bt, Ensembles.In (Setminus S X) y -> le s y).
    intros y h9. specialize (h7 _ h9).
    apply h2l; assumption.
  assert (h9: lb (Setminus S X) s). red. assumption.
  apply h3r; assumption.
(* greatest lower bound *)
intros a h5.
assert (h6: lb S a).
  red in h5.
  pose proof (Couple_l _ x x') as h6.
  pose proof (Couple_r _ x x') as h7.
  pose proof (h5 _ h6) as h8.
  pose proof (h5 _ h7) as h9.
  red in h3l. red in h4l.
  red.
  intros c h10.
  case (classic (Ensembles.In X c)) as [h11 | h12].
  (* In X c *)
  pose proof (h4l _ h11) as h13.
  apply trans_le with x; assumption.
  (* ~ In X c *)
  assert (h13: Ensembles.In (Setminus S X) c). auto with sets.
  pose proof (h3l _ h13) as h14.
  apply trans_le with x'; assumption.
apply h2r; assumption.
Qed.

Lemma decompose_inf': forall (S X:Ensemble Bt) (x x':Bt), Included X S -> 
  inf X x -> inf (Setminus S X) x' -> inf S (x * x').
intros S X x x' h1 h2 h3.
pose proof decompose_inf.
case (classic (exists s, inf S s)) as [h4 | h5].
(* h4 *)
destruct h4 as [s h4].
assert (h5:s = x * x').
  pose proof (decompose_inf _ _ _ _ _ h1 h4 h3 h2) as h5.
  pose proof (lat_prod x x') as h6.
  pose proof (inf_unq _ _ _ h5 h6).
  assumption.
rewrite h5 in h4.
assumption.
(* h5 *)
contradict h5.
exists (x * x').
(* maybe turn into a separate lemma *)
assert (h6:S = Union (Setminus S X) X).
  apply Extensionality_Ensembles. red. split.
  (* <= *)
  red. intros y h7.
  case (classic (Ensembles.In X y)) as [h8 | h9].
  (* h8 *) apply Union_intror; assumption.
  (* h9 *) apply Union_introl. unfold Setminus. red. split; assumption.
  (* >= *)
  red. intros y h7.
  pose proof (Union_inv _ _ _ _ h7) as h8.
  destruct h8 as [h8l | h8r].  
  unfold Setminus in h8l. red in h8l. tauto.
  auto with sets.
rewrite h6.
pose proof (inf_union _ _ _ _ h3 h2) as h7.
rewrite comm_prod in h7.
assumption. 
Qed.


Lemma inf_times_cons : forall (E:Ensemble Bt) (a b:Bt), Ensembles.In E a -> 
  inf (Subtract E a) b -> inf E (a * b).
intros E a b h1 h2.
pose proof decompose_inf'.
pose proof (inf_singleton _ a) as h3.
unfold Subtract in h2.
assert (h4:Included (Singleton a) E). 
  red. intros y h5. destruct h5; assumption. 
apply (decompose_inf' _ _ _ _ h4 h3 h2).
Qed.


Lemma inf_times_finite : forall (l:list Bt), 
  inf (list_to_set l) (times_list l).
intros l.
induction l.
(* nil *)
simpl.
red. split. 
red.  intros. contradiction.
intros; apply one_max.
(* a::l *)
simpl.
unfold Add.
rewrite comm_sum_psa.
apply inf_union.
apply inf_singleton.
assumption.
Qed.

Lemma sup_plus_finite : forall (l:list Bt), 
  sup (list_to_set l) (plus_list l).
intros l.
induction l.
(* nil *)
simpl.
red. split. 
red.  intros. contradiction.
intros;  apply zero_min. 
(* a::l *)
simpl.
unfold Add.
rewrite comm_sum_psa.
apply sup_union.
apply sup_singleton.
assumption.
Qed.
End ListOperations.

Arguments plus_list_app [B] _ _.
Arguments times_list_app [B] _ _.
Arguments dist_list_sing_plus [B] _ _.
Arguments dist_list_sing_times [B] _ _.
Arguments dist_list_2_plus [B] _ _.
Arguments dist_list_2_times [B] _ _.
Arguments plus_times_list_of_lists [B] _.
Arguments times_plus_list_of_lists [B] _.
Arguments times_fun [B] [T] _ _.
Arguments plus_fun [B] [T] _ _.
Arguments plus_times_fun1 [B] [T] [U] _ _ _.
Arguments plus_times_fun2 [B] [T] [U] _ _ _.
Arguments times_plus_fun1 [B] [T] [U] _ _ _.
Arguments times_plus_fun2 [B] [T] [U] _ _ _.
Arguments plus_times_all_funs1 [B] [T] [U] _ _ _ _ _ _.
Arguments plus_times_all_funs2 [B] [T] [U] _ _ _ _ _ _.
Arguments times_plus_all_funs1 [B] [T] [U] _ _ _ _ _ _.
Arguments times_plus_all_funs2 [B] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_list_times [B] _.
Arguments complete_dist_list_plus [B] _.
Arguments inf_union [B] _ _ _ _ _ _.
Arguments sup_union [B] _ _ _ _ _ _.
Arguments decompose_inf [B] _ _ _ _ _ _ _ _ _.
Arguments decompose_inf' [B] _ _ _ _ _ _ _.
Arguments inf_times_cons [B] _ _ _ _ _.
Arguments inf_times_finite [B] _.
Arguments sup_plus_finite [B] _.


Section OpProp.
Variable B:Bool_Alg.
Let Bt := bt B.
Lemma le_times_finite_member : forall (l:list Bt) (x:Bt), 
  (In x l) -> le (times_list l) x.
intros l x h1.
induction l as [|a l h2].
(* nil *)
simpl in h1.
contradiction.
(* a::l *)
apply in_inv in h1.
destruct h1 as [h1l | h1r].
  (* a = x *)
  subst.
  simpl.
  apply times_le.
  (* In x l *)
  specialize (h2 h1r).
  simpl.
  pose proof (times_le (times_list l) a) as h3.
  rewrite comm_prod in h3.
  apply trans_le with (times_list l); assumption.
Qed.

Lemma le_member_plus_finite : forall (l:list Bt) (x:Bt), 
  (In x l) -> le x (plus_list l).
intros l x h1.
induction l as [|a l h2].
(* nil *)
simpl in h1.
contradiction.
(* a:: l*)
apply in_inv in h1.
destruct h1 as [h1l | h1r].
  (* a = x *)
  subst.
  simpl.
  apply le_plus.
  (* In x l *)
  specialize (h2 h1r).
  simpl.
  pose proof (le_plus (plus_list l) a) as h3.
  rewrite comm_sum in h3.
  apply trans_le with (plus_list l); assumption.
Qed.


End OpProp.

Arguments le_times_finite_member [B] _ _ _.
Arguments le_member_plus_finite [B] _ _ _.


Section SetOperations.
Variable B:Bool_Alg.
Let Bt := bt B.


Lemma prod_list_dup_eq : 
  forall (l:list Bt) (x:Bt), 
    In x l -> times_list l = times_list (x::l).
  intros l.  induction l as [|a l h1].
  intros x h2; try contradiction.
  intros x h2.
  destruct h2 as [h2l | h2r]. subst.
  simpl.
  rewrite assoc_prod.
  rewrite idem_prod. reflexivity.
  simpl.
  rewrite assoc_prod.
  rewrite (comm_prod _ x a).
  rewrite <- assoc_prod.
  simpl in h1.
  rewrite <- (h1 x h2r).
  auto.
Qed.

Lemma sum_list_dup_eq : 
  forall (l:list Bt) (x:Bt), 
    In x l -> plus_list l = plus_list (x::l).
  intro l.  induction l as [|a l h1].
  intros x h2; try contradiction.
  intros x h2.
  destruct h2 as [h2l | h2r]. subst.
  simpl.
  rewrite assoc_sum.
  rewrite idem_sum. reflexivity.
  simpl.
  rewrite assoc_sum.
  rewrite (comm_sum _ x a).
  rewrite <- assoc_sum.
  simpl in h1.
  rewrite <- (h1 x h2r).
  auto.
Qed.


Lemma prod_preserves_list_singularize :
  forall (l:list Bt),
    times_list l = times_list (list_singularize l nil).
  intro l.
  induction l as [|a l h1].
  simpl; auto.
  simpl.
  rewrite h1.
  destruct (in_dec eq_dec a l) as [h2 | h3].
  rewrite <- h1.
  pose proof (prod_list_dup_eq _ _ h2) as h3.
  simpl in h3.
  rewrite <- h3. auto.
  simpl.
  rewrite <- h1.
  assert (h4:forall x:Bt, In x (a::nil) -> ~In x l).
    intros x h4.
    destruct h4; [subst | contradiction]; auto.
  rewrite (list_singularize_not_in _ _ h4).
  rewrite <- h1.
  reflexivity.
Qed.  

Lemma sum_preserves_list_singularize :
  forall (l:list Bt),
    plus_list l = plus_list (list_singularize l nil).
  intro l.
  induction l as [|a l h1].
  simpl; auto.
  simpl.
  rewrite h1.
  destruct (in_dec eq_dec a l) as [h2 | h3].
  rewrite <- h1.
  pose proof (sum_list_dup_eq _ _ h2) as h3.
  simpl in h3.
  rewrite <- h3. auto.
  simpl.
  rewrite <- h1.
  assert (h4:forall x:Bt, In x (a::nil) -> ~In x l).
    intros x h4.
    destruct h4; [subst | contradiction]; auto.
  rewrite (list_singularize_not_in _ _ h4).
  rewrite <- h1.
  reflexivity.
Qed.  

      

Lemma times_sing_preserves_new_head :
  forall (l:list Bt) (x:Bt),
    In x l ->
    times_list (list_singularize l nil) =
    times_list (new_head (list_singularize l nil) x).
intro l. induction l as [|a l h1].
intros; contradiction.
intros x h2.
simpl. simpl in h1.
destruct (in_dec eq_dec a l) as [h3 | h4].
destruct h2 as [h2a | h2b]. subst.
rewrite <- (h1 _ h3). auto.
rewrite <- (h1 _ h2b). auto.
simpl.
assert (h5:forall x:Bt, In x (a::nil) -> ~In x l).
  intros y h6.
  destruct h6; [subst | contradiction]; auto.
rewrite (list_singularize_not_in _ _ h5).
destruct (eq_dec a x) as [h6 | h7].
subst.
rewrite <- list_singularize_new_head_not_in_compat; auto.
simpl in h2.
assert (h8:In x l). tauto.
simpl.
rewrite (h1 _ h8).
rewrite assoc_prod.
rewrite (comm_prod _ a x).
rewrite assoc_prod.
auto.
Qed.

Lemma plus_sing_preserves_new_head :
  forall (l:list Bt) (x:Bt),
    In x l ->
    plus_list (list_singularize l nil) =
    plus_list (new_head (list_singularize l nil) x).
intro l. induction l as [|a l h1].
intros; contradiction.
intros x h2.
simpl. simpl in h1.
destruct (in_dec eq_dec a l) as [h3 | h4].
destruct h2 as [h2a | h2b]. subst.
rewrite <- (h1 _ h3). auto.
rewrite <- (h1 _ h2b). auto.
simpl.
assert (h5:forall x:Bt, In x (a::nil) -> ~In x l).
  intros y h6.
  destruct h6; [subst | contradiction]; auto.
rewrite (list_singularize_not_in _ _ h5).
destruct (eq_dec a x) as [h6 | h7].
subst.
rewrite <- list_singularize_new_head_not_in_compat; auto.
simpl in h2.
assert (h8:In x l). tauto.
simpl.
rewrite (h1 _ h8).
rewrite assoc_sum.
rewrite (comm_sum _ a x).
rewrite assoc_sum.
auto.
Qed.


Lemma times_list_sing_cons :
  forall (l:list Bt) (x:Bt), 
    In x (list_singularize l nil) ->
    times_list (list_singularize l nil) =
    x*times_list (list_singularize (new_head_aux l x) nil).
intros l. induction l as [|a l h1].
intros x h1. simpl in h1. contradiction.
intros x h2. simpl in h2. simpl.
destruct (in_dec eq_dec a l) as [h3 | h4]; 
  destruct (eq_dec a x) as [h5 | h6]. subst.
apply h1. auto.
simpl.
destruct (in_dec eq_dec a (new_head_aux l x)) as [h7 | h8].
auto.
pose proof (in_l_neq_a_x_in_new_head_aux _ _ _ h3 h6).
contradiction.
subst. simpl.
rewrite nin_new_head_aux_eq.
assert (h6:forall c:Bt, In c (x::nil) -> ~In c l).
  intros y h8.
  destruct h8; [subst | contradiction]; auto.
rewrite list_singularize_not_in. auto. auto. auto.
simpl.
destruct (in_dec eq_dec a (new_head_aux l x)) as [h7 | h8].
pose proof (nin_nin_new_head_aux _ _ x h4).
contradiction.
assert (h9:forall c:Bt, In c (a::nil) -> ~In c l).
  intros y h10.
  destruct h10; [subst | contradiction]; auto.
rewrite list_singularize_not_in.
rewrite list_singularize_not_in in h2.
assert (h10:forall c:Bt, In c (a::nil) -> 
                         ~In c (new_head_aux l x)).
  intros y h10.
  destruct h10; [subst | contradiction]; auto.
rewrite (list_singularize_not_in _ _ h10).
destruct h2 as [h2l | h2r]. contradiction.
rewrite h1 with x.
simpl.
rewrite assoc_prod. rewrite (comm_prod _ a x).
rewrite assoc_prod. reflexivity.
auto. auto. auto.
Qed.

Lemma plus_list_sing_cons :
  forall (l:list Bt) (x:Bt), 
    In x (list_singularize l nil) ->
    plus_list (list_singularize l nil) =
    x+plus_list (list_singularize (new_head_aux l x) nil).
intros l. induction l as [|a l h1].
intros x h1. simpl in h1. contradiction.
intros x h2. simpl in h2. simpl.
destruct (in_dec eq_dec a l) as [h3 | h4]; 
  destruct (eq_dec a x) as [h5 | h6]. subst.
apply h1. auto.
simpl.
destruct (in_dec eq_dec a (new_head_aux l x)) as [h7 | h8].
auto.
pose proof (in_l_neq_a_x_in_new_head_aux _ _ _ h3 h6).
contradiction.
subst. simpl.
rewrite nin_new_head_aux_eq.
assert (h6:forall c:Bt, In c (x::nil) -> ~In c l).
  intros y h8.
  destruct h8; [subst | contradiction]; auto.
rewrite list_singularize_not_in. auto. auto. auto.
simpl.
destruct (in_dec eq_dec a (new_head_aux l x)) as [h7 | h8].
pose proof (nin_nin_new_head_aux _ _ x h4).
contradiction.
assert (h9:forall c:Bt, In c (a::nil) -> ~In c l).
  intros y h10.
  destruct h10; [subst | contradiction]; auto.
rewrite list_singularize_not_in.
rewrite list_singularize_not_in in h2.
assert (h10:forall c:Bt, In c (a::nil) -> 
                         ~In c (new_head_aux l x)).
  intros y h10.
  destruct h10; [subst | contradiction]; auto.
rewrite (list_singularize_not_in _ _ h10).
destruct h2 as [h2l | h2r]. contradiction.
rewrite h1 with x.
simpl.
rewrite assoc_sum. rewrite (comm_sum _ a x).
rewrite assoc_sum. reflexivity.
auto. auto. auto.
Qed.

Lemma list_to_sets_eq_times_sing_eq :
  forall (E:Ensemble Bt), 
    forall (l l':list Bt), 
      E = list_to_set (list_singularize l nil) ->
      E = list_to_set (list_singularize l' nil) ->
      times_list (list_singularize l nil) = 
      times_list (list_singularize l' nil).
  intros E l l' h1.
  pose proof (list_to_set_finite (list_singularize l nil)) as h2.
  rewrite <- h1 in h2.
  revert h1. revert l l'.
  induction h2 as [|A h0 h4 x h5].
  intros l l' h1 h3.
  symmetry in h1. apply empty_set_nil in h1.
  symmetry in h3. apply empty_set_nil in h3.
  rewrite h1. rewrite h3.
  reflexivity.
  intros l l' h6 h7.
  pose proof (add_list_to_set_new_head _ _ _ h5 h6) as h8.
  pose proof (add_list_to_set_new_head _ _ _ h5 h7) as h9.
  rewrite new_head_singularize_comm in h8.
  rewrite new_head_singularize_comm in h9.
  specialize (h4 _ _ h8 h9).
  pose proof (Add_intro2 _ A x) as h10.
  pose proof h10 as h11.
  rewrite h6 in h10.
  rewrite h7 in h11.
  rewrite <- list_to_set_in_iff in h10.
  rewrite <- list_to_set_in_iff in h11.
  rewrite (times_list_sing_cons _ x h10).
  rewrite (times_list_sing_cons _ x h11).
  rewrite h4. reflexivity.
Qed.

Lemma list_to_sets_eq_plus_sing_eq :
  forall (E:Ensemble Bt), 
    forall (l l':list Bt), 
      E = list_to_set (list_singularize l nil) ->
      E = list_to_set (list_singularize l' nil) ->
      plus_list (list_singularize l nil) = 
      plus_list (list_singularize l' nil).
  intros E l l' h1.
  pose proof (list_to_set_finite (list_singularize l nil)) as h2.
  rewrite <- h1 in h2.
  revert h1. revert l l'.
  induction h2 as [|A h0 h4 x h5].
  intros l l' h1 h3.
  symmetry in h1. apply empty_set_nil in h1.
  symmetry in h3. apply empty_set_nil in h3.
  rewrite h1. rewrite h3.
  reflexivity.
  intros l l' h6 h7.
  pose proof (add_list_to_set_new_head _ _ _ h5 h6) as h8.
  pose proof (add_list_to_set_new_head _ _ _ h5 h7) as h9.
  rewrite new_head_singularize_comm in h8.
  rewrite new_head_singularize_comm in h9.
  specialize (h4 _ _ h8 h9).
  pose proof (Add_intro2 _ A x) as h10.
  pose proof h10 as h11.
  rewrite h6 in h10.
  rewrite h7 in h11.
  rewrite <- list_to_set_in_iff in h10.
  rewrite <- list_to_set_in_iff in h11.
  rewrite (plus_list_sing_cons _ x h10).
  rewrite (plus_list_sing_cons _ x h11).
  rewrite h4. reflexivity.
Qed.



Lemma list_to_sets_eq_times_eq :
  forall (E:Ensemble Bt), 
    forall (l l':list Bt), 
      E = list_to_set l ->
      E = list_to_set l' ->
      times_list l = times_list l'.
intros E l l' h1 h2.
rewrite list_to_set_singularize_compat in h1.
rewrite list_to_set_singularize_compat in h2.
rewrite prod_preserves_list_singularize.
rewrite (prod_preserves_list_singularize l').
apply list_to_sets_eq_times_sing_eq with E; assumption.
Qed.

Lemma list_to_sets_eq_plus_eq :
  forall (E:Ensemble Bt), 
    forall (l l':list Bt), 
      E = list_to_set l ->
      E = list_to_set l' ->
      plus_list l = plus_list l'.
intros E l l' h1 h2.
rewrite list_to_set_singularize_compat in h1.
rewrite list_to_set_singularize_compat in h2.
rewrite sum_preserves_list_singularize.
rewrite (sum_preserves_list_singularize l').
apply list_to_sets_eq_plus_sing_eq with E; assumption.
Qed.

Lemma times_list_functional : 
  forall (l l':list Bt),
    list_to_set l = list_to_set l' ->
    times_list l = times_list l'.
intros l l' h1.
apply (list_to_sets_eq_times_eq _ l l' (eq_refl _) h1).
Qed.


Lemma plus_list_functional : 
  forall (l l':list Bt),
    list_to_set l = list_to_set l' ->
    plus_list l = plus_list l'.
intros l l' h1.
apply (list_to_sets_eq_plus_eq _ l l' (eq_refl _) h1).
Qed.



Lemma times_list_unq : forall (E:Ensemble Bt)
                       (pf:Finite E), exists! x, 
                         forall (l:list Bt), E = list_to_set l ->
                       x = times_list l. 
intros E h1.
pose proof (finite_set_list _ h1) as h2.
destruct h2 as [l h3].
exists (times_list l).
red.
split.
intros l' h4.
apply list_to_sets_eq_times_eq with E; assumption.
intros x h4.
rewrite (h4 _ h3).
reflexivity.
Qed.

Lemma plus_list_unq : forall (E:Ensemble Bt)
                       (pf:Finite E), exists! x, 
                         forall (l:list Bt), E = list_to_set l ->
                       x = plus_list l. 
intros E h1.
pose proof (finite_set_list _ h1) as h2.
destruct h2 as [l h3].
exists (plus_list l).
red.
split.
intros l' h4.
apply list_to_sets_eq_plus_eq with E; assumption.
intros x h4.
rewrite (h4 _ h3).
reflexivity.
Qed.
 

Lemma times_times_list_remove : 
  forall (l:list Bt), 
    NoDup l ->
    forall (a:Bt),
      In a l -> 
      times_list l = a * times_list (remove eq_dec a l).
intro l.
induction l as [|b l h1]; auto; intros h2 x h3.
simpl in h3. contradiction.
simpl.
pose proof (no_dup_cons_nin _ _ h2) as h4.
apply no_dup_cons in h2.
destruct h3 as [h3a | h3b]. subst.
f_equal. destruct (eq_dec x x).
rewrite <- remove_not_in'; auto.
contradict n. reflexivity.
destruct (eq_dec x b) as [h5 | h6]. subst. contradiction.
simpl.
rewrite assoc_prod.
rewrite (comm_prod _ x b). 
specialize (h1 h2 _ h3b).
rewrite h1 at 1.
rewrite assoc_prod.
reflexivity.
Qed.

Lemma plus_plus_list_remove : 
  forall (l:list Bt), 
    NoDup l ->
    forall (a:Bt),
      In a l -> 
      plus_list l = a + plus_list (remove eq_dec a l).
intro l.
induction l as [|b l h1]; auto; intros h2 x h3.
simpl in h3. contradiction.
simpl.
pose proof (no_dup_cons_nin _ _ h2) as h4.
apply no_dup_cons in h2.
destruct h3 as [h3a | h3b]. subst.
f_equal. destruct (eq_dec x x).
rewrite <- remove_not_in'; auto.
contradict n. reflexivity.
destruct (eq_dec x b) as [h5 | h6]. subst. contradiction.
simpl.
rewrite assoc_sum.
rewrite (comm_sum _ x b). 
specialize (h1 h2 _ h3b).
rewrite h1 at 1.
rewrite assoc_sum.
reflexivity.
Qed.






(*This is the fruit of all the "singularize" code above, a function
that returns the product (sum) of a finite set, without the user having to deal with lists at all.*)
Definition times_set (E:Ensemble Bt) (pf:Finite E) : Bt.
refine (proj1_sig (constructive_definite_description 
            _ (times_list_unq E pf))).
Defined.

Definition plus_set (E:Ensemble Bt) (pf:Finite E) : Bt.
refine (proj1_sig (constructive_definite_description 
            _ (plus_list_unq E pf))).
Defined.

Lemma times_set_compat : forall (E:Ensemble Bt)
                                (pf:Finite E),
                           exists l:list Bt, 
                             times_set E pf = times_list l.
intros E h1.
unfold times_set.
destruct constructive_definite_description as [x h2]. simpl.
pose proof (finite_set_list _ h1) as h3.
destruct h3 as [l h4].
exists l.
apply h2. assumption.
Qed.


Lemma times_set_compat' : forall (E:Ensemble Bt) (l:list Bt)
                            (pf:Finite E),
                            E = list_to_set l ->
                            times_set E pf = times_list l.
intros E l pf h1.
unfold times_set.
destruct constructive_definite_description as [x h2]. simpl.
apply h2; assumption.
Qed.

Lemma plus_set_compat : forall (E:Ensemble Bt)
                                (pf:Finite E),
                           exists l:list Bt, 
                             plus_set E pf = plus_list l.
intros E h1.
unfold plus_set.
destruct constructive_definite_description as [x h2]. simpl.
pose proof (finite_set_list _ h1) as h3.
destruct h3 as [l h4].
exists l.
apply h2. assumption.
Qed.

Lemma plus_set_compat' : forall (E:Ensemble Bt) (l:list Bt)
                            (pf:Finite E),
                            E = list_to_set l ->
                            plus_set E pf = plus_list l.
intros E l pf h1.
unfold plus_set.
destruct constructive_definite_description as [x h2]. simpl.
apply h2; assumption.
Qed.

 

Lemma times_set_functional : 
  forall (A C:Ensemble Bt) (pfa:Finite A) (pfc:Finite C),
    A = C -> times_set A pfa = times_set C pfc.
intros A C h1 h2 h3.
pose proof (subsetT_eq_compat _ _ _ _ h1 h2 h3) as h4.
dependent rewrite -> h4.
reflexivity.
Qed.

Lemma plus_set_functional : 
  forall (A C:Ensemble Bt) (pfa:Finite A) (pfc:Finite C),
    A = C -> plus_set A pfa = plus_set C pfc.
intros A C h1 h2 h3.
pose proof (subsetT_eq_compat _ _ _ _ h1 h2 h3) as h4.
dependent rewrite -> h4.
reflexivity.
Qed.

Lemma times_set_empty : 
  times_set (Empty_set Bt) (Empty_is_finite Bt) = 1.
unfold times_set.
destruct constructive_definite_description as [x h1]. simpl.
specialize (h1 nil).
simpl in h1. 
apply h1. reflexivity.
Qed.

Lemma times_set_empty' : 
  forall (pf:Finite (Empty_set Bt)),
    times_set (Empty_set Bt) pf = 1.
intro h1.
pose (Empty_is_finite Bt) as h2.
assert (h3:h1 = h2). apply proof_irrelevance.
rewrite h3.
apply times_set_empty.
Qed.

Lemma plus_set_empty :
  plus_set (Empty_set Bt) (Empty_is_finite Bt) = 0.
unfold plus_set.
destruct constructive_definite_description as [x h1]. simpl.
specialize (h1 nil).
simpl in h1. 
apply h1. reflexivity.
Qed.

Lemma plus_set_empty' : 
  forall (pf:Finite (Empty_set Bt)),
    plus_set (Empty_set Bt) pf = 0.
intro h1.
pose (Empty_is_finite Bt) as h2.
assert (h3:h1 = h2). apply proof_irrelevance.
rewrite h3.
apply plus_set_empty.
Qed.

Lemma times_set_add : 
  forall (E:Ensemble Bt) (pf:Finite E) (x:Bt),
    times_set (Add E x) (Add_preserves_Finite _ _ _ pf) = 
    x * (times_set E pf).
intros E pf x.
unfold times_set.
destruct constructive_definite_description as [y h1].
simpl.
destruct constructive_definite_description as [z h2]. simpl.
pose proof (finite_set_list _ pf) as h3.
destruct h3 as [l h3].
pose proof (h2 l h3) as h5.
pose proof (finite_set_list _ (Add_preserves_Finite _ _ x pf)) as h4.
destruct h4 as [l' h4].
pose proof (h1 l' h4) as h6.
destruct (in_dec eq_dec x l) as [h9 | h10].
(* h9 *)
pose proof h9 as h9'.
rewrite (list_to_set_in_iff l x) in h9'.
rewrite <- h3 in h9'.
pose proof (in_add_eq E x h9') as h10.
rewrite h10 in h4.
specialize (h2 _ h4).
pose proof (prod_list_dup_eq _ _ h9) as h11.
simpl in h11.
subst.
rewrite h5.
assumption.
(* h10 *)
pose proof h10 as h10'.
rewrite (list_to_set_in_iff l x) in h10'.
rewrite <- h3 in h10'.
assert (h7:Subtract (Add E x) x = Subtract (list_to_set l') x).
  f_equal. auto.
pose proof (subtract_new_head_compat l' x) as h8.
rewrite h8 in h7.
rewrite (sub_add_compat_nin _ _ h10') in h7.
pose proof (h2 _ h7) as h11.
subst.
rewrite h11.
pose proof (Add_intro2 _ (list_to_set l) x) as h12.
rewrite h4 in h12.
rewrite <- list_to_set_in_iff in h12.
rewrite prod_preserves_list_singularize.
rewrite (prod_preserves_list_singularize (new_head_aux l' x)).
apply times_list_sing_cons.
pose proof (list_to_set_singularize_compat l') as h13.
rewrite list_to_set_in_iff in h12.
rewrite h13 in h12.
rewrite <- list_to_set_in_iff in h12.
assumption.
Qed.

Lemma times_set_add' : 
  forall (E:Ensemble Bt) (pf1:Finite E) (x:Bt)
         (pf2:Finite (Add E x)),
    times_set (Add E x) pf2 = 
    x * (times_set E pf1).
intros E h1 x h2.
pose (Add_preserves_Finite _ _ x h1) as h3.
assert (h4:h2 = h3).  apply proof_irrelevance.
rewrite h4.
unfold h3.
rewrite times_set_add. reflexivity.
Qed.

Lemma plus_set_add : 
  forall (E:Ensemble Bt) (pf:Finite E) (x:Bt),
    plus_set (Add E x) (Add_preserves_Finite _ _ _ pf) = 
    x + (plus_set E pf).
intros E pf x.
unfold plus_set.
destruct constructive_definite_description as [y h1].
simpl.
destruct constructive_definite_description as [z h2]. simpl.
pose proof (finite_set_list _ pf) as h3.
destruct h3 as [l h3].
pose proof (h2 l h3) as h5.
pose proof (finite_set_list _ (Add_preserves_Finite _ _ x pf)) as h4.
destruct h4 as [l' h4].
pose proof (h1 l' h4) as h6.
destruct (in_dec eq_dec x l) as [h9 | h10].
(* h9 *)
pose proof h9 as h9'.
rewrite (list_to_set_in_iff l x) in h9'.
rewrite <- h3 in h9'.
pose proof (in_add_eq E x h9') as h10.
rewrite h10 in h4.
specialize (h2 _ h4).
pose proof (sum_list_dup_eq _ _ h9) as h11.
simpl in h11.
subst.
rewrite h5.
assumption.
(* h10 *)
pose proof h10 as h10'.
rewrite (list_to_set_in_iff l x) in h10'.
rewrite <- h3 in h10'.
assert (h7:Subtract (Add E x) x = Subtract (list_to_set l') x).
  f_equal. auto.
pose proof (subtract_new_head_compat l' x) as h8.
rewrite h8 in h7.
rewrite (sub_add_compat_nin _ _ h10') in h7.
pose proof (h2 _ h7) as h11.
subst.
rewrite h11.
pose proof (Add_intro2 _ (list_to_set l) x) as h12.
rewrite h4 in h12.
rewrite <- list_to_set_in_iff in h12.
rewrite sum_preserves_list_singularize.
rewrite (sum_preserves_list_singularize (new_head_aux l' x)).
apply plus_list_sing_cons.
pose proof (list_to_set_singularize_compat l') as h13.
rewrite list_to_set_in_iff in h12.
rewrite h13 in h12.
rewrite <- list_to_set_in_iff in h12.
assumption.
Qed.

Lemma plus_set_add' : 
  forall (E:Ensemble Bt) (pf1:Finite E) (x:Bt)
         (pf2:Finite (Add E x)),
    plus_set (Add E x) pf2 = 
    x + (plus_set E pf1).
intros E h1 x h2.
pose (Add_preserves_Finite _ _ x h1) as h3.
assert (h4:h2 = h3).  apply proof_irrelevance.
rewrite h4.
unfold h3.
rewrite plus_set_add. reflexivity.
Qed.


Lemma times_set_sing : 
  forall x:Bt, times_set (Singleton x) (Singleton_is_finite _ x) = x.
intro x.
pose proof (Empty_is_finite Bt) as h1.
pose proof (Add_preserves_Finite _ _ x h1) as h2.
pose proof (add_empty_sing x) as h3.
symmetry in h3.
pose proof (subsetT_eq_compat _ _ _ _ (Singleton_is_finite _ x)
                              h2 h3) as h4.
dependent rewrite -> h4.
rewrite (times_set_add' _ h1 x h2). 
rewrite times_set_empty'.
apply one_prod.
Qed.

Lemma times_set_sing' : 
  forall (x:Bt) (pf:Finite (Singleton x)), times_set _ pf = x.
intros x h1.
pose proof (times_set_sing x) as h2.
assert (h3:h1 = (Singleton_is_finite Bt x)). apply proof_irrelevance.
rewrite <- h3 in h2.
assumption.
Qed.

Lemma times_set_one_or : 
  forall (E:Ensemble Bt) (pf:Finite E),
         (E = Empty_set _) \/ (E = Singleton 1) ->
    1 = times_set E pf.
intros E h1 h2.
destruct h2.
subst. rewrite times_set_empty'.
reflexivity.
subst.
rewrite times_set_sing'.
reflexivity.
Qed.


Lemma plus_set_sing : 
  forall x:Bt, plus_set (Singleton x) (Singleton_is_finite _ x) = x.
intro x.
pose proof (Empty_is_finite Bt) as h1.
pose proof (Add_preserves_Finite _ _ x h1) as h2.
pose proof (add_empty_sing x) as h3.
symmetry in h3.
pose proof (subsetT_eq_compat _ _ _ _ (Singleton_is_finite _ x)
                              h2 h3) as h4.
dependent rewrite -> h4.
rewrite (plus_set_add' _ h1 x h2). 
rewrite plus_set_empty'.
apply zero_sum.
Qed.

Lemma plus_set_sing' : 
  forall (x:Bt) (pf:Finite (Singleton x)), plus_set _ pf = x.
intros x h1.
pose proof (plus_set_sing x) as h2.
assert (h3:h1 = (Singleton_is_finite Bt x)). apply proof_irrelevance.
rewrite <- h3 in h2.
assumption.
Qed.

Lemma plus_set_zero_or : 
  forall (E:Ensemble Bt) (pf:Finite E),
         (E = Empty_set _) \/ (E = Singleton 0) ->
    0 = plus_set E pf.
intros E h1 h2.
destruct h2.
subst. rewrite plus_set_empty'.
reflexivity.
subst.
rewrite plus_set_sing'.
reflexivity.
Qed.

Lemma times_set_couple : 
  forall (x y:Bt),
    times_set (Couple x y) (finite_couple x y) = x * y.
intros x y.
pose proof (couple_add_sing x y) as h1.
pose proof (Add_preserves_Finite _ _ y (Singleton_is_finite _ x)) as h2.
pose proof (subsetT_eq_compat _ _ _ _ (finite_couple x y) h2 h1) as h3.
dependent rewrite -> h3.
rewrite (times_set_add' _ (Singleton_is_finite _ x) _ h2).
rewrite times_set_sing.
rewrite comm_prod.
reflexivity.
Qed.

Lemma times_set_couple' : 
  forall (x y:Bt) (pf:Finite (Couple x y)),
    times_set (Couple x y) pf = x * y.
intros x y h1.
rewrite <- times_set_couple.
apply times_set_functional.
reflexivity.
Qed.

Lemma plus_set_couple : 
  forall (x y:Bt),
    plus_set (Couple x y) (finite_couple x y) = x + y.
intros x y.
pose proof (couple_add_sing x y) as h1.
pose proof (Add_preserves_Finite _ _ y (Singleton_is_finite _ x)) as h2.
pose proof (subsetT_eq_compat _ _ _ _ (finite_couple x y) h2 h1) as h3.
dependent rewrite -> h3.
rewrite (plus_set_add' _ (Singleton_is_finite _ x) _ h2).
rewrite plus_set_sing.
rewrite comm_sum.
reflexivity.
Qed.

Lemma plus_set_couple' : 
  forall (x y:Bt) (pf:Finite (Couple x y)),
    plus_set (Couple x y) pf = x + y.
intros x y h1.
rewrite <- plus_set_couple.
apply plus_set_functional.
reflexivity.
Qed.


Lemma le_times_set : forall (E:Ensemble Bt) (pf:Finite E) 
                            (x:Bt), 
                       (Ensembles.In E x) -> 
                       le (times_set E pf) x.
intros E h1 x h2.
pose proof h1 as h3.
induction h3 as [|E h4 h5 a].
contradiction.
destruct h2 as [x h2a | x h2b].
assert (h6:Included E (Add E a)). auto with sets.
pose proof (Finite_downward_closed _ _ h1 _ h6) as h7.
rewrite (times_set_add' _ h7 _ h1).
pose proof (times_le (times_set E h7) a) as h3.
specialize (h5 h7 h2a).
apply trans_le with (times_set E h7).
rewrite comm_prod.
assumption. assumption.
destruct h2b.
rewrite (times_set_add' _ h4 _ h1).
apply times_le.
Qed.


Definition times_set_def 
           (def:Bt) (E:Ensemble Bt) :=
  match (classic_dec (Finite E)) with
    | left pf => times_set _ pf
    | _ => def
  end.

Definition plus_set_def 
           (def:Bt)  (E:Ensemble Bt)  :=
  match (classic_dec (Finite E)) with
    | left pf => plus_set _ pf
    | _ => def
  end.


Lemma le_plus_set : forall (E:Ensemble Bt) (pf:Finite E) 
                            (x:Bt), 
                       (Ensembles.In E x) -> 
                       le x (plus_set E pf).
intros E h1 x h2.
pose proof h1 as h3.
induction h3 as [|E h4 h5 a].
contradiction.
destruct h2 as [x h2a | x h2b].
assert (h6:Included E (Add E a)). auto with sets.
pose proof (Finite_downward_closed _ _ h1 _ h6) as h7.
rewrite (plus_set_add' _ h7 _ h1).
pose proof (le_plus (plus_set E h7) a) as h3.
specialize (h5 h7 h2a).
rewrite comm_sum.
apply trans_le with (plus_set E h7). 
assumption.  assumption.
destruct h2b.
rewrite (plus_set_add' _ h4 _ h1).
apply le_plus.
Qed.

Lemma inf_times_set : forall (E:Ensemble Bt)
  (pf:Finite E), inf E (times_set E pf).
intros E h1.
pose proof h1 as h2.
induction h2 as [|E h3 h4 e h5].
(*Empty*)
rewrite times_set_empty'.
red. split. red. intros; contradiction. 
intros; apply one_max.
(*Add*)
assert (h6:Included E (Add E e)). auto with sets.
specialize (h4 h3).
rewrite (times_set_add' _ h3).
unfold Add.
rewrite comm_sum_psa.
apply inf_union.
apply inf_singleton.
assumption.
Qed.

Lemma sup_plus_set : forall (E:Ensemble Bt)
  (pf:Finite E), sup E (plus_set E pf).
intros E h1.
pose proof h1 as h2.
induction h2 as [|E h3 h4 e h5].
(*Empty*)
rewrite plus_set_empty'.
red. split. red. intros; contradiction. 
intros; apply zero_min.
(*Add*)
assert (h6:Included E (Add E e)). auto with sets.
specialize (h4 h3).
rewrite (plus_set_add' _ h3).
unfold Add.
rewrite comm_sum_psa.
apply sup_union.
apply sup_singleton.
assumption.
Qed.

Lemma times_set_union : 
  forall (C D:Ensemble Bt) (pfc:Finite C)
         (pfd: Finite D),
    times_set (Union C D) (Union_preserves_Finite _ _ _ pfc pfd) =
    (times_set C pfc) * (times_set D pfd).
intros C D h1.
pose proof h1 as h2.
induction h2 as [|C h3 h4].
intros h3.
rewrite times_set_empty'.
pose proof (empty_union D) as h4.
pose proof (subsetT_eq_compat _ _ _ _
  (Union_preserves_Finite Bt (Empty_set Bt) D h1 h3) h3 h4) as h5.
dependent rewrite -> h5.
rewrite comm_prod.
rewrite one_prod. reflexivity.
intro h5.
pose proof (union_add_comm C D x) as h6.
pose proof (Union_preserves_Finite _ _ _ h3 h5) as h7.
pose proof (Add_preserves_Finite _ _ x h7) as h8.
pose proof (subsetT_eq_compat _ _ _ _
  (Union_preserves_Finite Bt (Add C x) D h1 h5) h8 h6) as h9.
dependent rewrite -> h9.
rewrite (times_set_add' C h3 x h1).
rewrite (times_set_add' (Union C D) h7 x h8).
specialize (h4 h3 h5).
assert (h10:(Union_preserves_Finite Bt C D h3 h5) = h7).
  apply proof_irrelevance.
pose proof 
     (subsetT_eq_compat _ _ _ h7
                        (Union_preserves_Finite Bt C D h3 h5)
                        h7 h10) as h11. 
dependent rewrite <- h11.
rewrite h4.
apply assoc_prod.
Qed.

Lemma times_set_union' : 
  forall (C D:Ensemble Bt) (pfc:Finite C)
         (pfd: Finite D) (pfu:Finite (Union C D)),
    times_set (Union C D) pfu =
    (times_set C pfc) * (times_set D pfd).
intros C D h1 h2 h3.
pose proof (times_set_union _ _ h1 h2) as h4.
rewrite <- h4.
apply times_set_functional.
reflexivity.
Qed.


Lemma times_set_inc_le :
  forall (C D:Ensemble Bt) (pfc:Finite C)
         (pfd:Finite D) (pfi:Included C D),
    le (times_set D pfd) (times_set C pfc).
intros C D h1 h2 h3.  pose proof h2 as h2'.
pose proof (decompose_setminus_inc D _ h3) as h4.
generalize dependent h2. rewrite h4.
intro h2.
pose proof (setminus_inc D C) as h5.
pose proof (Finite_downward_closed _ _ h2' _ h5) as h6.
rewrite (times_set_union' _ _ h1 h6).
apply times_le.
Qed.


Lemma plus_set_union : 
  forall (C D:Ensemble Bt) (pfc:Finite C)
         (pfd: Finite D),
    plus_set (Union C D) (Union_preserves_Finite _ _ _ pfc pfd) =
    (plus_set C pfc) + (plus_set D pfd).
intros C D h1.
pose proof h1 as h2.
induction h2 as [|C h3 h4].
intros h3.
rewrite plus_set_empty'.
pose proof (empty_union D) as h4.
pose proof (subsetT_eq_compat _ _ _ _
  (Union_preserves_Finite Bt (Empty_set Bt) D h1 h3) h3 h4) as h5.
dependent rewrite -> h5.
rewrite comm_sum.
rewrite zero_sum. reflexivity.
intro h5.
pose proof (union_add_comm C D x) as h6.
pose proof (Union_preserves_Finite _ _ _ h3 h5) as h7.
pose proof (Add_preserves_Finite _ _ x h7) as h8.
pose proof (subsetT_eq_compat _ _ _ _
  (Union_preserves_Finite Bt (Add C x) D h1 h5) h8 h6) as h9.
dependent rewrite -> h9.
rewrite (plus_set_add' C h3 x h1).
rewrite (plus_set_add' (Union C D) h7 x h8).
specialize (h4 h3 h5).
assert (h10:(Union_preserves_Finite Bt C D h3 h5) = h7).
  apply proof_irrelevance.
pose proof 
     (subsetT_eq_compat _ _ _ h7
                        (Union_preserves_Finite Bt C D h3 h5)
                        h7 h10) as h11.  
dependent rewrite <- h11.
rewrite h4.
apply assoc_sum.
Qed.

Lemma plus_set_union' : 
  forall (C D:Ensemble Bt) (pfc:Finite C)
         (pfd: Finite D) (pfu:Finite (Union C D)),
    plus_set (Union C D) pfu =
    (plus_set C pfc) + (plus_set D pfd).
intros C D h1 h2 h3.
pose proof (plus_set_union _ _ h1 h2) as h4.
rewrite <- h4.
apply plus_set_functional.
reflexivity.
Qed.

Lemma plus_set_inc_le :
  forall (C D:Ensemble Bt) (pfc:Finite C)
         (pfd:Finite D) (pfi:Included C D),
    le (plus_set C pfc) (plus_set D pfd).
intros C D h1 h2 h3.  pose proof h2 as h2'.
pose proof (decompose_setminus_inc D _ h3) as h4.
generalize dependent h2. rewrite h4.
intro h2.
pose proof (setminus_inc D C) as h5.
pose proof (Finite_downward_closed _ _ h2' _ h5) as h6.
rewrite (plus_set_union' _ _ h1 h6).
apply le_plus.
Qed.



Lemma plus_set_im_add : 
  forall (E:Ensemble Bt) (pf:Finite E) (x:Bt) (f:Bt->Bt),
    plus_set (Im (Add E x) f) 
                 (finite_image _ _ _ f 
                               (Add_preserves_Finite _ _ x pf)) =
    f x + plus_set (Im E f) (finite_image _ _ _ f pf).
intros E h1 x f.
pose proof (Im_add _ _ E x f) as h2.
pose proof (finite_image _ _ _ f h1) as h3.
pose proof (Add_preserves_Finite _ _ (f x) h3) as h4.
pose proof (subsetT_eq_compat _ _ _ _ 
                              (finite_image Bt Bt (Add E x) f (Add_preserves_Finite Bt E x h1))
                              h4 h2) as h7.
dependent rewrite -> h7.
rewrite (plus_set_add' _ (finite_image Bt Bt E f h1)  _ h4).
reflexivity.
Qed.

Lemma plus_set_im_add' : forall (E:Ensemble Bt) (x:Bt) (f:Bt->Bt)
                                (pf0: Finite E) 
                                (pf1:Finite (Im (Add E x) f))
                                (pf2:Finite (Im E f)),
                           plus_set (Im (Add E x) f) pf1 =
                           f x + (plus_set (Im E f) pf2).
intros E x f h1 h2 h3.
pose proof (plus_set_im_add _ h1 x f) as h4.
assert (h5:h2 = (finite_image Bt Bt (Add E x) f 
                              (Add_preserves_Finite Bt E x h1))).
  apply proof_irrelevance.
assert (h6:h3 = (finite_image Bt Bt E f h1)).
  apply proof_irrelevance.
rewrite <- h5 in h4.
rewrite <- h6 in h4.
assumption.
Qed.

Lemma times_set_im_add : 
  forall (E:Ensemble Bt) (pf:Finite E) (x:Bt) (f:Bt->Bt),
    times_set (Im (Add E x) f) 
                 (finite_image _ _ _ f 
                               (Add_preserves_Finite _ _ x pf)) =
    f x * times_set (Im E f) (finite_image _ _ _ f pf).
intros E h1 x f.
pose proof (Im_add _ _ E x f) as h2.
pose proof (finite_image _ _ _ f h1) as h3.
pose proof (Add_preserves_Finite _ _ (f x) h3) as h4.
pose proof (subsetT_eq_compat _ _ _ _ 
                              (finite_image Bt Bt (Add E x) f (Add_preserves_Finite Bt E x h1))
                              h4 h2) as h7.
dependent rewrite -> h7.
rewrite (times_set_add' _ (finite_image Bt Bt E f h1)  _ h4).
reflexivity.
Qed.

Lemma times_set_im_add' : forall (E:Ensemble Bt) (x:Bt) (f:Bt->Bt)
                                (pf0: Finite E) 
                                (pf1:Finite (Im (Add E x) f))
                                (pf2:Finite (Im E f)),
                           times_set (Im (Add E x) f) pf1 =
                           f x * (times_set (Im E f) pf2).
intros E x f h1 h2 h3.
pose proof (times_set_im_add _ h1 x f) as h4.
assert (h5:h2 = (finite_image Bt Bt (Add E x) f 
                              (Add_preserves_Finite Bt E x h1))).
  apply proof_irrelevance.
assert (h6:h3 = (finite_image Bt Bt E f h1)).
  apply proof_irrelevance.
rewrite <- h5 in h4.
rewrite <- h6 in h4.
assumption.
Qed.
                                            
Lemma dist_set_plus1 : 
  forall (E:Ensemble Bt) (pf:Finite E) (x:Bt),
    let f := (fun y:Bt => x *y) in
    x * (plus_set E pf) =
    plus_set (Im E f) (finite_image _ _ _ f pf).
intros E h1.
pose proof h1 as h2.
induction h2 as [|E h3 h4 a h5].
(* Empty *)
intros x f.
rewrite plus_set_empty'.
rewrite zero_prod.
pose proof (image_empty Bt Bt f) as h3.
pose proof 
     (subsetT_eq_compat _ _ (Im (Empty_set Bt) f) (Empty_set Bt) (finite_image _ _ (Empty_set Bt) 
                                                                               f h1) h1 h3) as h4.
dependent rewrite -> h4.
rewrite plus_set_empty'. reflexivity.
(* Add *)
intros x f.
pose proof (plus_set_im_add' E a f h3 
                          (finite_image Bt (Btype (Bc B)) 
                                        (Add E a) f h1)
                          (finite_image Bt (Btype (Bc B)) E f h3)) as h6.
replace ( plus_set (Im (Add E a) f) (finite_image Bt (Btype (Bc B)) (Add E a) f h1)) with ( f a + plus_set (Im E f) (finite_image Bt (Btype (Bc B)) E f h3)).
rewrite (plus_set_add' _ h3 _ h1).
rewrite dist_sum.
specialize (h4 h3 x).
rewrite h4.
unfold f.
reflexivity.
Qed.

Lemma dist_set_plus1' : 
  forall (E:Ensemble Bt) (pf1:Finite E) (x:Bt),
    let f := (fun y:Bt => x *y) in
    forall (pf2:Finite (Im E f)),
    x * (plus_set E pf1) =
    plus_set (Im E f) pf2.
intros E h1 x f h2.
pose proof (dist_set_plus1 _ h1 x) as h3.
assert (h4:h2 = (finite_image Bt (Btype (Bc B)) E f h1)).
  apply proof_irrelevance.
replace h2 with (finite_image Bt (Btype (Bc B)) E f h1).
assumption.
Qed.

Lemma dist_set_times1 : 
  forall (E:Ensemble Bt) (pf:Finite E) (x:Bt),
    let f := (fun y:Bt => x + y) in
    x + (times_set E pf) =
    times_set (Im E f) (finite_image _ _ _ f pf).
intros E h1.
pose proof h1 as h2.
induction h2 as [|E h3 h4 a h5].
(* Empty *)
intros x f.
rewrite times_set_empty'.
rewrite one_sum.
pose proof (image_empty Bt Bt f) as h3.
pose proof 
     (subsetT_eq_compat _ _ (Im (Empty_set Bt) f) (Empty_set Bt) (finite_image _ _ (Empty_set Bt) 
                                                                               f h1) h1 h3) as h4.
dependent rewrite -> h4.
rewrite times_set_empty'. reflexivity.
(* Add *)
intros x f.
pose proof (times_set_im_add' E a f h3 
                          (finite_image Bt (Btype (Bc B)) 
                                        (Add E a) f h1)
                          (finite_image Bt (Btype (Bc B)) E f h3)) as h6.
replace ( times_set (Im (Add E a) f) (finite_image Bt (Btype (Bc B)) (Add E a) f h1)) with ( f a * times_set (Im E f) (finite_image Bt (Btype (Bc B)) E f h3)).
rewrite (times_set_add' _ h3 _ h1).
rewrite dist_prod.
specialize (h4 h3 x).
rewrite h4.
unfold f.
reflexivity.
Qed.

Lemma dist_set_times1' : 
  forall (E:Ensemble Bt) (pf1:Finite E) (x:Bt),
    let f := (fun y:Bt => x + y) in
    forall (pf2:Finite (Im E f)), x + (times_set E pf1) =
    times_set (Im E f) pf2.
intros E h1 x f h2.
pose proof (dist_set_times1 E h1 x) as h3.
assert (h4:h2 = (finite_image Bt (Btype (Bc B)) E f h1)).
  apply proof_irrelevance.
replace h2 with (finite_image Bt (Btype (Bc B)) E f h1).
apply h3.
Qed.

Lemma dist_set_plus2 : 
  forall (D E:Ensemble Bt)
    (pfd: Finite D) (pfe: Finite E),
    let f := (fun p:Bt*Bt => (fst p) * (snd p)) in
    (plus_set D pfd) * (plus_set E pfe) =
    plus_set (Im (cart_prod D E) f)
             (finite_image _ _ _ f 
                           (cart_prod_fin _ _ pfd pfe)).
intros D E h1.
pose proof h1 as h2.
induction h2 as [|D h3 h4 d h5].
  (*Empty D*)
  pose proof (@cart_prod_empty Bt _ E) as h3.
  intros h4 f.
  pose proof (image_empty _ _ f) as h5.
  rewrite <- h3 in h5.
  pose proof (subsetT_eq_compat _ _ _ _ 
(finite_image (Bt * Bt) (Bt ) (cart_prod (Empty_set Bt) E) f
        (cart_prod_fin (Empty_set Bt) E h1 h4)) (Empty_is_finite Bt) h5) as h6.
  unfold Bt, bt in h6.
  unfold Bt, bt. dependent rewrite -> h6. 
  pose proof (plus_set_empty) as he.
  unfold Bt in he. rewrite he at 1.
  rewrite plus_set_empty'.
  rewrite comm_prod.
  rewrite zero_prod.
  reflexivity.
  (*Add D*)
  intros h6 f.
  pose proof h6 as h7.
  rewrite (plus_set_add' _ h3).
  rewrite dist_sum_r.
  specialize (h4 h3 h6).
  rewrite h4.
  pose (fun y:Bt => d * y) as g.
  pose proof (finite_image _ _ _ g h7) as h8.
  rewrite (dist_set_plus1' _ h6 _ h8).
  pose proof (cart_prod_eq' (Add D d) E) as h9.
  pose proof (feq_im _ _ f h9) as h10.
  pose proof (cart_prod_fin (Add D d) E h1 h6) as h11.
  pose proof (finite_image _ _ _ f h11) as h12.
  rewrite h10 in h12.
  pose proof (subsetT_eq_compat _ _ _ _
 (finite_image (Bt * Bt) (Btype (Bc B)) (cart_prod (Add D d) E) f
        (cart_prod_fin (Add D d) E h1 h6)) h12 h10) as h13.
  dependent rewrite -> h13.
  pose proof (add_ex_family D d 
                            (fun y:Bt => cart_prod (Singleton y) E)) as h14.
  pose proof (f_equal (@FamilyUnion _) h14) as h15.
  pose proof (feq_im _ _ f h15) as h16.
  simpl in h16.
  pose proof (cart_prod_fin D E h3 h6) as h17.
 pose proof h17 as h17'.
  rewrite cart_prod_eq in h17.
  pose proof (family_union_add ( [S : Ensemble (Bt * Bt)
                | exists u : Bt,
                    Ensembles.In D u /\ S = cart_prod (Singleton u) E]) (cart_prod (Singleton d) E)) as h18.
  rewrite h18 in h16.
  rewrite im_union in h16.
  pose proof (cart_prod_sing_fin' E d h7) as h19.
  pose proof (cart_prod_fin D E h3 h6) as h20.
  pose proof (finite_image _ _ _ f h19) as h21.
  rewrite cart_prod_eq' in h20.
  pose proof (finite_image _ _ _ f h20) as h22.

  pose proof (plus_set_union _ _ h21 h22) as h23.
  pose proof  (subsetT_eq_compat _ _ _ _ 
                                 h12
                                (Union_preserves_Finite Bt (Im (cart_prod (Singleton d) E) f)
             (Im
                (FamilyUnion
                   [S : Ensemble (Bt * Bt)
                   | exists x : Bt,
                       Ensembles.In D x /\ S = cart_prod (Singleton x) E]) f) h21 h22) h16) as h24. 
  dependent rewrite -> h24.
  rewrite plus_set_union.
  pose proof (cart_prod_eq' D E) as h25.
  pose proof (feq_im _ _ f h25) as h26.
  pose proof (finite_image _ _ _ f h20) as h27.
 
  pose proof (finite_image _ _ _ f h17') as h28.
  pose proof (subsetT_eq_compat _ _ _ _ h28 h27 h26) as h29.
  assert (h30:h27 = h22). apply proof_irrelevance.
  rewrite h30 in h29.
  dependent rewrite <- h29.
  assert (h31:plus_set (Im E (fun y : Bt => d * y)) h8 =
              plus_set (Im (cart_prod (Singleton d) E) f) h21).
    assert (h32:(Im E (fun y : Bt => d * y)) =
                (Im (cart_prod (Singleton d) E) f)).
      apply Extensionality_Ensembles.
      red. split.
      (* <= *)
      red.
      intros z h31.
      destruct h31 as [z h31]. subst.
      apply Im_intro with (d, z).
      constructor. simpl. split. constructor. assumption.
      unfold f. simpl. reflexivity.
      (* >= *)
      red. intros pr h31.
      destruct h31 as [pr h31]. subst.
      destruct h31 as [h31].
      apply Im_intro with (snd pr). tauto.
      unfold f.
      destruct h31 as [h31l h31r].
      destruct h31l. reflexivity.
    pose proof (subsetT_eq_compat _ _ _ _ h8 h21 h32) as h33.
    dependent rewrite -> h33.
    reflexivity.
  rewrite h31.
  assert (h32:h28 =  (finite_image (Bt * Bt) (Btype (Bc B)) (cart_prod D E)
        (fun p : Bt * Bt => fst p * snd p) (cart_prod_fin D E h3 h6))).  
    apply proof_irrelevance.
  rewrite h32.
  reflexivity.
Qed.

Lemma dist_set_plus2' : 
  forall (D E:Ensemble Bt)
    (pfd: Finite D) (pfe: Finite E),
    let f := (fun p:Bt*Bt => (fst p) * (snd p)) in
    forall (pfc: Finite (Im (cart_prod D E) f)),
      (plus_set D pfd) * (plus_set E pfe) =
      plus_set (Im (cart_prod D E) f) pfc.
intros D E h1 h2 f h3.
pose proof (dist_set_plus2 D E h1 h2) as h4.
assert (h5:h3 = (finite_image (Bt * Bt) (Btype (Bc B)) (cart_prod D E) f
            (cart_prod_fin D E h1 h2))).
  apply proof_irrelevance.
rewrite h5. assumption.
Qed.

Lemma dist_set_times2 : 
  forall (D E:Ensemble Bt)
    (pfd: Finite D) (pfe: Finite E),
    let f := (fun p:Bt*Bt => (fst p) + (snd p)) in
    (times_set D pfd) + (times_set E pfe) =
    times_set (Im (cart_prod D E) f)
             (finite_image _ _ _ f 
                           (cart_prod_fin _ _ pfd pfe)).
intros D E h1.
pose proof h1 as h2.
induction h2 as [|D h3 h4 d h5].
  (*Empty D*)
  pose proof (@cart_prod_empty Bt _ E) as h3.
  intros h4 f.
  pose proof (image_empty _ _ f) as h5.
  rewrite <- h3 in h5.
  pose proof (subsetT_eq_compat _ _ _ _ 
(finite_image (Bt * Bt) (Btype (Bc B)) (cart_prod (Empty_set Bt) E) f
        (cart_prod_fin (Empty_set Bt) E h1 h4)) (Empty_is_finite Bt) h5) as h6.
  dependent rewrite -> h6.
  fold Bt.  
  pose proof times_set_empty as he. 
  unfold Bt in he. unfold Bt. rewrite he at 1.
  rewrite times_set_empty'.
  rewrite comm_sum.
  rewrite one_sum.
  reflexivity.
  (*Add D*)
  intros h6 f.
  pose proof h6 as h7.
  rewrite (times_set_add' _ h3).
  rewrite dist_prod_r.
  specialize (h4 h3 h6).
  rewrite h4.
  pose (fun y:Bt => d + y) as g.
  pose proof (finite_image _ _ _ g h7) as h8.
  rewrite (dist_set_times1' _ h6 _ h8).
  pose proof (cart_prod_eq' (Add D d) E) as h9.
  pose proof (feq_im _ _ f h9) as h10.
  pose proof (cart_prod_fin (Add D d) E h1 h6) as h11.
  pose proof (finite_image _ _ _ f h11) as h12.
  rewrite h10 in h12.
  pose proof (subsetT_eq_compat _ _ _ _
 (finite_image (Bt * Bt) (Btype (Bc B)) (cart_prod (Add D d) E) f
        (cart_prod_fin (Add D d) E h1 h6)) h12 h10) as h13.
  dependent rewrite -> h13.
  pose proof (add_ex_family D d 
                            (fun y:Bt => cart_prod (Singleton y) E)) as h14.
  pose proof (f_equal (@FamilyUnion _) h14) as h15.
  pose proof (feq_im _ _ f h15) as h16.
  simpl in h16.
  pose proof (cart_prod_fin D E h3 h6) as h17.
 pose proof h17 as h17'.
  rewrite cart_prod_eq in h17.
  pose proof (family_union_add ( [S : Ensemble (Bt * Bt)
                | exists u : Bt,
                    Ensembles.In D u /\ S = cart_prod (Singleton u) E]) (cart_prod (Singleton d) E)) as h18.
  rewrite h18 in h16.
  rewrite im_union in h16.
  pose proof (cart_prod_sing_fin' E d h7) as h19.
  pose proof (cart_prod_fin D E h3 h6) as h20.
  pose proof (finite_image _ _ _ f h19) as h21.
  rewrite cart_prod_eq' in h20.
  pose proof (finite_image _ _ _ f h20) as h22.

  pose proof (times_set_union _ _ h21 h22) as h23.
  pose proof  (subsetT_eq_compat _ _ _ _ 
                                 h12
                                (Union_preserves_Finite Bt (Im (cart_prod (Singleton d) E) f)
             (Im
                (FamilyUnion
                   [S : Ensemble (Bt * Bt)
                   | exists x : Bt,
                       Ensembles.In D x /\ S = cart_prod (Singleton x) E]) f) h21 h22) h16) as h24. 
  dependent rewrite -> h24.
  rewrite times_set_union.
  pose proof (cart_prod_eq' D E) as h25.
  pose proof (feq_im _ _ f h25) as h26.
  pose proof (finite_image _ _ _ f h20) as h27.
 
  pose proof (finite_image _ _ _ f h17') as h28.
  pose proof (subsetT_eq_compat _ _ _ _ h28 h27 h26) as h29.
  assert (h30:h27 = h22). apply proof_irrelevance.
  rewrite h30 in h29.
  dependent rewrite <- h29.
  assert (h31:times_set (Im E (fun y : Bt => d + y)) h8 =
              times_set (Im (cart_prod (Singleton d) E) f) h21).
    assert (h32:(Im E (fun y : Bt => d + y)) =
                (Im (cart_prod (Singleton d) E) f)).
      apply Extensionality_Ensembles.
      red. split.
      (* <= *)
      red.
      intros z h31.
      destruct h31 as [z h31]. subst.
      apply Im_intro with (d, z).
      constructor. simpl. split. constructor. assumption.
      unfold f. simpl. reflexivity.
      (* >= *)
      red. intros pr h31.
      destruct h31 as [pr h31]. subst.
      destruct h31 as [h31].
      apply Im_intro with (snd pr). tauto.
      unfold f.
      destruct h31 as [h31l h31r].
      destruct h31l. reflexivity.
    pose proof (subsetT_eq_compat _ _ _ _ h8 h21 h32) as h33.
    dependent rewrite -> h33.
    reflexivity.
  rewrite h31.
  assert (h32:h28 =  (finite_image (Bt * Bt) (Btype (Bc B)) (cart_prod D E)
        (fun p : Bt * Bt => fst p + snd p) (cart_prod_fin D E h3 h6))).  
    apply proof_irrelevance.
  rewrite h32.
  reflexivity.
Qed.

Lemma dist_set_times2' : 
  forall (D E:Ensemble Bt)
    (pfd: Finite D) (pfe: Finite E),
    let f := (fun p:Bt*Bt => (fst p) + (snd p)) in
    forall (pfc: Finite (Im (cart_prod D E) f)),
      (times_set D pfd) + (times_set E pfe) =
      times_set (Im (cart_prod D E) f) pfc.
intros D E h1 h2 f h3.
pose proof (dist_set_times2 D E h1 h2) as h4.
assert (h5:h3 = (finite_image (Bt * Bt) (Btype (Bc B)) (cart_prod D E) f
            (cart_prod_fin D E h1 h2))).
  apply proof_irrelevance.
rewrite h5. assumption.
Qed. 

Definition plus_fin_pair_map1  
           {T U:Type} {C:Ensemble T} {D:Ensemble U} 
           {E:Ensemble Bt} (pfc:Finite C) 
           (F:Fin_map (cart_prod C D) E 0) := 
    fun_to_fin_map C 0 pfc 
                   (fun x:T => plus_set (im1 F x)
                                         (im1_fin F x)).



Definition plus_fin_pair_map2  
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Bt} (pfd:Finite D) 
           (F:Fin_map (cart_prod C D) E 0) := 
    fun_to_fin_map D 0 pfd 
                   (fun y:U => plus_set (im2 F y)
                                         (im2_fin F y)).

Definition times_fin_pair_map1  
           {T U:Type} {C:Ensemble T} {D:Ensemble U} 
           {E:Ensemble Bt} (pfc:Finite C) 
           (F:Fin_map (cart_prod C D) E 0) := 
    fun_to_fin_map C 0 pfc 
                   (fun x:T => times_set (im1 F x)
                                         (im1_fin F x)).

Definition times_fin_pair_map2  
           {T U:Type} {C:Ensemble T} {D:Ensemble U} 
           {E:Ensemble Bt} (pfd:Finite D) 
           (F:Fin_map (cart_prod C D) E 0) := 
    fun_to_fin_map D 0 pfd 
                   (fun y:U => times_set (im2 F y)
                                         (im2_fin F y)).


Lemma plus_fin_pair_map1_list_compat :
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U} 
         {E:Ensemble Bt} (pfc:Finite C) 
         (F:Fin_map (cart_prod C D) E 0)
         (nml:nice_map_lists2 F),
    fin_map_eq (plus_fin_pair_map1 pfc F) 
    (fun_to_fin_map C 0 pfc (fun x:T =>(plus_list (im1l (fpl2 F nml) 0 x)))).
intros T U C D E h1 F nml.
red.
assert (h2:Included (Im C (fun x:T => plus_set (im1 F x) (im1_fin F x)))
            (Im C (fun x:T => plus_list (im1l (fpl2 F nml) 0 x)))).
  red.
  intros x h3.
  destruct h3 as [x h3].
  subst.
  apply Im_intro with x.
  assumption.
  apply plus_set_compat'.
  apply im1_im1l_compat. assumption.
exists h2.
apply fin_map_ext.
intro x.
destruct (classic_dec (Ensembles.In C x)) as [h3 | h4].
rewrite <- (fin_map_new_ran_compat
(plus_fin_pair_map1 h1 F)
     (fin_map_fin_ran
        (fun_to_fin_map C 0 h1
           (fun x0 : T => plus_list (im1l (fpl2 F nml) 0 x0)))) h2); auto.
rewrite fun_to_fin_map_compat; auto.
unfold plus_fin_pair_map1.
rewrite fun_to_fin_map_compat; auto.
apply plus_set_compat'; auto.
apply im1_im1l_compat; auto.
rewrite fin_map_app_def.
rewrite fin_map_app_def.
reflexivity.
assumption. assumption.
Qed.

Lemma plus_fin_pair_map2_list_compat :
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U} {E:Ensemble Bt}
         (pfd:Finite D) 
         (F:Fin_map (cart_prod C D) E 0)
         (nml:nice_map_lists2 F),
    fin_map_eq (plus_fin_pair_map2 pfd F) 
    (fun_to_fin_map D 0 pfd (fun y:U =>(plus_list (im2l (fpl2 F nml) 0 y)))).
intros T U C D E h1 F nml.
red.
assert (h2:Included (Im D (fun y : U => plus_set (im2 F y) (im2_fin F y)))
            (Im D (fun y : U => plus_list (im2l (fpl2 F nml) 0 y)))).
  red.
  intros x h3.
  destruct h3 as [x h3].
  subst.
  apply Im_intro with x.
  assumption.
  apply plus_set_compat'.
  apply im2_im2l_compat. assumption. 
exists h2.
apply fin_map_ext.
intro y.
destruct (classic_dec (Ensembles.In D y)) as [h3 | h4].
rewrite <- (fin_map_new_ran_compat
(plus_fin_pair_map2 h1 F)
     (fin_map_fin_ran
        (fun_to_fin_map D 0 h1
           (fun y:U => plus_list (im2l (fpl2 F nml) 0 y)))) h2); auto.
rewrite fun_to_fin_map_compat; auto.
unfold plus_fin_pair_map2.
rewrite fun_to_fin_map_compat; auto.
apply plus_set_compat'; auto.
apply im2_im2l_compat; auto.
rewrite fin_map_app_def.
rewrite fin_map_app_def.
reflexivity.
assumption. assumption.
Qed.

Lemma times_fin_pair_map1_list_compat :
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U} {E:Ensemble Bt} 
         (pfc:Finite C) 
         (F:Fin_map (cart_prod C D) E 0)
         (nml:nice_map_lists2 F),
    fin_map_eq (times_fin_pair_map1 pfc F) 
    (fun_to_fin_map C 0 pfc (fun x:T =>(times_list (im1l (fpl2 F nml) 0 x)))).
intros T U C D E h1 F nml.
red.
assert (h2:Included (Im C (fun x : T => times_set (im1 F x) (im1_fin F x)))
            (Im C (fun x : T => times_list (im1l (fpl2 F nml) 0 x)))).
  red.
  intros x h3.
  destruct h3 as [x h3].
  subst.
  apply Im_intro with x.
  assumption.
  apply times_set_compat'.
  apply im1_im1l_compat. assumption.
exists h2.
apply fin_map_ext.
intro x.
destruct (classic_dec (Ensembles.In C x)) as [h3 | h4].
rewrite <- (fin_map_new_ran_compat
(times_fin_pair_map1 h1 F)
     (fin_map_fin_ran
        (fun_to_fin_map C 0 h1
           (fun x0 : T => times_list (im1l (fpl2 F nml) 0 x0)))) h2); auto.
rewrite fun_to_fin_map_compat; auto.
unfold times_fin_pair_map1.
rewrite fun_to_fin_map_compat; auto.
apply times_set_compat'; auto.
apply im1_im1l_compat; auto.
rewrite fin_map_app_def.
rewrite fin_map_app_def.
reflexivity.
assumption. assumption.
Qed.

Lemma times_fin_pair_map2_list_compat :
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U} {E:Ensemble Bt} 
         (pfd:Finite D) 
         (F:Fin_map (cart_prod C D) E 0)
         (nml:nice_map_lists2 F),
    fin_map_eq (times_fin_pair_map2 pfd F) 
               (fun_to_fin_map D 0 pfd (fun y:U =>(times_list (im2l (fpl2 F nml) 0 y)))).
intros T U C D E h1 F nml.
red.
assert (h2:Included (Im D (fun y : U => times_set (im2 F y) (im2_fin F y)))
            (Im D (fun y : U => times_list (im2l (fpl2 F nml) 0 y)))).
  red.
  intros x h3.
  destruct h3 as [x h3].
  subst.
  apply Im_intro with x.
  assumption.
  apply times_set_compat'.
  apply im2_im2l_compat. assumption. 
exists h2.
apply fin_map_ext.
intro y.
destruct (classic_dec (Ensembles.In D y)) as [h3 | h4].
rewrite <- (fin_map_new_ran_compat
(times_fin_pair_map2 h1 F)
     (fin_map_fin_ran
        (fun_to_fin_map D 0 h1
           (fun y : U => times_list (im2l (fpl2 F nml) 0 y)))) h2); auto.
rewrite fun_to_fin_map_compat; auto.
unfold times_fin_pair_map2.
rewrite fun_to_fin_map_compat; auto.
apply times_set_compat'; auto.
apply im2_im2l_compat; auto.
rewrite fin_map_app_def.
rewrite fin_map_app_def.
reflexivity.
assumption. assumption.
Qed.




Lemma plus_fin_pair_map2_functional : 
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U} {E:Ensemble Bt}
         (pf:Finite D),
    forall (F G:Fin_map (cart_prod C D) E 0),
      F = G -> fin_map_eq (plus_fin_pair_map2 pf F)
                          (plus_fin_pair_map2 pf G).
intros T U C D E h1 F G h2.
red. rewrite h2.
assert (h3: Included (Im D (fun y : U => plus_set (im2 G y) (im2_fin G y)))
            (Im D (fun y : U => plus_set (im2 G y) (im2_fin G y)))). auto with sets.
exists h3.
apply fin_map_ext.
intros x.
symmetry.
apply fin_map_new_ran_compat.
Qed.



Definition fin_map_times {T:Type} {A:Ensemble T} 
           {C:Ensemble Bt}
           (f:Fin_map A C 0) : Bt.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ A (fin_map_app f) h1) as h2.
refine (times_set _ h2).
Defined.

Lemma fin_map_times_list_compat : 
  forall {T:Type} {A:Ensemble T} {C:Ensemble Bt}
         (F:Fin_map A C 0) (nml:nice_map_lists F), 
    fin_map_times F = times_list (n_im F nml).
intros T A C F nml.
unfold fin_map_times.
pose proof (n_im_im_fin_map_compat F nml) as h1.
apply times_set_compat'.
rewrite h1.
unfold im_fin_map.
reflexivity.
Qed.



Lemma fin_map_times_empty1 : 
  forall {T:Type} {C:Ensemble Bt} (F:Fin_map (Empty_set T) C 0),
    fin_map_times F = 1.
intros T C F.
pose proof (fin_map_empty1 F) as h1.
unfold fin_map_times.
pose proof (image_empty T Bt (fin_map_app F)) as h2.
pose proof (Empty_is_finite Bt) as h3.
pose proof (subsetT_eq_compat _ _ _ _ 
                              (finite_image T Bt (Empty_set T) 
                                            (fin_map_app F) 
                                            (fin_map_fin_dom F))
                              h3
                              h2) as h4.
dependent rewrite -> h4.
apply times_set_empty'.
Qed.

Lemma fin_map_eq_times : forall {T:Type} (A:Ensemble T) 
                                (C E:Ensemble Bt) 
                                (F:Fin_map A C 0) 
                                (G:Fin_map A E 0),
        fin_map_eq F G -> fin_map_times F = fin_map_times G.
intros T A C E F G h1.
destruct h1 as [h1 h2].
pose proof (fin_map_new_ran_compat F (fin_map_fin_ran G) h1) as h3.
rewrite h2 in h3.
unfold fin_map_times.
assert (h4:Im A (fin_map_app F) = Im A (fin_map_app G)).
  assert (h5:fin_map_app F = fin_map_app G).
    apply functional_extensionality.
    apply h3.
  rewrite h5.
  reflexivity.
pose proof (subsetT_eq_compat _ _ _ _
                              (finite_image T Bt A (fin_map_app F) (fin_map_fin_dom F))
                              (finite_image T Bt A (fin_map_app G) (fin_map_fin_dom G)) h4) as h5.
dependent rewrite -> h5.
reflexivity.
Qed.

Lemma im2_empty1 : forall {T U:Type} (D:Ensemble U) (C:Ensemble Bt) 
                          (pfd:Finite D) (pfc:Finite C) 
                          (y:U),
                     im2 (cart_empty_map11 T U Bt 0 D C pfd pfc) y =
                     Empty_set _.
intros T U D C pfd pfc x.
unfold im2.
destruct (eq_dec D (Empty_set _)) as [h1 | h2]; auto.
apply image_empty.
Qed.


Lemma im2_empty2 : forall {T U:Type} (A:Ensemble T) (C:Ensemble Bt) 
                          (pfa:Finite A) (pfc:Finite C) 
                          (y:U),
                     im2 (cart_empty_map21 T U Bt 0 A C pfa pfc) y =
                     Empty_set _.
intros T U A C pfd pfc x.
unfold im2.
destruct (eq_dec (Empty_set _) (Empty_set _)) as [h1 | h2]; auto.
contradict h2. reflexivity.
Qed.


Lemma plus_fin_pair_map2_cart_empty_eq1 : 
  forall {T U:Type} (C:Ensemble Bt) (D:Ensemble U)
         (pfd:Finite D)
         (F:Fin_map (cart_prod (Empty_set T) D) C 0),

    fin_map_eq (plus_fin_pair_map2 pfd F) (fun_to_fin_map D 0 pfd
                                              (fun x => 0)).
intros T U C D h1  F.
apply fps_eq_fin_map_eq.
pose proof h1 as h3.
induction h3. 
pose proof (image_empty _ _  (fun y : U => plus_set (im2 F y) (im2_fin F y))) as h3.
pose proof (image_empty U Bt (fun _ : U => 0)) as h4.
assert (h5:Included (Empty_set Bt) (Empty_set Bt)). auto with sets.
rewrite <- h3 in h5 at 1.
rewrite <- h4 in h5.
assumption. 
red.
intros a h4.
destruct h4 as [a h5 b h6].
apply Im_intro with a. assumption. 
assert (h7:im2 F a = Empty_set _).
  unfold im2.
  destruct (eq_dec (Add A x) (Empty_set _)). auto.
  apply (image_empty _ _ (fun x0 : T => F |-> (x0, a))).
pose proof (Empty_is_finite Bt) as h2'.
pose proof (subsetT_eq_compat _ _ _ _ (im2_fin F a) h2' h7) as h8.
dependent rewrite -> h8 in h6.
rewrite plus_set_empty' in h6.
assumption.
unfold plus_fin_pair_map2.
assert (h3:(fun y:U =>plus_set (im2 F y) (im2_fin F y)) =
           (fun _:U => 0)).
  apply functional_extensionality.
  intro y.
  pose proof (cart_empty_map11_compat F h1) as h3.
  assert (h4:im2 F y = im2 (cart_empty_map11 T U Bt 0 D C h1 (fin_map_fin_ran F)) y). f_equal. assumption.
  rewrite im2_empty1 in h4.
  pose proof (Empty_is_finite Bt) as h2'.
  pose proof (subsetT_eq_compat _ _ _ _ (im2_fin F y) h2' h4) as h5.
  dependent rewrite -> h5.
  apply plus_set_empty'.
rewrite h3.
reflexivity.
Qed.


Lemma plus_fin_pair_map2_cart_empty : 
  forall {T U:Type} (A:Ensemble T)  (C:Ensemble Bt) (pfe:Finite (Empty_set _)) 
         (F:Fin_map (cart_prod A (Empty_set U)) C 0) (y:U), 
    Finite A -> plus_fin_pair_map2 pfe F |-> y = 0.
intros T U A C h1 F y h2.
unfold plus_fin_pair_map2.
rewrite fun_to_fin_map_empty_set1.
apply empty_map1_def.
Qed.

Lemma plus_fin_pair_map2_cart_empty_eq2 : 
  forall {T U:Type} (A:Ensemble T) (C:Ensemble Bt)
         (pfa:Finite A)
         (pfe:Finite (Empty_set U))
         (F:Fin_map (cart_prod A (Empty_set U)) C 0),

    fin_map_eq (plus_fin_pair_map2 pfe F) 
               (empty_map1 U Bt 0 _ (fin_map_fin_ran F)).
intros T U A C h1 h3 F.
apply fps_eq_fin_map_eq.
rewrite image_empty. auto with sets.
pose proof (@fps_to_f_inj).
apply fin_map_to_fps_ext. 
intro x.
rewrite plus_fin_pair_map2_cart_empty.
rewrite empty_map1_def. reflexivity.
assumption.
Qed.                                                                                  


Lemma fin_map_times_sing : forall {T:Type} (A:Ensemble T) (pf:Finite A)
                                  (val:Bt), A <> Empty_set _ ->
                             fin_map_times (fin_map_sing A pf 0 val) = val.
intros T A h1 val h2.
pose proof (im_fin_map_sing A 0 val h1 h2) as h3.
rewrite <- times_set_sing.
unfold fin_map_times.
apply times_set_functional.
assumption.
Qed.


Lemma fin_map_times_cart_empty11 : 
  forall {T U:Type} (C:Ensemble Bt) (D:Ensemble U)
        (pfd:Finite D)
        (pfde:D <> Empty_set _)
         (F:Fin_map (cart_prod (Empty_set T) D) C 0),
    fin_map_times 
                  (plus_fin_pair_map2 pfd F) = 0.
intros T U C D h1 h0 F.
pose proof (fin_map_cart_empty11 F) as h3.
pose proof (plus_fin_pair_map2_cart_empty_eq1 C D h1 F) as h4.
rewrite (fin_map_eq_times _ _ _ _ _ h4).
pose proof (fun_to_fin_map_sing_im  D 0 0 h1 (U:=Bt)) as h5.
pose proof (fin_map_eq_times _ _ _ _ _ h5) as h6.
replace ( fin_map_times (fun_to_fin_map D 0 h1 (fun _ : U => 0)))
        with ( fin_map_times (fin_map_sing D h1 0 0)).
apply fin_map_times_sing. assumption.
Qed.

Lemma fin_map_times_cart_empty21 : 
  forall {T U:Type} (A:Ensemble T) (C:Ensemble Bt) (pfa:Finite A)
         (pfe:Finite (Empty_set U))
         (F:Fin_map (cart_prod A (Empty_set U)) C 0),
    fin_map_times (plus_fin_pair_map2 pfe F) = 1.
intros T U A C h1 h2 F. 
pose proof (plus_fin_pair_map2_cart_empty_eq2 _ _ h1 h2 F) as h3.
rewrite (fin_map_eq_times _ _ _ _ _ h3).
rewrite fin_map_times_empty1.
reflexivity.
Qed.

Lemma fin_map_times_empty2 : 
  forall {T:Type} {A:Ensemble T} (F:Fin_map A (Empty_set Bt) 0),
    fin_map_times F = 1.
intros T A F.
inversion F as [h1 h2 S h3].
induction h1 as [|A h4 h5 x h6].
apply fin_map_times_empty1.
pose proof (add_not_empty A x) as h7.
pose proof (Add_preserves_Finite _ _ x h4) as h8.
pose proof (empty_map_ex2_fin _ 0 h8 h7 F).
contradiction.
Qed.

Definition fin_map_plus {T:Type} {A:Ensemble T} {C:Ensemble Bt}
           (f:Fin_map A C 0) : Bt.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ A (fin_map_app f) h1) as h2.
refine (plus_set _ h2).
Defined.

Lemma fin_map_plus_list_compat : 
  forall {T:Type} {A:Ensemble T} {C:Ensemble Bt}
         (F:Fin_map A C 0) (nml:nice_map_lists F), 
    fin_map_plus F = plus_list (n_im F nml).
intros T A C F nml.
unfold fin_map_times.
pose proof (n_im_im_fin_map_compat F nml) as h1.
apply plus_set_compat'.
rewrite h1.
unfold im_fin_map.
reflexivity.
Qed.


Lemma fin_map_plus_empty1 : 
  forall {T:Type} {C:Ensemble Bt} (F:Fin_map (Empty_set T) C 0),
    fin_map_plus F = 0.
intros T C F.
pose proof (fin_map_empty1 F) as h1.
unfold fin_map_plus.
pose proof (image_empty T Bt (fin_map_app F)) as h2.
pose proof (Empty_is_finite Bt) as h3.
pose proof (subsetT_eq_compat _ _ _ _ 
                              (finite_image T Bt (Empty_set T) 
                                            (fin_map_app F) 
                                            (fin_map_fin_dom F))
                              h3
                              h2) as h4.
dependent rewrite -> h4.
apply plus_set_empty'.
Qed.

Lemma fin_map_plus_empty2 : 
  forall {T:Type} {A:Ensemble T} (F:Fin_map A (Empty_set Bt) 0),
    fin_map_plus F = 0.
intros T A F.
inversion F as [h1 h2 S h3].
induction h1 as [|A h4 h5 x h6].
apply fin_map_plus_empty1.
pose proof (add_not_empty A x) as h7.
pose proof (Add_preserves_Finite _ _ x h4) as h8.
pose proof (empty_map_ex2_fin _ 0 h8 h7 F).
contradiction.
Qed.

Lemma fin_map_eq_plus : forall {T:Type} (A:Ensemble T) 
                               (C E:Ensemble Bt) 
                               (F:Fin_map A C 0) 
                               (G:Fin_map A E 0),
        fin_map_eq F G -> fin_map_plus F = fin_map_plus G.
intros T A C E F G h1.
destruct h1 as [h1 h2].
pose proof (fin_map_new_ran_compat F (fin_map_fin_ran G) h1) as h3.
rewrite h2 in h3.
unfold fin_map_plus.
assert (h4:Im A (fin_map_app F) = Im A (fin_map_app G)).
  assert (h5:fin_map_app F = fin_map_app G).
    apply functional_extensionality.
    apply h3.
  rewrite h5.
  reflexivity.
pose proof (subsetT_eq_compat _ _ _ _
                              (finite_image T Bt A (fin_map_app F) (fin_map_fin_dom F))
                              (finite_image T Bt A (fin_map_app G) (fin_map_fin_dom G)) h4) as h5.
dependent rewrite -> h5.
reflexivity.
Qed.

 
Definition times_plus_fin_pair_map1
           {T U:Type} {C:Ensemble T} {D:Ensemble U} 
           {E:Ensemble Bt} (pfc:Finite C) 
           (F:Fin_map (cart_prod C D) E 0) := 
  fin_map_times (plus_fin_pair_map1 pfc F).

Definition times_plus_fin_pair_map2
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Bt} (pfd:Finite D) 
           (F:Fin_map (cart_prod C D) E 0) := 
  fin_map_times (plus_fin_pair_map2 pfd F).

Definition plus_times_fin_pair_map1  
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Bt} (pfc:Finite C) 
           (F:Fin_map (cart_prod C D) E 0) := 
  fin_map_plus (times_fin_pair_map1 pfc F).

Definition plus_times_fin_pair_map2  
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Bt} (pfd:Finite D) 
           (F:Fin_map (cart_prod C D) E 0) := 
  fin_map_plus (times_fin_pair_map2 pfd F).


Lemma times_plus_fin_pair_map1_list_compat : 
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U} 
         {E:Ensemble Bt} (pfc:Finite C) 
         (F:Fin_map (cart_prod C D) E 0)
         (nml:nice_map_lists (plus_fin_pair_map1 pfc F)),
    times_plus_fin_pair_map1 pfc F = 
    times_list (n_im _ nml).
intros T U C D E pfc F nml.
apply fin_map_times_list_compat.
Qed.


Lemma plus_fun_fin_map_to_fun_comm : 
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (F:Fin_map (cart_prod C D) E 0) 
         (pfc:Finite C) (x:T) (lb:list U),
    list_to_set lb = D -> Ensembles.In C x -> 
    plus_fun (fun y:U => fin_map_to_fun F (x, y)) lb =
    fun_to_fin_map C 0 pfc (fun x:T => plus_set (im1 F x) (im1_fin F x)) |-> x.
intros T U C D E F h1 x lb h0 h2.
unfold plus_fun.
rewrite fun_to_fin_map_compat.
pose proof (plus_set_compat').
rewrite <- (plus_set_compat' _ _ (list_to_set_finite _) (eq_refl _)).
apply plus_set_functional.
 unfold im1.
destruct (eq_dec C (Empty_set _)) as [h3 | h4].
rewrite h3 in h2. contradiction.
apply Extensionality_Ensembles.
red. split.
red.
intros z h5.
rewrite <- list_to_set_in_iff in h5.
unfold Bt in h5.
rewrite in_map_iff in h5.
destruct h5 as [b h5].
destruct h5 as [h5l h5r].
apply Im_intro with b.
rewrite list_to_set_in_iff in h5r.
unfold Bt in h0.
rewrite h0 in h5r.
assumption.
subst.
symmetry.
apply fin_map_to_fun_compat.
red.
intros z h5.
destruct h5 as [b h5 z h6].
rewrite <- list_to_set_in_iff.
unfold Bt.
rewrite in_map_iff.
exists b.
split.
subst. symmetry.
apply fin_map_to_fun_compat.
unfold Bt in h0.
rewrite <- h0 in h5.
unfold Bt in h5.
rewrite <- list_to_set_in_iff in h5.
assumption. assumption.
Qed.

Lemma plus_fun_fin_map_to_fun_comm' : 
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (F:Fin_map (cart_prod C D) E 0) 
         (pfd:Finite D) (y:U) (la:list T),
    list_to_set la = C -> Ensembles.In D y -> 
    plus_fun (fun x:T => fin_map_to_fun F (x, y)) la =
    fun_to_fin_map D 0 pfd (fun x:U => plus_set (im2 F x) (im2_fin F x)) |-> y.
intros T U C D E F h1 y la h0 h2.
unfold plus_fun.
rewrite fun_to_fin_map_compat.
pose proof (plus_set_compat').
rewrite <- (plus_set_compat' _ _ (list_to_set_finite _) (eq_refl _)).
apply plus_set_functional.
 unfold im2.
destruct (eq_dec D (Empty_set _)) as [h3 | h4].
rewrite h3 in h2. contradiction.
apply Extensionality_Ensembles.
red. split.
red. 
intros z h5.
rewrite <- list_to_set_in_iff in h5.
unfold Bt in h5.
rewrite in_map_iff in h5.
destruct h5 as [a h5].
destruct h5 as [h5l h5r].
apply Im_intro with a.
rewrite list_to_set_in_iff in h5r.
unfold Bt in h0.
rewrite h0 in h5r.
assumption.
subst.
symmetry.
apply fin_map_to_fun_compat.
red.
intros z h5.
destruct h5 as [a h5 z h6].
rewrite <- list_to_set_in_iff.
unfold Bt.
rewrite in_map_iff.
exists a.
split.
subst. symmetry.
apply fin_map_to_fun_compat.
unfold Bt in h0.
rewrite <- h0 in h5.
unfold Bt in h5.
rewrite <- list_to_set_in_iff in h5.
assumption. assumption.
Qed.



 
Lemma times_plus_fin_pair_map1_list_compat' :
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (pfc:Finite C) 
    (F:Fin_map (cart_prod C D) E 0)
    (nml2:nice_map_lists2 F)
    (nml:nice_map_lists (plus_fin_pair_map1 pfc F)),
    (n_la2 _ nml2) = (n_la _ nml) ->
    times_list (n_im _ nml) = times_plus_fun1 (n_la2 _ nml2) (n_lb2 _ nml2) (f_no_pr (fin_map_to_fun F)).
intros T U C D E pfc F nml2 nml h0. 
unfold times_plus_fun1.
unfold times_fun.
do 2 rewrite <- (times_set_compat' _ _ (list_to_set_finite _) (eq_refl _)).
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
red.
(* <= *)
intros y h1. 
rewrite <- list_to_set_in_iff in h1.
rewrite <- list_to_set_in_iff.
unfold Bt.
rewrite in_map_iff.
unfold n_im in h1.
unfold Bt in h1. simpl in h1.
pose proof in_map_iff.
pose proof (in_map_iff  (snd (B:=Btype (Bc B))) (n_lp (plus_fin_pair_map1 pfc F) nml) y) as hi.
unfold bt, Bt  in hi, h1.
simpl in hi, h1.
rewrite hi in h1.
destruct h1 as [pr h1].
destruct h1 as [h1l h1r].
exists (fst pr).
split.
unfold f_no_pr. 
rewrite list_to_set_in_iff in h1r.
pose proof (lp_compat _ nml) as h2.
unfold Bt in h2. unfold Bt in h1r.
rewrite h2 in h1r.
pose proof (fin_map_to_fps_fin_map_app_compat _ _ h1r) as h3.
subst.
unfold plus_fin_pair_map1 in h3.
pose proof (fin_map_to_fps_compat (plus_fin_pair_map1 pfc F)) as h4.
destruct h4 as [h4 h5].
destruct h4 as [h4a h4b].
pose proof (h4b _ h1r) as h6.
destruct h6 as [h6l h6r].
rewrite <- h3.
apply plus_fun_fin_map_to_fun_comm.
destruct h6r as [y h6r x h8]. 
pose proof (lab_empty2 _ nml2) as h11.
destruct (eq_dec (n_lab2 _ nml2) nil) as [h9 | h10].
destruct h11 as [h11l | h11r].
destruct h11l as [h11a h11b].
rewrite h11b in h6r. contradiction.
destruct h11r as [h11a h11b].
unfold Bt. unfold Bt in h11a.
rewrite h11a.
rewrite h11b.
simpl. reflexivity.
destruct h11 as [h11l h11r].
assumption. assumption. 
pose proof (fin_map_to_fps_compat (plus_fin_pair_map1 pfc F)) as h4.
destruct h4 as [h4l h4r].
destruct h4l as [h4a h4b].
pose proof (lp_compat _ nml) as h2.
pose proof h1r as h1'.
rewrite list_to_set_in_iff in h1r. 
unfold Bt in h2. unfold Bt in h1r.
rewrite h2 in h1r.
pose proof (h4b _ h1r) as h5.
destruct h5 as [h5l h5r]. 
unfold Bt. unfold Bt in h0.
rewrite h0.
rewrite list_to_set_in_iff.
pose proof (la_compat _ nml) as h6.
unfold Bt in h6.
rewrite h6. assumption.
(* >= *)
red. 
intros y h1.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h1.
unfold Bt in h1.
rewrite in_map_iff in h1.
destruct h1 as [x h1].
destruct h1 as [h1l h1r].
unfold n_im.
unfold Bt, bt.
rewrite in_map_iff at 1.
exists (x, y). simpl. split; auto. 
rewrite list_to_set_in_iff.
pose proof (lp_compat _ nml) as h2.
unfold Bt, bt in h2.
rewrite h2.
unfold Bt. unfold Bt in h0.
rewrite h0 in h1r.
pose proof (la_compat _ nml) as h3.
unfold Bt in h3.
rewrite list_to_set_in_iff in h1r.
rewrite h3 in h1r.
apply fin_map_app_fin_map_to_fps_compat. simpl; auto.
simpl.
rewrite <- h1l.
unfold f_no_pr.
unfold plus_fin_pair_map1.
symmetry.
apply plus_fun_fin_map_to_fun_comm.
pose proof (lab_empty2 _ nml2) as h5.
destruct (eq_dec (n_lab2 F nml2)) as [h6 | h7].
destruct h5 as [h5l | h5r].
destruct h5l as [h5a h5b].
rewrite h5b in h1r. contradiction.
destruct h5r as [h5a h5b].
unfold Bt in h5a.
rewrite h5a. rewrite h5b. simpl. reflexivity.
destruct h5. assumption.
assumption.
Qed.


Lemma times_plus_fin_pair_map2_list_compat : 
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (pfd:Finite D) 
         (F:Fin_map (cart_prod C D) E 0)
         (nml:nice_map_lists (plus_fin_pair_map2 pfd F)),
    times_plus_fin_pair_map2 pfd F = 
    times_list (n_im _ nml).
intros T U C D E pfc F nml.
apply fin_map_times_list_compat.
Qed.

 
Lemma times_plus_fin_pair_map2_list_compat' :
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (pfd:Finite D) 
    (F:Fin_map (cart_prod C D) E 0)
    (nml2:nice_map_lists2 F)
    (nml:nice_map_lists (plus_fin_pair_map2 pfd F)),
    (n_lb2 _ nml2) = (n_la _ nml) ->
    times_list (n_im _ nml) = times_plus_fun2 (n_la2 _ nml2) (n_lb2 _ nml2) (f_no_pr (fin_map_to_fun F)).
intros T U C D E pfd F nml2 nml h0. 
unfold times_plus_fun2.
unfold times_fun.
do 2 rewrite <- (times_set_compat' _ _ (list_to_set_finite _) (eq_refl _)).
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
red.
(* <= *)
intros y h1. 
rewrite <- list_to_set_in_iff in h1.
rewrite <- list_to_set_in_iff.
unfold Bt.
rewrite in_map_iff.
unfold n_im in h1.
unfold Bt, bt in h1.
rewrite in_map_iff in h1.
destruct h1 as [pr h1].
destruct h1 as [h1l h1r].
exists (fst pr).
split.
unfold f_no_pr. 
rewrite list_to_set_in_iff in h1r.
pose proof (lp_compat _ nml) as h2.
unfold Bt, bt in h2. unfold Bt, bt in h1r.
rewrite h2 in h1r.
pose proof (fin_map_to_fps_fin_map_app_compat _ _ h1r) as h3.
subst.
unfold plus_fin_pair_map2 in h3.
pose proof (fin_map_to_fps_compat (plus_fin_pair_map2 pfd F)) as h4.
destruct h4 as [h4 h5].
destruct h4 as [h4a h4b].
pose proof (h4b _ h1r) as h6.
destruct h6 as [h6l h6r].
rewrite <- h3.
apply plus_fun_fin_map_to_fun_comm'.
destruct h6r as [y h6r x h8]. 
pose proof (lab_empty2 _ nml2) as h11.
destruct (eq_dec (n_lab2 _ nml2) nil) as [h9 | h10]. 
destruct h11 as [h11l | h11r].
Focus 2.
destruct h11r as [h11a h11b].
rewrite h11b in h6r. contradiction.
destruct h11l as [h11a h11b].
unfold Bt. unfold Bt in h11a.
rewrite h11a.
rewrite h11b.
simpl. reflexivity.
destruct h11 as [h11l h11r].
assumption. assumption. 
pose proof (fin_map_to_fps_compat (plus_fin_pair_map2 pfd F)) as h4. 
destruct h4 as [h4l h4r].
destruct h4l as [h4a h4b].
pose proof (lp_compat _ nml) as h2.
pose proof h1r as h1'.
rewrite list_to_set_in_iff in h1r. 
unfold Bt, bt in h2. unfold Bt, bt in h1r.
rewrite h2 in h1r.
pose proof (h4b _ h1r) as h5.
destruct h5 as [h5l h5r]. 
unfold Bt. unfold Bt in h0.
rewrite h0.
rewrite list_to_set_in_iff.
pose proof (la_compat _ nml) as h6.
unfold Bt in h6.
rewrite h6. assumption.
(* >= *)
red. 
intros y h1.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h1.
unfold Bt in h1.
rewrite in_map_iff in h1.
destruct h1 as [x h1].
destruct h1 as [h1l h1r].
unfold n_im.
unfold Bt, bt.
rewrite in_map_iff. 
exists (x, y). simpl. split; auto. 
rewrite list_to_set_in_iff.
pose proof (lp_compat _ nml) as h2.
unfold Bt, bt in h2.
rewrite h2.
unfold Bt. unfold Bt in h0.
rewrite h0 in h1r.
pose proof (la_compat _ nml) as h3.
unfold Bt in h3.
rewrite list_to_set_in_iff in h1r.
rewrite h3 in h1r.
apply fin_map_app_fin_map_to_fps_compat. simpl; auto.
simpl.
rewrite <- h1l.
unfold f_no_pr.
unfold plus_fin_pair_map2.
symmetry.
apply plus_fun_fin_map_to_fun_comm'.
pose proof (lab_empty2 _ nml2) as h5.
destruct (eq_dec (n_lab2 F nml2)) as [h6 | h7]. 
destruct h5 as [h5l | h5r]. 
Focus 2.
destruct h5r as [h5a h5b].
rewrite h5b in h1r. contradiction.
destruct h5l as [h5a h5b].
unfold Bt in h5a.
rewrite h5a. rewrite h5b. simpl. reflexivity.
destruct h5. assumption.
assumption.
Qed.


Lemma plus_times_fin_pair_map1_list_compat : 
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (pfc:Finite C) 
    (F:Fin_map (cart_prod C D) E 0)
    (nml:nice_map_lists (times_fin_pair_map1 pfc F)),
    plus_times_fin_pair_map1 pfc F = 
    plus_list (n_im _ nml).
intros T U C D E pfc F nml.
apply fin_map_plus_list_compat.
Qed.

Lemma times_fun_fin_map_to_fun_comm : 
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (F:Fin_map (cart_prod C D) E 0) 
    (pfc:Finite C) (x:T) (lb:list U),
    list_to_set lb = D -> Ensembles.In C x -> 
    times_fun (fun y:U => fin_map_to_fun F (x, y)) lb =
    fun_to_fin_map C 0 pfc (fun x:T => times_set (im1 F x) (im1_fin F x)) |-> x.
intros T U C D E F h1 x lb h0 h2.
unfold times_fun.
rewrite fun_to_fin_map_compat.
pose proof (times_set_compat').
rewrite <- (times_set_compat' _ _ (list_to_set_finite _) (eq_refl _)).
apply times_set_functional.
 unfold im1.
destruct (eq_dec C (Empty_set _)) as [h3 | h4].
rewrite h3 in h2. contradiction.
apply Extensionality_Ensembles.
red. split.
red.
intros z h5.
rewrite <- list_to_set_in_iff in h5.
unfold Bt in h5.
rewrite in_map_iff in h5.
destruct h5 as [b h5].
destruct h5 as [h5l h5r].
apply Im_intro with b.
rewrite list_to_set_in_iff in h5r.
unfold Bt in h0.
rewrite h0 in h5r.
assumption.
subst.
symmetry.
apply fin_map_to_fun_compat.
red.
intros z h5.
destruct h5 as [b h5 z h6].
rewrite <- list_to_set_in_iff.
unfold Bt.
rewrite in_map_iff.
exists b.
split.
subst. symmetry.
apply fin_map_to_fun_compat.
unfold Bt in h0.
rewrite <- h0 in h5.
unfold Bt in h5.
rewrite <- list_to_set_in_iff in h5.
assumption. assumption.
Qed.


Lemma plus_times_fin_pair_map1_list_compat' :
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (pfc:Finite C) 
    (F:Fin_map (cart_prod C D) E 0)
    (nml2:nice_map_lists2 F)
    (nml:nice_map_lists (times_fin_pair_map1 pfc F)),
    (n_la2 _ nml2) = (n_la _ nml) ->
    plus_list (n_im _ nml) = plus_times_fun1 (n_la2 _ nml2) (n_lb2 _ nml2) (f_no_pr (fin_map_to_fun F)).
intros T U C D E pfc F nml2 nml h0. 
unfold plus_times_fun1.
unfold plus_fun.
do 2 rewrite <- (plus_set_compat' _ _ (list_to_set_finite _) (eq_refl _)).
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
red.
(* <= *)
intros y h1. 
rewrite <- list_to_set_in_iff in h1.
rewrite <- list_to_set_in_iff.
unfold Bt.
rewrite in_map_iff.
unfold n_im in h1.
unfold Bt, bt in h1.
rewrite in_map_iff in h1.
destruct h1 as [pr h1].
destruct h1 as [h1l h1r].
exists (fst pr).
split.
unfold f_no_pr. 
rewrite list_to_set_in_iff in h1r.
pose proof (lp_compat _ nml) as h2.
unfold Bt, bt in h2. unfold Bt, bt in h1r.
rewrite h2 in h1r.
pose proof (fin_map_to_fps_fin_map_app_compat _ _ h1r) as h3.
subst.
unfold times_fin_pair_map1 in h3.
pose proof (fin_map_to_fps_compat (times_fin_pair_map1 pfc F)) as h4.
destruct h4 as [h4 h5].
destruct h4 as [h4a h4b].
pose proof (h4b _ h1r) as h6.
destruct h6 as [h6l h6r].
rewrite <- h3.
apply times_fun_fin_map_to_fun_comm.
destruct h6r as [y h6r x h8]. 
pose proof (lab_empty2 _ nml2) as h11.
destruct (eq_dec (n_lab2 _ nml2) nil) as [h9 | h10].
destruct h11 as [h11l | h11r].
destruct h11l as [h11a h11b].
rewrite h11b in h6r. contradiction.
destruct h11r as [h11a h11b].
unfold Bt. unfold Bt in h11a.
rewrite h11a.
rewrite h11b.
simpl. reflexivity.
destruct h11 as [h11l h11r].
assumption. assumption. 
pose proof (fin_map_to_fps_compat (times_fin_pair_map1 pfc F)) as h4.
destruct h4 as [h4l h4r].
destruct h4l as [h4a h4b].
pose proof (lp_compat _ nml) as h2.
pose proof h1r as h1'.
rewrite list_to_set_in_iff in h1r. 
unfold Bt, bt in h2. unfold Bt, bt in h1r.
rewrite h2 in h1r.
pose proof (h4b _ h1r) as h5.
destruct h5 as [h5l h5r]. 
unfold Bt. unfold Bt in h0.
rewrite h0.
rewrite list_to_set_in_iff.
pose proof (la_compat _ nml) as h6.
unfold Bt in h6.
rewrite h6. assumption.
(* >= *)
red. 
intros y h1.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h1.
unfold Bt in h1.
rewrite in_map_iff in h1.
destruct h1 as [x h1].
destruct h1 as [h1l h1r].
unfold n_im.
unfold Bt, bt.
rewrite in_map_iff.
exists (x, y). simpl. split; auto. 
rewrite list_to_set_in_iff.
pose proof (lp_compat _ nml) as h2.
unfold Bt, bt in h2.
rewrite h2.
unfold Bt. unfold Bt in h0.
rewrite h0 in h1r.
pose proof (la_compat _ nml) as h3.
unfold Bt in h3.
rewrite list_to_set_in_iff in h1r.
rewrite h3 in h1r.
apply fin_map_app_fin_map_to_fps_compat. simpl; auto.
simpl.
rewrite <- h1l.
unfold f_no_pr.
unfold times_fin_pair_map1.
symmetry.
apply times_fun_fin_map_to_fun_comm.
pose proof (lab_empty2 _ nml2) as h5.
destruct (eq_dec (n_lab2 F nml2)) as [h6 | h7].
destruct h5 as [h5l | h5r].
destruct h5l as [h5a h5b].
rewrite h5b in h1r. contradiction.
destruct h5r as [h5a h5b].
unfold Bt in h5a.
rewrite h5a. rewrite h5b. simpl. reflexivity.
destruct h5. assumption.
assumption.
Qed.



Lemma plus_times_fin_pair_map2_list_compat : 
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (pfd:Finite D) 
    (F:Fin_map (cart_prod C D) E 0)
    (nml:nice_map_lists (times_fin_pair_map2 pfd F)),
    plus_times_fin_pair_map2 pfd F = 
    plus_list (n_im _ nml).
intros T U C D E pfc F nml.
apply fin_map_plus_list_compat.
Qed.

Lemma times_fun_fin_map_to_fun_comm' : 
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (F:Fin_map (cart_prod C D) E 0) 
    (pfd:Finite D) (y:U) (la:list T),
    list_to_set la = C -> Ensembles.In D y -> 
    times_fun (fun x:T => fin_map_to_fun F (x, y)) la =
    fun_to_fin_map D 0 pfd (fun x:U => times_set (im2 F x) (im2_fin F x)) |-> y.
intros T U C D E F h1 y la h0 h2.
unfold times_fun.
rewrite fun_to_fin_map_compat.
pose proof (times_set_compat').
rewrite <- (times_set_compat' _ _ (list_to_set_finite _) (eq_refl _)).
apply times_set_functional.
 unfold im2.
destruct (eq_dec D (Empty_set _)) as [h3 | h4].
rewrite h3 in h2. contradiction.
apply Extensionality_Ensembles.
red. split.
red. 
intros z h5.
rewrite <- list_to_set_in_iff in h5.
unfold Bt in h5.
rewrite in_map_iff in h5.
destruct h5 as [a h5].
destruct h5 as [h5l h5r].
apply Im_intro with a.
rewrite list_to_set_in_iff in h5r.
unfold Bt in h0.
rewrite h0 in h5r.
assumption.
subst.
symmetry.
apply fin_map_to_fun_compat.
red.
intros z h5.
destruct h5 as [a h5 z h6].
rewrite <- list_to_set_in_iff.
unfold Bt.
rewrite in_map_iff.
exists a.
split.
subst. symmetry.
apply fin_map_to_fun_compat.
unfold Bt in h0.
rewrite <- h0 in h5.
unfold Bt in h5.
rewrite <- list_to_set_in_iff in h5.
assumption. assumption.
Qed.

Lemma plus_times_fin_pair_map2_list_compat' :
  forall 
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Bt} (pfd:Finite D) 
    (F:Fin_map (cart_prod C D) E 0)
    (nml2:nice_map_lists2 F)
    (nml:nice_map_lists (times_fin_pair_map2 pfd F)),
    (n_lb2 _ nml2) = (n_la _ nml) ->
    plus_list (n_im _ nml) = plus_times_fun2 (n_la2 _ nml2) (n_lb2 _ nml2) (f_no_pr (fin_map_to_fun F)).
intros T U C D E pfd F nml2 nml h0. 
unfold plus_times_fun2.
unfold plus_fun.
do 2 rewrite <- (plus_set_compat' _ _ (list_to_set_finite _) (eq_refl _)).
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
red.
(* <= *)
intros y h1. 
rewrite <- list_to_set_in_iff in h1.
rewrite <- list_to_set_in_iff.
unfold Bt.
rewrite in_map_iff.
unfold n_im in h1.
unfold Bt, bt in h1.
rewrite in_map_iff in h1.
destruct h1 as [pr h1].
destruct h1 as [h1l h1r].
exists (fst pr).
split.
unfold f_no_pr. 
rewrite list_to_set_in_iff in h1r.
pose proof (lp_compat _ nml) as h2.
unfold Bt, bt in h2. unfold Bt, bt in h1r.
rewrite h2 in h1r.
pose proof (fin_map_to_fps_fin_map_app_compat _ _ h1r) as h3.
subst.
unfold times_fin_pair_map2 in h3.
pose proof (fin_map_to_fps_compat (times_fin_pair_map2 pfd F)) as h4.
destruct h4 as [h4 h5].
destruct h4 as [h4a h4b].
pose proof (h4b _ h1r) as h6.
destruct h6 as [h6l h6r].
rewrite <- h3.
apply times_fun_fin_map_to_fun_comm'.
destruct h6r as [y h6r x h8]. 
pose proof (lab_empty2 _ nml2) as h11.
destruct (eq_dec (n_lab2 _ nml2) nil) as [h9 | h10]. 
destruct h11 as [h11l | h11r].
Focus 2.
destruct h11r as [h11a h11b].
rewrite h11b in h6r. contradiction.
destruct h11l as [h11a h11b].
unfold Bt. unfold Bt in h11a.
rewrite h11a.
rewrite h11b.
simpl. reflexivity.
destruct h11 as [h11l h11r].
assumption. assumption. 
pose proof (fin_map_to_fps_compat (times_fin_pair_map2 pfd F)) as h4. 
destruct h4 as [h4l h4r].
destruct h4l as [h4a h4b].
pose proof (lp_compat _ nml) as h2.
pose proof h1r as h1'.
rewrite list_to_set_in_iff in h1r. 
unfold Bt, bt in h2. unfold Bt, bt in h1r.
rewrite h2 in h1r.
pose proof (h4b _ h1r) as h5.
destruct h5 as [h5l h5r]. 
unfold Bt. unfold Bt in h0.
rewrite h0.
rewrite list_to_set_in_iff.
pose proof (la_compat _ nml) as h6.
unfold Bt in h6.
rewrite h6. assumption.
(* >= *)
red. 
intros y h1.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h1.
unfold Bt in h1.
rewrite in_map_iff in h1.
destruct h1 as [x h1].
destruct h1 as [h1l h1r].
unfold n_im.
unfold Bt, bt.
rewrite in_map_iff. 
exists (x, y). simpl. split; auto. 
rewrite list_to_set_in_iff.
pose proof (lp_compat _ nml) as h2.
unfold Bt, bt in h2.
rewrite h2.
unfold Bt. unfold Bt in h0.
rewrite h0 in h1r.
pose proof (la_compat _ nml) as h3.
unfold Bt in h3.
rewrite list_to_set_in_iff in h1r.
rewrite h3 in h1r.
apply fin_map_app_fin_map_to_fps_compat. simpl; auto.
simpl.
rewrite <- h1l.
unfold f_no_pr.
unfold times_fin_pair_map2.
symmetry.
apply times_fun_fin_map_to_fun_comm'.
pose proof (lab_empty2 _ nml2) as h5.
destruct (eq_dec (n_lab2 F nml2)) as [h6 | h7]. 
destruct h5 as [h5l | h5r]. 
Focus 2.
destruct h5r as [h5a h5b].
rewrite h5b in h1r. contradiction.
destruct h5l as [h5a h5b].
unfold Bt in h5a.
rewrite h5a. rewrite h5b. simpl. reflexivity.
destruct h5. assumption.
assumption.
Qed.



(*The meet of (p (i, a i)) as i ranges over I*)
Definition times_fun_fin_map1 
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
           (p:(T*U)->Bt) (a:Fin_map I J def) : Bt.
pose proof (fin_map_fin_dom a) as h1.
pose proof (fin_map_fin_ran a) as h2.
pose proof (cart_prod_fin _ _ h1 h2) as h4.
pose (fun_to_fin_map _ 0 h4 p) as P.
refine 
  (fin_map_times 
     (fun_to_fin_map I 0 h1 
                     (fun i:T => (P |-> (i, (a |-> i)))))).
Defined.

(*The meet of (p (a j, j)) as j ranges over J*)
Definition times_fun_fin_map2 
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
           (p:(T*U)->Bt) (a:Fin_map J I def) : Bt.
pose proof (fin_map_fin_dom a) as h1.
pose proof (fin_map_fin_ran a) as h2.
pose proof (cart_prod_fin _ _ h2 h1) as h4.
pose (fun_to_fin_map _ 0 h4 p) as P.
refine 
  (fin_map_times 
     (fun_to_fin_map J 0 h1 
                     (fun j:U => (P |-> (a |-> j, j))))).
Defined.

Definition times_fun_fin_map1_l 
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
           (p:(T*U)->Bt) (a:Fin_map I J def) 
           (nml2:nice_map_lists2 (fun_to_fin_map _ 0 
                              (cart_prod_fin _ _ (fin_map_fin_dom a) (fin_map_fin_ran a)) p))
  :=
times_list (map (fun i:T => (p (i, a |-> i))) (n_la2 _ nml2)).

Definition times_fun_fin_map2_l 
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
           (p:(T*U)->Bt) (a:Fin_map J I def) 
           (nml2:nice_map_lists2 (fun_to_fin_map _ 0 
                              (cart_prod_fin _ _ (fin_map_fin_ran a) (fin_map_fin_dom a)) p))
  :=
times_list (map (fun j:U => (p (a |-> j, j))) (n_lb2 _ nml2)).

Lemma im_sing_times_fun_fin_map1 : 
  forall 
    {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
    (p:T*U->Bt) (a:Fin_map I J def),
    Im (Singleton a)
       (times_fun_fin_map1 p) = 
    Singleton 
      ((times_fun_fin_map1 p) 
         a).                
intros T U I J def p a.
rewrite <- add_empty_sing.
rewrite Im_add.
rewrite image_empty.
rewrite add_empty_sing.
reflexivity.
Qed.

Lemma times_fun_fin_map1_empty : 
  forall {T U:Type} (def:U) (p:(T*U)->Bt),
    times_fun_fin_map1 p (empty_map T U def) = 1.
intros T U def p.
unfold times_fun_fin_map1.
destruct (empty_map Bt Bt 0) as [h1 h2 S h3].
apply fin_map_times_empty1.
Qed.


Lemma times_fun_fin_map2_empty : 
  forall 
    {T U:Type} (def:T)
    (p:(T*U)->Bt),
    times_fun_fin_map2 p (empty_map U T def) = 1.
intros T U def p.
unfold times_fun_fin_map2.
destruct (empty_map Bt Bt 0) as [h1 h2 S h3].
apply fin_map_times_empty1.
Qed.


Lemma times_fun_fin_map1_empty1 : 
  forall 
    {T U:Type} (def:U)
    (p:(T*U)->Bt)
    (J:Ensemble U) (pf:Finite J),
    times_fun_fin_map1 p (empty_map1 T U def J pf) = 1.
intros T U def p J h1.
unfold times_fun_fin_map1.
destruct (empty_map1 T U def J h1) as [h2 h3 S h4].
apply fin_map_times_empty1.
Qed.


Lemma times_fun_fin_map2_empty1 : 
  forall 
    {T U:Type}  (def:T)
    (p:(T*U)->Bt)
    (I:Ensemble T) (pf:Finite I),
    times_fun_fin_map2 p (empty_map1 U T def I pf) = 1.
intros T U def p I h1.
unfold times_fun_fin_map2.
destruct (empty_map1 U T def I h1) as [h2 h3 S h4].
apply fin_map_times_empty1.
Qed.


Lemma times_fun_fin_map1_list_compat : 
  forall 
    {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
    (p:(T*U)->Bt) (a:Fin_map I J def)
    (nml2:nice_map_lists2 (fun_to_fin_map _ 0 
                                          (cart_prod_fin _ _ (fin_map_fin_dom a) (fin_map_fin_ran a)) p)),
    times_fun_fin_map1 p a =
    times_fun_fin_map1_l p a nml2. 
intros T U I J def p a nml2.  
unfold times_fun_fin_map1_l.
pose proof (lab_empty2 _ nml2) as h1.
destruct (eq_dec (n_lab2
               (fun_to_fin_map (cart_prod I J) 0
                  (cart_prod_fin I J (fin_map_fin_dom a) (fin_map_fin_ran a))
                  p) nml2) nil) as [h2 | h3].
destruct h1 as [h4 | h5].
(* h4 *)
destruct h4 as [h4l h4r].
rewrite h4l. simpl.
generalize dependent a.
rewrite h4r.
intros. 
pose proof (fin_map_fin_ran a) as h5.
pose proof (empty_map1_compat a h5) as h6.
rewrite h6.
apply times_fun_fin_map1_empty1.
(* h5 *) 
destruct h5 as [h5l h5r].
generalize dependent a.
rewrite h5r.
intros.
pose proof (fin_map_fin_dom a) as h6.
induction h6 as [|I h7 h8 x h9].
pose proof (empty_map_compat a) as h6.
rewrite h6 at 1.
rewrite times_fun_fin_map1_empty. 
pose proof (lab_empty2' _ nml2) as h7.
destruct (eq_dec 
            (n_lab2
               (fun_to_fin_map (cart_prod (Empty_set T) (Empty_set U)) 0
                  (cart_prod_fin (Empty_set T) (Empty_set U)
                     (fin_map_fin_dom a) (fin_map_fin_ran a)) p) nml2) nil) as [h8 | h9].
destruct h7 as [h7l h7r].
specialize (h7l (eq_refl _)).
rewrite h7l.
simpl. reflexivity.
contradiction.
pose proof (fin_map_fin_dom a) as h10.
pose proof (Add_intro2 _ I x) as h11.
pose proof (Inhabited_intro _ _ _ h11) as h12.
pose proof (Inhabited_not_empty _ _ h12) as h13.
pose proof (empty_map_ex2_fin _ _ h10 h13 a).
contradiction. 
unfold times_fun_fin_map1.
unfold fin_map_times.
pose proof (list_to_set_finite (map (fun i : T => p (i, a |-> i))
        (n_la2
           (fun_to_fin_map (cart_prod I J) 0
              (cart_prod_fin I J (fin_map_fin_dom a) (fin_map_fin_ran a)) p)
           nml2))) as h2.
pose proof (times_set_compat' _ 
     (map (fun i : T => p (i, a |-> i))
        (n_la2
           (fun_to_fin_map (cart_prod I J) 0
              (cart_prod_fin I J (fin_map_fin_dom a) (fin_map_fin_ran a)) p)
           nml2)) h2 (eq_refl _)) as h4.
rewrite <- h4.
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <=*)
red.
intros y h5.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
destruct h5 as [x h5 y h6].
rewrite fun_to_fin_map_compat in h6.
rewrite fun_to_fin_map_compat in h6.
exists x. split.
rewrite h6. reflexivity.
destruct h1 as [h1a h1b].
rewrite list_to_set_in_iff.
rewrite h1a.
assumption.
constructor.
simpl.
split. assumption.
apply fin_map_app_in.
assumption. assumption.
(* >= *)
red.
intros y h5.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5.
destruct h5 as [x h5].
destruct h5 as [h5l h5r].
destruct h1 as [h1l h1r].
rewrite list_to_set_in_iff in h5r.
rewrite h1l in h5r.
apply Im_intro with x. assumption.
rewrite fun_to_fin_map_compat.
rewrite fun_to_fin_map_compat.
rewrite h5l.
reflexivity.
constructor. simpl.
split.
assumption.
apply fin_map_app_in. assumption. assumption.
Qed.

Lemma times_fun_fin_map2_list_compat : 
  forall 
    {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
    (p:(T*U)->Bt) (a:Fin_map J I def)
    (nml2:nice_map_lists2 (fun_to_fin_map _ 0 
                              (cart_prod_fin _ _ (fin_map_fin_ran a) (fin_map_fin_dom a)) p)),
    times_fun_fin_map2 p a =
    times_fun_fin_map2_l p a nml2. 
intros T U I J def p a nml2.  
unfold times_fun_fin_map2_l.
pose proof (lab_empty2 _ nml2) as h1.
destruct (eq_dec (n_lab2
               (fun_to_fin_map (cart_prod I J) 0
                  (cart_prod_fin I J (fin_map_fin_ran a) (fin_map_fin_dom a) )
                  p) nml2) nil) as [h2 | h3]. 
destruct h1 as [h4 | h5].    
(* h5 *)
Focus 2.
destruct h5 as [h5l h5r].
rewrite h5l. simpl.
generalize dependent a.
rewrite h5r.
intros. 
pose proof (fin_map_fin_ran a) as h5.
pose proof (empty_map1_compat a h5) as h6.
rewrite h6.
apply times_fun_fin_map2_empty1.
(* h4 *)  
destruct h4 as [h4l h4r].
generalize dependent a.
rewrite h4r.
intros.
pose proof (fin_map_fin_dom a) as h6.
induction h6 as [|J h7 h8 y h9].
pose proof (empty_map_compat a) as h6.
rewrite h6 at 1.
rewrite times_fun_fin_map2_empty. 
pose proof (lab_empty2' _ nml2) as h7.
destruct (eq_dec 
            (n_lab2
               (fun_to_fin_map (cart_prod (Empty_set T) (Empty_set U)) 0
                  (cart_prod_fin (Empty_set T) (Empty_set U)
                     (fin_map_fin_ran a) (fin_map_fin_dom a)) p) nml2) nil) as [h8 | h9].
destruct h7 as [h7l h7r]. 
specialize (h7r (eq_refl _)).
rewrite h7r.
simpl. reflexivity.
contradiction.
pose proof (fin_map_fin_dom a) as h10.
pose proof (Add_intro2 _ J y) as h11.
pose proof (Inhabited_intro _ _ _ h11) as h12.
pose proof (Inhabited_not_empty _ _ h12) as h13.
pose proof (empty_map_ex2_fin _ _ h10 h13 a).
contradiction. 
unfold times_fun_fin_map2.
unfold fin_map_times.
pose proof (list_to_set_finite (map (fun j : U => p (a |-> j, j))
        (n_lb2
           (fun_to_fin_map (cart_prod I J) 0
              (cart_prod_fin I J (fin_map_fin_ran a) (fin_map_fin_dom a)) p)
           nml2))) as h2.
pose proof (times_set_compat' _ 
     (map (fun j : U => p (a |-> j, j))
        (n_lb2
           (fun_to_fin_map (cart_prod I J) 0
              (cart_prod_fin I J (fin_map_fin_ran a) (fin_map_fin_dom a)) p)
           nml2)) h2 (eq_refl _)) as h4.
rewrite <- h4.
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <=*)
red.
intros x h5.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
destruct h5 as [y h5 x h6].
rewrite fun_to_fin_map_compat in h6.
rewrite fun_to_fin_map_compat in h6.
exists y. split.
rewrite h6. reflexivity.
destruct h1 as [h1a h1b].
rewrite list_to_set_in_iff.
rewrite h1b.
assumption.
constructor.
simpl.
split. 
Focus 2. assumption.
apply fin_map_app_in.
assumption. assumption.
(* >= *)
red.
intros z h5.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5.
destruct h5 as [y h5].
destruct h5 as [h5l h5r].
destruct h1 as [h1l h1r].
rewrite list_to_set_in_iff in h5r.
rewrite h1r in h5r.
apply Im_intro with y. assumption.
rewrite fun_to_fin_map_compat.
rewrite fun_to_fin_map_compat.
rewrite h5l.
reflexivity.
constructor. simpl.
split.
Focus 2.
assumption.
apply fin_map_app_in. assumption. assumption.
Qed.


Definition plus_fun_fin_map1 
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
           (p:(T*U)->Bt) (a:Fin_map I J def) : Bt.
pose proof (fin_map_fin_dom a) as h1.
pose proof (fin_map_fin_ran a) as h2.
pose proof (cart_prod_fin _ _ h1 h2) as h4.
pose (fun_to_fin_map _ 0 h4 p) as P.
refine 
  (fin_map_plus 
     (fun_to_fin_map I 0 h1 
                     (fun i:T => (P |-> (i, (a |-> i)))))).
Defined.

(*The join of (p (a j, j)) as j ranges over J*)
Definition plus_fun_fin_map2 
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
           (p:(T*U)->Bt) (a:Fin_map J I def) : Bt.
pose proof (fin_map_fin_dom a) as h1.
pose proof (fin_map_fin_ran a) as h2.
pose proof (cart_prod_fin _ _ h2 h1) as h4.
pose (fun_to_fin_map _ 0 h4 p) as P.
refine 
  (fin_map_plus
     (fun_to_fin_map J 0 h1 
                     (fun j:U => (P |-> (a |-> j, j))))).
Defined.

Lemma plus_fun_fin_map1_empty : 
  forall {T U:Type} (def:U) (p:(T*U)->Bt),
    plus_fun_fin_map1 p (empty_map T U def) = 0.
intros T U def p.
unfold plus_fun_fin_map1.
destruct (empty_map Bt Bt 0) as [h1 h2 S h3].
apply fin_map_plus_empty1.
Qed.

Lemma plus_fun_fin_map1_empty1 : 
  forall 
    {T U:Type} (def:U) (p:(T*U)->Bt)
  (J:Ensemble U) (pf:Finite J),
    plus_fun_fin_map1 p (empty_map1 T U def J pf) = 0.
intros T U def p J h1.
unfold plus_fun_fin_map1.
destruct (empty_map1 T U def J h1) as [h2 h3 S h4].
apply fin_map_plus_empty1.
Qed.


Definition plus_fun_fin_map1_l 
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
           (p:(T*U)->Bt) (a:Fin_map I J def) 
           (nml2:nice_map_lists2 (fun_to_fin_map _ 0 
                              (cart_prod_fin _ _ (fin_map_fin_dom a) (fin_map_fin_ran a)) p))
  :=
plus_list (map (fun i:T => (p (i, a |-> i))) (n_la2 _ nml2)).

Definition plus_fun_fin_map2_l 
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
           (p:(T*U)->Bt) (a:Fin_map J I def) 
           (nml2:nice_map_lists2 (fun_to_fin_map _ 0 
                              (cart_prod_fin _ _ (fin_map_fin_ran a) (fin_map_fin_dom a)) p))
  :=
plus_list (map (fun j:U => (p (a |-> j, j))) (n_lb2 _ nml2)).


Lemma plus_fun_fin_map2_empty : 
  forall 
    {T U:Type} (def:T) (p:(T*U)->Bt),
    plus_fun_fin_map2 p (empty_map U T def) = 0.
intros T U def p.
unfold plus_fun_fin_map2.
destruct (empty_map Bt Bt 0) as [h1 h2 S h3].
apply fin_map_plus_empty1.
Qed.


Lemma plus_fun_fin_map2_empty1 : 
  forall 
    {T U:Type} (def:T) (p:(T*U)->Bt)
    (I:Ensemble T) (pf:Finite I),
    plus_fun_fin_map2 p (empty_map1 U T def I pf) = 0.
intros T U def p I h1.
unfold plus_fun_fin_map2.
destruct (empty_map1 U T def I h1) as [h2 h3 S h4].
apply fin_map_plus_empty1.
Qed.

Lemma plus_fun_fin_map1_list_compat : 
  forall
    {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
         (p:(T*U)->Bt) (a:Fin_map I J def)
         (nml2:nice_map_lists2 (fun_to_fin_map _ 0 
                              (cart_prod_fin _ _ (fin_map_fin_dom a) (fin_map_fin_ran a)) p)),
    plus_fun_fin_map1 p a =
    plus_fun_fin_map1_l p a nml2. 
intros T U I J def p a nml2.  
unfold plus_fun_fin_map1_l. 
pose proof (lab_empty2 _ nml2) as h1.
destruct (eq_dec (n_lab2
               (fun_to_fin_map (cart_prod I J) 0
                  (cart_prod_fin I J (fin_map_fin_dom a) (fin_map_fin_ran a))
                  p) nml2) nil) as [h2 | h3].
destruct h1 as [h4 | h5]. 
(* h4 *)
destruct h4 as [h4l h4r].
rewrite h4l. simpl.
generalize dependent a.
rewrite h4r.
intros. 
pose proof (fin_map_fin_ran a) as h5.
pose proof (empty_map1_compat a h5) as h6.
rewrite h6.
apply plus_fun_fin_map1_empty1.
(* h5 *) 
destruct h5 as [h5l h5r].
generalize dependent a.
rewrite h5r.
intros.
pose proof (fin_map_fin_dom a) as h6.
induction h6 as [|I h7 h8 x h9].
pose proof (empty_map_compat a) as h6.
rewrite h6 at 1.
rewrite plus_fun_fin_map1_empty. 
pose proof (lab_empty2' _ nml2) as h7.
destruct (eq_dec 
            (n_lab2
               (fun_to_fin_map (cart_prod (Empty_set T) (Empty_set U)) 0
                  (cart_prod_fin (Empty_set T) (Empty_set U)
                     (fin_map_fin_dom a) (fin_map_fin_ran a)) p) nml2) nil) as [h8 | h9].
destruct h7 as [h7l h7r].
specialize (h7l (eq_refl _)).
rewrite h7l.
simpl. reflexivity.
contradiction.
pose proof (fin_map_fin_dom a) as h10.
pose proof (Add_intro2 _ I x) as h11.
pose proof (Inhabited_intro _ _ _ h11) as h12.
pose proof (Inhabited_not_empty _ _ h12) as h13.
pose proof (empty_map_ex2_fin _ _ h10 h13 a).
contradiction. 
unfold plus_fun_fin_map1.
unfold fin_map_plus.
pose proof (list_to_set_finite (map (fun i : T => p (i, a |-> i))
        (n_la2
           (fun_to_fin_map (cart_prod I J) 0
              (cart_prod_fin I J (fin_map_fin_dom a) (fin_map_fin_ran a)) p)
           nml2))) as h2.
pose proof (plus_set_compat' _ 
     (map (fun i : T => p (i, a |-> i))
        (n_la2
           (fun_to_fin_map (cart_prod I J) 0
              (cart_prod_fin I J (fin_map_fin_dom a) (fin_map_fin_ran a)) p)
           nml2)) h2 (eq_refl _)) as h4.
rewrite <- h4.
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <=*)
red.
intros y h5.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
destruct h5 as [x h5 y h6].
rewrite fun_to_fin_map_compat in h6.
rewrite fun_to_fin_map_compat in h6.
exists x. split.
rewrite h6. reflexivity.
destruct h1 as [h1a h1b].
rewrite list_to_set_in_iff.
rewrite h1a.
assumption.
constructor.
simpl.
split. assumption.
apply fin_map_app_in.
assumption. assumption.
(* >= *)
red.
intros y h5.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5.
destruct h5 as [x h5].
destruct h5 as [h5l h5r].
destruct h1 as [h1l h1r].
rewrite list_to_set_in_iff in h5r.
rewrite h1l in h5r.
apply Im_intro with x. assumption.
rewrite fun_to_fin_map_compat.
rewrite fun_to_fin_map_compat.
rewrite h5l.
reflexivity.
constructor. simpl.
split.
assumption.
apply fin_map_app_in. assumption. assumption.
Qed.

Lemma plus_fun_fin_map2_list_compat : 
  forall 
    {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
         (p:(T*U)->Bt) (a:Fin_map J I def)
         (nml2:nice_map_lists2 (fun_to_fin_map _ 0 
                              (cart_prod_fin _ _ (fin_map_fin_ran a) (fin_map_fin_dom a)) p)),
    plus_fun_fin_map2 p a =
    plus_fun_fin_map2_l p a nml2. 
intros T U I J def p a nml2.  
unfold plus_fun_fin_map2_l.
pose proof (lab_empty2 _ nml2) as h1.
destruct (eq_dec (n_lab2
               (fun_to_fin_map (cart_prod I J) 0
                  (cart_prod_fin I J (fin_map_fin_ran a) (fin_map_fin_dom a) )
                  p) nml2) nil) as [h2 | h3]. 
destruct h1 as [h4 | h5].    
(* h5 *)
Focus 2.
destruct h5 as [h5l h5r].
rewrite h5l. simpl.
generalize dependent a.
rewrite h5r.
intros. 
pose proof (fin_map_fin_ran a) as h5.
pose proof (empty_map1_compat a h5) as h6.
rewrite h6.
apply plus_fun_fin_map2_empty1.
(* h4 *)  
destruct h4 as [h4l h4r].
generalize dependent a.
rewrite h4r.
intros.
pose proof (fin_map_fin_dom a) as h6.
induction h6 as [|J h7 h8 y h9].
pose proof (empty_map_compat a) as h6.
rewrite h6 at 1.
rewrite plus_fun_fin_map2_empty. 
pose proof (lab_empty2' _ nml2) as h7.
destruct (eq_dec 
            (n_lab2
               (fun_to_fin_map (cart_prod (Empty_set T) (Empty_set U)) 0
                  (cart_prod_fin (Empty_set T) (Empty_set U)
                     (fin_map_fin_ran a) (fin_map_fin_dom a)) p) nml2) nil) as [h8 | h9].
destruct h7 as [h7l h7r]. 
specialize (h7r (eq_refl _)).
rewrite h7r.
simpl. reflexivity.
contradiction.
pose proof (fin_map_fin_dom a) as h10.
pose proof (Add_intro2 _ J y) as h11.
pose proof (Inhabited_intro _ _ _ h11) as h12.
pose proof (Inhabited_not_empty _ _ h12) as h13.
pose proof (empty_map_ex2_fin _ _ h10 h13 a).
contradiction.  
unfold times_fun_fin_map2.
unfold fin_map_times.
pose proof (list_to_set_finite (map (fun j : U => p (a |-> j, j))
        (n_lb2
           (fun_to_fin_map (cart_prod I J) 0
              (cart_prod_fin I J (fin_map_fin_ran a) (fin_map_fin_dom a)) p)
           nml2))) as h2.
pose proof (plus_set_compat' _ 
     (map (fun j : U => p (a |-> j, j))
        (n_lb2
           (fun_to_fin_map (cart_prod I J) 0
              (cart_prod_fin I J (fin_map_fin_ran a) (fin_map_fin_dom a)) p)
           nml2)) h2 (eq_refl _)) as h4.
rewrite <- h4.
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <=*)
red.
intros x h5.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
destruct h5 as [y h5 x h6].
rewrite fun_to_fin_map_compat in h6.
rewrite fun_to_fin_map_compat in h6.
exists y. split.
rewrite h6. reflexivity.
destruct h1 as [h1a h1b].
rewrite list_to_set_in_iff.
rewrite h1b.
assumption.
constructor.
simpl.
split. 
Focus 2. assumption.
apply fin_map_app_in.
assumption. assumption.
(* >= *)
red.
intros z h5.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5.
destruct h5 as [y h5].
destruct h5 as [h5l h5r].
destruct h1 as [h1l h1r].
rewrite list_to_set_in_iff in h5r.
rewrite h1r in h5r.
apply Im_intro with y. assumption.
rewrite fun_to_fin_map_compat.
rewrite fun_to_fin_map_compat.
rewrite h5l.
reflexivity.
constructor. simpl.
split.
Focus 2.
assumption.
apply fin_map_app_in. assumption. assumption.
Qed.


Definition plus_times_fun_all_maps1
           {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
           (pfi:Finite I) (pfj:Finite J) 
           (p:(T*U)->Bt): Bt

:= let f:= (@times_fun_fin_map1 T U I J def p) in
   let S := Full_set (Fin_map I J def) in
   plus_set (Im S f) (finite_image _ _ S f (finite_fin_maps _ _ def pfi pfj)).

Definition plus_times_fun_all_maps2 
           {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
           (pfi:Finite I) 
           (pfj:Finite J) (p:(T*U)->Bt): Bt

:= let f:= (@times_fun_fin_map2 T U I J def p) in
   let S := Full_set (Fin_map J I def) in
   plus_set (Im S f) (finite_image _ _ S f (finite_fin_maps _ _ def pfj pfi)).


Definition times_plus_fun_all_maps1 
           {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
           (pfi:Finite I) (pfj:Finite J) (p:T*U->Bt) : Bt

:= let f:= (@plus_fun_fin_map1 T U I J def p) in
   let S := Full_set (Fin_map I J def) in
   times_set (Im S f) (finite_image _ _ S f (finite_fin_maps _ _ def pfi pfj)).

Definition times_plus_fun_all_maps2 
           {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
           (pfi:Finite I) (pfj:Finite J) (p:T*U->Bt) : Bt

:= let f:= (@plus_fun_fin_map2 T U I J def p) in
   let S := Full_set (Fin_map J I def) in
   times_set (Im S f) (finite_image _ _ S f (finite_fin_maps _ _ def pfj pfi)).


Lemma complete_dist_list_times1' : 
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj)
         (p:T->U->Bt) (def:U),
      times_plus_fun1 li lj p = plus_times_all_funs1 li lj pfi pfj p def.
intros T U li lj h1 h2 p def. 
pose proof (complete_dist_list_times) as h3.
unfold times_plus_fun1.
unfold times_fun. unfold plus_fun.
unfold times_plus_list_of_lists in h3.
specialize (h3 _ (map (fun i : T => (map (p i) lj)) li)). 
rewrite map_map in h3.  
unfold Bt. unfold Bt in h3.
rewrite h3.
clear h3.
unfold plus_times_list_of_lists. 
unfold plus_times_all_funs1. unfold plus_fun.
pose proof (list_to_set_finite (map (fun l' : list (Btype (Bc B)) => times_list l')
        (list_of_lists_seqs (map (fun i : T => map (p i) lj) li)))) as h3.
pose proof (list_to_set_finite  (map
        (fun l : {x : list (T * U) | In x (list_power li lj)} =>
         times_fun
           (fun i : T =>
            p i
              ((in_list_power_fpl li lj (proj1_sig l) h1
                  (list_power_no_dup li lj (proj1_sig l) h1 h2 (proj2_sig l))
                  (proj2_sig l), def) l-> i)) li)
        (map_sig (list_power li lj)))) as h4.
pose proof (plus_set_compat' _ _ h3 (eq_refl _)) as h5.
pose proof (plus_set_compat' _ _ h4 (eq_refl _)) as h6.
unfold Bt, bt. unfold Bt, bt in h5. unfold Bt, bt in h6.
rewrite <- h5. rewrite <- h6.
clear h5 h6.
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <= *) 
red.
intros b h5.
unfold Bt, bt in h5.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5. 
destruct h5 as [l h5]. 
destruct h5 as [h5l h5r]. 
unfold Bt, bt.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
unfold times_fun.
pose proof (in_list_of_lists_seqs_map_map p _ _ h1 h2 _ def h5r) as h6.  
destruct h6 as [lp h7].
destruct h7 as [h7 h8].
exists (exist _ lp h7).
simpl.
rewrite <- h5l.
split.
f_equal.
symmetry.
apply h8.
apply map_sig_in.
(* >= *)
red.
intros x h5.
unfold Bt, bt. unfold Bt, bt in h5.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5. rewrite in_map_iff.
destruct h5 as [ls h6].
destruct h6 as [h6l h6r].
unfold times_fun in h6l. 
pose proof (in_map_list_of_lists_seqs_map p _ _ ls h1 h2 def h6r) as h7.
destruct h7 as [l [h7a h7b]].
exists l. split.
rewrite <- h6l.
f_equal.
assumption. 
assumption.
Qed.

Lemma complete_dist_list_times2' : 
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj)
         (p:T->U->Bt) (def:T),
      times_plus_fun2 li lj p = plus_times_all_funs2 li lj pfi pfj p def.
intros T U li lj h1 h2 p def. 
pose proof (complete_dist_list_times) as h3. 
unfold times_plus_fun2.
unfold times_fun. unfold plus_fun.
unfold times_plus_list_of_lists in h3.
specialize (h3 _ (map (fun j : U => map (fun i : T => p i j) li) lj)).
rewrite map_map in h3.  
unfold Bt. unfold Bt in h3.
rewrite h3.
clear h3.
unfold plus_times_list_of_lists. 
unfold plus_times_all_funs2. unfold plus_fun.
pose proof (list_to_set_finite (map (fun l' : list (Btype (Bc B)) => times_list l')
        (list_of_lists_seqs 
(map (fun j : U => map (fun i : T => p i j) li) lj)))) as h3.
pose proof (list_to_set_finite (map
        (fun l : {x : list (U * T) | In x (list_power lj li)} =>
         times_fun
           (fun j : U =>
            p
              ((in_list_power_fpl lj li (proj1_sig l) h2
                  (list_power_no_dup lj li (proj1_sig l) h2 h1 (proj2_sig l))
                  (proj2_sig l), def) l-> j) j) lj)
        (map_sig (list_power lj li)))) as h4.
pose proof (plus_set_compat' _ _ h3 (eq_refl _)) as h5.
pose proof (plus_set_compat' _ _ h4 (eq_refl _)) as h6.
unfold Bt, bt. unfold Bt, bt in h5. unfold Bt, bt in h6.
rewrite <- h5. rewrite <- h6.
clear h5 h6.
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <= *) 
red.
intros b h5.
unfold Bt, bt in h5.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5. 
destruct h5 as [l h5]. 
destruct h5 as [h5l h5r]. 
unfold Bt, bt.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
unfold times_fun.
pose proof (in_list_of_lists_seqs_map_map' _ _ _ h1 h2 _ def h5r) as h6.
destruct h6 as [lp h7].
destruct h7 as [h7 h8].
exists (exist _ lp h7).
simpl.
rewrite <- h5l.
split.
f_equal.
symmetry.
apply h8.
apply map_sig_in.
(* >= *)
red.
intros x h5.
unfold Bt, bt. unfold Bt, bt in h5.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5. rewrite in_map_iff.
destruct h5 as [ls h6].
destruct h6 as [h6l h6r].
unfold times_fun in h6l. 
pose proof (in_map_list_of_lists_seqs_map' p _ _ ls h1 h2 def h6r) as h7.
destruct h7 as [l [h7a h7b]].
exists l. split.
rewrite <- h6l.
f_equal.
assumption. 
assumption.
Qed.


Lemma complete_dist_list_plus1' : 
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj)
         (p:T->U->Bt) (def:U),
      plus_times_fun1 li lj p = times_plus_all_funs1 li lj pfi pfj p def.
intros T U li lj h1 h2 p def. 
pose proof (complete_dist_list_plus) as h3.
unfold plus_times_fun1.
unfold plus_fun. unfold times_fun. 
unfold plus_times_list_of_lists in h3.
specialize (h3 _ (map (fun i : T => (map (p i) lj)) li)). 
rewrite map_map in h3.  
unfold Bt. unfold Bt in h3.
rewrite h3.
clear h3.
unfold times_plus_list_of_lists. 
unfold times_plus_all_funs1. unfold times_fun.
pose proof (list_to_set_finite (map (fun l' : list (Btype (Bc B)) => plus_list l')
        (list_of_lists_seqs (map (fun i : T => map (p i) lj) li)))) as h3.
pose proof (list_to_set_finite  (map
        (fun l : {x : list (T * U) | In x (list_power li lj)} =>
         plus_fun
           (fun i : T =>
            p i
              ((in_list_power_fpl li lj (proj1_sig l) h1
                  (list_power_no_dup li lj (proj1_sig l) h1 h2 (proj2_sig l))
                  (proj2_sig l), def) l-> i)) li)
        (map_sig (list_power li lj)))) as h4.
pose proof (times_set_compat' _ _ h3 (eq_refl _)) as h5.
pose proof (times_set_compat' _ _ h4 (eq_refl _)) as h6.
unfold Bt, bt. unfold Bt, bt in h5. unfold Bt, bt in h6.
rewrite <- h5. rewrite <- h6.
clear h5 h6.
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <= *) 
red.
intros b h5.
unfold Bt, bt in h5.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5. 
destruct h5 as [l h5]. 
destruct h5 as [h5l h5r]. 
unfold Bt, bt.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
unfold plus_fun.
pose proof (in_list_of_lists_seqs_map_map p _ _ h1 h2 _ def h5r) as h6.  
destruct h6 as [lp h7].
destruct h7 as [h7 h8].
exists (exist _ lp h7).
simpl.
rewrite <- h5l.
split.
f_equal.
symmetry.
apply h8.
apply map_sig_in.
(* >= *)
red.
intros x h5.
unfold Bt, bt. unfold Bt, bt in h5.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5. rewrite in_map_iff.
destruct h5 as [ls h6].
destruct h6 as [h6l h6r]. 
unfold plus_fun in h6l. 
pose proof (in_map_list_of_lists_seqs_map p _ _ ls h1 h2 def h6r) as h7.
destruct h7 as [l [h7a h7b]].
exists l. split.
rewrite <- h6l.
f_equal.
assumption. 
assumption.
Qed.


Lemma complete_dist_list_plus2' : 
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj)
         (p:T->U->Bt) (def:T),
      plus_times_fun2 li lj p = times_plus_all_funs2 li lj pfi pfj p def.
intros T U li lj h1 h2 p def. 
pose proof (complete_dist_list_plus) as h3. 
unfold plus_times_fun2.
unfold plus_fun. unfold times_fun.
unfold plus_times_list_of_lists in h3.
specialize (h3 _ (map (fun j : U => map (fun i : T => p i j) li) lj)).
rewrite map_map in h3.  
unfold Bt. unfold Bt in h3.
rewrite h3.
clear h3.
unfold times_plus_list_of_lists. 
unfold times_plus_all_funs2. unfold times_fun.
pose proof (list_to_set_finite (map (fun l' : list (Btype (Bc B)) => plus_list l')
        (list_of_lists_seqs 
(map (fun j : U => map (fun i : T => p i j) li) lj)))) as h3.
pose proof (list_to_set_finite (map
        (fun l : {x : list (U * T) | In x (list_power lj li)} =>
         plus_fun
           (fun j : U =>
            p
              ((in_list_power_fpl lj li (proj1_sig l) h2
                  (list_power_no_dup lj li (proj1_sig l) h2 h1 (proj2_sig l))
                  (proj2_sig l), def) l-> j) j) lj)
        (map_sig (list_power lj li)))) as h4.
pose proof (times_set_compat' _ _ h3 (eq_refl _)) as h5.
pose proof (times_set_compat' _ _ h4 (eq_refl _)) as h6.
unfold Bt, bt. unfold Bt, bt in h5. unfold Bt, bt in h6.
rewrite <- h5. rewrite <- h6.
clear h5 h6.
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <= *) 
red.
intros b h5.
unfold Bt, bt in h5.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5. 
destruct h5 as [l h5]. 
destruct h5 as [h5l h5r]. 
unfold Bt, bt.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
unfold plus_fun.
pose proof (in_list_of_lists_seqs_map_map' _ _ _ h1 h2 _ def h5r) as h6.
destruct h6 as [lp h7].
destruct h7 as [h7 h8].
exists (exist _ lp h7).
simpl.
rewrite <- h5l.
split.
f_equal.
symmetry.
apply h8.
apply map_sig_in.
(* >= *)
red.
intros x h5.
unfold Bt, bt. unfold Bt, bt in h5.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h5.
rewrite in_map_iff in h5. rewrite in_map_iff.
destruct h5 as [ls h6].
destruct h6 as [h6l h6r].
unfold plus_fun in h6l. 
pose proof (in_map_list_of_lists_seqs_map' p _ _ ls h1 h2 def h6r) as h7.
destruct h7 as [l [h7a h7b]].
exists l. split.
rewrite <- h6l.
f_equal.
assumption. 
assumption.
Qed.


Lemma plus_times_all_maps1_funs_compat : 
  forall 
    {T U:Type} {I:Ensemble T} {J:Ensemble U} (def:U)
    (pfi:Finite I) (pfj: Finite J) 
    (li:list T) (lj:list U)  
    (pfndpi: NoDup li)
    (pfndpj: NoDup lj) (p:T->U->Bt),
    list_to_set li = I -> list_to_set lj = J ->
    plus_times_fun_all_maps1 _ _ def pfi pfj (f_pr p) =
    plus_times_all_funs1 _ _ pfndpi pfndpj p def.
intros T U I J def h1 h2 li lj h3 h4 p h5 h6.
unfold plus_times_fun_all_maps1.
unfold plus_times_all_funs1.
unfold plus_fun.
pose proof (list_to_set_finite (map
        (fun l : {x : list (T * U) | In x (list_power li lj)} =>
         times_fun
           (fun i : T =>
            p i
              ((in_list_power_fpl li lj (proj1_sig l) h3
                  (list_power_no_dup li lj (proj1_sig l) h3 h4 (proj2_sig l))
                  (proj2_sig l), def) l-> i)) li) (map_sig (list_power li lj)))) as h7.
pose proof (plus_set_compat' _ _ h7 (eq_refl _)) as h8.
unfold Bt.
unfold Bt in h8.
rewrite <- h8.
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros y h9.
destruct h9 as [F h9 x]. subst.
unfold Bt.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
pose proof (fin_map_ex_nice_map_lists_list_to_set F h3 h4) as h10.
destruct h10 as [nml h11].
destruct h11 as [h11l h11r].
pose proof (in_lp _ nml) as h12.
rewrite <- h11l in h12.
rewrite <- h11r in h12.
exists (exist _ (n_lp F nml) h12). 
simpl.
split.  
pose proof (fin_map_ex_nice_map_lists2_list_to_set_dom 
                _ (fun_to_fin_map (cart_prod (list_to_set li) (list_to_set lj)) 0
                               (cart_prod_fin (list_to_set li) (list_to_set lj)
                                              (fin_map_fin_dom F) (fin_map_fin_ran F)) (f_pr p)) h3 h4) as h13.
destruct h13 as [nml2 h13].
destruct h13 as [h13l h13r].
pose proof (times_fun_fin_map1_list_compat _ _ nml2) as h14.
unfold Bt. unfold Bt in h14.
rewrite h14.
unfold times_fun_fin_map1_l.
unfold times_fun.
f_equal.
unfold Bt. unfold Bt in h13l.
rewrite <- h13l.
apply map_ext_in.
intros x h15.
unfold f_pr.
simpl.
f_equal.
apply fpl_f_compat_list_to_set.
apply map_sig_in.
(* >= *)
red. intros x h9.
unfold Bt in h9.
rewrite <- list_to_set_in_iff in h9.
rewrite in_map_iff in h9.
destruct h9 as [lp h9].
destruct h9 as [h9l h9r].
destruct lp as [lp h10].
clear h9r.
simpl in h9l.
pose proof (in_list_power_synced _ _ _ h10) as hs.
pose proof (list_power_no_dup _ _ _ h3 h4 h10) as h11.
pose proof (in_list_power_fpl _ _ _ h3 h11 h10) as h12.
pose proof (fp_fpl_compat _ _ (list_to_set lp) _ _ lp h5 h6 (eq_refl _)) as h13.
unfold Bt in h12.
rewrite <- h13 in h12. 
apply Im_intro with (fin_map_intro _ _ def h1 h2 _ h12).
constructor.   
pose proof (fin_map_ex_nice_map_lists2_dom (fun_to_fin_map (cart_prod I J) 0
           (cart_prod_fin I J
              (fin_map_fin_dom
                 (fin_map_intro I J def h1 h2 (list_to_set lp) h12))
              (fin_map_fin_ran
                 (fin_map_intro I J def h1 h2 (list_to_set lp) h12))) 
           (f_pr p)) _ _ h5 h6 h3 h4) as h14.
destruct h14 as [nml2 h14].
destruct h14 as [h14l h14r].
pose proof (times_fun_fin_map1_list_compat _ _ nml2) as h14.
unfold Bt. unfold Bt in h14.
rewrite h14.
rewrite <- h9l.
unfold times_fun_fin_map1_l.
unfold times_fun.
f_equal.
unfold Bt. unfold Bt in h14l.
rewrite <- h14l.
apply map_ext_in.
intros i h15.
unfold f_pr. simpl.
f_equal.
pose proof (fin_map_ex_nice_map_lists_intro _ _ li lj lp def h1 h2 
                                            h10 h12 h5 h6 h3 h4) as h16. 
destruct h16 as [nml h16]. 
destruct h16 as [h16l [h16r h16f]]. 
pose proof (in_lp _ nml) as h17. 
rewrite <- h16l in h17.
rewrite <- h16r in h17.
rewrite <- h16f in h17.
pose proof (lp_compat _ nml) as h18.
rewrite <- h16f in h18.
pose proof (fpl_f_compat_pseudo_list_to_set _ li lj lp h10 h3 h4 i h5 h6 h18) as h19.
simpl in h19.
assumption.
Qed.

Lemma plus_times_all_maps2_funs_compat : 
  forall 
    {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
         (pfi:Finite I) (pfj: Finite J) 
         (li:list T) (lj:list U)  
         (pfndpi: NoDup li)
         (pfndpj: NoDup lj) (p:T->U->Bt),
    list_to_set li = I -> list_to_set lj = J ->
    plus_times_fun_all_maps2 _ _ def pfi pfj (f_pr p) =
    plus_times_all_funs2 _ _ pfndpi pfndpj p def.
intros T U I J def h1 h2 li lj h3 h4 p h5 h6.
unfold plus_times_fun_all_maps2. 
unfold plus_times_all_funs2. 
unfold plus_fun.
pose proof (list_to_set_finite (map
(fun l : {x : list (U * T) | In x (list_power lj li)} =>
         times_fun
           (fun j : U =>
            p
              ((in_list_power_fpl lj li (proj1_sig l) h4
                  (list_power_no_dup lj li (proj1_sig l) h4 h3 (proj2_sig l))
                  (proj2_sig l), def) l-> j) j) lj)
 (map_sig (list_power lj li)))) as h7.
pose proof (plus_set_compat' _ _ h7 (eq_refl _)) as h8.
unfold Bt.
unfold Bt in h8.
rewrite <- h8.
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <= *) 
red. 
intros x h9. 
destruct h9 as [F h9 y]. subst.
unfold Bt.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
pose proof (fin_map_ex_nice_map_lists_list_to_set F h4 h3) as h10.
destruct h10 as [nml h11].
destruct h11 as [h11l h11r].
pose proof (in_lp _ nml) as h12.
rewrite <- h11l in h12.
rewrite <- h11r in h12.
exists (exist _ (n_lp F nml) h12). 
simpl. 
split.  
pose proof (fin_map_ex_nice_map_lists2_list_to_set_dom 
                _ (fun_to_fin_map (cart_prod (list_to_set li) (list_to_set lj)) 0
                               (cart_prod_fin (list_to_set li) (list_to_set lj)
                                              (fin_map_fin_ran F) (fin_map_fin_dom F)) (f_pr p)) h3 h4) as h13.
destruct h13 as [nml2 h13].
destruct h13 as [h13l h13r].  
(* the below just needs a change or two here or there.*)
pose proof (times_fun_fin_map2_list_compat _ _ nml2) as h14.
unfold Bt. unfold Bt in h14.
rewrite h14.
unfold times_fun_fin_map2_l.
unfold times_fun.
f_equal.
unfold Bt. unfold Bt in h13r.
rewrite <- h13r.
apply map_ext_in.
intros y h15. 
unfold f_pr. 
simpl. 
f_equal.
apply fpl_f_compat_list_to_set.
apply map_sig_in.
(* >= *)
red. intros y h9.
unfold Bt in h9.
rewrite <- list_to_set_in_iff in h9.
rewrite in_map_iff in h9.
destruct h9 as [lp h9]. 
destruct h9 as [h9l h9r].
destruct lp as [lp h10].
clear h9r.
simpl in h9l.
pose proof (in_list_power_synced _ _ _ h10) as hs.
pose proof (list_power_no_dup _ _ _ h4 h3 h10) as h11.
pose proof (in_list_power_fpl _ _ _ h4 h11 h10) as h12.
pose proof (fp_fpl_compat _ _ (list_to_set lp) _ _ lp h6 h5 (eq_refl _)) as h13.
unfold Bt in h12.
rewrite <- h13 in h12. 
apply Im_intro with (fin_map_intro _ _ def h2 h1 _ h12).
constructor.   
pose proof (fin_map_ex_nice_map_lists2_dom (fun_to_fin_map (cart_prod I J) 0
           (cart_prod_fin I J
              (fin_map_fin_ran
                 (fin_map_intro J I def h2 h1 (list_to_set lp) h12))
              (fin_map_fin_dom
                 (fin_map_intro J I def h2 h1 (list_to_set lp) h12))
)
           (f_pr p)) _ _ h5 h6 h3 h4) as h14.
destruct h14 as [nml2 h14].
destruct h14 as [h14l h14r].
pose proof (times_fun_fin_map2_list_compat _ _ nml2) as h14.
unfold Bt. unfold Bt in h14.
rewrite h14.
rewrite <- h9l.
unfold times_fun_fin_map2_l.
unfold times_fun.
f_equal.
unfold Bt. unfold Bt in h14r.
rewrite <- h14r.
apply map_ext_in.
intros i h15.
unfold f_pr. simpl.
f_equal.
pose proof (fin_map_ex_nice_map_lists_intro _ _ lj li lp def h2 h1 
                                            h10 h12 h6 h5 h4 h3) as h16. 
destruct h16 as [nml h16]. 
destruct h16 as [h16l [h16r h16f]]. 
pose proof (in_lp _ nml) as h17. 
rewrite <- h16l in h17.
rewrite <- h16r in h17.
rewrite <- h16f in h17.
pose proof (lp_compat _ nml) as h18.
rewrite <- h16f in h18.
pose proof (fpl_f_compat_pseudo_list_to_set _ lj li lp h10 h4 h3 i h6 h5 h18) as h19.
simpl in h19.
assumption.
Qed.


Lemma times_plus_all_maps1_funs_compat : 
  forall
    {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
         (pfi:Finite I) (pfj: Finite J) 
         (li:list T) (lj:list U)  
         (pfndpi: NoDup li)
         (pfndpj: NoDup lj) (p:T->U->Bt),
    list_to_set li = I -> list_to_set lj = J ->
    times_plus_fun_all_maps1 _ _ def pfi pfj (f_pr p) =
    times_plus_all_funs1 _ _ pfndpi pfndpj p def.
intros T U I J def h1 h2 li lj h3 h4 p h5 h6.
unfold times_plus_fun_all_maps1.
unfold times_plus_all_funs1.
unfold times_fun.
pose proof (list_to_set_finite (map
        (fun l : {x : list (T * U) | In x (list_power li lj)} =>
         plus_fun
           (fun i : T =>
            p i
              ((in_list_power_fpl li lj (proj1_sig l) h3
                  (list_power_no_dup li lj (proj1_sig l) h3 h4 (proj2_sig l))
                  (proj2_sig l), def) l-> i)) li) (map_sig (list_power li lj)))) as h7.
pose proof (times_set_compat' _ _ h7 (eq_refl _)) as h8.
unfold Bt.
unfold Bt in h8.
rewrite <- h8.
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros y h9.
destruct h9 as [F h9 x]. subst.
unfold Bt.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
pose proof (fin_map_ex_nice_map_lists_list_to_set F h3 h4) as h10.
destruct h10 as [nml h11].
destruct h11 as [h11l h11r].
pose proof (in_lp _ nml) as h12.
rewrite <- h11l in h12.
rewrite <- h11r in h12.
exists (exist _ (n_lp F nml) h12). 
simpl.
split.  
pose proof (fin_map_ex_nice_map_lists2_list_to_set_dom 
                _ (fun_to_fin_map (cart_prod (list_to_set li) (list_to_set lj)) 0
                               (cart_prod_fin (list_to_set li) (list_to_set lj)
                                              (fin_map_fin_dom F) (fin_map_fin_ran F)) (f_pr p)) h3 h4) as h13.
destruct h13 as [nml2 h13].
destruct h13 as [h13l h13r].
pose proof (plus_fun_fin_map1_list_compat _ _ nml2) as h14.
unfold Bt. unfold Bt in h14.
rewrite h14.
unfold plus_fun_fin_map1_l.
unfold plus_fun.
f_equal.
unfold Bt. unfold Bt in h13l.
rewrite <- h13l.
apply map_ext_in.
intros x h15.
unfold f_pr.
simpl.
f_equal.
apply fpl_f_compat_list_to_set.
apply map_sig_in.
(* >= *)
red. intros x h9.
unfold Bt in h9.
rewrite <- list_to_set_in_iff in h9.
rewrite in_map_iff in h9.
destruct h9 as [lp h9].
destruct h9 as [h9l h9r].
destruct lp as [lp h10].
clear h9r.
simpl in h9l.
pose proof (in_list_power_synced _ _ _ h10) as hs.
pose proof (list_power_no_dup _ _ _ h3 h4 h10) as h11.
pose proof (in_list_power_fpl _ _ _ h3 h11 h10) as h12.
pose proof (fp_fpl_compat _ _ (list_to_set lp) _ _ lp h5 h6 (eq_refl _)) as h13.
unfold Bt in h12.
rewrite <- h13 in h12. 
apply Im_intro with (fin_map_intro _ _ def h1 h2 _ h12).
constructor.   
pose proof (fin_map_ex_nice_map_lists2_dom (fun_to_fin_map (cart_prod I J) 0
           (cart_prod_fin I J
              (fin_map_fin_dom
                 (fin_map_intro I J def h1 h2 (list_to_set lp) h12))
              (fin_map_fin_ran
                 (fin_map_intro I J def h1 h2 (list_to_set lp) h12))) 
           (f_pr p)) _ _ h5 h6 h3 h4) as h14.
destruct h14 as [nml2 h14].
destruct h14 as [h14l h14r].
pose proof (plus_fun_fin_map1_list_compat _ _ nml2) as h14.
unfold Bt. unfold Bt in h14.
rewrite h14.
rewrite <- h9l.
unfold plus_fun_fin_map1_l.
unfold plus_fun.
f_equal.
unfold Bt. unfold Bt in h14l.
rewrite <- h14l.
apply map_ext_in.
intros i h15.
unfold f_pr. simpl.
f_equal.
pose proof (fin_map_ex_nice_map_lists_intro _ _ li lj lp def h1 h2 
                                            h10 h12 h5 h6 h3 h4) as h16. 
destruct h16 as [nml h16]. 
destruct h16 as [h16l [h16r h16f]]. 
pose proof (in_lp _ nml) as h17. 
rewrite <- h16l in h17.
rewrite <- h16r in h17.
rewrite <- h16f in h17.
pose proof (lp_compat _ nml) as h18.
rewrite <- h16f in h18.
pose proof (fpl_f_compat_pseudo_list_to_set _ li lj lp h10 h3 h4 i h5 h6 h18) as h19.
simpl in h19.
assumption.
Qed.

Lemma times_plus_all_maps2_funs_compat : 
  forall 
    {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
    (pfi:Finite I) (pfj: Finite J) 
    (li:list T) (lj:list U)  
    (pfndpi: NoDup li)
    (pfndpj: NoDup lj) (p:T->U->Bt),
    list_to_set li = I -> list_to_set lj = J ->
    times_plus_fun_all_maps2 _ _ def pfi pfj (f_pr p) =
    times_plus_all_funs2 _ _ pfndpi pfndpj p def.
intros T U I J def h1 h2 li lj h3 h4 p h5 h6.
unfold times_plus_fun_all_maps2. 
unfold times_plus_all_funs2. 
unfold times_fun.
pose proof (list_to_set_finite (map
(fun l : {x : list (U * T) | In x (list_power lj li)} =>
         plus_fun
           (fun j : U =>
            p
              ((in_list_power_fpl lj li (proj1_sig l) h4
                  (list_power_no_dup lj li (proj1_sig l) h4 h3 (proj2_sig l))
                  (proj2_sig l), def) l-> j) j) lj)
 (map_sig (list_power lj li)))) as h7.
pose proof (times_set_compat' _ _ h7 (eq_refl _)) as h8.
unfold Bt.
unfold Bt in h8.
rewrite <- h8.
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
(* <= *) 
red. 
intros x h9. 
destruct h9 as [F h9 y]. subst.
unfold Bt.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
pose proof (fin_map_ex_nice_map_lists_list_to_set F h4 h3) as h10.
destruct h10 as [nml h11].
destruct h11 as [h11l h11r].
pose proof (in_lp _ nml) as h12.
rewrite <- h11l in h12.
rewrite <- h11r in h12.
exists (exist _ (n_lp F nml) h12). 
simpl. 
split.  
pose proof (fin_map_ex_nice_map_lists2_list_to_set_dom 
                _ (fun_to_fin_map (cart_prod (list_to_set li) (list_to_set lj)) 0
                               (cart_prod_fin (list_to_set li) (list_to_set lj)
                                              (fin_map_fin_ran F) (fin_map_fin_dom F)) (f_pr p)) h3 h4) as h13.
destruct h13 as [nml2 h13].
destruct h13 as [h13l h13r].  
pose proof (plus_fun_fin_map2_list_compat _ _ nml2) as h14.
unfold Bt. unfold Bt in h14.
rewrite h14.
unfold plus_fun_fin_map2_l.
unfold plus_fun.
f_equal.
unfold Bt. unfold Bt in h13r.
rewrite <- h13r.
apply map_ext_in.
intros y h15. 
unfold f_pr. 
simpl. 
f_equal.
apply fpl_f_compat_list_to_set.
apply map_sig_in.
(* >= *)
red. intros y h9.
unfold Bt in h9.
rewrite <- list_to_set_in_iff in h9.
rewrite in_map_iff in h9.
destruct h9 as [lp h9]. 
destruct h9 as [h9l h9r].
destruct lp as [lp h10].
clear h9r.
simpl in h9l.
pose proof (in_list_power_synced _ _ _ h10) as hs.
pose proof (list_power_no_dup _ _ _ h4 h3 h10) as h11.
pose proof (in_list_power_fpl _ _ _ h4 h11 h10) as h12.
pose proof (fp_fpl_compat _ _ (list_to_set lp) _ _ lp h6 h5 (eq_refl _)) as h13.
unfold Bt in h12.
rewrite <- h13 in h12. 
apply Im_intro with (fin_map_intro _ _ def h2 h1 _ h12).
constructor.   
pose proof (fin_map_ex_nice_map_lists2_dom (fun_to_fin_map (cart_prod I J) 0
           (cart_prod_fin I J
              (fin_map_fin_ran
                 (fin_map_intro J I def h2 h1 (list_to_set lp) h12))
              (fin_map_fin_dom
                 (fin_map_intro J I def h2 h1 (list_to_set lp) h12))
)
           (f_pr p)) _ _ h5 h6 h3 h4) as h14.
destruct h14 as [nml2 h14].
destruct h14 as [h14l h14r].
pose proof (plus_fun_fin_map2_list_compat _ _ nml2) as h14.
unfold Bt. unfold Bt in h14.
rewrite h14.
rewrite <- h9l.
unfold plus_fun_fin_map2_l.
unfold plus_fun.
f_equal.
unfold Bt. unfold Bt in h14r.
rewrite <- h14r.
apply map_ext_in.
intros i h15.
unfold f_pr. simpl.
f_equal.
pose proof (fin_map_ex_nice_map_lists_intro _ _ lj li lp def h2 h1 
                                            h10 h12 h6 h5 h4 h3) as h16. 
destruct h16 as [nml h16]. 
destruct h16 as [h16l [h16r h16f]]. 
pose proof (in_lp _ nml) as h17. 
rewrite <- h16l in h17.
rewrite <- h16r in h17.
rewrite <- h16f in h17.
pose proof (lp_compat _ nml) as h18.
rewrite <- h16f in h18.
pose proof (fpl_f_compat_pseudo_list_to_set _ lj li lp h10 h4 h3 i h6 h5 h18) as h19.
simpl in h19.
assumption.
Qed.



Lemma times_plus_fun1_fin_map_to_fun_undoes_fun_to_fin_map :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U)  
         (li:list T) (lj:list U) (pf:Finite (cart_prod I J))
         (p:T*U->Bt),
    list_to_set li = I -> list_to_set lj = J ->
    times_plus_fun1 li lj 
                    (f_no_pr (fin_map_to_fun (fun_to_fin_map (cart_prod I J) 0 pf p))) = times_plus_fun1 li lj (f_no_pr p).
intros T U I J li lj pf p h1 h2.
unfold times_plus_fun1. unfold times_fun.
f_equal.
apply map_ext_in.
intros x h3.  
unfold f_no_pr.  simpl. unfold plus_fun. 
f_equal. 
apply map_ext_in.
intros j h4.
apply fin_map_to_fun_undoes_fun_to_fin_map.
constructor. simpl. split.
rewrite <- h1. rewrite <- list_to_set_in_iff. assumption.
rewrite <- h2. rewrite <- list_to_set_in_iff. assumption.
Qed.

Lemma times_plus_fun2_fin_map_to_fun_undoes_fun_to_fin_map :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) 
         (li:list T) (lj:list U) (pf:Finite (cart_prod I J))
         (p:T*U->Bt),
    list_to_set li = I -> list_to_set lj = J ->
    times_plus_fun2 li lj 
                    (f_no_pr (fin_map_to_fun (fun_to_fin_map (cart_prod I J) 0 pf p))) = times_plus_fun2 li lj (f_no_pr p).
intros T U I J li lj pf p h1 h2.
unfold times_plus_fun2. unfold times_fun.
f_equal.
apply map_ext_in.
intros x h3.  
unfold f_no_pr.  simpl. unfold plus_fun. 
f_equal. 
apply map_ext_in.
intros j h4.
apply fin_map_to_fun_undoes_fun_to_fin_map.
constructor. simpl. split.
rewrite <- h1. rewrite <- list_to_set_in_iff. assumption.
rewrite <- h2. rewrite <- list_to_set_in_iff. assumption.
Qed.

Lemma plus_times_fun1_fin_map_to_fun_undoes_fun_to_fin_map :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) 
         (li:list T) (lj:list U) (pf:Finite (cart_prod I J))
         (p:T*U->Bt),
    list_to_set li = I -> list_to_set lj = J ->
    plus_times_fun1 li lj 
                    (f_no_pr (fin_map_to_fun (fun_to_fin_map (cart_prod I J) 0 pf p))) = plus_times_fun1 li lj (f_no_pr p).
intros T U I J li lj pf p h1 h2.
unfold plus_times_fun1. unfold plus_fun.
f_equal.
apply map_ext_in.
intros x h3.  
unfold f_no_pr.  simpl. unfold times_fun. 
f_equal. 
apply map_ext_in.
intros j h4.
apply fin_map_to_fun_undoes_fun_to_fin_map.
constructor. simpl. split.
rewrite <- h1. rewrite <- list_to_set_in_iff. assumption.
rewrite <- h2. rewrite <- list_to_set_in_iff. assumption.
Qed.

Lemma plus_times_fun2_fin_map_to_fun_undoes_fun_to_fin_map :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) 
         (li:list T)
         (lj:list U) (pf:Finite (cart_prod I J))
         (p:T*U->Bt),
    list_to_set li = I -> list_to_set lj = J ->
    plus_times_fun2 li lj 
                    (f_no_pr (fin_map_to_fun (fun_to_fin_map (cart_prod I J) 0 pf p))) = plus_times_fun2 li lj (f_no_pr p).
intros T U I J li lj pf p h1 h2.
unfold plus_times_fun2. unfold plus_fun.
f_equal.
apply map_ext_in.
intros x h3.  
unfold f_no_pr.  simpl. unfold times_fun. 
f_equal. 
apply map_ext_in.
intros j h4.
apply fin_map_to_fun_undoes_fun_to_fin_map.
constructor. simpl. split.
rewrite <- h1. rewrite <- list_to_set_in_iff. assumption.
rewrite <- h2. rewrite <- list_to_set_in_iff. assumption.
Qed.


Lemma complete_dist_times_plus1 : 
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
    (pfi:Finite I) (pfj:Finite J) (p:T*U->Bt),
    times_plus_fin_pair_map1 
      pfi (fun_to_fin_map _ 0 (cart_prod_fin _ _ pfi pfj) p) =
    plus_times_fun_all_maps1 _ _ def pfi pfj p.
intros T U I J def h1 h2 p.
pose proof fin_map_ex_nice_map_lists2 (fun_to_fin_map (cart_prod I J) 0 (cart_prod_fin I J h1 h2) p) as h3. 
destruct h3 as [nml2 h4]. clear h4.
pose proof (la_lb_compat2 _ nml2) as h3.
specialize (h3 h1 h2).
destruct h3 as [h3l h3r].
pose proof (nda2 _ nml2) as h4.
pose proof (ndb2 _ nml2) as h5.
rewrite (f_pr_f_no_pr p).
rewrite (plus_times_all_maps1_funs_compat def h1 h2 _ _ h4 h5 (f_no_pr p) h3l h3r).  
rewrite <- complete_dist_list_times1'.  
pose proof (fin_map_ex_nice_map_lists_list_to_set_dom (plus_fin_pair_map1 h1 (fun_to_fin_map (cart_prod I J) 0 (cart_prod_fin I J h1 h2) p)) _  h3l h4) as h6.
destruct h6 as [nml h6].
pose proof (times_plus_fin_pair_map1_list_compat'  _  _ nml2 nml h6) as h7. 
rewrite times_plus_fun1_fin_map_to_fun_undoes_fun_to_fin_map in h7.
unfold Bt, bt. unfold Bt, bt in h7.
rewrite <- h7. 
unfold Bt. unfold Bt in nml. unfold Bt in h2.
pose proof (times_plus_fin_pair_map1_list_compat _ _ nml) as h8.
unfold Bt, bt in h8.
rewrite <- h8.
unfold Bt, bt.
pose proof (f_pr_f_no_pr p) as h9.
unfold Bt, bt in h9.
rewrite <- h9.
reflexivity. assumption. assumption.
Qed. 


Lemma complete_dist_times_plus2 : 
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
    (pfi:Finite I) (pfj:Finite J)
    (p:T*U->Bt),
    times_plus_fin_pair_map2 
      pfj (fun_to_fin_map _ 0 (cart_prod_fin _ _ pfi pfj) p) =
    plus_times_fun_all_maps2 _ _ def pfi pfj p.
intros T U I J def h1 h2 p.
pose proof fin_map_ex_nice_map_lists2 (fun_to_fin_map (cart_prod I J ) 0 (cart_prod_fin I J h1 h2) p) as h3. 
destruct h3 as [nml2 h4]. clear h4.
pose proof (la_lb_compat2 _ nml2) as h3.
specialize (h3 h1 h2).
destruct h3 as [h3l h3r].
pose proof (nda2 _ nml2) as h4.
pose proof (ndb2 _ nml2) as h5.
rewrite (f_pr_f_no_pr p).
rewrite (plus_times_all_maps2_funs_compat _ _ def h1 h2 _ _ h4 h5 (f_no_pr p) h3l h3r).
rewrite <- complete_dist_list_times2'.  
pose proof (fin_map_ex_nice_map_lists_list_to_set_dom (plus_fin_pair_map2 h2 (fun_to_fin_map (cart_prod I J) 0 (cart_prod_fin I J h1 h2) p)) _  h3r h5) as h6.
destruct h6 as [nml h6].
pose proof (times_plus_fin_pair_map2_list_compat'  _  _ nml2 nml h6) as h7. 
rewrite times_plus_fun2_fin_map_to_fun_undoes_fun_to_fin_map in h7.
unfold Bt, bt. unfold Bt, bt in h7.
rewrite <- h7. 
unfold Bt. unfold Bt in nml. unfold Bt in h2.
pose proof (times_plus_fin_pair_map2_list_compat _ _ nml) as h8.
unfold Bt, bt in h8.
rewrite <- h8.
unfold Bt.
pose proof (f_pr_f_no_pr p) as h9.
unfold Bt, bt in h9.
rewrite <- h9.
reflexivity. assumption. assumption.
Qed. 

Lemma complete_dist_plus_times1 : 
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
    (pfi:Finite I) (pfj:Finite J)
    (p:T*U->Bt),
    plus_times_fin_pair_map1 
      pfi (fun_to_fin_map _ 0 (cart_prod_fin _ _ pfi pfj) p) =
    times_plus_fun_all_maps1 _ _ def pfi pfj p.
intros T U  I J def h1 h2 p.
pose proof fin_map_ex_nice_map_lists2 (fun_to_fin_map (cart_prod I J) 0 (cart_prod_fin I J h1 h2) p) as h3. 
destruct h3 as [nml2 h4]. clear h4.
pose proof (la_lb_compat2 _ nml2) as h3.
specialize (h3 h1 h2).
destruct h3 as [h3l h3r].
pose proof (nda2 _ nml2) as h4.
pose proof (ndb2 _ nml2) as h5.
rewrite (f_pr_f_no_pr p).
rewrite (times_plus_all_maps1_funs_compat _ _ def h1 h2 _ _ h4 h5 (f_no_pr p) h3l h3r).  
rewrite <- complete_dist_list_plus1'.  
pose proof (fin_map_ex_nice_map_lists_list_to_set_dom (times_fin_pair_map1 h1 (fun_to_fin_map (cart_prod I J) 0 (cart_prod_fin I J h1 h2) p)) _  h3l h4) as h6.
destruct h6 as [nml h6].
pose proof (plus_times_fin_pair_map1_list_compat'  _  _ nml2 nml h6) as h7. 
rewrite plus_times_fun1_fin_map_to_fun_undoes_fun_to_fin_map in h7.
unfold Bt, bt. unfold Bt, bt in h7.
rewrite <- h7. 
unfold Bt. unfold Bt in nml. unfold Bt in h2.
pose proof (plus_times_fin_pair_map1_list_compat _ _ nml) as h8.
unfold Bt, bt in h8.
rewrite <- h8.
unfold Bt.
pose proof (f_pr_f_no_pr p) as h9.
unfold Bt, bt in h9.
rewrite <- h9.
reflexivity. assumption. assumption.
Qed. 


Lemma complete_dist_plus_times2 : 
  forall {T U:Type}  (I:Ensemble T) (J:Ensemble U) (def:T)
         (pfi:Finite I) (pfj:Finite J)
         (p:T*U->Bt),
    plus_times_fin_pair_map2 
      pfj (fun_to_fin_map _ 0 (cart_prod_fin _ _ pfi pfj) p) =
    times_plus_fun_all_maps2 _ _ def pfi pfj p.
intros T U I J def h1 h2 p.
pose proof fin_map_ex_nice_map_lists2 (fun_to_fin_map (cart_prod I J ) 0 (cart_prod_fin I J h1 h2) p) as h3. 
destruct h3 as [nml2 h4]. clear h4.
pose proof (la_lb_compat2 _ nml2) as h3.
specialize (h3 h1 h2).
destruct h3 as [h3l h3r].
pose proof (nda2 _ nml2) as h4.
pose proof (ndb2 _ nml2) as h5.
rewrite (f_pr_f_no_pr p).
rewrite (times_plus_all_maps2_funs_compat _ _ def h1 h2 _ _ h4 h5 (f_no_pr p) h3l h3r).
rewrite <- complete_dist_list_plus2'.  
pose proof (fin_map_ex_nice_map_lists_list_to_set_dom (times_fin_pair_map2 h2 (fun_to_fin_map (cart_prod I J) 0 (cart_prod_fin I J h1 h2) p)) _  h3r h5) as h6.
destruct h6 as [nml h6].
pose proof (plus_times_fin_pair_map2_list_compat'  _  _ nml2 nml h6) as h7. 
rewrite plus_times_fun2_fin_map_to_fun_undoes_fun_to_fin_map in h7.
unfold Bt, bt. unfold Bt, bt in h7.
rewrite <- h7. 
unfold Bt. unfold Bt in nml. unfold Bt in h2.
pose proof (plus_times_fin_pair_map2_list_compat _ _ nml) as h8.
unfold Bt, bt in h8.
rewrite <- h8. 
unfold Bt.
pose proof (f_pr_f_no_pr p) as h9.
unfold Bt, bt in h9.
rewrite <- h9.
reflexivity. assumption. assumption.
Qed. 
End SetOperations.

Arguments prod_list_dup_eq [B] _ _ _.
Arguments sum_list_dup_eq [B] _ _ _.
Arguments prod_preserves_list_singularize [B] _.
Arguments sum_preserves_list_singularize [B] _.
Arguments times_sing_preserves_new_head [B] _ _ _.
Arguments plus_sing_preserves_new_head [B] _ _ _.
Arguments times_list_sing_cons [B] _ _ _.
Arguments plus_list_sing_cons [B] _ _ _.
Arguments list_to_sets_eq_times_sing_eq [B] _ _ _ _ _.
Arguments list_to_sets_eq_plus_sing_eq [B] _ _ _ _ _.
Arguments list_to_sets_eq_times_eq [B] _ _ _ _ _.
Arguments list_to_sets_eq_plus_eq [B] _ _ _ _ _.
Arguments times_list_unq [B] _ _.
Arguments plus_list_unq [B] _ _.
Arguments times_set [B] _ _.
Arguments plus_set [B] _ _.
Arguments times_set_compat [B] _ _.
Arguments times_set_compat' [B] _ _ _ _.
Arguments plus_set_compat [B] _ _.
Arguments plus_set_compat' [B] _ _ _ _.
Arguments times_set_functional [B] _ _ _ _ _.
Arguments plus_set_functional [B] _ _ _ _ _.
Arguments times_set_empty [B].
Arguments times_set_empty' [B] _.
Arguments plus_set_empty [B].
Arguments plus_set_empty' [B] _.
Arguments times_set_add [B] _ _ _.
Arguments times_set_add' [B] _ _ _ _.
Arguments plus_set_add [B] _ _ _.
Arguments plus_set_add' [B] _ _ _ _.
Arguments times_set_sing [B] _.
Arguments times_set_sing' [B] _ _.
Arguments times_set_one_or [B] _ _ _.
Arguments plus_set_sing [B] _.
Arguments plus_set_sing' [B] _ _.
Arguments plus_set_zero_or [B] _ _ _.
Arguments times_set_couple [B] _ _.
Arguments times_set_couple' [B] _ _ _.
Arguments plus_set_couple [B] _ _.
Arguments plus_set_couple' [B] _ _ _.
Arguments le_times_set [B] _ _ _ _.
Arguments le_plus_set [B] _ _ _ _.
Arguments inf_times_set [B] _ _.
Arguments sup_plus_set [B] _ _.
Arguments times_set_union [B] _ _ _ _.
Arguments times_set_union' [B] _ _ _ _ _.
Arguments times_set_inc_le [B] _ _ _ _ _.
Arguments plus_set_union [B] _ _ _ _.
Arguments plus_set_union' [B] _ _ _ _ _.
Arguments plus_set_inc_le [B] _ _ _ _ _.
Arguments plus_set_im_add [B] _ _ _ _.
Arguments plus_set_im_add' [B] _ _ _ _ _ _.
Arguments times_set_im_add [B] _ _ _ _.
Arguments times_set_im_add' [B] _ _ _ _ _ _.
Arguments dist_set_plus1 [B] _ _ _.
Arguments dist_set_plus1' [B] _ _ _ _.
Arguments dist_set_times1 [B] _ _ _.
Arguments dist_set_times1' [B] _ _ _ _.
Arguments dist_set_plus2 [B] _ _ _ _.
Arguments dist_set_plus2' [B] _ _ _ _ _.
Arguments dist_set_times2 [B] _ _ _ _.
Arguments dist_set_times2' [B] _ _ _ _ _.
Arguments plus_fin_pair_map1 [B] [T] [U] [C] [D] [E] _ _.
Arguments plus_fin_pair_map2 [B] [T] [U] [C] [D] [E] _ _.
Arguments times_fin_pair_map1 [B] [T] [U] [C] [D] [E] _ _.
Arguments times_fin_pair_map2 [B] [T] [U] [C] [D] [E] _ _.
Arguments plus_fin_pair_map1_list_compat [B] [T] [U] [C] [D] [E] _ _ _.
Arguments plus_fin_pair_map2_list_compat [B] [T] [U] [C] [D] [E] _ _ _.
Arguments times_fin_pair_map1_list_compat [B] [T] [U] [C] [D] [E] _ _ _.
Arguments times_fin_pair_map2_list_compat [B] [T] [U] [C] [D] [E] _ _ _.
Arguments plus_fin_pair_map2_functional [B] [T] [U] [C] [D] [E] _ _ _ _.
Arguments fin_map_times [B] [T] [A] [C] _.
Arguments fin_map_times_list_compat [B] [T] [A] [C] _ _.
Arguments fin_map_times_empty1 [B] [T] [C] _.
Arguments fin_map_eq_times [B] [T] _ _ _ _ _ _.
Arguments im2_empty1 [B] [T] [U] _ _ _ _ _.
Arguments im2_empty2 [B] [T] [U] _ _ _ _ _.
Arguments plus_fin_pair_map2_cart_empty_eq1 [B] [T] [U] _ _ _ _.
Arguments plus_fin_pair_map2_cart_empty [B] [T] [U] _ _ _ _ _ _.
Arguments plus_fin_pair_map2_cart_empty_eq2 [B] [T] [U] _ _ _ _ _.
Arguments fin_map_times_sing [B] [T] _ _ _ _.
Arguments fin_map_times_cart_empty11 [B] [T] [U] _ _ _ _ _.
Arguments fin_map_times_cart_empty21 [B] [T] [U] _ _ _ _ _.
Arguments fin_map_times_empty2 [B] [T] _ _.
Arguments fin_map_plus [B] [T] [A] [C] _.
Arguments fin_map_plus_list_compat [B] [T] [A] [C] _ _.
Arguments fin_map_plus_empty1 [B] [T] [C] _.
Arguments fin_map_plus_empty2 [B] [T] [A] _.
Arguments fin_map_eq_plus [B] [T] _ _ _ _ _ _.
Arguments times_plus_fin_pair_map1 [B] [T] [U] [C] [D] [E] _ _.
Arguments times_plus_fin_pair_map2 [B] [T] [U] [C] [D] [E] _ _.
Arguments times_plus_fin_pair_map2 [B] [T] [U] [C] [D] [E] _ _.
Arguments plus_times_fin_pair_map1 [B] [T] [U] [C] [D] [E] _ _.
Arguments plus_times_fin_pair_map2 [B] [T] [U] [C] [D] [E] _ _.
Arguments times_plus_fin_pair_map1_list_compat [B] [T] [U] [C] [D] [E] _ _ _.
Arguments plus_fun_fin_map_to_fun_comm [B] [T] [U] [C] [D] [E] _ _ _ _ _ _.
Arguments plus_fun_fin_map_to_fun_comm' [B] [T] [U] [C] [D] [E] _ _ _ _ _ _.
Arguments times_plus_fin_pair_map1_list_compat' [B] [T] [U] [C] [D] [E] _ _ _ _ _.
Arguments times_plus_fin_pair_map2_list_compat [B] [T] [U] [C] [D] [E] _ _ _.
Arguments times_plus_fin_pair_map2_list_compat' [B] [T] [U] [C] [D] [E] _ _ _ _ _ .
Arguments plus_times_fin_pair_map1_list_compat [B] [T] [U] [C] [D] [E] _ _ _.
Arguments times_fun_fin_map_to_fun_comm [B] [T] [U] [C] [D] [E] _ _ _ _ _ _.
Arguments plus_times_fin_pair_map1_list_compat' [B] [T] [U] [C] [D] [E] _ _ _ _ _.
Arguments plus_times_fin_pair_map2_list_compat [B] [T] [U] [C] [D] [E] _ _ _.
Arguments times_fun_fin_map_to_fun_comm' [B] [T] [U] [C] [D] [E] _ _ _ _ _ _.
Arguments plus_times_fin_pair_map2_list_compat' [B] [T] [U] [C] [D] [E] _ _ _ _ _.
Arguments times_fun_fin_map1 [B] [T] [U] [I] [J] [def] _ _.
Arguments times_fun_fin_map2 [B] [T] [U] [I] [J] [def] _ _.
Arguments times_fun_fin_map1_l [B] [T] [U] [I] [J] [def] _ _ _.
Arguments times_fun_fin_map2_l [B] [T] [U] [I] [J] [def] _ _ _.
Arguments im_sing_times_fun_fin_map1 [B] [T] [U] [I] [J] [def] _ _.
Arguments times_fun_fin_map1_empty [B] [T] [U] _ _.
Arguments times_fun_fin_map2_empty [B] [T] [U] _ _.
Arguments times_fun_fin_map1_empty1 [B] [T] [U] _ _ _ _.
Arguments times_fun_fin_map2_empty1 [B] [T] [U] _ _ _ _.
Arguments times_fun_fin_map1_list_compat [B] [T] [U] [I] [J] [def] _ _ _.
Arguments times_fun_fin_map2_list_compat [B] [T] [U] [I] [J] [def] _ _ _.
Arguments plus_fun_fin_map1 [B] [T] [U] [I] [J] [def] _ _.
Arguments plus_fun_fin_map2 [B] [T] [U] [I] [J] [def] _ _.
Arguments plus_fun_fin_map1_empty [B] [T] [U] _ _.
Arguments plus_fun_fin_map1_empty1 [B] [T] [U] _ _ _ _.
Arguments plus_fun_fin_map1_l [B] [T] [U] [I] [J] [def] _ _ _.
Arguments plus_fun_fin_map2_l [B] [T] [U] [I] [J] [def] _ _ _.
Arguments plus_fun_fin_map2_empty [B] [T] [U] _ _.
Arguments plus_fun_fin_map2_empty1 [B] [T] [U] _ _ _ _.
Arguments plus_fun_fin_map1_list_compat [B] [T] [U] [I] [J] [def] _ _ _.
Arguments plus_fun_fin_map2_list_compat [B] [T] [U] [I] [J] [def] _ _ _.
Arguments plus_times_fun_all_maps1 [B] [T] [U] _ _ _ _ _ _.
Arguments plus_times_fun_all_maps2 [B] [T] [U] _ _ _ _ _ _.
Arguments times_plus_fun_all_maps1 [B] [T] [U] _ _ _ _ _ _.
Arguments times_plus_fun_all_maps2 [B] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_list_times1' [B] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_list_times2' [B] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_list_plus1' [B] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_list_plus2' [B] [T] [U] _ _ _ _ _ _.
Arguments plus_times_all_maps1_funs_compat [B] [T] [U] [I] [J] _ _ _ _ _ _ _ _ _ _.
Arguments plus_times_all_maps2_funs_compat [B] [T] [U] _ _ _ _ _ _ _ _ _ _ _ _.
Arguments times_plus_all_maps1_funs_compat [B] [T] [U] _ _ _ _ _ _ _ _ _ _ _ _.
Arguments times_plus_all_maps2_funs_compat [B] [T] [U] _ _ _ _ _ _ _ _ _ _ _ _.
Arguments times_plus_fun1_fin_map_to_fun_undoes_fun_to_fin_map [B] [T] [U] _ _ _ _ _ _ _ _.
Arguments times_plus_fun2_fin_map_to_fun_undoes_fun_to_fin_map [B] [T] [U] _ _ _ _ _ _ _ _.
Arguments plus_times_fun1_fin_map_to_fun_undoes_fun_to_fin_map [B] [T] [U] _ _ _ _ _ _ _ _.
Arguments plus_times_fun2_fin_map_to_fun_undoes_fun_to_fin_map [B] [T] [U] _ _ _ _ _ _ _ _.
Arguments complete_dist_times_plus1 [B] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_times_plus2 [B] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_plus_times1 [B] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_plus_times2 [B] [T] [U] _ _ _ _ _ _.




(*Versions of all of the above theorems and definitions with
  [Bool_Alg_p]s instead of [Bool_Alg]s.*)
Section ParametricAnalogues.
Variable T':Type.
Variable Bp:Bool_Alg_p T'.
Let Btp := btp Bp.

Fixpoint times_list_p {T0:Type} {Bp0:Bool_Alg_p T0} 
         (l:list (btp  Bp0)) : 
  Btype_p T0 (Bc_p T0 Bp0) :=
  match l with
  | nil => %1
  | cons a l' => a %* (times_list_p l')
  end.

Fixpoint plus_list_p {T0:Type} {Bp0:Bool_Alg_p T0} 
         (l:list (btp Bp0)) :
  Btype_p T0 (Bc_p T0 Bp0) :=
  match l with
  | nil => %0
  | cons a l' => a %+ (plus_list_p l')
  end.

Lemma times_list_p_eq : 
  forall {T0:Type} {Bp0:Bool_Alg_p T0} 
         (l:list (btp Bp0)),
    times_list_p l = times_list (ba_conv_list l).
intros T0 Bp0 l.
induction l as [|a l h1]. simpl. reflexivity.
simpl. rewrite h1 at 1.
reflexivity.
Qed.

Lemma plus_list_p_eq : 
  forall {T0:Type} {Bp0:Bool_Alg_p T0} 
         (l:list (btp Bp0)),
    plus_list_p l = plus_list (ba_conv_list l).
intros T0 Bp0 l.
induction l as [|a l h1]. simpl. reflexivity.
simpl. rewrite h1 at 1.
reflexivity.
Qed.

Lemma plus_list_app_p : forall (l1 l2:list Btp),
  plus_list_p (l1 ++ l2) = plus_list_p l1 %+ plus_list_p l2.
intros l1 l2.
do 3 rewrite plus_list_p_eq.
apply (@plus_list_app (ba_conv Bp)).
Qed.

Lemma times_list_app_p : forall (l1 l2:list Btp),
  times_list_p (l1 ++ l2) = times_list_p l1 %* times_list_p l2.
intros l1 l2.
do 3 rewrite times_list_p_eq.
apply (@times_list_app (ba_conv Bp)).
Qed.

Lemma dist_list_sing_plus_p : forall (l:list Btp) (x:Btp),
  x %* (plus_list_p l) = plus_list_p (map (fun y:Btp => (x%*y)) l).
intros l x.
do 2 rewrite plus_list_p_eq.
apply (@dist_list_sing_plus (ba_conv Bp)).
Qed.

Lemma dist_list_sing_times_p : forall (l:list Btp) (x:Btp),
  x %+ (times_list_p l) = times_list_p (map (fun y:Btp => (x%+y)) l).
intros l x.
do 2 rewrite times_list_p_eq.
apply (@dist_list_sing_times (ba_conv Bp)).
Qed.

Lemma dist_list_2_plus_p : forall (l1 l2:list Btp),
  (plus_list_p l1) %* (plus_list_p l2) =
  plus_list_p (map (fun p:Btp*Btp => (fst p %* snd p)) (list_prod l1 l2)).
intros l1 l2.
do 3 rewrite plus_list_p_eq.
apply (@dist_list_2_plus (ba_conv Bp)).
Qed.

Lemma dist_list_2_times_p : forall (l1 l2:list Btp),
  (times_list_p l1) %+ (times_list_p l2) =
  times_list_p (map (fun p:Btp*Btp => (fst p %+ snd p)) (list_prod l1 l2)).
intros l1 l2.
do 3 rewrite times_list_p_eq.
apply (@dist_list_2_times (ba_conv Bp)).
Qed.

Definition plus_times_list_of_lists_p (l:list (list Btp)) : Btp :=
  plus_list_p (map (fun l':(list Btp) => times_list_p l') l).

Lemma plus_times_list_of_lists_p_eq : 
  forall (l:list (list Btp)),
    plus_times_list_of_lists_p l = plus_times_list_of_lists (ba_conv_list_list l).
intro l.
induction l as [|a l h1]; simpl. reflexivity.
unfold plus_times_list_of_lists_p, plus_times_list_of_lists.
rewrite plus_list_p_eq.
assert (h2:(fun l' : list Btp => times_list_p l') = 
            (fun l' : list (Btype (Bc (ba_conv Bp))) =>
               times_list l')).
  apply functional_extensionality.
  intro x. rewrite times_list_p_eq. reflexivity.
rewrite h2.
reflexivity.
Qed.

Definition times_plus_list_of_lists_p (l:list (list Btp)) : Btp :=
  times_list_p (map (fun l':(list Btp) => plus_list_p l') l).


Lemma times_plus_list_of_lists_p_eq : 
  forall (l:list (list Btp)),
    times_plus_list_of_lists_p l = times_plus_list_of_lists (ba_conv_list_list l).
intro l.
induction l as [|a l h1]; simpl. reflexivity.
unfold times_plus_list_of_lists_p, plus_times_list_of_lists.
rewrite times_list_p_eq.
assert (h2:(fun l' : list Btp => plus_list_p l') = 
            (fun l' : list (Btype (Bc (ba_conv Bp))) =>
               plus_list l')).
  apply functional_extensionality.
  intro x. rewrite plus_list_p_eq. reflexivity.
rewrite h2.
reflexivity.
Qed.

Definition times_fun_p {T:Type} (p:T->Btp) (l:list T) :=
  times_list_p (map p l).


Lemma times_fun_p_eq : 
  forall {T:Type} (p:T->Btp) (l:list T),
    times_fun_p p l = times_fun (ba_conv_ind p) l.
intros T p l.
unfold times_fun_p, times_fun.
rewrite times_list_p_eq.
reflexivity.
Qed.


Definition plus_fun_p {T:Type} (p:T->Btp) (l:list T) :=
  plus_list_p (map p l).

Lemma plus_fun_p_eq : 
  forall {T:Type} (p:T->Btp) (l:list T),
    plus_fun_p p l = plus_fun (ba_conv_ind p) l.
intros T p l.
unfold plus_fun_p, times_fun.
rewrite plus_list_p_eq.
reflexivity.
Qed.

Definition ba_conv_ind_ind {T U:Type} (p:T->U->Btp) :=
  transfer_dep (U:=fun V=>T->U->V) (ba_conv_type Bp) p.

Definition plus_times_fun1_p {T U:Type} (li:list T) (lj:list U)
           (p:T->U->Btp) :=
  plus_fun_p (fun i:T => times_fun_p (p i) lj) li.

Lemma plus_times_fun1_p_eq : 
  forall {T U:Type} (li:list T) (lj:list U)
           (p:T->U->Btp),
    plus_times_fun1_p li lj p =
    plus_times_fun1 li lj (ba_conv_ind_ind p).
intros T U li lj p.
unfold plus_times_fun1_p, plus_times_fun1.
rewrite plus_fun_p_eq.
assert (h1: (fun i : T => times_fun_p (p i) lj) = 
            (fun i : T => times_fun (ba_conv_ind_ind p i) lj)).
  apply functional_extensionality.
  intro x.
  rewrite times_fun_p_eq. reflexivity.
rewrite h1.
reflexivity.
Qed.

Definition plus_times_fun2_p {T U:Type} (li:list T) (lj:list U)
           (p:T->U->Btp) :=
  plus_fun_p (fun j:U => times_fun_p (fun i:T => (p i j)) li) lj.

Lemma plus_times_fun2_p_eq : 
  forall {T U:Type} (li:list T) (lj:list U)
           (p:T->U->Btp),
    plus_times_fun2_p li lj p =
    plus_times_fun2 li lj (ba_conv_ind_ind p).
intros T U li lj p.
unfold plus_times_fun2_p, plus_times_fun2.
rewrite plus_fun_p_eq.
assert (h1:
          (fun j : U => times_fun_p (fun i : T => p i j) li) =
          (fun j : U => times_fun (fun i : T => ba_conv_ind_ind p i j) li)).
  apply functional_extensionality.
  intro x.
  rewrite times_fun_p_eq. reflexivity.
rewrite h1.
reflexivity.
Qed.


Definition times_plus_fun1_p {T U:Type} (li:list T) (lj:list U)
           (p:T->U->Btp) :=
  times_fun_p (fun i:T => plus_fun_p (p i) lj) li.


Lemma times_plus_fun1_p_eq : 
  forall {T U:Type} (li:list T) (lj:list U)
           (p:T->U->Btp),
    times_plus_fun1_p li lj p =
    times_plus_fun1 li lj (ba_conv_ind_ind p).
intros T U li lj p.
unfold times_plus_fun1_p, times_plus_fun1.
rewrite times_fun_p_eq.
assert (h1: (fun i : T => plus_fun_p (p i) lj) = 
            (fun i : T => plus_fun (ba_conv_ind_ind p i) lj)).
  apply functional_extensionality.
  intro x.
  rewrite plus_fun_p_eq. reflexivity.
rewrite h1.
reflexivity.
Qed.


Definition times_plus_fun2_p {T U:Type} (li:list T) (lj:list U)
           (p:T->U->Btp) :=
  times_fun_p (fun j:U => plus_fun_p (fun i:T => (p i j)) li) lj.


Lemma times_plus_fun2_p_eq : 
  forall {T U:Type} (li:list T) (lj:list U)
           (p:T->U->Btp),
    times_plus_fun2_p li lj p =
    times_plus_fun2 li lj (ba_conv_ind_ind p).
intros T U li lj p.
unfold times_plus_fun2_p, times_plus_fun2.
rewrite times_fun_p_eq.
assert (h1:
          (fun j : U => plus_fun_p (fun i : T => p i j) li) =
          (fun j : U => plus_fun (fun i : T => ba_conv_ind_ind p i j) li)).
  apply functional_extensionality.
  intro x.
  rewrite plus_fun_p_eq. reflexivity.
rewrite h1.
reflexivity.
Qed.


Definition plus_times_all_funs1_p {T U:Type} (li:list T) (lj:list U)
           (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Btp) (def:U): Btp.
pose (list_power li lj) as lp.
pose (map_sig lp) as lp'.
pose (fun l:{x:list (T*U) | In x lp} =>
        times_fun_p (fun i:T =>
                     (p i (((in_list_power_fpl _ _ (proj1_sig l) pfi (list_power_no_dup _ _ _ pfi pfj (proj2_sig l)) (proj2_sig l)), def) l-> i))) li) as f.
refine (plus_fun_p f lp').
Defined.

Lemma plus_times_all_funs1_p_eq : 
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Btp) (def:U),
    plus_times_all_funs1_p li lj pfi pfj p def =
    plus_times_all_funs1 li lj pfi pfj (ba_conv_ind_ind p) def.
intros T U li lj h1 h2 p def.
unfold plus_times_all_funs1_p. unfold plus_times_all_funs1.
rewrite plus_fun_p_eq.
assert (h3: (fun l : {x : list (T * U) | In x (list_power li lj)} =>
         times_fun_p
           (fun i : T =>
            p i
              ((in_list_power_fpl li lj (proj1_sig l) h1
                  (list_power_no_dup li lj (proj1_sig l) h1 h2 (proj2_sig l))
                  (proj2_sig l), def) l-> i)) li) =
(fun l : {x : list (T * U) | In x (list_power li lj)} =>
      times_fun
        (fun i : T =>
         ba_conv_ind_ind p i
           ((in_list_power_fpl li lj (proj1_sig l) h1
               (list_power_no_dup li lj (proj1_sig l) h1 h2 (proj2_sig l))
               (proj2_sig l), def) l-> i)) li)).
  apply functional_extensionality.
  intro x.
  rewrite times_fun_p_eq. reflexivity.
rewrite h3.
reflexivity.
Qed.



Definition plus_times_all_funs2_p {T U:Type} (li:list T) (lj:list U)
           (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Btp) (def:T): Btp.
pose (list_power lj li) as lp.
pose (map_sig lp) as lp'.
pose (fun l:{x:list (U*T) | In x lp} =>
        times_fun_p (fun j:U =>
                     (p (((in_list_power_fpl _ _ (proj1_sig l) pfj (list_power_no_dup _ _ _ pfj pfi (proj2_sig l)) (proj2_sig l)), def) l-> j) j)) lj) as f.
refine (plus_fun_p f lp').
Defined.

Lemma plus_times_all_funs2_p_eq :
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Btp) (def:T),
    plus_times_all_funs2_p li lj pfi pfj p def =
    plus_times_all_funs2 li lj pfi pfj (ba_conv_ind_ind p) def.
intros T U li lj h1 h2 p def.
unfold plus_times_all_funs2_p, plus_times_all_funs2.
rewrite plus_fun_p_eq.
assert (h3: (fun l : {x : list (U * T) | In x (list_power lj li)} =>
         times_fun_p
           (fun j : U =>
            p
              ((in_list_power_fpl lj li (proj1_sig l) h2
                  (list_power_no_dup lj li (proj1_sig l) h2 h1 (proj2_sig l))
                  (proj2_sig l), def) l-> j) j) lj) =
           (fun l : {x : list (U * T) | In x (list_power lj li)} =>
      times_fun
        (fun j : U =>
         ba_conv_ind_ind p
           ((in_list_power_fpl lj li (proj1_sig l) h2
               (list_power_no_dup lj li (proj1_sig l) h2 h1 (proj2_sig l))
               (proj2_sig l), def) l-> j) j) lj)).
  apply functional_extensionality.
  intro x. rewrite times_fun_p_eq. reflexivity.
rewrite h3.
reflexivity.
Qed.


Definition times_plus_all_funs1_p {T U:Type} (li:list T) (lj:list U)
           (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Btp) (def:U): Btp.
pose (list_power li lj) as lp.
pose (map_sig lp) as lp'.
pose (fun l:{x:list (T*U) | In x lp} =>
        plus_fun_p (fun i:T =>
                     (p i (((in_list_power_fpl _ _ (proj1_sig l) pfi (list_power_no_dup _ _ _ pfi pfj (proj2_sig l)) (proj2_sig l)), def) l-> i))) li) as f.
refine (times_fun_p f lp').
Defined.

Lemma times_plus_all_funs1_p_eq : 
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Btp) (def:U),
    times_plus_all_funs1_p li lj pfi pfj p def =
    times_plus_all_funs1 li lj pfi pfj (ba_conv_ind_ind p) def.
intros T U li lj h1 h2 p def.
unfold times_plus_all_funs1_p, times_plus_all_funs1.
rewrite times_fun_p_eq.
assert (h3:(fun l : {x : list (T * U) | In x (list_power li lj)} =>
         plus_fun_p
           (fun i : T =>
            p i
              ((in_list_power_fpl li lj (proj1_sig l) h1
                  (list_power_no_dup li lj (proj1_sig l) h1 h2 (proj2_sig l))
                  (proj2_sig l), def) l-> i)) li) =
          (fun l : {x : list (T * U) | In x (list_power li lj)} =>
      plus_fun
        (fun i : T =>
         ba_conv_ind_ind p i
           ((in_list_power_fpl li lj (proj1_sig l) h1
               (list_power_no_dup li lj (proj1_sig l) h1 h2 (proj2_sig l))
               (proj2_sig l), def) l-> i)) li)).
  apply functional_extensionality.
  intro x.
  rewrite plus_fun_p_eq.
  reflexivity.
rewrite h3.
reflexivity.
Qed.

Definition times_plus_all_funs2_p {T U:Type} (li:list T) (lj:list U)
           (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Btp) (def:T): Btp.
pose (list_power lj li) as lp.
pose (map_sig lp) as lp'.
pose (fun l:{x:list (U*T) | In x lp} =>
        plus_fun_p (fun j:U =>
                     (p (((in_list_power_fpl _ _ (proj1_sig l) pfj (list_power_no_dup _ _ _ pfj pfi (proj2_sig l)) (proj2_sig l)), def) l-> j) j)) lj) as f.
refine (times_fun_p f lp').
Defined.   

Lemma times_plus_all_funs2_p_eq : 
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj) (p:T->U->Btp) (def:T),
    times_plus_all_funs2_p li lj pfi pfj p def =
    times_plus_all_funs2 li lj pfi pfj (ba_conv_ind_ind p) def.
intros T U li lj h1 h2 p def.
unfold times_plus_all_funs2_p, times_plus_all_funs2.
rewrite times_fun_p_eq.
assert (h3:(fun l : {x : list (U * T) | In x (list_power lj li)} =>
         plus_fun_p
           (fun j : U =>
            p
              ((in_list_power_fpl lj li (proj1_sig l) h2
                  (list_power_no_dup lj li (proj1_sig l) h2 h1 (proj2_sig l))
                  (proj2_sig l), def) l-> j) j) lj) =
         (fun l : {x : list (U * T) | In x (list_power lj li)} =>
      plus_fun
        (fun j : U =>
         ba_conv_ind_ind p
           ((in_list_power_fpl lj li (proj1_sig l) h2
               (list_power_no_dup lj li (proj1_sig l) h2 h1 (proj2_sig l))
               (proj2_sig l), def) l-> j) j) lj)).
  apply functional_extensionality.
  intro x.
  rewrite plus_fun_p_eq.
  reflexivity.
rewrite h3.
reflexivity.
Qed.

Lemma complete_dist_list_times_p : forall (l:list (list Btp)),
  times_plus_list_of_lists_p l = plus_times_list_of_lists_p (list_of_lists_seqs l).
intro l.
rewrite times_plus_list_of_lists_p_eq.
rewrite plus_times_list_of_lists_p_eq.
apply (@complete_dist_list_times (ba_conv Bp)).
Qed.

Lemma complete_dist_list_plus_p : forall (l:list (list Btp)),
  plus_times_list_of_lists_p l = times_plus_list_of_lists_p (list_of_lists_seqs l).
intro l.
rewrite plus_times_list_of_lists_p_eq.
rewrite times_plus_list_of_lists_p_eq.
apply (@complete_dist_list_plus (ba_conv Bp)).
Qed.


Lemma inf_union_p : forall (U V:Ensemble Btp) (u v:Btp), inf_p U u -> inf_p V v ->
  (inf_p (Union U V) (u %* v)).
intros U V u v.
do 3 rewrite inf_p_iff.
apply (@inf_union (ba_conv Bp)).
Qed.

Lemma sup_union_p : forall (U V:Ensemble Btp) (u v:Btp), sup_p U u -> sup_p V v ->
  (sup_p (Union U V) (u %+ v)).
intros U V u v.
do 3 rewrite sup_p_iff.
apply (@sup_union (ba_conv Bp)).
Qed.


Lemma decompose_inf_p : forall (S X:Ensemble Btp) (s x x':Btp), Included X S -> inf_p S s ->
  inf_p (Setminus S X) x' -> inf_p X x -> inf_p (Couple x x') s.
intros S X s x x'.
do 4 rewrite inf_p_iff.
apply (@decompose_inf (ba_conv Bp)).
Qed.


Lemma decompose_inf_p': forall (S X:Ensemble Btp) (x x':Btp), Included X S ->
  inf_p X x -> inf_p (Setminus S X) x' -> inf_p S (x %* x').
intros S X x x'.
do 3 rewrite inf_p_iff.
apply (@decompose_inf' (ba_conv Bp)).
Qed.


Lemma inf_times_cons_p : forall (E:Ensemble Btp) (a b:Btp), Ensembles.In E a ->
  inf_p (Subtract E a) b -> inf_p E (a %* b).
intros E a b.
do 2 rewrite inf_p_iff. apply (@inf_times_cons (ba_conv Bp)).
Qed.


Lemma inf_times_finite_p : forall (l:list Btp),
  inf_p (list_to_set l) (times_list_p l).
intro l. rewrite inf_p_iff. rewrite times_list_p_eq.
apply (@inf_times_finite (ba_conv Bp)).
Qed.


Lemma sup_plus_finite_p : forall (l:list Btp),
  sup_p (list_to_set l) (plus_list_p l).
intro l. rewrite sup_p_iff. rewrite plus_list_p_eq.
apply (@sup_plus_finite (ba_conv Bp)).
Qed.


Lemma le_times_finite_member_p : forall (l:list Btp) (x:Btp),
  (In x l) -> le_p (times_list_p l) x.
intros l x.
rewrite le_p_iff. rewrite times_list_p_eq.
apply (@le_times_finite_member (ba_conv Bp)).
Qed.


Lemma le_member_plus_finite_p : forall (l:list Btp) (x:Btp),
  (In x l) -> le_p x (plus_list_p l).
intros l x.
rewrite le_p_iff. rewrite plus_list_p_eq.
apply (@le_member_plus_finite (ba_conv Bp)).
Qed.

Lemma prod_list_dup_eq_p :
  forall (l:list Btp) (x:Btp),
    In x l -> times_list_p l = times_list_p (x::l).
intros l x. do 2 rewrite times_list_p_eq.
apply (@prod_list_dup_eq (ba_conv Bp)).
Qed.


Lemma sum_list_dup_eq_p :
  forall (l:list Btp) (x:Btp),
    In x l -> plus_list_p l = plus_list_p (x::l).
intros l x.
rewrite plus_list_p_eq. rewrite plus_list_p_eq.
apply (@sum_list_dup_eq (ba_conv Bp)).
Qed.


Lemma prod_preserves_list_singularize_p :
  forall (l:list Btp),
    times_list_p l = times_list_p (list_singularize l nil).
intro l. do 2 rewrite times_list_p_eq.
apply (@prod_preserves_list_singularize (ba_conv Bp)).
Qed.


Lemma sum_preserves_list_singularize_p :
  forall (l:list Btp),
    plus_list_p l = plus_list_p (list_singularize l nil).
intro l. do 2 rewrite plus_list_p_eq.
apply (@sum_preserves_list_singularize (ba_conv Bp)).
Qed.


Lemma times_sing_preserves_new_head_p :
  forall (l:list Btp) (x:Btp),
    In x l ->
    times_list_p (list_singularize l nil) =
    times_list_p (new_head (list_singularize l nil) x).
intros l x.
do 2 rewrite times_list_p_eq.
apply (@times_sing_preserves_new_head (ba_conv Bp)).
Qed.


Lemma plus_sing_preserves_new_head_p :
  forall (l:list Btp) (x:Btp),
    In x l ->
    plus_list_p (list_singularize l nil) =
    plus_list_p (new_head (list_singularize l nil) x).
intros l x.
do 2 rewrite plus_list_p_eq.
apply (@plus_sing_preserves_new_head (ba_conv Bp)).
Qed.


Lemma times_list_sing_cons_p :
  forall (l:list Btp) (x:Btp),
    In x (list_singularize l nil) ->
    times_list_p (list_singularize l nil) =
    x%*times_list_p (list_singularize (new_head_aux l x) nil).
intros l x.
do 2 rewrite times_list_p_eq.
apply (@times_list_sing_cons (ba_conv Bp)).
Qed.


Lemma plus_list_sing_cons_p :
  forall (l:list Btp) (x:Btp),
    In x (list_singularize l nil) ->
    plus_list_p (list_singularize l nil) =
    x%+plus_list_p (list_singularize (new_head_aux l x) nil).
intros l x.
do 2 rewrite plus_list_p_eq.
apply (@plus_list_sing_cons (ba_conv Bp)).
Qed.


Lemma list_to_sets_eq_times_sing_eq_p :
  forall (E:Ensemble Btp),
    forall (l l':list Btp),
      E = list_to_set (list_singularize l nil) ->
      E = list_to_set (list_singularize l' nil) ->
      times_list_p (list_singularize l nil) =
      times_list_p (list_singularize l' nil).
intros E l l'.
do 2 rewrite times_list_p_eq.
apply (@list_to_sets_eq_times_sing_eq (ba_conv Bp)).
Qed.


Lemma list_to_sets_eq_plus_sing_eq_p :
  forall (E:Ensemble Btp),
    forall (l l':list Btp),
      E = list_to_set (list_singularize l nil) ->
      E = list_to_set (list_singularize l' nil) ->
      plus_list_p (list_singularize l nil) =
      plus_list_p (list_singularize l' nil).
intros E l l'.
do 2 rewrite plus_list_p_eq.
apply (@list_to_sets_eq_plus_sing_eq (ba_conv Bp)).
Qed.

Lemma list_to_sets_eq_times_eq_p :
  forall (E:Ensemble Btp),
    forall (l l':list Btp),
      E = list_to_set l ->
      E = list_to_set l' ->
      times_list_p l = times_list_p l'.
intros E l l'.
do 2 rewrite times_list_p_eq.
apply (@list_to_sets_eq_times_eq (ba_conv Bp)).
Qed.


Lemma list_to_sets_eq_plus_eq_p :
  forall (E:Ensemble Btp),
    forall (l l':list Btp),
      E = list_to_set l ->
      E = list_to_set l' ->
      plus_list_p l = plus_list_p l'.
intros E l l'.
do 2 rewrite plus_list_p_eq.
apply (@list_to_sets_eq_plus_eq (ba_conv Bp)).
Qed.

Lemma times_list_unq_p : forall (E:Ensemble Btp)
                       (pf:Finite E), exists! x,
                         forall (l:list Btp), E = list_to_set l ->
                       x = times_list_p l.
intros E h1.
pose proof (@times_list_unq (ba_conv Bp) E h1) as h2.
destruct h2 as [x h2].
exists x.
red in h2. destruct h2 as [h2l h2r].
red. split.
intros l h3. rewrite times_list_p_eq.
apply h2l; auto. 
intros x' h3.
assert (h4:forall l:list (Btype (Bc (ba_conv Bp))), E = list_to_set l -> x' = times_list l).
  intros l h4.
  specialize (h3 _ h4).
  rewrite times_list_p_eq in h3. assumption.
apply h2r; auto.
Qed.


Lemma plus_list_unq_p : forall (E:Ensemble Btp)
                       (pf:Finite E), exists! x,
                         forall (l:list Btp), E = list_to_set l ->
                       x = plus_list_p l.
intros E h1.
pose proof (@plus_list_unq (ba_conv Bp) E h1) as h2.
destruct h2 as [x h2].
exists x.
red in h2. destruct h2 as [h2l h2r].
red. split.
intros l h3. rewrite plus_list_p_eq.
apply h2l; auto. 
intros x' h3.
assert (h4:forall l:list (Btype (Bc (ba_conv Bp))), E = list_to_set l -> x' = plus_list l).
  intros l h4.
  specialize (h3 _ h4).
  rewrite plus_list_p_eq in h3. assumption.
apply h2r; auto.
Qed.


Definition times_set_p (E:Ensemble Btp) (pf:Finite E) : Btp.
refine (proj1_sig (constructive_definite_description
            _ (times_list_unq_p E pf))).
Defined.

Lemma times_set_p_eq : 
  forall (E:Ensemble Btp) (pf:Finite E),
    times_set_p E pf = ba_conv_elt (times_set (ba_conv_set E) pf).
intros E h1.
unfold times_set_p. unfold times_set.
destruct constructive_definite_description as [x h2].
destruct constructive_definite_description as [y h3].
simpl.
pose proof (finite_set_list _ h1) as h4.
destruct h4 as [l h4].
specialize (h2 _ h4). specialize (h3 _ h4). subst.
rewrite times_list_p_eq.
reflexivity.
Qed.

Definition plus_set_p (E:Ensemble Btp) (pf:Finite E) : Btp.
refine (proj1_sig (constructive_definite_description
            _ (plus_list_unq_p E pf))).
Defined.


Lemma plus_set_p_eq : 
  forall (E:Ensemble Btp) (pf:Finite E),
    plus_set_p E pf = ba_conv_elt (plus_set (ba_conv_set E) pf).
intros E h1.
unfold plus_set_p. unfold plus_set.
destruct constructive_definite_description as [x h2].
destruct constructive_definite_description as [y h3].
simpl.
pose proof (finite_set_list _ h1) as h4.
destruct h4 as [l h4].
specialize (h2 _ h4). specialize (h3 _ h4). subst.
rewrite plus_list_p_eq.
reflexivity.
Qed.


Lemma times_set_compat_p : forall (E:Ensemble Btp)
                                  (pf:Finite E),
                           exists l:list Btp,
                             times_set_p E pf = times_list_p l.
intros E h1.
pose proof (@times_set_compat (ba_conv Bp) _ h1) as h2.
destruct h2 as [l h2].
exists l.
rewrite times_set_p_eq. rewrite times_list_p_eq.
assumption.
Qed.


Lemma times_set_compat_p' : forall (E:Ensemble Btp) (l:list Btp)
                            (pf:Finite E),
                            E = list_to_set l ->
                            times_set_p E pf = times_list_p l.
intros E l h1.
rewrite times_set_p_eq. rewrite times_list_p_eq.
apply (@times_set_compat' (ba_conv Bp)).
Qed.


Lemma plus_set_compat_p : forall (E:Ensemble Btp)
                                (pf:Finite E),
                           exists l:list Btp,
                             plus_set_p E pf = plus_list_p l.
intros E h1.
pose proof (@plus_set_compat (ba_conv Bp) _ h1) as h2.
destruct h2 as [l h2].
exists l. rewrite plus_list_p_eq. rewrite plus_set_p_eq. assumption.
Qed.


Lemma plus_set_compat_p' : forall (E:Ensemble Btp) (l:list Btp)
                            (pf:Finite E),
                            E = list_to_set l ->
                            plus_set_p E pf = plus_list_p l.
intros E l h1.
rewrite plus_set_p_eq. rewrite plus_list_p_eq.
apply (@plus_set_compat' (ba_conv Bp)).
Qed.


Lemma times_set_functional_p :
  forall (A C:Ensemble Btp) (pfa:Finite A) (pfc:Finite C),
    A = C -> times_set_p A pfa = times_set_p C pfc.
intros A C h1 h2. do 2 rewrite times_set_p_eq.
apply (@times_set_functional (ba_conv Bp)).
Qed.


Lemma plus_set_functional_p :
  forall (A C:Ensemble Btp) (pfa:Finite A) (pfc:Finite C),
    A = C -> plus_set_p A pfa = plus_set_p C pfc.
intros A C h1 h2. do 2 rewrite plus_set_p_eq.
apply (@plus_set_functional (ba_conv Bp)).
Qed.

Lemma times_set_empty_p :
  times_set_p (Empty_set Btp) (Empty_is_finite Btp) = %1.
rewrite times_set_p_eq.
apply (@times_set_empty (ba_conv Bp)).
Qed.


Lemma times_set_empty_p' :
  forall (pf:Finite (Empty_set Btp)),
    times_set_p (Empty_set Btp) pf = %1.
intro h1. rewrite times_set_p_eq.
apply (@times_set_empty' (ba_conv Bp)).
Qed.

Lemma plus_set_empty_p :
  plus_set_p (Empty_set Btp) (Empty_is_finite Btp) = %0.
rewrite plus_set_p_eq.
apply (@plus_set_empty (ba_conv Bp)).
Qed.


Lemma plus_set_empty_p' :
  forall (pf:Finite (Empty_set Btp)),
    plus_set_p (Empty_set Btp) pf = %0.
intro h1. rewrite plus_set_p_eq.
apply (@plus_set_empty' (ba_conv Bp)).
Qed.


Lemma times_set_add_p :
  forall (E:Ensemble Btp) (pf:Finite E) (x:Btp),
    times_set_p (Add E x) (Add_preserves_Finite _ _ _ pf) =
    x %* (times_set_p E pf).
intros E h1 x.
do 2 rewrite times_set_p_eq.
apply (@times_set_add (ba_conv Bp)).
Qed.


Lemma times_set_add_p' :
  forall (E:Ensemble Btp) (pf1:Finite E) (x:Btp)
         (pf2:Finite (Add E x)),
    times_set_p (Add E x) pf2 =
    x %* (times_set_p E pf1).
intros E h1 x h2.
do 2 rewrite times_set_p_eq.
apply (@times_set_add' (ba_conv Bp)).
Qed.


Lemma plus_set_add_p :
  forall (E:Ensemble Btp) (pf:Finite E) (x:Btp),
    plus_set_p (Add E x) (Add_preserves_Finite _ _ _ pf) =
    x %+ (plus_set_p E pf).
intros E h1 x. do 2 rewrite plus_set_p_eq.
apply (@plus_set_add (ba_conv Bp)).
Qed.


Lemma plus_set_add_p' :
  forall (E:Ensemble Btp) (pf1:Finite E) (x:Btp)
         (pf2:Finite (Add E x)),
    plus_set_p (Add E x) pf2 =
    x %+ (plus_set_p E pf1).
intros E h1 x h2.
do 2 rewrite plus_set_p_eq.
apply (@plus_set_add' (ba_conv Bp)).
Qed.


Lemma times_set_sing_p :
  forall x:Btp, times_set_p (Singleton x) (Singleton_is_finite _ x) = x.
intro x. rewrite times_set_p_eq. apply (@times_set_sing (ba_conv Bp)).
Qed.


Lemma times_set_sing_p' :
  forall (x:Btp) (pf:Finite (Singleton x)), times_set_p _ pf = x.
intros x h1. rewrite times_set_p_eq. apply (@times_set_sing' (ba_conv Bp)).
Qed.


Lemma times_set_one_or_p :
  forall (E:Ensemble Btp) (pf:Finite E),
         (E = Empty_set _) \/ (E = Singleton %1) ->
    %1 = times_set_p E pf.
intros E h1. rewrite times_set_p_eq.
apply (@times_set_one_or (ba_conv Bp)).
Qed.


Lemma plus_set_sing_p :
  forall x:Btp, plus_set_p (Singleton x) (Singleton_is_finite _ x) = x.
intro x. rewrite plus_set_p_eq.  apply (@plus_set_sing (ba_conv Bp)).
Qed.


Lemma plus_set_sing_p' :
  forall (x:Btp) (pf:Finite (Singleton x)), plus_set_p _ pf = x.
intros x h1. rewrite plus_set_p_eq.
apply (@plus_set_sing' (ba_conv Bp)).
Qed.


Lemma plus_set_zero_or_p :
  forall (E:Ensemble Btp) (pf:Finite E),
         (E = Empty_set _) \/ (E = Singleton %0) ->
    %0 = plus_set_p E pf.
intros E h1. rewrite plus_set_p_eq.
apply (@plus_set_zero_or (ba_conv Bp)).
Qed.


Lemma times_set_couple_p :
  forall (x y:Btp),
    times_set_p (Couple x y) (finite_couple x y) = x %* y.
intros x y. rewrite times_set_p_eq.
apply (@times_set_couple (ba_conv Bp)).
Qed.


Lemma times_set_couple_p' :
  forall (x y:Btp) (pf:Finite (Couple x y)),
    times_set_p (Couple x y) pf = x %* y.
intros x y h1. rewrite times_set_p_eq.
apply (@times_set_couple' (ba_conv Bp)).
Qed.


Lemma plus_set_couple_p :
  forall (x y:Btp),
    plus_set_p (Couple x y) (finite_couple x y) = x %+ y.
intros x y. rewrite plus_set_p_eq. apply (@plus_set_couple (ba_conv Bp)).
Qed.


Lemma plus_set_couple_p' :
  forall (x y:Btp) (pf:Finite (Couple x y)),
    plus_set_p (Couple x y) pf = x %+ y.
intros x y h1. rewrite plus_set_p_eq. apply (@plus_set_couple' (ba_conv Bp)).
Qed.


Lemma le_times_set_p : 
  forall (E:Ensemble Btp) (pf:Finite E)
         (x:Btp),
    (Ensembles.In E x) ->
    le_p (times_set_p E pf) x.
intros E h1 x. rewrite le_p_iff. rewrite times_set_p_eq.
apply (@le_times_set (ba_conv Bp)).
Qed.


Lemma le_plus_set_p : 
  forall (E:Ensemble Btp) (pf:Finite E)
         (x:Btp),
    (Ensembles.In E x) ->
    le_p x (plus_set_p E pf).
intros E h1 x h2. rewrite le_p_iff. rewrite plus_set_p_eq.
apply (@le_plus_set (ba_conv Bp)); auto.
Qed.


Lemma inf_times_set_p : forall (E:Ensemble Btp)
  (pf:Finite E), inf_p E (times_set_p E pf).
intros E h1. rewrite inf_p_iff. rewrite times_set_p_eq.
apply (@inf_times_set (ba_conv Bp)).
Qed.


Lemma sup_plus_set_p : forall (E:Ensemble Btp)
  (pf:Finite E), sup_p E (plus_set_p E pf).
intros E h1. rewrite sup_p_iff. rewrite plus_set_p_eq.
apply (@sup_plus_set (ba_conv Bp)).
Qed.


Lemma times_set_union_p :
  forall (C D:Ensemble Btp) (pfc:Finite C)
         (pfd: Finite D),
    times_set_p (Union C D) (Union_preserves_Finite _ _ _ pfc pfd) =
    (times_set_p C pfc) %* (times_set_p D pfd).
intros C D h1 h2.
rewrite times_set_p_eq. do 2 rewrite times_set_p_eq.
apply (@times_set_union (ba_conv Bp)).
Qed.


Lemma times_set_union_p' :
  forall (C D:Ensemble Btp) (pfc:Finite C)
         (pfd: Finite D) (pfu:Finite (Union C D)),
    times_set_p (Union C D) pfu =
    (times_set_p C pfc) %* (times_set_p D pfd).
intros C D h1 h2 h3.
do 3 rewrite times_set_p_eq.
apply (@times_set_union' (ba_conv Bp)).
Qed.


Lemma times_set_inc_le_p :
  forall (C D:Ensemble Btp) (pfc:Finite C)
         (pfd:Finite D) (pfi:Included C D),
    le_p (times_set_p D pfd) (times_set_p C pfc).
intros C D h1 h2 h3.
rewrite le_p_iff. do 2 rewrite times_set_p_eq.
apply (@times_set_inc_le (ba_conv Bp)); auto.
Qed.


Lemma plus_set_union_p :
  forall (C D:Ensemble Btp) (pfc:Finite C)
         (pfd: Finite D),
    plus_set_p (Union C D) (Union_preserves_Finite _ _ _ pfc pfd) =
    (plus_set_p C pfc) %+ (plus_set_p D pfd).
intros C D h1 h2.
do 3 rewrite plus_set_p_eq.
apply (@plus_set_union (ba_conv Bp)).
Qed.


Lemma plus_set_union_p' :
  forall (C D:Ensemble Btp) (pfc:Finite C)
         (pfd: Finite D) (pfu:Finite (Union C D)),
    plus_set_p (Union C D) pfu =
    (plus_set_p C pfc) %+ (plus_set_p D pfd).
intros C D h1 h2 h3. 
do 3 rewrite plus_set_p_eq.
apply (@plus_set_union' (ba_conv Bp)).
Qed.

Lemma plus_set_inc_le_p :
  forall (C D:Ensemble Btp) (pfc:Finite C)
         (pfd:Finite D) (pfi:Included C D),
    le_p (plus_set_p C pfc) (plus_set_p D pfd).
intros C D h1 h2 h3.
rewrite le_p_iff. do 2 rewrite plus_set_p_eq.
apply (@plus_set_inc_le (ba_conv Bp)); auto.
Qed.

Lemma plus_set_im_add_p :
  forall (E:Ensemble Btp) (pf:Finite E) (x:Btp) (f:Btp->Btp),
    plus_set_p (Im (Add E x) f)
                 (finite_image _ _ _ f
                               (Add_preserves_Finite _ _ x pf)) =
    f x %+ plus_set_p (Im E f) (finite_image _ _ _ f pf).
intros E h1 x f.
do 2 rewrite plus_set_p_eq.
apply (@plus_set_im_add (ba_conv Bp)).
Qed.

Lemma plus_set_im_add_p' : 
  forall (E:Ensemble Btp) (x:Btp) (f:Btp->Btp)
         (pf0: Finite E)
         (pf1:Finite (Im (Add E x) f))
         (pf2:Finite (Im E f)),
    plus_set_p (Im (Add E x) f) pf1 =
    f x %+ (plus_set_p (Im E f) pf2).
intros E x f h1 h2 h3.
do 2 rewrite plus_set_p_eq.
apply (@plus_set_im_add' (ba_conv Bp)).
assumption.
Qed.


Lemma times_set_im_add_p :
  forall (E:Ensemble Btp) (pf:Finite E) (x:Btp) (f:Btp->Btp),
    times_set_p (Im (Add E x) f)
                 (finite_image _ _ _ f
                               (Add_preserves_Finite _ _ x pf)) =
    f x %* times_set_p (Im E f) (finite_image _ _ _ f pf).
intros E h1 x f.
do 2 rewrite times_set_p_eq.
apply (@times_set_im_add (ba_conv Bp)).
Qed.

Lemma times_set_im_add_p' : 
  forall (E:Ensemble Btp) (x:Btp) (f:Btp->Btp)
         (pf0: Finite E)
         (pf1:Finite (Im (Add E x) f))
         (pf2:Finite (Im E f)),
    times_set_p (Im (Add E x) f) pf1 =
    f x %* (times_set_p (Im E f) pf2).
intros E x f h1 h2 h3.
do 2 rewrite times_set_p_eq.
apply (@times_set_im_add' (ba_conv Bp)).
assumption.
Qed.


Lemma dist_set_plus1_p :
  forall (E:Ensemble Btp) (pf:Finite E) (x:Btp),
    let f := (fun y:Btp => x %* y) in
    x %* (plus_set_p E pf) =
    plus_set_p (Im E f) (finite_image _ _ _ f pf).
intros E h1 x f.
do 2 rewrite plus_set_p_eq.
apply (@dist_set_plus1 (ba_conv Bp)); auto.
Qed.


Lemma dist_set_plus1_p' :
  forall (E:Ensemble Btp) (pf1:Finite E) (x:Btp),
    let f := (fun y:Btp => x %* y) in
    forall (pf2:Finite (Im E f)),
    x %* (plus_set_p E pf1) =
    plus_set_p (Im E f) pf2.
intros E h1 x f h2.
do 2 rewrite plus_set_p_eq.
apply (@dist_set_plus1' (ba_conv Bp)).
Qed.


Lemma dist_set_times1_p :
  forall (E:Ensemble Btp) (pf:Finite E) (x:Btp),
    let f := (fun y:Btp => x %+ y) in
    x %+ (times_set_p E pf) =
    times_set_p (Im E f) (finite_image _ _ _ f pf).
intros E h1 x f.
do 2 rewrite times_set_p_eq.
apply (@dist_set_times1 (ba_conv Bp)).
Qed.


Lemma dist_set_times1_p' :
  forall (E:Ensemble Btp) (pf1:Finite E) (x:Btp),
    let f := (fun y:Btp => x %+ y) in
    forall (pf2:Finite (Im E f)), x %+ (times_set_p E pf1) =
    times_set_p (Im E f) pf2.
intros E h1 x f h2.
do 2 rewrite times_set_p_eq.
apply (@dist_set_times1' (ba_conv Bp)).
Qed.


Lemma dist_set_plus2_p :
  forall (D E:Ensemble Btp)
    (pfd: Finite D) (pfe: Finite E),
    let f := (fun p:Btp*Btp => (fst p) %* (snd p)) in
    (plus_set_p D pfd) %* (plus_set_p E pfe) =
    plus_set_p (Im (cart_prod D E) f)
             (finite_image _ _ _ f
                           (cart_prod_fin _ _ pfd pfe)).
intros D E h1 h2 f.
do 3 rewrite plus_set_p_eq.
apply (@dist_set_plus2 (ba_conv Bp)).
Qed.


Lemma dist_set_plus2_p' :
  forall (D E:Ensemble Btp)
    (pfd: Finite D) (pfe: Finite E),
    let f := (fun p:Btp*Btp => (fst p) %* (snd p)) in
    forall (pfc: Finite (Im (cart_prod D E) f)),
      (plus_set_p D pfd) %* (plus_set_p E pfe) =
      plus_set_p (Im (cart_prod D E) f) pfc.
intros D E h1 h2 f h3.
do 3 rewrite plus_set_p_eq.
apply (@dist_set_plus2' (ba_conv Bp)).
Qed.


Lemma dist_set_times2_p :
  forall (D E:Ensemble Btp)
    (pfd: Finite D) (pfe: Finite E),
    let f := (fun p:Btp*Btp => (fst p) %+ (snd p)) in
    (times_set_p D pfd) %+ (times_set_p E pfe) =
    times_set_p (Im (cart_prod D E) f)
             (finite_image _ _ _ f
                           (cart_prod_fin _ _ pfd pfe)).
intros D E h1 h2 f.
do 3 rewrite times_set_p_eq.
apply (@dist_set_times2 (ba_conv Bp)).
Qed.


Lemma dist_set_times2_p' :
  forall (D E:Ensemble Btp)
    (pfd: Finite D) (pfe: Finite E),
    let f := (fun p:Btp*Btp => (fst p) %+ (snd p)) in
    forall (pfc: Finite (Im (cart_prod D E) f)),
      (times_set_p D pfd) %+ (times_set_p E pfe) =
      times_set_p (Im (cart_prod D E) f) pfc.
intros D E h1 h2 f h3.
do 3 rewrite times_set_p_eq.
apply (@dist_set_times2' (ba_conv Bp)); auto.
Qed.

Definition ba_conv_fin_map_dom 
           {U:Type} {C:Ensemble Btp} {D:Ensemble U} {def:U}
           (F:Fin_map C D def) : Fin_map (ba_conv_set C) D def.
unfold ba_conv_set, ba_conv_type, transfer_dep_eq_refl.
refine F.
Defined.


Definition ba_conv_fin_map_ran 
           {T:Type} {C:Ensemble T} {D:Ensemble Btp}
           (F:Fin_map C D %0) : Fin_map C (ba_conv_set D) 0.
unfold ba_conv_set. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
refine F.
Defined.

Definition ba_conv_ens_fin_map_dom
           {U:Type} {C:Ensemble Btp} {D:Ensemble U} {def:U}
           (A:Ensemble (Fin_map C D def)) : Ensemble (Fin_map (ba_conv_set C) D def). 
unfold ba_conv_set, ba_conv_type. rewrite transfer_dep_eq_refl.
refine A.
Defined.

Definition ba_conv_ens_fin_map_ran
           {T:Type} {C:Ensemble T} {D:Ensemble Btp}
           (A:Ensemble (Fin_map C D %0)) : Ensemble (Fin_map C (ba_conv_set D) 0).
unfold ba_conv_set, ba_conv_type. rewrite transfer_dep_eq_refl.
refine A.
Defined.

Lemma incl_ba_conv_ens_fin_map_dom_iff : 
  forall  {U:Type} {C:Ensemble Btp} {D:Ensemble U} {def:U}
          (A B:Ensemble (Fin_map C D def)),
    Included A B <-> Included (ba_conv_ens_fin_map_dom A)
                              (ba_conv_ens_fin_map_dom B).
intros U C D def A B.
split.
intro h1. red. intros x h2.
apply h1; auto.
intro h1. red. intros x h2.
apply h1; auto.
Qed.


Definition plus_fin_pair_map1_p
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Btp} (pfc:Finite C)
           (F:Fin_map (cart_prod C D) E %0) : 
  Fin_map C
    (Im C
       (fun x : T =>
        plus_set (im1 (ba_conv_fin_map_ran F) x) (im1_fin (ba_conv_fin_map_ran F) x)))
    %0 :=
  (fun_to_fin_map C %0 pfc
                         (fun x:T => plus_set (im1 (ba_conv_fin_map_ran F) x)  
                                              (im1_fin (ba_conv_fin_map_ran F) x))).


Lemma plus_fin_pair_map1_p_eq : 
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U}
         {E:Ensemble Btp} (pfc:Finite C)
         (F:Fin_map (cart_prod C D) E %0),
    plus_fin_pair_map1_p pfc F = plus_fin_pair_map1 pfc (ba_conv_fin_map_ran F).
auto.
Qed.


Definition plus_fin_pair_map2_p
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Btp} (pfd:Finite D)
           (F:Fin_map (cart_prod C D) E %0) : 
  Fin_map D
    (Im D
       (fun y : U =>
        plus_set (im2 (ba_conv_fin_map_ran F) y) (im2_fin (ba_conv_fin_map_ran F) y)))
    %0 :=
  (fun_to_fin_map D %0 pfd
                         (fun y:U => plus_set (im2 (ba_conv_fin_map_ran F) y)  
                                              (im2_fin (ba_conv_fin_map_ran F) y))).



Lemma plus_fin_pair_map2_p_eq : 
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U}
         {E:Ensemble Btp} (pfd:Finite D)
         (F:Fin_map (cart_prod C D) E %0),
    plus_fin_pair_map2_p pfd F = plus_fin_pair_map2 pfd (ba_conv_fin_map_ran F).
auto.
Qed.


Definition times_fin_pair_map1_p
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Btp} (pfc:Finite C)
           (F:Fin_map (cart_prod C D) E %0) : 
  Fin_map C
    (Im C
       (fun x : T =>
        times_set (im1 (ba_conv_fin_map_ran F) x) (im1_fin (ba_conv_fin_map_ran F) x)))
    %0 :=
  (fun_to_fin_map C %0 pfc
                         (fun x:T => times_set (im1 (ba_conv_fin_map_ran F) x)  
                                              (im1_fin (ba_conv_fin_map_ran F) x))).


Lemma times_fin_pair_map1_p_eq : 
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U}
         {E:Ensemble Btp} (pfc:Finite C)
         (F:Fin_map (cart_prod C D) E %0),
    times_fin_pair_map1_p pfc F = times_fin_pair_map1 pfc (ba_conv_fin_map_ran F).
auto.
Qed.


Definition times_fin_pair_map2_p
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Btp} (pfd:Finite D)
           (F:Fin_map (cart_prod C D) E %0) : 
  Fin_map D
    (Im D
       (fun y : U =>
        times_set (im2 (ba_conv_fin_map_ran F) y) (im2_fin (ba_conv_fin_map_ran F) y)))
    %0 :=
  (fun_to_fin_map D %0 pfd
                         (fun y:U => times_set (im2 (ba_conv_fin_map_ran F) y)  
                                              (im2_fin (ba_conv_fin_map_ran F) y))).



Lemma times_fin_pair_map2_p_eq : 
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U}
         {E:Ensemble Btp} (pfd:Finite D)
         (F:Fin_map (cart_prod C D) E %0),
    times_fin_pair_map2_p pfd F = times_fin_pair_map2 pfd (ba_conv_fin_map_ran F).
auto.
Qed.


Lemma plus_fin_pair_map1_list_compat_p :
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U}
         {E:Ensemble Btp} (pfc:Finite C)
         (F:Fin_map (cart_prod C D) E %0)
         (nml:nice_map_lists2 F),
    fin_map_eq (plus_fin_pair_map1_p pfc F)
    (fun_to_fin_map C %0 pfc (fun x:T =>(plus_list_p (im1l (fpl2 F nml) %0 x)))).
intros T U C D E h1 F nml. 
rewrite plus_fin_pair_map1_p_eq.
pose proof (@plus_fin_pair_map1_list_compat (ba_conv Bp) _ _ _ _ _ h1 (ba_conv_fin_map_ran F) nml) as h2.  
red in h2. destruct h2 as [h2 h3].
red. 
assert (h0:forall x:T, Ensembles.In C x ->
                       fin_map_new_ran (plus_fin_pair_map1 h1 (ba_conv_fin_map_ran F))
         (fin_map_fin_ran
            (fun_to_fin_map C 0 h1
               (fun x : T =>
                plus_list (im1l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x)))) h2 |-> x =
       fun_to_fin_map C 0 h1
         (fun x : T => plus_list (im1l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x)) |-> x).
  rewrite h3. auto. 
assert (h4: Included
            (Im C
               (fun x : T =>
                plus_set (im1 (ba_conv_fin_map_ran F) x)
                  (im1_fin (ba_conv_fin_map_ran F) x)))
            (Im C (fun x : T => plus_list_p (im1l (fpl2 F nml) %0 x)))).
  red. intros x h4. apply h2 in h4.
  destruct h4 as [x h4 ]. subst.
  pose proof (plus_list_p_eq (im1l (fpl2 F nml) %0 x)) as h5. 
  unfold ba_conv_list in h5. unfold ba_conv_type in h5. rewrite transfer_dep_eq_refl in h5. simpl.
  unfold ba_conv_fin_map_ran. unfold eq_rect_r. simpl.
  unfold Btp, btp in h5. unfold Btype_p in h5. unfold ba_conv_set.
    unfold transfer_dep. unfold eq_rect_r.
  simpl.
  rewrite <- h5 at 1.
  apply Im_intro with x; auto.
exists h4.
apply fin_map_ext_in.
intros x h5.  
rewrite <- fin_map_new_ran_compat; auto.
rewrite fun_to_fin_map_compat.
rewrite plus_list_p_eq.
assert (h6: fun_to_fin_map C 0 h1
         (fun x0 : T => plus_list (im1l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x0))
       |-> x = 
 plus_list (ba_conv_list (im1l (fpl2 F nml) %0 x))).
  rewrite fun_to_fin_map_compat. simpl. reflexivity. assumption.
rewrite <- h6.
rewrite <- h0.
rewrite <- fin_map_new_ran_compat.
reflexivity.
assumption.
assumption.
Qed.


Lemma plus_fin_pair_map2_list_compat_p :
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U} {E:Ensemble Btp}
         (pfd:Finite D)
         (F:Fin_map (cart_prod C D) E %0)
         (nml:nice_map_lists2 F),
    fin_map_eq (plus_fin_pair_map2_p pfd F)
    (fun_to_fin_map D %0 pfd (fun y:U =>(plus_list_p (im2l (fpl2 F nml) %0 y)))).
intros T U C D E h1 F nml. 
rewrite plus_fin_pair_map2_p_eq.
pose proof (@plus_fin_pair_map2_list_compat (ba_conv Bp) _ _ _ _ _ h1 (ba_conv_fin_map_ran F) nml) as h2.  
red in h2. destruct h2 as [h2 h3].
red. 
assert (h0:forall x:U, Ensembles.In D x ->
                       fin_map_new_ran (plus_fin_pair_map2 h1 (ba_conv_fin_map_ran F))
         (fin_map_fin_ran
            (fun_to_fin_map D 0 h1
               (fun x : U =>
                plus_list (im2l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x)))) h2 |-> x =
       fun_to_fin_map D 0 h1
         (fun x : U => plus_list (im2l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x)) |-> x).
  rewrite h3. auto. 
assert (h4: Included
            (Im D
               (fun x : U =>
                plus_set (im2 (ba_conv_fin_map_ran F) x)
                  (im2_fin (ba_conv_fin_map_ran F) x)))
            (Im D (fun x : U => plus_list_p (im2l (fpl2 F nml) %0 x)))).
  red. intros x h4. apply h2 in h4.
  destruct h4 as [x h4 ]. subst.
  pose proof (plus_list_p_eq (im2l (fpl2 F nml) %0 x)) as h5. 
  unfold ba_conv_list in h5. unfold ba_conv_type in h5. rewrite transfer_dep_eq_refl in h5. simpl.
  unfold ba_conv_fin_map_ran. unfold eq_rect_r. simpl.
  unfold Btp, btp in h5. unfold Btype_p in h5. unfold ba_conv_set.
    unfold transfer_dep. unfold eq_rect_r.
  simpl.
  rewrite <- h5 at 1.
  apply Im_intro with x; auto.
exists h4.
apply fin_map_ext_in.
intros x h5.  
rewrite <- fin_map_new_ran_compat; auto.
rewrite fun_to_fin_map_compat.
rewrite plus_list_p_eq.
assert (h6: fun_to_fin_map D 0 h1
         (fun x0 : U => plus_list (im2l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x0))
       |-> x = 
 plus_list (ba_conv_list (im2l (fpl2 F nml) %0 x))).
  rewrite fun_to_fin_map_compat. simpl. reflexivity. assumption.
rewrite <- h6.
rewrite <- h0.
rewrite <- fin_map_new_ran_compat.
reflexivity.
assumption.
assumption.
Qed.


Lemma times_fin_pair_map1_list_compat_p :
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U} {E:Ensemble Btp}
         (pfc:Finite C)
         (F:Fin_map (cart_prod C D) E %0)
         (nml:nice_map_lists2 F),
    fin_map_eq (times_fin_pair_map1_p pfc F)
    (fun_to_fin_map C %0 pfc (fun x:T =>(times_list_p (im1l (fpl2 F nml) %0 x)))).
intros T U C D E h1 F nml. 
rewrite times_fin_pair_map1_p_eq.
pose proof (@times_fin_pair_map1_list_compat (ba_conv Bp) _ _ _ _ _ h1 (ba_conv_fin_map_ran F) nml) as h2.  
red in h2. destruct h2 as [h2 h3].
red. 
assert (h0:forall x:T, Ensembles.In C x ->
                       fin_map_new_ran (times_fin_pair_map1 h1 (ba_conv_fin_map_ran F))
         (fin_map_fin_ran
            (fun_to_fin_map C 0 h1
               (fun x : T =>
                times_list (im1l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x)))) h2 |-> x =
       fun_to_fin_map C 0 h1
         (fun x : T => times_list (im1l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x)) |-> x).
  rewrite h3. auto. 
assert (h4: Included
            (Im C
               (fun x : T =>
                times_set (im1 (ba_conv_fin_map_ran F) x)
                  (im1_fin (ba_conv_fin_map_ran F) x)))
            (Im C (fun x : T => times_list_p (im1l (fpl2 F nml) %0 x)))).
  red. intros x h4. apply h2 in h4.
  destruct h4 as [x h4 ]. subst.
  pose proof (times_list_p_eq (im1l (fpl2 F nml) %0 x)) as h5. 
  unfold ba_conv_list in h5. unfold ba_conv_type in h5. rewrite transfer_dep_eq_refl in h5. simpl.
  unfold ba_conv_fin_map_ran. unfold eq_rect_r. simpl.
  unfold Btp, btp in h5. unfold Btype_p in h5. unfold ba_conv_set.
    unfold transfer_dep. unfold eq_rect_r.
  simpl.
  rewrite <- h5 at 1.
  apply Im_intro with x; auto.
exists h4.
apply fin_map_ext_in.
intros x h5.  
rewrite <- fin_map_new_ran_compat; auto.
rewrite fun_to_fin_map_compat.
rewrite times_list_p_eq.
assert (h6: fun_to_fin_map C 0 h1
         (fun x0 : T => times_list (im1l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x0))
       |-> x = 
 times_list (ba_conv_list (im1l (fpl2 F nml) %0 x))).
  rewrite fun_to_fin_map_compat. simpl. reflexivity. assumption.
rewrite <- h6.
rewrite <- h0.
rewrite <- fin_map_new_ran_compat.
reflexivity.
assumption.
assumption.
Qed.


Lemma times_fin_pair_map2_list_compat_p :
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U} {E:Ensemble Btp}
         (pfd:Finite D)
         (F:Fin_map (cart_prod C D) E %0)
         (nml:nice_map_lists2 F),
    fin_map_eq (times_fin_pair_map2_p pfd F)
               (fun_to_fin_map D %0 pfd (fun y:U =>(times_list_p (im2l (fpl2 F nml) %0 y)))).
intros T U C D E h1 F nml. 
rewrite times_fin_pair_map2_p_eq.
pose proof (@times_fin_pair_map2_list_compat (ba_conv Bp) _ _ _ _ _ h1 (ba_conv_fin_map_ran F) nml) as h2.  
red in h2. destruct h2 as [h2 h3].
red. 
assert (h0:forall x:U, Ensembles.In D x ->
                       fin_map_new_ran (times_fin_pair_map2 h1 (ba_conv_fin_map_ran F))
         (fin_map_fin_ran
            (fun_to_fin_map D 0 h1
               (fun x : U =>
                times_list (im2l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x)))) h2 |-> x =
       fun_to_fin_map D 0 h1
         (fun x : U => times_list (im2l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x)) |-> x).
  rewrite h3. auto. 
assert (h4: Included
            (Im D
               (fun x : U =>
                times_set (im2 (ba_conv_fin_map_ran F) x)
                  (im2_fin (ba_conv_fin_map_ran F) x)))
            (Im D (fun x : U => times_list_p (im2l (fpl2 F nml) %0 x)))).
  red. intros x h4. apply h2 in h4.
  destruct h4 as [x h4 ]. subst.
  pose proof (times_list_p_eq (im2l (fpl2 F nml) %0 x)) as h5. 
  unfold ba_conv_list in h5. unfold ba_conv_type in h5. rewrite transfer_dep_eq_refl in h5. simpl.
  unfold ba_conv_fin_map_ran. unfold eq_rect_r. simpl.
  unfold Btp, btp in h5. unfold Btype_p in h5. unfold ba_conv_set.
    unfold transfer_dep. unfold eq_rect_r.
  simpl.
  rewrite <- h5 at 1.
  apply Im_intro with x; auto.
exists h4.
apply fin_map_ext_in.
intros x h5.  
rewrite <- fin_map_new_ran_compat; auto.
rewrite fun_to_fin_map_compat.
rewrite times_list_p_eq.
assert (h6: fun_to_fin_map D 0 h1
         (fun x0 : U => times_list (im2l (fpl2 (ba_conv_fin_map_ran F) nml) 0 x0))
       |-> x = 
 times_list (ba_conv_list (im2l (fpl2 F nml) %0 x))).
  rewrite fun_to_fin_map_compat. simpl. reflexivity. assumption.
rewrite <- h6.
rewrite <- h0.
rewrite <- fin_map_new_ran_compat.
reflexivity.
assumption.
assumption.
Qed.


Lemma plus_fin_pair_map2_functional_p :
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U} {E:Ensemble Btp}
         (pf:Finite D),
    forall (F G:Fin_map (cart_prod C D) E %0),
      F = G -> fin_map_eq (plus_fin_pair_map2_p pf F)
                          (plus_fin_pair_map2_p pf G).
intros; subst.
red.
exists (inclusion_reflexive _).
apply fin_map_ext.
intros x.
symmetry.
apply fin_map_new_ran_compat.
Qed.

Definition fin_map_times_p {T:Type} {A:Ensemble T}
           {C:Ensemble Btp}
           (f:Fin_map A C %0) : Btp.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ A (fin_map_app f) h1) as h2.
refine (times_set_p _ h2).
Defined.

Definition fin_map_times_p_eq : 
  forall {T:Type} {A:Ensemble T}
         {C:Ensemble Btp}
         (f:Fin_map A C %0),
  fin_map_times_p f = fin_map_times (ba_conv_fin_map_ran f).
intros.
unfold fin_map_times_p. unfold fin_map_times.
rewrite times_set_p_eq.
reflexivity.
Qed.


Lemma fin_map_times_list_compat_p :
  forall {T:Type} {A:Ensemble T} {C:Ensemble Btp}
         (F:Fin_map A C %0) (nml:nice_map_lists F),
    fin_map_times_p F = times_list_p (n_im F nml).
intros T A C F nml.
unfold fin_map_times_p.
pose proof (n_im_im_fin_map_compat F nml) as h1.
apply times_set_compat_p'.
rewrite h1.
unfold im_fin_map.
reflexivity.
Qed.


Lemma fin_map_times_empty1_p :
  forall {T:Type} {C:Ensemble Btp} (F:Fin_map (Empty_set T) C %0),
    fin_map_times_p F = %1.
intros. rewrite fin_map_times_p_eq.
apply (@fin_map_times_empty1 (ba_conv Bp)).
Qed.


Lemma fin_map_eq_times_p : forall {T:Type} (A:Ensemble T)
                                  (C E:Ensemble Btp)
                                  (F:Fin_map A C %0)
                                  (G:Fin_map A E %0),
        fin_map_eq F G -> fin_map_times_p F = fin_map_times_p G.
intros. do 2 rewrite fin_map_times_p_eq.
apply (@fin_map_eq_times (ba_conv Bp)).
assumption.
Qed.


Lemma im2_empty1_p : 
  forall {T U:Type} (D:Ensemble U) (C:Ensemble Btp)
         (pfd:Finite D) (pfc:Finite C)
         (y:U),
    im2 (cart_empty_map11 T U Btp %0 D C pfd pfc) y =
                     Empty_set _.
intros.
apply (@im2_empty1 (ba_conv Bp)).
Qed.

Lemma im2_empty2_p : 
  forall {T U:Type} (A:Ensemble T) (C:Ensemble Btp)
         (pfa:Finite A) (pfc:Finite C)
         (y:U),
    im2 (cart_empty_map21 T U Btp %0 A C pfa pfc) y =
                     Empty_set _.
intros.
apply (@im2_empty2 (ba_conv Bp)).
Qed.


Lemma plus_fin_pair_map2_cart_empty_eq1_p :
  forall {T U:Type} (C:Ensemble Btp) (D:Ensemble U)
         (pfd:Finite D)
         (F:Fin_map (cart_prod (Empty_set T) D) C %0),

    fin_map_eq (plus_fin_pair_map2_p pfd F) (fun_to_fin_map D %0 pfd
                                              (fun x => %0)).
intros. rewrite plus_fin_pair_map2_p_eq.
apply (@plus_fin_pair_map2_cart_empty_eq1 (ba_conv Bp)).
Qed.


Lemma plus_fin_pair_map2_cart_empty_p :
  forall {T U:Type} (A:Ensemble T)  (C:Ensemble Btp) (pfe:Finite (Empty_set _))
         (F:Fin_map (cart_prod A (Empty_set U)) C %0) (y:U),
    Finite A -> plus_fin_pair_map2_p pfe F |-> y = 0.
intros. rewrite plus_fin_pair_map2_p_eq.
apply (@plus_fin_pair_map2_cart_empty (ba_conv Bp)); auto.
Qed.


Lemma plus_fin_pair_map2_cart_empty_eq2_p :
  forall {T U:Type} (A:Ensemble T) (C:Ensemble Btp)
         (pfa:Finite A)
         (pfe:Finite (Empty_set U))
         (F:Fin_map (cart_prod A (Empty_set U)) C %0),

    fin_map_eq (plus_fin_pair_map2_p pfe F)
               (empty_map1 U Btp %0 _ (fin_map_fin_ran F)).
intros. rewrite plus_fin_pair_map2_p_eq.
apply (@plus_fin_pair_map2_cart_empty_eq2 (ba_conv Bp)); auto.
Qed.


Lemma fin_map_times_sing_p : 
  forall {T:Type} (A:Ensemble T) (pf:Finite A)
         (val:Btp), A <> Empty_set _ ->
                   fin_map_times_p (fin_map_sing A pf %0 val) = val.
intros. rewrite fin_map_times_p_eq.
apply (@fin_map_times_sing (ba_conv Bp)); auto.
Qed.



Lemma fin_map_times_cart_empty11_p :
  forall {T U:Type} (C:Ensemble Btp) (D:Ensemble U)
        (pfd:Finite D)
        (pfde:D <> Empty_set _)
         (F:Fin_map (cart_prod (Empty_set T) D) C %0),
    fin_map_times_p
                  (plus_fin_pair_map2_p pfd F) = %0.
intros. rewrite fin_map_times_p_eq.
apply (@fin_map_times_cart_empty11 (ba_conv Bp)); auto.
Qed.


Lemma fin_map_times_cart_empty21_p :
  forall {T U:Type} (A:Ensemble T) (C:Ensemble Btp) (pfa:Finite A)
         (pfe:Finite (Empty_set U))
         (F:Fin_map (cart_prod A (Empty_set U)) C %0),
    fin_map_times_p (plus_fin_pair_map2_p pfe F) = %1.
intros. rewrite fin_map_times_p_eq.
apply (@fin_map_times_cart_empty21 (ba_conv Bp)); auto.
Qed.

Lemma fin_map_times_empty2_p :
  forall {T:Type} {A:Ensemble T} (F:Fin_map A (Empty_set Btp) %0),
    fin_map_times_p F = %1.
intros. rewrite fin_map_times_p_eq.
apply (@fin_map_times_empty2 (ba_conv Bp)); auto.
Qed.


Definition fin_map_plus_p {T:Type} {A:Ensemble T} {C:Ensemble Btp}
           (f:Fin_map A C %0) : Btp.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ A (fin_map_app f) h1) as h2.
refine (plus_set_p _ h2).
Defined.


Definition fin_map_plus_p_eq :
  forall {T:Type} {A:Ensemble T}
         {C:Ensemble Btp}
         (f:Fin_map A C %0),
  fin_map_plus_p f = fin_map_plus (ba_conv_fin_map_ran f).
intros.
unfold fin_map_plus_p. unfold fin_map_plus.
rewrite plus_set_p_eq.
reflexivity.
Qed.


Lemma fin_map_plus_list_compat_p :
  forall {T:Type} {A:Ensemble T} {C:Ensemble Btp}
         (F:Fin_map A C %0) (nml:nice_map_lists F),
    fin_map_plus_p F = plus_list_p (n_im F nml).
intros. rewrite fin_map_plus_p_eq. rewrite plus_list_p_eq.
apply (@fin_map_plus_list_compat (ba_conv Bp)).
Qed.


Lemma fin_map_plus_empty1_p :
  forall {T:Type} {C:Ensemble Btp} (F:Fin_map (Empty_set T) C %0),
    fin_map_plus_p F = %0.
intros. rewrite fin_map_plus_p_eq.
apply (@fin_map_plus_empty1 (ba_conv Bp)).
Qed.


Lemma fin_map_plus_empty2_p :
  forall {T:Type} {A:Ensemble T} (F:Fin_map A (Empty_set Btp) %0),
    fin_map_plus_p F = %0.
intros. rewrite fin_map_plus_p_eq.
apply (@fin_map_plus_empty2 (ba_conv Bp)).
Qed.


Lemma fin_map_eq_plus_p : forall {T:Type} (A:Ensemble T)
                                 (C E:Ensemble Btp)
                                 (F:Fin_map A C %0)
                                 (G:Fin_map A E %0),
        fin_map_eq F G -> fin_map_plus_p F = fin_map_plus_p G.
intros. do 2 rewrite fin_map_plus_p_eq.
apply (@fin_map_eq_plus (ba_conv Bp)); auto.
Qed.


Definition times_plus_fin_pair_map1_p
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Btp} (pfc:Finite C)
           (F:Fin_map (cart_prod C D) E %0) :=
  fin_map_times_p (plus_fin_pair_map1_p pfc F).

Lemma times_plus_fin_pair_map1_p_eq : 
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U}
         {E:Ensemble Btp} (pfc:Finite C)
         (F:Fin_map (cart_prod C D) E %0),
    times_plus_fin_pair_map1_p pfc F =
    times_plus_fin_pair_map1 pfc (ba_conv_fin_map_ran F).
intros. unfold times_plus_fin_pair_map1_p, times_plus_fin_pair_map1.
rewrite fin_map_times_p_eq.
reflexivity.
Qed.


Definition times_plus_fin_pair_map2_p
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Btp} (pfd:Finite D)
           (F:Fin_map (cart_prod C D) E %0) :=
  fin_map_times_p (plus_fin_pair_map2_p pfd F).


Definition times_plus_fin_pair_map2_p_eq : 
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U}
         {E:Ensemble Btp} (pfd:Finite D)
         (F:Fin_map (cart_prod C D) E %0),
    times_plus_fin_pair_map2_p pfd F =
    times_plus_fin_pair_map2 pfd (ba_conv_fin_map_ran F).
intros. unfold times_plus_fin_pair_map2_p, times_plus_fin_pair_map2.
rewrite fin_map_times_p_eq.
reflexivity.
Qed.


Definition plus_times_fin_pair_map1_p
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Btp} (pfc:Finite C)
           (F:Fin_map (cart_prod C D) E %0) :=
  fin_map_plus_p (times_fin_pair_map1_p pfc F).

Lemma plus_times_fin_pair_map1_p_eq : 
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U}
         {E:Ensemble Btp} (pfc:Finite C)
         (F:Fin_map (cart_prod C D) E %0),
    plus_times_fin_pair_map1_p pfc F =
    plus_times_fin_pair_map1 pfc (ba_conv_fin_map_ran F).
intros. unfold plus_times_fin_pair_map1_p, plus_times_fin_pair_map1.
rewrite fin_map_plus_p_eq.
reflexivity.
Qed.


Definition plus_times_fin_pair_map2_p
           {T U:Type} {C:Ensemble T} {D:Ensemble U}
           {E:Ensemble Btp} (pfd:Finite D)
           (F:Fin_map (cart_prod C D) E %0) :=
  fin_map_plus_p (times_fin_pair_map2_p pfd F).


Lemma plus_times_fin_pair_map2_p_eq : 
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U}
         {E:Ensemble Btp} (pfd:Finite D)
         (F:Fin_map (cart_prod C D) E %0),
    plus_times_fin_pair_map2_p pfd F =
    plus_times_fin_pair_map2 pfd (ba_conv_fin_map_ran F).
intros. unfold plus_times_fin_pair_map2_p, plus_times_fin_pair_map2.
rewrite fin_map_plus_p_eq.
reflexivity.
Qed.


Lemma times_plus_fin_pair_map1_list_compat_p :
  forall {T U:Type} {C:Ensemble T} {D:Ensemble U}
         {E:Ensemble Btp} (pfc:Finite C)
         (F:Fin_map (cart_prod C D) E %0)
         (nml:nice_map_lists (plus_fin_pair_map1_p pfc F)),
    times_plus_fin_pair_map1_p pfc F =
    times_list_p (n_im _ nml).
intros. rewrite times_plus_fin_pair_map1_p_eq.
rewrite times_list_p_eq.
apply (@times_plus_fin_pair_map1_list_compat (ba_conv Bp)).
Qed.


Lemma plus_fun_fin_map_to_fun_comm_p :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (F:Fin_map (cart_prod C D) E %0)
    (pfc:Finite C) (x:T) (lb:list U),
    list_to_set lb = D -> Ensembles.In C x ->
    plus_fun_p (fun y:U => fin_map_to_fun F (x, y)) lb =
    fun_to_fin_map C %0 pfc (fun x:T => plus_set_p (im1 F x) (im1_fin F x)) |-> x.
intros T U C D E F h1 x lb h2 h3.
pose proof (plus_fun_fin_map_to_fun_comm (ba_conv_fin_map_ran F) h1 x lb h2 h3) as h4.
rewrite plus_fun_p_eq.
unfold ba_conv_ind. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
unfold ba_conv_fin_map_ran in h4. unfold eq_rect_r in h4.  simpl in h4. 
unfold Btp, btp. unfold Btype_p.  unfold ba_conv_set in h4.
unfold transfer_dep in h4.  unfold eq_rect_r in h4. 
simpl in h4. 
rewrite h4 at 1.
assert (h5:(fun x0 : T => @plus_set (ba_conv Bp) (im1 F x0) (im1_fin F x0)) =
           (fun x0 : T => plus_set_p (im1 F x0) (im1_fin F x0))).
  apply functional_extensionality.
  intro. rewrite plus_set_p_eq. reflexivity.
unfold Btp, btp, Btype_p in h5. (* unfold Btype_p in h5.*)
unfold bt.

simpl. simpl in h5.
rewrite h5.
reflexivity.
Qed.

Lemma plus_fun_fin_map_to_fun_comm_p' :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (F:Fin_map (cart_prod C D) E %0)
    (pfd:Finite D) (y:U) (la:list T),
    list_to_set la = C -> Ensembles.In D y ->
    plus_fun_p (fun x:T => fin_map_to_fun F (x, y)) la =
    fun_to_fin_map D %0 pfd (fun x:U => plus_set_p (im2 F x) (im2_fin F x)) |-> y.
intros T U C D E F h1 y la h2 h3.
pose proof (plus_fun_fin_map_to_fun_comm' (ba_conv_fin_map_ran F) h1 y la h2 h3) as h4.
rewrite plus_fun_p_eq. 
unfold ba_conv_ind. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
unfold ba_conv_fin_map_ran in h4. unfold eq_rect_r in h4.  simpl in h4. 
unfold Btp, btp. unfold Btype_p.  unfold ba_conv_set in h4.
unfold transfer_dep in h4.  unfold eq_rect_r in h4. 
simpl in h4. 
rewrite h4 at 1.
assert (h5:(fun x : U => @plus_set (ba_conv Bp) (im2 F x) (im2_fin F x)) =
           (fun x : U => plus_set_p (im2 F x) (im2_fin F x))).
  apply functional_extensionality.
  intro x. rewrite plus_set_p_eq. reflexivity.
unfold Btp, btp in h5. unfold Btype_p in h5.
unfold bt.
simpl. simpl in h5.
rewrite h5.
reflexivity.
Qed.

Lemma times_plus_fin_pair_map1_list_compat_p' :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (pfc:Finite C)
    (F:Fin_map (cart_prod C D) E %0)
    (nml2:nice_map_lists2 F)
    (nml:nice_map_lists (plus_fin_pair_map1_p pfc F)),
    (n_la2 _ nml2) = (n_la _ nml) ->
    times_list_p (n_im _ nml) = times_plus_fun1_p (n_la2 _ nml2) (n_lb2 _ nml2) (f_no_pr (fin_map_to_fun F)).
intros.
rewrite times_list_p_eq, times_plus_fun1_p_eq.
apply (@times_plus_fin_pair_map1_list_compat' (ba_conv Bp)); auto.
Qed.

Lemma times_plus_fin_pair_map2_list_compat_p :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (pfd:Finite D)
         (F:Fin_map (cart_prod C D) E %0)
         (nml:nice_map_lists (plus_fin_pair_map2_p pfd F)),
    times_plus_fin_pair_map2_p pfd F =
    times_list_p (n_im _ nml).
intros.
rewrite times_plus_fin_pair_map2_p_eq, times_list_p_eq.
apply (@times_plus_fin_pair_map2_list_compat (ba_conv Bp)).
Qed.

Lemma times_plus_fin_pair_map2_list_compat_p' :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (pfd:Finite D)
    (F:Fin_map (cart_prod C D) E %0)
    (nml2:nice_map_lists2 F)
    (nml:nice_map_lists (plus_fin_pair_map2_p pfd F)),
    (n_lb2 _ nml2) = (n_la _ nml) ->
    times_list_p (n_im _ nml) = times_plus_fun2_p (n_la2 _ nml2) (n_lb2 _ nml2) (f_no_pr (fin_map_to_fun F)).
intros.
rewrite times_list_p_eq, times_plus_fun2_p_eq.
apply (@times_plus_fin_pair_map2_list_compat' (ba_conv Bp)); auto.
Qed.


Lemma plus_times_fin_pair_map1_list_compat_p :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (pfc:Finite C)
    (F:Fin_map (cart_prod C D) E %0)
    (nml:nice_map_lists (times_fin_pair_map1_p pfc F)),
    plus_times_fin_pair_map1_p pfc F =
    plus_list_p (n_im _ nml).
intros T U C D E pfc F nml.
apply fin_map_plus_list_compat_p.
Qed.



Lemma times_fun_fin_map_to_fun_comm_p :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (F:Fin_map (cart_prod C D) E %0)
    (pfc:Finite C) (x:T) (lb:list U),
    list_to_set lb = D -> Ensembles.In C x ->
    times_fun_p (fun y:U => fin_map_to_fun F (x, y)) lb =
    fun_to_fin_map C %0 pfc (fun x:T => times_set_p (im1 F x) (im1_fin F x)) |-> x.
intros T U C D E F h1 x lb h2 h3.
pose proof (times_fun_fin_map_to_fun_comm (ba_conv_fin_map_ran F) h1 x lb h2 h3) as h4.
rewrite times_fun_p_eq.
unfold ba_conv_ind. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
unfold ba_conv_fin_map_ran in h4. unfold eq_rect_r in h4.  simpl in h4. 
unfold Btp, btp. unfold Btype_p.  unfold ba_conv_set in h4.
unfold transfer_dep in h4.  unfold eq_rect_r in h4. 
simpl in h4. 
rewrite h4 at 1.
assert (h5:(fun x0 : T => @times_set (ba_conv Bp) (im1 F x0) (im1_fin F x0)) = 
           (fun x0 : T => times_set_p (im1 F x0) (im1_fin F x0))).
  apply functional_extensionality.
  intro. rewrite times_set_p_eq. reflexivity.
unfold Btp, btp in h5. unfold Btype_p in h5.
unfold bt.
simpl in h5. simpl.
rewrite h5.
reflexivity.
Qed.


Lemma plus_times_fin_pair_map1_list_compat_p' :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (pfc:Finite C)
    (F:Fin_map (cart_prod C D) E %0)
    (nml2:nice_map_lists2 F)
    (nml:nice_map_lists (times_fin_pair_map1_p pfc F)),
    (n_la2 _ nml2) = (n_la _ nml) ->
    plus_list_p (n_im _ nml) = plus_times_fun1_p (n_la2 _ nml2) (n_lb2 _ nml2) (f_no_pr (fin_map_to_fun F)).
intros. rewrite plus_list_p_eq, plus_times_fun1_p_eq.
apply (@plus_times_fin_pair_map1_list_compat' (ba_conv Bp)); auto.
Qed.



Lemma plus_times_fin_pair_map2_list_compat_p :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (pfd:Finite D)
    (F:Fin_map (cart_prod C D) E %0)
    (nml:nice_map_lists (times_fin_pair_map2_p pfd F)),
    plus_times_fin_pair_map2_p pfd F =
    plus_list_p (n_im _ nml).
intros T U C D E pfc F nml.
apply fin_map_plus_list_compat_p.
Qed.


Lemma times_fun_fin_map_to_fun_comm_p' :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (F:Fin_map (cart_prod C D) E %0)
    (pfd:Finite D) (y:U) (la:list T),
    list_to_set la = C -> Ensembles.In D y ->
    times_fun_p (fun x:T => fin_map_to_fun F (x, y)) la =
    fun_to_fin_map D %0 pfd (fun x:U => times_set_p (im2 F x) (im2_fin F x)) |-> y.
intros T U C D E F h1 y la h2 h3.
rewrite times_fun_p_eq.
pose proof (times_fun_fin_map_to_fun_comm' (ba_conv_fin_map_ran F) h1 y la h2 h3) as h4. 
unfold ba_conv_ind. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
unfold ba_conv_fin_map_ran in h4. unfold eq_rect_r in h4.  simpl in h4. 
unfold Btp, bt. unfold Btype_p.  unfold ba_conv_set in h4.
unfold transfer_dep in h4.  unfold eq_rect_r in h4. 
simpl in h4. 
rewrite h4 at 1.
assert (h5: (fun x : U => @times_set (ba_conv Bp) (im2 F x) (im2_fin F x)) =
            (fun x : U => times_set_p (im2 F x) (im2_fin F x))).
  apply functional_extensionality. intro. rewrite times_set_p_eq.
  reflexivity.
unfold Btp, btp in h5. unfold Btype_p in h5.
unfold bt. simpl in h5. simpl.
rewrite h5.
reflexivity.
Qed.


Lemma plus_times_fin_pair_map2_list_compat_p' :
  forall
    {T U:Type} {C:Ensemble T} {D:Ensemble U}
    {E:Ensemble Btp} (pfd:Finite D)
    (F:Fin_map (cart_prod C D) E %0)
    (nml2:nice_map_lists2 F)
    (nml:nice_map_lists (times_fin_pair_map2_p pfd F)),
    (n_lb2 _ nml2) = (n_la _ nml) ->
    plus_list_p (n_im _ nml) = plus_times_fun2_p (n_la2 _ nml2) (n_lb2 _ nml2) (f_no_pr (fin_map_to_fun F)).
intros. rewrite plus_list_p_eq, plus_times_fun2_p_eq.
apply (@plus_times_fin_pair_map2_list_compat' (ba_conv Bp)); auto.
Qed.


Definition times_fun_fin_map1_p
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
           (p:(T*U)->Btp) (a:Fin_map I J def) : Btp.
pose proof (fin_map_fin_dom a) as h1.
pose proof (fin_map_fin_ran a) as h2.
pose proof (cart_prod_fin _ _ h1 h2) as h4.
pose (fun_to_fin_map _ 0 h4 (ba_conv_ind p)) as P.
refine
  (fin_map_times_p
     (fun_to_fin_map I 0 h1
                     (fun i:T => (P |-> (i, (a |-> i)))))).
Defined.


Lemma times_fun_fin_map1_p_eq : 
  forall {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
         (p:(T*U)->Btp) (a:Fin_map I J def),
    times_fun_fin_map1_p p a =
    times_fun_fin_map1 (ba_conv_ind p) a.
intros.
unfold times_fun_fin_map1_p, times_fun_fin_map1.
rewrite fin_map_times_p_eq.
reflexivity.
Qed.


Definition times_fun_fin_map2_p
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
           (p:(T*U)->Btp) (a:Fin_map J I def) : Btp.
pose proof (fin_map_fin_dom a) as h1.
pose proof (fin_map_fin_ran a) as h2.
pose proof (cart_prod_fin _ _ h2 h1) as h4.
pose (fun_to_fin_map _ 0 h4 (ba_conv_ind p)) as P.
refine
  (fin_map_times_p
     (fun_to_fin_map J 0 h1
                     (fun j:U => (P |-> (a |-> j, j))))).
Defined.



Lemma times_fun_fin_map2_p_eq : 
  forall {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
         (p:(T*U)->Btp) (a:Fin_map J I def),
    times_fun_fin_map2_p p a =
    times_fun_fin_map2 (ba_conv_ind p) a.
intros. unfold times_fun_fin_map2_p, times_fun_fin_map2.
rewrite fin_map_times_p_eq.
reflexivity.
Qed.



Definition times_fun_fin_map1_l_p
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
           (p:(T*U)->Btp) (a:Fin_map I J def)
           (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                              (cart_prod_fin _ _ (fin_map_fin_dom a) (fin_map_fin_ran a)) (ba_conv_ind p)))
  :=
times_list_p (map (fun i:T => (p (i, a |-> i))) (n_la2 _ nml2)).



Lemma times_fun_fin_map1_l_p_eq : 
  forall {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
         (p:(T*U)->Btp) (a:Fin_map I J def)
         (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                                               (cart_prod_fin _ _ (fin_map_fin_dom a) (fin_map_fin_ran a)) (ba_conv_ind p))),
    times_fun_fin_map1_l_p p a nml2 =
    times_fun_fin_map1_l (ba_conv_ind p) a nml2.
intros.
unfold times_fun_fin_map1_l_p. unfold times_fun_fin_map1_l.
rewrite times_list_p_eq.
reflexivity.
Qed.


Definition times_fun_fin_map2_l_p
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
           (p:(T*U)->Btp) (a:Fin_map J I def)
           (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                              (cart_prod_fin _ _ (fin_map_fin_ran a) (fin_map_fin_dom a)) (ba_conv_ind p)))
  :=
times_list_p (map (fun j:U => (p (a |-> j, j))) (n_lb2 _ nml2)).



Lemma times_fun_fin_map2_l_p_eq : 
  forall {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
         (p:(T*U)->Btp) (a:Fin_map J I def)
         (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                                               (cart_prod_fin _ _ (fin_map_fin_ran a) (fin_map_fin_dom a)) (ba_conv_ind p))),
  times_fun_fin_map2_l_p p a nml2 =
  times_fun_fin_map2_l (ba_conv_ind p) a nml2.
intros. unfold times_fun_fin_map2_l_p, times_fun_fin_map2_l.
rewrite times_list_p_eq.
reflexivity.
Qed.


Lemma im_sing_times_fun_fin_map1_p :
  forall
    {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
    (p:T*U->Btp) (a:Fin_map I J def),
    Im (Singleton a)
       (times_fun_fin_map1_p p) =
    Singleton
      ((times_fun_fin_map1_p p)
         a).
intros T U I J def p a. rewrite times_fun_fin_map1_p_eq.
pose proof (im_sing_times_fun_fin_map1 (ba_conv_ind p) a) as h1.
simpl in h1.
unfold Btp, btp.  unfold Btype_p.  rewrite <- h1 at 1.
f_equal.
apply functional_extensionality.
intro F.
rewrite times_fun_fin_map1_p_eq.
reflexivity.
Qed.



Lemma times_fun_fin_map1_empty_p :
  forall {T U:Type} (def:U) (p:(T*U)->Btp),
    times_fun_fin_map1_p p (empty_map T U def) = %1.
intros.
rewrite times_fun_fin_map1_p_eq.
apply (@times_fun_fin_map1_empty (ba_conv Bp)).
Qed.


Lemma times_fun_fin_map2_empty_p :
  forall
    {T U:Type} (def:T)
    (p:(T*U)->Btp),
    times_fun_fin_map2_p p (empty_map U T def) = %1.
intros.
rewrite times_fun_fin_map2_p_eq.
apply (@times_fun_fin_map2_empty (ba_conv Bp)).
Qed.


Lemma times_fun_fin_map1_empty1_p :
  forall
    {T U:Type} (def:U)
    (p:(T*U)->Btp)
    (J:Ensemble U) (pf:Finite J),
    times_fun_fin_map1_p p (empty_map1 T U def J pf) = %1.
intros. rewrite times_fun_fin_map1_p_eq.
apply (@times_fun_fin_map1_empty1 (ba_conv Bp)).
Qed.


Lemma times_fun_fin_map2_empty1_p :
  forall
    {T U:Type}  (def:T)
    (p:(T*U)->Btp)
    (I:Ensemble T) (pf:Finite I),
    times_fun_fin_map2_p p (empty_map1 U T def I pf) = %1.
intros. rewrite times_fun_fin_map2_p_eq.
apply (@times_fun_fin_map2_empty1 (ba_conv Bp)).
Qed.


Lemma times_fun_fin_map1_list_compat_p :
  forall
    {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
    (p:(T*U)->Btp) (a:Fin_map I J def)
    (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                                          (cart_prod_fin _ _ (fin_map_fin_dom a) (fin_map_fin_ran a)) (ba_conv_ind p))),
    times_fun_fin_map1_p p a =
    times_fun_fin_map1_l_p p a nml2.
intros.
rewrite times_fun_fin_map1_p_eq, times_fun_fin_map1_l_p_eq.
apply (@times_fun_fin_map1_list_compat (ba_conv Bp)).
Qed.


Lemma times_fun_fin_map2_list_compat_p :
  forall
    {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
    (p:(T*U)->Btp) (a:Fin_map J I def)
    (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                              (cart_prod_fin _ _ (fin_map_fin_ran a) (fin_map_fin_dom a)) (ba_conv_ind p))),
    times_fun_fin_map2_p p a =
    times_fun_fin_map2_l_p p a nml2.
intros.
rewrite times_fun_fin_map2_p_eq, times_fun_fin_map2_l_p_eq.
apply (@times_fun_fin_map2_list_compat (ba_conv Bp)).
Qed.


Definition plus_fun_fin_map1_p
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
           (p:(T*U)->Btp) (a:Fin_map I J def) : Btp.
pose proof (fin_map_fin_dom a) as h1.
pose proof (fin_map_fin_ran a) as h2.
pose proof (cart_prod_fin _ _ h1 h2) as h4.
pose (fun_to_fin_map _ 0 h4 (ba_conv_ind p)) as P.
refine
  (fin_map_plus_p
     (fun_to_fin_map I 0 h1
                     (fun i:T => (P |-> (i, (a |-> i)))))).
Defined.

Lemma plus_fun_fin_map1_p_eq : 
  forall {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
         (p:(T*U)->Btp) (a:Fin_map I J def),
    plus_fun_fin_map1_p p a =
    plus_fun_fin_map1 (ba_conv_ind p) a.
intros. unfold plus_fun_fin_map1_p, plus_fun_fin_map1.
rewrite fin_map_plus_p_eq.
reflexivity.
Qed.


Definition plus_fun_fin_map2_p
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
           (p:(T*U)->Btp) (a:Fin_map J I def) : Btp.
pose proof (fin_map_fin_dom a) as h1.
pose proof (fin_map_fin_ran a) as h2.
pose proof (cart_prod_fin _ _ h2 h1) as h4.
pose (fun_to_fin_map _ 0 h4 (ba_conv_ind p)) as P.
refine
  (fin_map_plus_p
     (fun_to_fin_map J 0 h1
                     (fun j:U => (P |-> (a |-> j, j))))).
Defined.


Lemma plus_fun_fin_map2_p_eq : 
  forall {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
         (p:(T*U)->Btp) (a:Fin_map J I def),
    plus_fun_fin_map2_p p a =
    plus_fun_fin_map2 (ba_conv_ind p) a.
intros. unfold plus_fun_fin_map2_p, plus_fun_fin_map2.
rewrite fin_map_plus_p_eq.
reflexivity.
Qed.


Lemma plus_fun_fin_map1_empty_p :
  forall {T U:Type} (def:U) (p:(T*U)->Btp),
    plus_fun_fin_map1_p p (empty_map T U def) = %0.
intros.
rewrite plus_fun_fin_map1_p_eq.
apply (@plus_fun_fin_map1_empty (ba_conv Bp)).
Qed.


Lemma plus_fun_fin_map1_empty1_p :
  forall
    {T U:Type} (def:U) (p:(T*U)->Btp)
    (J:Ensemble U) (pf:Finite J),
    plus_fun_fin_map1_p p (empty_map1 T U def J pf) = %0.
intros.
rewrite plus_fun_fin_map1_p_eq.
apply (@plus_fun_fin_map1_empty1 (ba_conv Bp)).
Qed.


Definition plus_fun_fin_map1_l_p
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
           (p:(T*U)->Btp) (a:Fin_map I J def)
           (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                              (cart_prod_fin _ _ (fin_map_fin_dom a) (fin_map_fin_ran a)) (ba_conv_ind p)))
  :=
plus_list_p (map (fun i:T => (p (i, a |-> i))) (n_la2 _ nml2)).


Lemma plus_fun_fin_map1_l_p_eq : 
  forall {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
         (p:(T*U)->Btp) (a:Fin_map I J def)
         (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                                               (cart_prod_fin _ _ (fin_map_fin_dom a) (fin_map_fin_ran a)) (ba_conv_ind p))),
    plus_fun_fin_map1_l_p p a nml2 =
    plus_fun_fin_map1_l (ba_conv_ind p) a nml2.
intros.
unfold plus_fun_fin_map1_l_p, plus_fun_fin_map1_l.
rewrite plus_list_p_eq.
reflexivity.
Qed.


Definition plus_fun_fin_map2_l_p
           {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
           (p:(T*U)->Btp) (a:Fin_map J I def)
           (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                              (cart_prod_fin _ _ (fin_map_fin_ran a) (fin_map_fin_dom a)) (ba_conv_ind p)))
  :=
plus_list_p (map (fun j:U => (p (a |-> j, j))) (n_lb2 _ nml2)).


Lemma plus_fun_fin_map2_l_p_eq : 
  forall {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
         (p:(T*U)->Btp) (a:Fin_map J I def)
         (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                                               (cart_prod_fin _ _ (fin_map_fin_ran a) (fin_map_fin_dom a)) (ba_conv_ind p))),
    plus_fun_fin_map2_l_p p a nml2 =
    plus_fun_fin_map2_l (ba_conv_ind p) a nml2.
intros. unfold plus_fun_fin_map2_l_p, plus_fun_fin_map2_l.
rewrite plus_list_p_eq.
reflexivity.
Qed.

Lemma plus_fun_fin_map2_empty_p :
  forall
    {T U:Type} (def:T) (p:(T*U)->Btp),
    plus_fun_fin_map2_p p (empty_map U T def) = %0.
intros. rewrite plus_fun_fin_map2_p_eq.
apply (@plus_fun_fin_map2_empty (ba_conv Bp)).
Qed.


Lemma plus_fun_fin_map2_empty1_p :
  forall
    {T U:Type} (def:T) (p:(T*U)->Btp)
    (I:Ensemble T) (pf:Finite I),
    plus_fun_fin_map2_p p (empty_map1 U T def I pf) = %0.
intros. rewrite plus_fun_fin_map2_p_eq.
apply (@plus_fun_fin_map2_empty1 (ba_conv Bp)).
Qed.


Lemma plus_fun_fin_map1_list_compat_p :
  forall
    {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
         (p:(T*U)->Btp) (a:Fin_map I J def)
         (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                                               (cart_prod_fin _ _ (fin_map_fin_dom a) (fin_map_fin_ran a)) (ba_conv_ind p))),
    plus_fun_fin_map1_p p a =
    plus_fun_fin_map1_l_p p a nml2.
intros.
rewrite plus_fun_fin_map1_p_eq, plus_fun_fin_map1_l_p_eq.
apply (@plus_fun_fin_map1_list_compat (ba_conv Bp)).
Qed.


Lemma plus_fun_fin_map2_list_compat_p :
  forall
    {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:T}
    (p:(T*U)->Btp) (a:Fin_map J I def)
    (nml2:nice_map_lists2 (fun_to_fin_map _ 0
                                          (cart_prod_fin _ _ (fin_map_fin_ran a) (fin_map_fin_dom a)) (ba_conv_ind p))),
    plus_fun_fin_map2_p p a =
    plus_fun_fin_map2_l_p p a nml2.
intros. rewrite plus_fun_fin_map2_p_eq, plus_fun_fin_map2_l_p_eq.
apply (@plus_fun_fin_map2_list_compat (ba_conv Bp)).
Qed.


Definition plus_times_fun_all_maps1_p
           {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
           (pfi:Finite I) (pfj:Finite J)
           (p:(T*U)->Btp): Btp

:= let f:= (@times_fun_fin_map1_p T U I J def p) in
   let S := Full_set (Fin_map I J def) in
   plus_set_p (Im S f) (finite_image _ _ S f (finite_fin_maps _ _ def pfi pfj)).

Lemma plus_times_fun_all_maps1_p_eq : 
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
         (pfi:Finite I) (pfj:Finite J)
         (p:(T*U)->Btp),
    plus_times_fun_all_maps1_p I J def pfi pfj p =
    plus_times_fun_all_maps1 I J def pfi pfj (ba_conv_ind p).
intros. unfold plus_times_fun_all_maps1_p, plus_times_fun_all_maps1.
rewrite plus_set_p_eq.
assert (h1:(ba_conv_set
           (Im (Full_set (Fin_map I J def)) (times_fun_fin_map1_p p))) =  (Im (Full_set (Fin_map I J def)) (times_fun_fin_map1 (ba_conv_ind p)))).
  apply Extensionality_Ensembles.
  red. split. red.
  intros x h1.
  destruct h1 as [x h1]. subst. apply Im_intro with x; auto.
  rewrite times_fun_fin_map1_p_eq. reflexivity.
  red. intros x h1.
  destruct h1 as [x h1]. subst.
  apply Im_intro with x; auto.
  rewrite times_fun_fin_map1_p_eq. reflexivity.
apply (plus_set_functional _ _ _ _ h1).
Qed.


Definition plus_times_fun_all_maps2_p
           {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
           (pfi:Finite I)
           (pfj:Finite J) (p:(T*U)->Btp): Btp

:= let f:= (@times_fun_fin_map2_p T U I J def p) in
   let S := Full_set (Fin_map J I def) in
   plus_set_p (Im S f) (finite_image _ _ S f (finite_fin_maps _ _ def pfj pfi)).


Lemma plus_times_fun_all_maps2_p_eq : 
  forall 
    {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
    (pfi:Finite I)
    (pfj:Finite J) (p:(T*U)->Btp),
    plus_times_fun_all_maps2_p I J def pfi pfj p =
    plus_times_fun_all_maps2   I J def pfi pfj (ba_conv_ind p).
intros. unfold plus_times_fun_all_maps2_p, plus_times_fun_all_maps1.
rewrite plus_set_p_eq.
assert (h1:(ba_conv_set
           (Im (Full_set (Fin_map J I def)) (times_fun_fin_map2_p p))) =  (Im (Full_set (Fin_map J I def)) (times_fun_fin_map2 (ba_conv_ind p)))).
  apply Extensionality_Ensembles.
  red. split. red.
  intros x h1.
  destruct h1 as [x h1]. subst. apply Im_intro with x; auto.
  rewrite times_fun_fin_map2_p_eq. reflexivity.
  red. intros x h1.
  destruct h1 as [x h1]. subst.
  apply Im_intro with x; auto.
  rewrite times_fun_fin_map2_p_eq. reflexivity.
apply (plus_set_functional _ _ _ _ h1).
Qed.


Definition times_plus_fun_all_maps1_p
           {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
           (pfi:Finite I) (pfj:Finite J) (p:T*U->Btp) : Btp

:= let f:= (@plus_fun_fin_map1_p T U I J def p) in
   let S := Full_set (Fin_map I J def) in
   times_set_p (Im S f) (finite_image _ _ S f (finite_fin_maps _ _ def pfi pfj)).


Lemma times_plus_fun_all_maps1_p_eq : 
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
         (pfi:Finite I) (pfj:Finite J) (p:T*U->Btp),
    times_plus_fun_all_maps1_p I J def pfi pfj p =
    times_plus_fun_all_maps1   I J def pfi pfj (ba_conv_ind p).
intros. unfold times_plus_fun_all_maps1_p, times_plus_fun_all_maps1.
rewrite times_set_p_eq.
assert (h1:(ba_conv_set
           (Im (Full_set (Fin_map I J def)) (plus_fun_fin_map1_p p))) =  (Im (Full_set (Fin_map I J def)) (plus_fun_fin_map1 (ba_conv_ind p)))).
  apply Extensionality_Ensembles.
  red. split. red.
  intros x h1.
  destruct h1 as [x h1]. subst. apply Im_intro with x; auto.
  rewrite plus_fun_fin_map1_p_eq. reflexivity.
  red. intros x h1.
  destruct h1 as [x h1]. subst.
  apply Im_intro with x; auto.
  rewrite plus_fun_fin_map1_p_eq. reflexivity.
apply (times_set_functional _ _ _ _ h1).
Qed.


Definition times_plus_fun_all_maps2_p
           {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
           (pfi:Finite I) (pfj:Finite J) (p:T*U->Btp) : Btp

:= let f:= (@plus_fun_fin_map2_p T U I J def p) in
   let S := Full_set (Fin_map J I def) in
   times_set_p (Im S f) (finite_image _ _ S f (finite_fin_maps _ _ def pfj pfi)).

Lemma times_plus_fun_all_maps2_p_eq : 
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
         (pfi:Finite I) (pfj:Finite J) (p:T*U->Btp),
    times_plus_fun_all_maps2_p I J def pfi pfj p =
    times_plus_fun_all_maps2   I J def pfi pfj (ba_conv_ind p).
intros.
unfold times_plus_fun_all_maps2_p, times_plus_fun_all_maps2.
rewrite times_set_p_eq.
assert (h1:(ba_conv_set
           (Im (Full_set (Fin_map J I def)) (plus_fun_fin_map2_p p))) =  (Im (Full_set (Fin_map J I def)) (plus_fun_fin_map2 (ba_conv_ind p)))).
  apply Extensionality_Ensembles.
  red. split. red.
  intros x h1.
  destruct h1 as [x h1]. subst. apply Im_intro with x; auto.
  rewrite plus_fun_fin_map2_p_eq. reflexivity.
  red. intros x h1.
  destruct h1 as [x h1]. subst.
  apply Im_intro with x; auto.
  rewrite plus_fun_fin_map2_p_eq. reflexivity.
apply (times_set_functional _ _ _ _ h1).
Qed.

Lemma complete_dist_list_times1_p' :
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj)
         (p:T->U->Btp) (def:U),
      times_plus_fun1_p li lj p = plus_times_all_funs1_p li lj pfi pfj p def.
intros. rewrite times_plus_fun1_p_eq, plus_times_all_funs1_p_eq.
apply (@complete_dist_list_times1' (ba_conv Bp)).
Qed.

Lemma complete_dist_list_times2_p' :
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj)
         (p:T->U->Btp) (def:T),
      times_plus_fun2_p li lj p = plus_times_all_funs2_p li lj pfi pfj p def.

intros. rewrite times_plus_fun2_p_eq, plus_times_all_funs2_p_eq.
apply (@complete_dist_list_times2' (ba_conv Bp)).
Qed.


Lemma complete_dist_list_plus1_p' :
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj)
         (p:T->U->Btp) (def:U),
      plus_times_fun1_p li lj p = times_plus_all_funs1_p li lj pfi pfj p def.
intros. rewrite plus_times_fun1_p_eq, times_plus_all_funs1_p_eq.
apply (@complete_dist_list_plus1' (ba_conv Bp)).
Qed.


Lemma complete_dist_list_plus2_p' :
  forall {T U:Type} (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj)
         (p:T->U->Btp) (def:T),
      plus_times_fun2_p li lj p = times_plus_all_funs2_p li lj pfi pfj p def.
intros. rewrite plus_times_fun2_p_eq, times_plus_all_funs2_p_eq.
apply (@complete_dist_list_plus2' (ba_conv Bp)).
Qed.


Lemma plus_times_all_maps1_funs_compat_p :
  forall
    {T U:Type} {I:Ensemble T} {J:Ensemble U} (def:U)
    (pfi:Finite I) (pfj: Finite J)
    (li:list T) (lj:list U)
    (pfndpi: NoDup li)
    (pfndpj: NoDup lj) (p:T->U->Btp),
    list_to_set li = I -> list_to_set lj = J ->
    plus_times_fun_all_maps1_p _ _ def pfi pfj (f_pr p) =
    plus_times_all_funs1_p _ _ pfndpi pfndpj p def.
intros. rewrite plus_times_fun_all_maps1_p_eq, plus_times_all_funs1_p_eq.
apply (@plus_times_all_maps1_funs_compat (ba_conv Bp)); auto.
Qed.


Lemma plus_times_all_maps2_funs_compat_p :
  forall
    {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
         (pfi:Finite I) (pfj: Finite J)
         (li:list T) (lj:list U)
         (pfndpi: NoDup li)
         (pfndpj: NoDup lj) (p:T->U->Btp),
    list_to_set li = I -> list_to_set lj = J ->
    plus_times_fun_all_maps2_p _ _ def pfi pfj (f_pr p) =
    plus_times_all_funs2_p _ _ pfndpi pfndpj p def.
intros. rewrite plus_times_fun_all_maps2_p_eq, plus_times_all_funs2_p_eq.
apply (@plus_times_all_maps2_funs_compat (ba_conv Bp)); auto.
Qed.


Lemma times_plus_all_maps1_funs_compat_p :
  forall
    {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
         (pfi:Finite I) (pfj: Finite J)
         (li:list T) (lj:list U)
         (pfndpi: NoDup li)
         (pfndpj: NoDup lj) (p:T->U->Btp),
    list_to_set li = I -> list_to_set lj = J ->
    times_plus_fun_all_maps1_p _ _ def pfi pfj (f_pr p) =
    times_plus_all_funs1_p _ _ pfndpi pfndpj p def.
intros. rewrite times_plus_fun_all_maps1_p_eq, times_plus_all_funs1_p_eq.
apply (@times_plus_all_maps1_funs_compat (ba_conv Bp)); auto.
Qed.




Lemma times_plus_all_maps2_funs_compat_p :
  forall
    {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
    (pfi:Finite I) (pfj: Finite J)
    (li:list T) (lj:list U)
    (pfndpi: NoDup li)
    (pfndpj: NoDup lj) (p:T->U->Btp),
    list_to_set li = I -> list_to_set lj = J ->
    times_plus_fun_all_maps2_p _ _ def pfi pfj (f_pr p) =
    times_plus_all_funs2_p _ _ pfndpi pfndpj p def.
intros. rewrite times_plus_fun_all_maps2_p_eq, times_plus_all_funs2_p_eq.
apply (@times_plus_all_maps2_funs_compat (ba_conv Bp)); auto.
Qed.



Lemma times_plus_fun1_fin_map_to_fun_undoes_fun_to_fin_map_p :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U)
         (li:list T) (lj:list U) (pf:Finite (cart_prod I J))
         (p:T*U->Btp),
    list_to_set li = I -> list_to_set lj = J ->
    times_plus_fun1_p li lj
                      (f_no_pr (fin_map_to_fun (fun_to_fin_map (cart_prod I J) 0 pf (ba_conv_ind p)))) = times_plus_fun1_p li lj (f_no_pr p).
intros.
do 2 rewrite times_plus_fun1_p_eq.
apply (@times_plus_fun1_fin_map_to_fun_undoes_fun_to_fin_map (ba_conv Bp)); auto.
Qed.


Lemma times_plus_fun2_fin_map_to_fun_undoes_fun_to_fin_map_p :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U)
         (li:list T) (lj:list U) (pf:Finite (cart_prod I J))
         (p:T*U->Btp),
    list_to_set li = I -> list_to_set lj = J ->
    times_plus_fun2_p li lj
                      (f_no_pr (fin_map_to_fun (fun_to_fin_map (cart_prod I J) 0 pf (ba_conv_ind p)))) = times_plus_fun2_p li lj (f_no_pr p).
intros. do 2 rewrite times_plus_fun2_p_eq.
apply (@times_plus_fun2_fin_map_to_fun_undoes_fun_to_fin_map (ba_conv Bp)); auto.
Qed.


Lemma plus_times_fun1_fin_map_to_fun_undoes_fun_to_fin_map_p :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U)
         (li:list T) (lj:list U) (pf:Finite (cart_prod I J))
         (p:T*U->Btp),
    list_to_set li = I -> list_to_set lj = J ->
    plus_times_fun1_p li lj
                    (f_no_pr (fin_map_to_fun (fun_to_fin_map (cart_prod I J) 0 pf (ba_conv_ind p)))) = plus_times_fun1_p li lj (f_no_pr p).
intros. do 2 rewrite plus_times_fun1_p_eq.
apply (@plus_times_fun1_fin_map_to_fun_undoes_fun_to_fin_map (ba_conv Bp)); auto.
Qed.

Lemma plus_times_fun2_fin_map_to_fun_undoes_fun_to_fin_map_p :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U)
         (li:list T)
         (lj:list U) (pf:Finite (cart_prod I J))
         (p:T*U->Btp),
    list_to_set li = I -> list_to_set lj = J ->
    plus_times_fun2_p li lj
                    (f_no_pr (fin_map_to_fun (fun_to_fin_map (cart_prod I J) 0 pf (ba_conv_ind p)))) = plus_times_fun2_p li lj (f_no_pr p).
intros. do 2 rewrite plus_times_fun2_p_eq.
apply (@plus_times_fun2_fin_map_to_fun_undoes_fun_to_fin_map (ba_conv Bp)); auto.
Qed.


Lemma complete_dist_times_plus1_p :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
    (pfi:Finite I) (pfj:Finite J) (p:T*U->Btp),
    times_plus_fin_pair_map1_p
      pfi (fun_to_fin_map _ 0 (cart_prod_fin _ _ pfi pfj) (ba_conv_ind p)) =
    plus_times_fun_all_maps1_p _ _ def pfi pfj p.
intros. rewrite times_plus_fin_pair_map1_p_eq, plus_times_fun_all_maps1_p_eq.
apply (@complete_dist_times_plus1 (ba_conv Bp)).
Qed.



Lemma complete_dist_times_plus2_p :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:T)
    (pfi:Finite I) (pfj:Finite J)
    (p:T*U->Btp),
    times_plus_fin_pair_map2_p
      pfj (fun_to_fin_map _ 0 (cart_prod_fin _ _ pfi pfj) (ba_conv_ind p)) =
    plus_times_fun_all_maps2_p _ _ def pfi pfj p.
intros. rewrite times_plus_fin_pair_map2_p_eq, plus_times_fun_all_maps2_p_eq.
apply (@complete_dist_times_plus2 (ba_conv Bp)).
Qed.



Lemma complete_dist_plus_times1_p :
  forall {T U:Type} (I:Ensemble T) (J:Ensemble U) (def:U)
    (pfi:Finite I) (pfj:Finite J)
    (p:T*U->Btp),
    plus_times_fin_pair_map1_p
      pfi (fun_to_fin_map _ 0 (cart_prod_fin _ _ pfi pfj) (ba_conv_ind p)) =
    times_plus_fun_all_maps1_p _ _ def pfi pfj p.
intros. 
rewrite plus_times_fin_pair_map1_p_eq, times_plus_fun_all_maps1_p_eq.
apply (@complete_dist_plus_times1 (ba_conv Bp)).
Qed.


Lemma complete_dist_plus_times2_p :
  forall {T U:Type}  (I:Ensemble T) (J:Ensemble U) (def:T)
         (pfi:Finite I) (pfj:Finite J)
         (p:T*U->Btp),
    plus_times_fin_pair_map2_p
      pfj (fun_to_fin_map _ 0 (cart_prod_fin _ _ pfi pfj) (ba_conv_ind p)) =
    times_plus_fun_all_maps2_p _ _ def pfi pfj p.
intros.
rewrite plus_times_fin_pair_map2_p_eq, times_plus_fun_all_maps2_p_eq.
apply (@complete_dist_plus_times2 (ba_conv Bp)).
Qed.


     

End ParametricAnalogues.

Arguments plus_list_app_p [T'] [Bp] _ _.
Arguments times_list_app_p [T'] [Bp] _ _.
Arguments dist_list_sing_plus_p [T'] [Bp] _ _.
Arguments dist_list_sing_times_p [T'] [Bp] _ _.
Arguments dist_list_2_plus_p [T'] [Bp] _ _.
Arguments dist_list_2_times_p [T'] [Bp] _ _.
Arguments plus_times_list_of_lists_p [T'] [Bp] _.
Arguments times_plus_list_of_lists_p [T'] [Bp] _.
Arguments times_fun_p [T'] [Bp] [T] _ _.
Arguments plus_fun_p [T'] [Bp] [T] _ _.
Arguments plus_times_fun1_p [T'] [Bp] [T] [U] _ _ _.
Arguments plus_times_fun2_p [T'] [Bp] [T] [U] _ _ _.
Arguments times_plus_fun1_p [T'] [Bp] [T] [U] _ _ _.
Arguments times_plus_fun2_p [T'] [Bp] [T] [U] _ _ _.
Arguments plus_times_all_funs1_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments plus_times_all_funs2_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments times_plus_all_funs1_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments times_plus_all_funs2_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_list_times_p [T'] [Bp] _.
Arguments complete_dist_list_plus_p [T'] [Bp] _.
Arguments inf_union_p [T'] [Bp] _ _ _ _ _ _.
Arguments sup_union_p [T'] [Bp] _ _ _ _ _ _.
Arguments decompose_inf_p [T'] [Bp] _ _ _ _ _ _ _ _ _.
Arguments decompose_inf_p' [T'] [Bp] _ _ _ _ _ _ _.
Arguments inf_times_cons_p [T'] [Bp] _ _ _ _ _.
Arguments inf_times_finite_p [T'] [Bp] _.
Arguments sup_plus_finite_p [T'] [Bp] _.
Arguments le_times_finite_member_p [T'] [Bp] _ _ _.
Arguments le_member_plus_finite_p [T'] [Bp] _ _ _.
Arguments prod_list_dup_eq_p [T'] [Bp] _ _ _.
Arguments sum_list_dup_eq_p [T'] [Bp] _ _ _.
Arguments prod_preserves_list_singularize_p [T'] [Bp] _.
Arguments sum_preserves_list_singularize_p [T'] [Bp] _.
Arguments times_sing_preserves_new_head_p [T'] [Bp] _ _ _.
Arguments plus_sing_preserves_new_head_p [T'] [Bp] _ _ _.
Arguments times_list_sing_cons_p [T'] [Bp] _ _ _.
Arguments plus_list_sing_cons_p [T'] [Bp] _ _ _.
Arguments list_to_sets_eq_times_sing_eq_p [T'] [Bp] _ _ _ _ _.
Arguments list_to_sets_eq_plus_sing_eq_p [T'] [Bp] _ _ _ _ _.
Arguments list_to_sets_eq_times_eq_p [T'] [Bp] _ _ _ _ _.
Arguments list_to_sets_eq_plus_eq_p [T'] [Bp] _ _ _ _ _.
Arguments times_list_unq_p [T'] [Bp] _ _.
Arguments plus_list_unq_p [T'] [Bp] _ _.
Arguments times_set_p [T'] [Bp] _ _.
Arguments plus_set_p [T'] [Bp] _ _.
Arguments times_set_compat_p [T'] [Bp] _ _.
Arguments times_set_compat_p' [T'] [Bp] _ _ _ _.
Arguments plus_set_compat_p [T'] [Bp] _ _.
Arguments plus_set_compat_p' [T'] [Bp] _ _ _ _.
Arguments times_set_functional_p [T'] [Bp] _ _ _ _ _.
Arguments plus_set_functional_p [T'] [Bp] _ _ _ _ _.
Arguments times_set_empty_p [T'] [Bp].
Arguments times_set_empty_p' [T'] [Bp] _.
Arguments plus_set_empty_p [T'] [Bp].
Arguments plus_set_empty_p' [T'] [Bp] _.
Arguments times_set_add_p [T'] [Bp] _ _ _.
Arguments times_set_add_p' [T'] [Bp] _ _ _ _.
Arguments plus_set_add_p [T'] [Bp] _ _ _.
Arguments plus_set_add_p' [T'] [Bp] _ _ _ _.
Arguments times_set_sing_p [T'] [Bp] _.
Arguments times_set_sing_p' [T'] [Bp] _ _.
Arguments times_set_one_or_p [T'] [Bp] _ _ _.
Arguments plus_set_sing_p [T'] [Bp] _.
Arguments plus_set_sing_p' [T'] [Bp] _ _.
Arguments plus_set_zero_or_p [T'] [Bp] _ _ _.
Arguments times_set_couple_p [T'] [Bp] _ _.
Arguments times_set_couple_p' [T'] [Bp] _ _ _.
Arguments plus_set_couple_p [T'] [Bp] _ _.
Arguments plus_set_couple_p' [T'] [Bp] _ _ _.
Arguments le_times_set_p [T'] [Bp] _ _ _ _.
Arguments le_plus_set_p [T'] [Bp] _ _ _ _.
Arguments inf_times_set_p [T'] [Bp] _ _.
Arguments sup_plus_set_p [T'] [Bp] _ _.
Arguments times_set_union_p [T'] [Bp] _ _ _ _.
Arguments times_set_union_p' [T'] [Bp] _ _ _ _ _.
Arguments times_set_inc_le_p [T'] [Bp] _ _ _ _ _.
Arguments plus_set_union_p [T'] [Bp] _ _ _ _.
Arguments plus_set_union_p' [T'] [Bp] _ _ _ _ _.
Arguments plus_set_inc_le_p [T'] [Bp] _ _ _ _ _.
Arguments plus_set_im_add_p [T'] [Bp] _ _ _ _.
Arguments plus_set_im_add_p' [T'] [Bp] _ _ _ _ _ _.
Arguments times_set_im_add_p [T'] [Bp] _ _ _ _.
Arguments times_set_im_add_p' [T'] [Bp] _ _ _ _ _ _.
Arguments dist_set_plus1_p [T'] [Bp] _ _ _.
Arguments dist_set_plus1_p' [T'] [Bp] _ _ _ _.
Arguments dist_set_times1_p [T'] [Bp] _ _ _.
Arguments dist_set_times1_p' [T'] [Bp] _ _ _ _.
Arguments dist_set_plus2_p [T'] [Bp] _ _ _ _.
Arguments dist_set_plus2_p' [T'] [Bp] _ _ _ _ _.
Arguments dist_set_times2_p [T'] [Bp] _ _ _ _.
Arguments dist_set_times2_p' [T'] [Bp] _ _ _ _ _.
Arguments plus_fin_pair_map1_p [T'] [Bp] [T] [U] [C] [D] [E] _ _.
Arguments plus_fin_pair_map2_p [T'] [Bp] [T] [U] [C] [D] [E] _ _.
Arguments times_fin_pair_map1_p [T'] [Bp] [T] [U] [C] [D] [E] _ _.
Arguments times_fin_pair_map2_p [T'] [Bp] [T] [U] [C] [D] [E] _ _.
Arguments plus_fin_pair_map1_list_compat_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _.
Arguments plus_fin_pair_map2_list_compat_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _.
Arguments times_fin_pair_map1_list_compat_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _.
Arguments times_fin_pair_map2_list_compat_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _.
Arguments plus_fin_pair_map2_functional_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _ _.
Arguments fin_map_times_p [T'] [Bp] [T] [A] [C] _.
Arguments fin_map_times_list_compat_p [T'] [Bp] [T] [A] [C] _ _.
Arguments fin_map_times_empty1_p [T'] [Bp] [T] [C] _.
Arguments fin_map_eq_times_p [T'] [Bp] [T] _ _ _ _ _ _.
Arguments im2_empty1_p [T'] [Bp] [T] [U] _ _ _ _ _.
Arguments im2_empty2_p [T'] [Bp] [T] [U] _ _ _ _ _.
Arguments plus_fin_pair_map2_cart_empty_eq1_p [T'] [Bp] [T] [U] _ _ _ _.
Arguments plus_fin_pair_map2_cart_empty_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments plus_fin_pair_map2_cart_empty_eq2_p [T'] [Bp] [T] [U] _ _ _ _ _.
Arguments fin_map_times_sing_p [T'] [Bp] [T] _ _ _ _.
Arguments fin_map_times_cart_empty11_p [T'] [Bp] [T] [U] _ _ _ _ _.
Arguments fin_map_times_cart_empty21_p [T'] [Bp] [T] [U] _ _ _ _ _.
Arguments fin_map_times_empty2_p [T'] [Bp] [T] _ _.
Arguments fin_map_plus_p [T'] [Bp] [T] [A] [C] _.
Arguments fin_map_plus_list_compat_p [T'] [Bp] [T] [A] [C] _ _.
Arguments fin_map_plus_empty1_p [T'] [Bp] [T] [C] _.
Arguments fin_map_plus_empty2_p [T'] [Bp] [T] [A] _.
Arguments fin_map_eq_plus_p [T'] [Bp] [T] _ _ _ _ _ _.
Arguments times_plus_fin_pair_map1_p [T'] [Bp] [T] [U] [C] [D] [E] _ _.
Arguments times_plus_fin_pair_map2_p [T'] [Bp] [T] [U] [C] [D] [E] _ _.
Arguments times_plus_fin_pair_map2_p [T'] [Bp] [T] [U] [C] [D] [E] _ _.
Arguments plus_times_fin_pair_map1_p [T'] [Bp] [T] [U] [C] [D] [E] _ _.
Arguments plus_times_fin_pair_map2_p [T'] [Bp] [T] [U] [C] [D] [E] _ _.
Arguments times_plus_fin_pair_map1_list_compat_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _.
Arguments plus_fun_fin_map_to_fun_comm_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _ _ _ _.
Arguments plus_fun_fin_map_to_fun_comm_p' [T'] [Bp] [T] [U] [C] [D] [E] _ _ _ _ _ _.
Arguments times_plus_fin_pair_map1_list_compat_p' [T'] [Bp] [T] [U] [C] [D] [E] _ _ _ _ _.
Arguments times_plus_fin_pair_map2_list_compat_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _.
Arguments times_plus_fin_pair_map2_list_compat_p' [T'] [Bp] [T] [U] [C] [D] [E] _ _ _ _ _ .
Arguments plus_times_fin_pair_map1_list_compat_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _.
Arguments times_fun_fin_map_to_fun_comm_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _ _ _ _.
Arguments plus_times_fin_pair_map1_list_compat_p' [T'] [Bp] [T] [U] [C] [D] [E] _ _ _ _ _.
Arguments plus_times_fin_pair_map2_list_compat_p [T'] [Bp] [T] [U] [C] [D] [E] _ _ _.
Arguments times_fun_fin_map1_p [T'] [Bp] [T] [U] [I] [J] [def] _ _.
Arguments times_fun_fin_map2_p [T'] [Bp] [T] [U] [I] [J] [def] _ _.
Arguments times_fun_fin_map1_l_p [T'] [Bp] [T] [U] [I] [J] [def] _ _ _.
Arguments times_fun_fin_map2_l_p [T'] [Bp] [T] [U] [I] [J] [def] _ _ _.
Arguments im_sing_times_fun_fin_map1_p [T'] [Bp] [T] [U] [I] [J] [def] _ _.
Arguments times_fun_fin_map1_empty_p [T'] [Bp] [T] [U] _ _.
Arguments times_fun_fin_map2_empty_p [T'] [Bp] [T] [U] _ _.
Arguments times_fun_fin_map1_empty1_p [T'] [Bp] [T] [U] _ _ _ _.
Arguments times_fun_fin_map2_empty1_p [T'] [Bp] [T] [U] _ _ _ _.
Arguments times_fun_fin_map1_list_compat_p [T'] [Bp] [T] [U] [I] [J] [def] _ _ _.
Arguments times_fun_fin_map2_list_compat_p [T'] [Bp] [T] [U] [I] [J] [def] _ _ _.
Arguments plus_fun_fin_map1_p [T'] [Bp] [T] [U] [I] [J] [def] _ _.
Arguments plus_fun_fin_map2_p [T'] [Bp] [T] [U] [I] [J] [def] _ _.
Arguments plus_fun_fin_map1_empty_p [T'] [Bp] [T] [U] _ _.
Arguments plus_fun_fin_map1_empty1_p [T'] [Bp] [T] [U] _ _ _ _.
Arguments plus_fun_fin_map1_l_p [T'] [Bp] [T] [U] [I] [J] [def] _ _ _.
Arguments plus_fun_fin_map2_l_p [T'] [Bp] [T] [U] [I] [J] [def] _ _ _.
Arguments plus_fun_fin_map2_empty_p [T'] [Bp] [T] [U] _ _.
Arguments plus_fun_fin_map2_empty1_p [T'] [Bp] [T] [U] _ _ _ _.
Arguments plus_fun_fin_map1_list_compat_p [T'] [Bp] [T] [U] [I] [J] [def] _ _ _.
Arguments plus_fun_fin_map2_list_compat_p [T'] [Bp] [T] [U] [I] [J] [def] _ _ _.
Arguments plus_times_fun_all_maps1_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments plus_times_fun_all_maps2_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments times_plus_fun_all_maps1_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments times_plus_fun_all_maps2_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_list_times1_p' [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_list_times2_p' [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_list_plus1_p' [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_list_plus2_p' [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments plus_times_all_maps1_funs_compat_p [T'] [Bp] [T] [U] [I] [J] _ _ _ _ _ _ _ _ _ _.
Arguments plus_times_all_maps2_funs_compat_p [T'] [Bp] [T] [U] _ _ _ _ _ _ _ _ _ _ _ _.
Arguments times_plus_all_maps1_funs_compat_p [T'] [Bp] [T] [U] _ _ _ _ _ _ _ _ _ _ _ _.
Arguments times_plus_all_maps2_funs_compat_p [T'] [Bp] [T] [U] _ _ _ _ _ _ _ _ _ _ _ _.
Arguments times_plus_fun1_fin_map_to_fun_undoes_fun_to_fin_map_p [T'] [Bp] [T] [U] _ _ _ _ _ _ _ _.
Arguments times_plus_fun2_fin_map_to_fun_undoes_fun_to_fin_map_p [T'] [Bp] [T] [U] _ _ _ _ _ _ _ _.
Arguments plus_times_fun1_fin_map_to_fun_undoes_fun_to_fin_map_p [T'] [Bp] [T] [U] _ _ _ _ _ _ _ _.
Arguments plus_times_fun2_fin_map_to_fun_undoes_fun_to_fin_map_p [T'] [Bp] [T] [U] _ _ _ _ _ _ _ _.
Arguments complete_dist_times_plus1_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_times_plus2_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_plus_times1_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments complete_dist_plus_times2_p [T'] [Bp] [T] [U] _ _ _ _ _ _.
Arguments ba_conv_fin_map_dom [T'] [Bp] [U] [C] [D] [def] _.
Arguments ba_conv_fin_map_ran [T'] [Bp] [T] [C] [D] _.
Arguments incl_ba_conv_ens_fin_map_dom_iff [T'] [Bp] [U] [C] [D] [def] _ _.
Arguments ba_conv_ens_fin_map_dom [T'] [Bp] [U] [C] [D] [def] _ _.
Arguments ba_conv_ens_fin_map_ran [T'] [Bp] [T] [C] [D] _ _.
