(* Copyright (C) 2014, Daniel Wyckoff, except for the portions so labeleled
which I got from Daniel Schepler*)
(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)

Require Import SetUtilities.
Require Import Image.
Require Import Powerset.
Require Import Classical_sets.
Require Import Classical.
Require Import ClassicalChoice.
Require Import FunctionalExtensionality.
Require Import LogicUtilities.
Require Import TypeUtilities.
Require Import FunctionProperties.
Require Import Basics.
Require Import ProofIrrelevance.

(*Much of this is copied and pasted from Daniel Schepler's Coq User contribution
  "Topology", specifically the files "TopologicalSpaces.v" 
  and "InteriorsClosures.v" from that package*)

(*Schepler*)
Record TopologicalSpace : Type := {
  point_set : Type;
  open : Ensemble point_set -> Prop;
  open_family_union : forall F : Family point_set,
    (forall S : Ensemble point_set, In F S -> open S) ->
    open (FamilyUnion F);
  open_intersection2: forall U V:Ensemble point_set,
    open U -> open V -> open (Intersection U V);
  open_full : open (Full_set point_set)
}.

Arguments open [_] _.

(*Schepler*)
Lemma open_empty: forall (X:TopologicalSpace),
  open (Empty_set (point_set X)).
Proof.
intros.
rewrite <- empty_family_union.
apply (open_family_union X).
intros.
destruct H.
Qed.

(*Schepler*)
Lemma open_union2: forall (X:TopologicalSpace)(U V:Ensemble (point_set X)),
  open U -> open V -> open (Union U V).
Proof.
intros.
assert (Union U V = FamilyUnion (Couple U V)).
apply Extensionality_Ensembles; split; red; intros.
destruct H1.
exists U; auto with sets.
exists V; auto with sets.
destruct H1.
destruct H1.
left; trivial.
right; trivial.

rewrite H1; apply open_family_union.
intros.
destruct H2; trivial.
Qed.

(*Schepler*)
Lemma open_indexed_union: forall {X:TopologicalSpace} {A:Type}
  (F:IndexedFamily A (point_set X)),
  (forall a:A, open (F a)) -> open (IndexedUnion F).
Proof.
intros.
rewrite indexed_to_family_union.
apply open_family_union.
intros.
destruct H0.
rewrite H1; apply H.
Qed.

(*Schepler*)
Lemma open_finite_indexed_intersection:
  forall {X:TopologicalSpace} {A:Type}
    (F:IndexedFamily A (point_set X)),
    FiniteT A -> (forall a:A, open (F a)) ->
    open (IndexedIntersection F).
Proof.
intros.
induction H.
rewrite empty_indexed_intersection.
apply open_full.

assert (IndexedIntersection F = Intersection
  (IndexedIntersection (fun x:T => F (Some x)))
  (F None)).
apply Extensionality_Ensembles; split; red; intros.
destruct H1.
constructor.
constructor.
intros; apply H1.
apply H1.
destruct H1.
destruct H1.
constructor.
destruct i.
apply H1.
apply H2.
rewrite H1.
apply open_intersection2.
apply IHFiniteT.
intros; apply H0.
apply H0.

destruct H1.
assert (IndexedIntersection F =
  IndexedIntersection (fun x:X0 => F (f x))).
apply Extensionality_Ensembles; split; red; intros.
constructor.
destruct H3.
intro; apply H3.
constructor.
destruct H3.
intro; rewrite <- H2 with i.
apply H3.
rewrite H3.
apply IHFiniteT.
intro; apply H0.
Qed.

Section Discrete.
Variable T : Type.

(*Wyckoff*)
Definition open_dscrt (S:(Ensemble T)) := True.

(*Wyckoff*)
Lemma open_family_union_dscrt : forall (F: Family T),  
    (forall S : Ensemble T, In F S -> open_dscrt S) ->
    open_dscrt (FamilyUnion F).
intros. compute. tauto.
Qed.

(*Wyckoff*)
Lemma open_intersection2_dscrt : forall (U V:Ensemble T),
  open_dscrt U -> open_dscrt V -> open_dscrt (Intersection U V).
intros. compute. tauto.
Qed.

(*Wyckoff*)
Lemma open_full_dscrt : open_dscrt (Full_set T).
intros. compute. tauto.
Qed.

(*Wyckoff*)  
Definition discrete_top := 
    Build_TopologicalSpace T open_dscrt open_family_union_dscrt
    open_intersection2_dscrt open_full_dscrt.

End Discrete.

Section Trivial.
Variable T: Type.
(*Wyckoff*)
Hypothesis T_ne : Full_set(T) <> Empty_set T.
Definition open_trivial (S:(Ensemble T)) := 
  (S = (Full_set T) \/ S = (Empty_set T)).

(*Wyckoff*)
Lemma open_family_union_trivial : forall (F: Family T),  
    (forall S : Ensemble T, In F S -> open_trivial S) ->
    open_trivial (FamilyUnion F).
intros F h1. compute.
unfold open_trivial in h1.
assert (h2: let UF := (FamilyUnion F = Full_set T) in UF \/ ~UF).
  intro UF.  tauto.
destruct h2 as [h3 | h4].
(*left*)
  left.  assumption.
(*right*)
assert (h5: forall S : Ensemble T, In F S -> S <> Full_set T).
  intros S h6.
  assert (h7: S = Full_set T \/ S = Empty_set T).
    apply h1. assumption.
  destruct h7 as [h8 | h9].
    (*left*)
    assert (h9 : FamilyUnion F = Full_set T).
      apply Extensionality_Ensembles.
      unfold Same_set. split.
        (*left*)
        unfold Included.
          intros. apply Full_intro.
        (*right*)
        rewrite <- h8.
        unfold Included.
          intros x h9. apply family_union_intro with (S := S).
          assumption.  assumption.
    contradiction.
    (*right*)
    intro h10.
    rewrite h10 in h9.
    contradiction.
assert (h13 : forall (S : Ensemble T), In F S -> S = Empty_set T).
  intros S h14.
    assert (h15 : S = Full_set T \/ S = Empty_set T).
      apply h1. assumption.
    assert (h16 : S <> Full_set T).
      apply h5. assumption.
  tauto.
right.
apply Extensionality_Ensembles. unfold Same_set.
split.
(*left*)
  unfold Included.
  intros x h17.
  inversion h17 as [S y h18 h19 h20].
  assert (h14 : S = Empty_set T).
    apply h13. assumption.
  rewrite h14 in h19. assumption.
(*right*)
  unfold Included.
  intros x h21.
assert (h22 : False).
  inversion h21.
tauto.
Qed.

(*Wyckoff*)
Lemma open_intersection2_trivial : forall (U V:Ensemble T),
  open_trivial U -> open_trivial V -> open_trivial (Intersection U V).
unfold open_trivial.
intros U V h1 h2. 

destruct h1 as [h3|h4]; destruct h2 as [h5|h6].
assert (h7: Intersection U V = U).
  apply Extensionality_Ensembles.  unfold Same_set.
    unfold Included.
  split.
  intros x h7.
  inversion h7 as [y h8 h9 h10]. assumption.
  rewrite <- h5 in h3.
  rewrite h3.
  intros x h11.
  apply Intersection_intro.
    assumption.  assumption.
  rewrite h7. left. apply h3.

assert (h7: Intersection U V = V).
  apply Extensionality_Ensembles.  unfold Same_set.
    unfold Included.
  split.
  intros x h7.
  inversion h7 as [y h8 h9 h10]. assumption.
  intros x h8. rewrite h6 in h8. inversion h8.
  rewrite h7.
  right. assumption.

assert (h8: Intersection U V = U).
  apply Extensionality_Ensembles.  unfold Same_set.
    unfold Included.
  split.
  intros x h7.
  inversion h7 as [y h8 h9 h10]. assumption.
  intros x h8. rewrite h4 in h8. inversion h8.
  rewrite h8. right. assumption.

assert (h8: Intersection U V = U).
  apply Extensionality_Ensembles.  unfold Same_set.
    unfold Included.
  split.
  intros x h7.
  inversion h7 as [y h8 h9 h10]. assumption.
  intros x h8. rewrite h4 in h8. inversion h8.
  rewrite h8. right. assumption.
Qed.
 
(*Wyckoff*)
Lemma open_full_trivial : open_trivial (Full_set T).
compute. left. reflexivity.
Qed.

(*Wyckoff*)
Definition trivial_top := 
    Build_TopologicalSpace T open_trivial open_family_union_trivial
    open_intersection2_trivial open_full_trivial.
End Trivial.



Section Cofinite.
Variable T:Type.

(*Wyckoff*)
Definition open_cof (S:Ensemble T) := (S = Empty_set _) \/ 
  Finite (Ensembles.Complement S).

(*Wyckoff*)
Lemma open_family_union_cof : forall F : Family T,
    (forall S : Ensemble T, In F S -> open_cof S) ->
    open_cof (FamilyUnion F).
red. intros F h1.
red in h1.
pose (h2 := (forall S:Ensemble T, In F S -> S = Empty_set _)).
assert (h3: h2 \/ 
  (exists S:Ensemble T, In F S /\ Finite (Ensembles.Complement S))).
  assert (h3: h2 \/ ~h2). tauto.
  destruct h3 as [h4|h5].
  tauto.
  unfold h2 in h5.
  pose proof (not_all_ex_not _ _ h5) as h6.
  elim h6.  intros S h7.
  right.
  exists S. 
  assert (h9: S <> Empty_set _).
    intro h10.
    tauto.
  assert (h10: ~~(In F S)).
    tauto.
  pose proof (NNPP _ h10) as h11.
  split. assumption.
  pose proof (h1 S h11) as h12.
  destruct h12. contradiction. assumption.
destruct h3 as [h12 | h13].
left.
apply Extensionality_Ensembles. unfold Same_set. unfold Included.
split.
intros S h14.
inversion h14.
unfold h2 in h12.
pose proof (h12 S0 H).
rewrite H2 in H0.
contradiction.
intros; contradiction.
right.
elim h13. intros S h15.
elim h15. intros h16 h17.
assert (h18: Included S (FamilyUnion F)).
  unfold Included. intros x h18. apply family_union_intro with (S).
  assumption. assumption.
pose proof (complement_inclusion _ _ h18) as h19.
apply Finite_downward_closed with (Ensembles.Complement S). 
assumption. assumption.
Qed.


(*Wyckoff*)
Lemma open_intersection2_cof: forall U V:Ensemble T,
    open_cof U -> open_cof V -> open_cof (Intersection U V).
intros U V h1 h2.
red in h1. red in h2.
destruct h1 as [h3 | h4]. 
destruct h2 as [h5 | h6].
(*case 1*)
assert (h7: Intersection U V = Empty_set _ ).
  rewrite h3. rewrite h5.
  auto with sets.
red. left. assumption.
(*case 2*)
assert (h8: Intersection U V = Empty_set _ ).
  rewrite h3.
  auto with sets.
red. left. assumption.
(*case 3*)
red.
destruct h2 as [h10 | h11].
  left.
  rewrite h10. auto with sets.
(*case 4*)
right.
pose proof (comp_int _ U V) as h12.
rewrite h12.
apply Union_preserves_Finite. assumption. assumption.
Qed.

Lemma open_full_cof : open_cof (Full_set T).
red.
right.
pose proof (complement_full T) as h1.
rewrite h1.
constructor.
Qed.

(*Wyckoff*)
Definition cof_top := 
    Build_TopologicalSpace T open_cof open_family_union_cof
    open_intersection2_cof open_full_cof.

(*Code "The discrete and cofinite topologies on X coincide when X is
finite, but they are obviously different when X is infinite."*)

End Cofinite.

Section Top_Space_Equiv.

(*Wyckoff*)
Definition set_iso (T1 T2 : Type) (heq : T1 = T2) (S1 : Ensemble T1) : 
  Ensemble T2.
rewrite heq in S1.
refine S1.
Defined.

(*Wyckoff*)
Definition ts_eq (X1 X2 : TopologicalSpace) : Prop :=
  exists heq:((point_set X1) = (point_set X2)), 
  (forall (S1:Ensemble (point_set X1)) (S2:Ensemble (point_set X2)), 
    S2 = set_iso _ _ heq S1 -> ((open S1) <-> (open S2))).
End Top_Space_Equiv.

Section Cofinite_Discrete_Finite.
Variable T : Type.
Definition X_dt := discrete_top T.
Definition X_ct := cof_top T.

(*Wyckoff*)
Lemma dscrt_cof_same_fin : FiniteT T -> ts_eq X_dt X_ct.
intro h1.
red. simpl.
exists (refl_equal T).
intros S1 S2 h2.
unfold open_dscrt. unfold open_cof.
split.
(*left*)
  intro h3.
  right.
  apply FiniteT_Finite. assumption.
(*right*)
intros. 
auto.
Qed.
End Cofinite_Discrete_Finite.

Section Inherited.
Variable X:TopologicalSpace.

Variable Y:(Ensemble (point_set X)).

(*Wyckoff*)
Definition Yt := {y:(point_set X) | In Y y}.

(*This just "casts" a Yt ensemble to a (point_set X) ensemble*)
Definition yte_to_ps (P:Ensemble (Yt)) : (Ensemble (point_set X)):= 
  (Im P (@proj1_sig (point_set X) 
    (fun(x:(point_set X)) => (In Y x)))).   

(*Wyckoff*)
Definition open_inh (P:(Ensemble Yt)) : Prop := 
  (exists Q:Ensemble (point_set X), open Q /\ 
         (yte_to_ps P) = Intersection Y Q).

(*Wyckoff*)
Definition Fam_sig (F:Family Yt) := {S:Ensemble Yt | In F S}.

(*Wyckoff*)
Definition F_to_Fs (F:Family Yt) := Full_set (Fam_sig F).

(*Wyckoff*)
Lemma open_family_union_inh : forall F : Family Yt,
    (forall S : (Fam_sig F), open_inh (proj1_sig S)) ->
    open_inh (FamilyUnion F).
intros F h1.
unfold open_inh in h1.
destruct (choice (fun (S:Fam_sig F) Q => open Q /\ yte_to_ps (proj1_sig S) = Intersection Y Q))
  as [choice_fun h2].
apply h1.
unfold open_inh.
exists (IndexedUnion choice_fun).
split.
apply open_indexed_union.
apply h2.
assert (h3:Intersection Y (IndexedUnion choice_fun) = 
  IndexedUnion (Int_Ind_Ens _ _ choice_fun Y)).
  apply dist_union_infnt_1_ind.
rewrite h3.
unfold Int_Ind_Ens.
assert (h4: (fun i : Fam_sig F => Intersection Y (choice_fun i)) = 
            (fun i : Fam_sig F => yte_to_ps (proj1_sig i))).
  apply functional_extensionality.
  intro S.
  pose proof (h2 S) as h5.
  elim h5.
    intros h6 h7.
  rewrite h7. reflexivity.
rewrite h4.
apply Extensionality_Ensembles. unfold Same_set. unfold Included.
split.
(*left*)
intros x h8.
inversion h8.
inversion H.
apply indexed_union_intro with (i := exist _ S H2).
unfold yte_to_ps.
simpl.
apply Im_intro with (x := x0).
assumption. assumption.
(*right*)
intros x h9.
inversion h9.
unfold yte_to_ps.
unfold yte_to_ps in H.

inversion H.
apply Im_intro with (x := x1).
unfold Fam_sig in i.
inversion i.
apply family_union_intro with (S := proj1_sig i).
apply (proj2_sig i).
assumption.
assumption.
Qed.

(*Wyckoff*)
Lemma open_family_union_inh' : forall F : Family Yt,
    (forall S : Ensemble Yt, In F S -> open_inh S) ->
    open_inh (FamilyUnion F).
intros F h1.
apply open_family_union_inh.
intro S'.
apply (h1 (proj1_sig S') (proj2_sig S')).
Qed.

(*Wyckoff*)
Lemma open_intersection2_inh: forall U V:Ensemble Yt,
    open_inh U -> open_inh V -> open_inh (Intersection U V).
unfold open_inh.
intros U V h1 h2.
elim h1. intros Q1 h3.
elim h2. intros Q2 h4.
exists (Intersection Q1 Q2).
split.
(*left*)
apply (open_intersection2 X). apply h3. apply h4.
(*right*)
elim h3. intros h5 h6.
elim h4. intros h7 h8.
assert (h9:Intersection Y (Intersection Q1 Q2) = 
  Intersection (Intersection Y Q1) (Intersection Y Q2)).
  assert (h10: Y = Intersection Y Y).    
    apply Extensionality_Ensembles. unfold Same_set. unfold Included.
    split.
    (*left*)
    intros x h11. auto with sets.
    (*right*)
    intros x h12. inversion h12. assumption.
  rewrite h10 at 1.
  pose proof (assoc_prod_psa (Intersection Y Y) Q1 Q2) as h11.
  pose proof (assoc_prod_psa Y Y Q1) as h12.
  rewrite <- h12 in h11.
  pose proof (comm_prod_psa Y Q1) as h13.
  rewrite h13 in h11.
  pose proof (assoc_prod_psa Y Q1 Y) as h14.
  rewrite h14 in h11.
  pose proof (assoc_prod_psa (Intersection Y Q1) Y Q2) as h15.
  rewrite <- h15 in h11.
  assumption.
rewrite h9.
rewrite <- h6. rewrite <- h8.
unfold yte_to_ps.
apply Extensionality_Ensembles. unfold Same_set. unfold Included.
split.
(*left*)
intros x h16. inversion h16.
inversion H.
apply Intersection_intro.
apply Im_intro with (x := x0). assumption. assumption.
apply Im_intro with (x := x0). assumption. assumption.
(*right*)
intros x h17.
inversion h17.
inversion H.
inversion H0.
assert (h20: x1 = x2).
  destruct x1. destruct x2.
  apply existTexist.
  pose proof (subsetT_eq_compat _ _ 
    (proj1_sig (exist (fun y : point_set X => In Y y) x1 i))
    (proj1_sig (exist (fun y : point_set X => In Y y) x2 i0))
    i i0) as h20.
  rewrite H3 in H6.
  pose proof (h20 H6) as h21.
  simpl in h21.
  apply h21.
assert (h22: In (Intersection U V) x1).
  apply Intersection_intro.
  assumption.
  rewrite <- h20 in H5. assumption.
apply Im_intro with (x := x1).
assumption. assumption.
Qed.

(*Wyckoff*)
Lemma open_full_inh : open_inh (Full_set Yt).
unfold open_inh.
exists (Full_set (point_set X)).
split.
(*left*)
apply (open_full X).
(*right*)
assert (h1 : yte_to_ps (Full_set Yt) = Y).
  unfold yte_to_ps.
  apply Extensionality_Ensembles. unfold Same_set. unfold Included.
  split.
  (*left*)
  intros x h2.
  inversion h2.
  pose proof (proj2_sig x0) as h3. simpl in h3.
  rewrite H0. assumption.
  (*right*)
  intros x h4.  
  apply Im_intro with (x := (exist _ x h4)).
  apply Full_intro.
  simpl. reflexivity.
assert (h5: Y = Intersection Y (Full_set (point_set X))).
  apply Extensionality_Ensembles. unfold Same_set. unfold Included.
  split.
  (*left*)
  intros x h6.
  apply Intersection_intro.
  assumption.
  apply Full_intro.
  (*right*)
  intros x h7.
  inversion h7. assumption.
congruence.
Qed.

(*Wyckoff*)
Definition inh_top := 
    Build_TopologicalSpace Yt open_inh open_family_union_inh'
    open_intersection2_inh open_full_inh.
End Inherited.

Section Closed.
Variable X : TopologicalSpace.

(*Schepler*)
Definition closed (F:Ensemble (point_set X)) :=
  open (Ensembles.Complement F).

(*Schepler*)
Lemma closed_complement_open: forall 
  (U:Ensemble (point_set X)), closed (Ensembles.Complement U) ->
  open U.
Proof.
intros.
red in H.
rewrite Complement_Complement in H.
assumption.
Qed.

(*Schepler*)
Lemma closed_empty : closed (Empty_set (point_set X)).
red.
pose proof (complement_empty (point_set X)) as h1.
rewrite h1.
apply open_full.
Qed.

(*Schepler*)
Lemma closed_full : closed (Full_set (point_set X)).
red.
pose proof (complement_full (point_set X)) as h1.
rewrite h1.
apply open_empty.
Qed.

(*Schepler*)
Lemma closed_union2: forall 
  (F G:Ensemble (point_set X)),
  closed F -> closed G -> closed (Union F G).
Proof.
intros.
red in H, H0.
red.
assert (Ensembles.Complement (Union F G) =
  Intersection (Ensembles.Complement F)
               (Ensembles.Complement G)).
unfold Ensembles.Complement.
apply Extensionality_Ensembles; split; red; intros.
constructor.
auto with sets.
auto with sets.
destruct H1.
red; red; intro.
destruct H3.
apply (H1 H3).
apply (H2 H3).

rewrite H1.
apply open_intersection2; assumption.
Qed.

(*Schepler*)
Lemma closed_intersection2: forall
  (F G:Ensemble (point_set X)),
  closed F -> closed G -> closed (Intersection F G).
Proof.
intros.
red in H, H0.
red.
assert (Ensembles.Complement (Intersection F G) =
  Union (Ensembles.Complement F)
        (Ensembles.Complement G)).
apply Extensionality_Ensembles; split; red; intros.
apply NNPP.
red; intro.
unfold Ensembles.Complement in H1.
unfold In in H1.
contradict H1.
constructor.
apply NNPP.
red; intro.
auto with sets.
apply NNPP.
red; intro.
auto with sets.
red; red; intro.
destruct H2.
destruct H1; auto with sets.

rewrite H1; apply open_union2; trivial.
Qed.

(*Schepler*)
Lemma closed_family_intersection: forall
  (F:Family (point_set X)),
  (forall S:Ensemble (point_set X), In F S -> closed S) ->
  closed (FamilyIntersection F).
Proof.
intros.
unfold closed in H.
red.
assert (Ensembles.Complement (FamilyIntersection F) =
  FamilyUnion [ S:Ensemble (point_set X) |
                  In F (Ensembles.Complement S) ]).
apply Extensionality_Ensembles; split; red; intros.
apply NNPP.
red; intro.
red in H0; red in H0.
contradict H0.
constructor.
intros.
apply NNPP.
red; intro.
contradict H1.
exists (Ensembles.Complement S).
constructor.
rewrite Complement_Complement; assumption.
assumption.
destruct H0.
red; red; intro.
destruct H2.
destruct H0.
pose proof (H2 _ H0).
contradiction H3.

rewrite H0; apply open_family_union.
intros.
destruct H1.
pose proof (H _ H1).
rewrite Complement_Complement in H2; assumption.
Qed.

(*Schepler*)
Lemma closed_indexed_intersection: forall 
  {A:Type} (F:IndexedFamily A (point_set X)),
  (forall a:A, closed (F a)) -> closed (IndexedIntersection F).
Proof.
intros.
rewrite indexed_to_family_intersection.
apply closed_family_intersection.
intros.
destruct H0.
rewrite H1; trivial.
Qed.


(*Schepler*)
Lemma closed_finite_indexed_union: forall 
  {A:Type} (F:IndexedFamily A (point_set X)),
  FiniteT A -> (forall a:A, closed (F a)) ->
  closed (IndexedUnion F).
Proof.
intros.
red.
assert (Ensembles.Complement (IndexedUnion F) =
  IndexedIntersection (fun a:A => Ensembles.Complement (F a))).
apply Extensionality_Ensembles; split; red; intros.
constructor.
intros.
red; red; intro.
contradiction H1.
exists i.
assumption.
destruct H1.
red; red; intro.
destruct H2.
contradiction (H1 i).

rewrite H1; apply open_finite_indexed_intersection; trivial.
Qed.
End Closed.

Arguments closed [_] _.

Section Closed_Discrete.
Variable T:Type.

(*Wyckoff*)
Lemma all_closed_dscrt : forall S:Ensemble (point_set (X_dt T)), closed S.
intro S.
red. simpl. red.
auto.
Qed.
End Closed_Discrete.

Section Closed_Cofinite.
Variable T:Type.

(*Wyckoff*)
Lemma closed_cof_fin : 
  (forall S:Ensemble (point_set (X_ct T)), closed S <-> (Finite S) \/
  (S = Full_set _)).
intro S.
split.
(* -> *)
intro h1.
red in h1. simpl in h1. red in h1.
destruct h1 as [h2 | h3].
  (*h2*)
  right. simpl.
  pose proof (complement_empty T) as h4.
  rewrite <- h2 in h4.
  rewrite Complement_Complement in h4.
  assumption.
  (*h3*)
  left.
  rewrite Complement_Complement in h3.
  simpl.
  assumption.
(* <- *)
intro h5.
simpl. red.
destruct h5 as [h6 | h7].
right.
simpl in h6.
rewrite <- Complement_Complement in h6. assumption.
left.
simpl in h7.
pose proof (complement_full T) as h8.
rewrite <- h7 in h8. assumption.
Qed.
End Closed_Cofinite.


Section Clopen.
Variable X:TopologicalSpace.

(*Wyckoff*)
Definition clopen (S:Ensemble (point_set X)) := open S /\ closed S.

(*Wyckoff*)
Lemma clopen_full : clopen (Full_set (point_set X)).
red. split.  apply open_full. apply closed_full.
Qed.

(*Wyckoff*)
Lemma clopen_empty : clopen (Full_set (point_set X)).
red. split. apply open_full. apply closed_full.
Qed.
End Clopen.

Arguments clopen [_] _.

Section Clopen_Discrete.

(*Wyckoff*)
Lemma clopen_discrete : forall (T:Type) (S:(Ensemble (point_set (X_dt T)))), clopen S.
intros T S.
simpl in S.
red. split.
simpl. red. auto.
simpl. red. auto.
Qed.

End Clopen_Discrete.

(*This section is copied and pasted directly from the file 
  InteriorsClosures.v from Daniel Schepler's Topology user contribution
  package *)
Section interior_closure.

Variable X:TopologicalSpace.
Variable S:Ensemble (point_set X).

(*Schepler*)
Definition interior := FamilyUnion
  [ U:Ensemble (point_set X) | open U /\ Included U S ].
(*Schepler*)
Definition closure := FamilyIntersection
  [ F:Ensemble (point_set X) | closed F /\ Included S F ].

(*Schepler*)
Lemma interior_open : open interior.
Proof.
apply open_family_union.
intros.
destruct H as [[]]; trivial.
Qed.

(*Schepler*)
Lemma interior_deflationary : Included interior S.
Proof.
red; intros.
destruct H.
destruct H as [[]]; auto with sets.
Qed.

(*Schepler*)
Lemma interior_fixes_open: open S -> interior = S.
Proof.
intros.
apply Extensionality_Ensembles; split.
apply interior_deflationary.
red; intros.
exists S; trivial.
constructor; auto with sets.
Qed.

(*Schepler*)
Lemma interior_maximal: forall U:Ensemble (point_set X),
  open U -> Included U S -> Included U interior.
Proof.
intros.
red; intros.
exists U; trivial.
constructor; split; trivial.
Qed.

(*Schepler*)
Lemma closure_closed : closed closure.
Proof.
apply closed_family_intersection.
intros.
destruct H as [[]]; trivial.
Qed.

(*Schepler*)
Lemma closure_inflationary : Included S closure.
Proof.
red; intros.
constructor; intros.
destruct H0 as [[]].
auto with sets.
Qed.

(*Schepler*)
Lemma closure_fixes_closed : closed S -> closure = S.
Proof.
intro.
apply Extensionality_Ensembles; split.
red; intros.
destruct H0.
apply H0; split; auto with sets.
apply closure_inflationary.
Qed.

(*Schepler*)
Lemma closure_minimal: forall F:Ensemble (point_set X),
  closed F -> Included S F -> Included closure F.
Proof.
intros; red; intros.
destruct H1.
apply H1.
constructor; split; trivial.
Qed.

End interior_closure.

(*Wyckoff*)
Notation "P 'o'" := (interior _ P) (at level 50).
Notation "P -" := (closure _ P) (at level 50).
Notation "P '" := (Ensembles.Complement P) (at level 50).

Section interior_examples.

Variable T:Type.

(*Wyckoff*)
Example discrete_int_eq : forall S:Ensemble (point_set (X_dt T)), S o = S.
intro S.
simpl in S.
unfold interior.
apply Extensionality_Ensembles. red. split.
(*left*)
red.
intros x h2.
inversion h2 as [U y h3 h4 h5].
inversion h3 as [h6].
destruct h6 as [h7 h8].
auto with sets.
(*right*)
simpl.
red.
intros x h1.
apply family_union_intro with (S).
constructor; split; red; trivial.
assumption.
Qed.

(*Wyckoff*)
(*In a space with the cofinite topology, the interior of a cofinite subset is itself, 
and the interior of every other subset is empty.*)
Example cof_int_eq_empty : forall S : Ensemble (point_set (X_ct T)),   
  (Finite (S ') -> S o = S) /\ (~ Finite (S ') -> S o = Empty_set _).
intro S.
simpl in S.
split.
(*left*)
intro h1.
simpl in h1.
unfold interior.
simpl.
apply Extensionality_Ensembles.
red. split.
  (* left *)
  unfold Included.
  intros x h2.
  inversion h2 as [U y h3].
  inversion h3 as [h4].
  destruct h4.
  auto.
  (* right *)
  unfold Included.
  intros x h5.
  apply family_union_intro with (S).
  constructor.
  split.
    (* left *)
    red. right; assumption.
    (* right *)
    auto.
  (* right *)
  assumption.
(*right*)
simpl.
intro h6.
apply Extensionality_Ensembles.
red. split.
  (* left *)
  red.
  intros x h7.
  unfold interior in h7.
  inversion h7.
  inversion H.
  destruct H2.
  simpl in H2. red in H2.
  destruct H2.
    (*left*)
    rewrite H2 in H0.
    contradiction.
    (*right*)
    pose proof (complement_inclusion _ _ H3).    
    pose proof (Finite_downward_closed _ _ H2).
      pose proof (H5 (S ')).
    pose proof (H6 H4).
    contradiction.
  (* right *)
  auto with sets.
Qed.

End interior_examples.

Section closure_examples.
Variable T:Type.

(*Wyckoff*)
Example discrete_clo_eq : forall S:Ensemble (point_set (X_dt T)), S - = S.
intro S.
simpl in S.
unfold closure.
apply Extensionality_Ensembles. red. split.
  (*left*)
  red.  simpl.
  intros x h1.
  inversion h1 as [y h2].
  apply h2.
  constructor.
  split.
  red. trivial.
  auto with sets.
  (*right*)
  red. simpl.
  intros x h2.
  constructor.
  intros S0 h3.
  inversion h3 as [h4].
  destruct h4 as [[]h6].
  auto with sets.
Qed.

(*Wyckoff*)
Example cof_clo_eq_empty : forall S : Ensemble (point_set (X_ct T)),   
  (Finite S -> S - = S) /\ (~ Finite S -> S - = Full_set _).
simpl.
intro S. 
split.
  (*left*)
  intro h1.
  unfold closure.
  simpl.
  apply Extensionality_Ensembles.  red.
  split.
    (*left*)
    red.
    intros x h2.
    inversion h2 as [y h3].
    apply h3.
    constructor.
    split.
      (*left*)
      red.
      right.
      pose proof (Complement_Complement _ S) as h4.
      rewrite h4. assumption.
      (*right*)
      auto with sets.
    (*right*)
    red.
    intros x h5.
    constructor.
    intros S0 h6.
    inversion h6 as [h7].
    destruct h7 as [[]h8]; auto with sets.
  (*right*)
  intro h8.
  unfold closure.
  apply Extensionality_Ensembles.  red. split.
    (*left*)
    red.
    intros x h9.
    constructor.
    (*right*)
    red.
    intros x h10.
    constructor.
    intros S0 h11.
    inversion h11 as [h12].
    destruct h12 as [h13 h14].
    red in h13. simpl in h13. red in h13.
    case h13.
      (*left*)
      intro h15.
      pose proof (complement_empty T) as h16.
      rewrite <- h15 in h16.
      pose proof (Complement_Complement _ S0) as h17.
      simpl in h17.
      rewrite h16 in h17.
      rewrite <- h17.
      auto with sets.
      (*right*)
      intro h18.
      pose proof (Complement_Complement _ S0) as h19.
      simpl in h19.
      rewrite h19 in h18.
      pose proof (Finite_downward_closed _ _ h18) as h20.
      pose proof (h20 S h14) as h21.
      contradiction.
Qed.  
  

End closure_examples.

Arguments interior [_] _ _.
Arguments closure [_] _ _.
Arguments interior_open [_] _.
Arguments interior_deflationary [_] _ _ _.
Arguments interior_fixes_open [_] _ _.
Arguments interior_maximal [_] _ _ _ _ _ _.
Arguments closure_closed [_] _.
Arguments closure_inflationary [_] _ _ _.
Arguments closure_fixes_closed [_] _ _.
Arguments closure_minimal [_] _ _ _ _ _ _.

(*This is also copied and pasted from the same file as mentioned above previous
  section *)
Section interior_closure_relations.

Variable X:TopologicalSpace.

(*Schepler*)
Lemma interior_increasing: forall S T:Ensemble (point_set X),
  Included S T -> Included (interior S) (interior T).
Proof.
intros.
apply interior_maximal.
apply interior_open.
assert (Included (interior S) S).
apply interior_deflationary.
auto with sets.
Qed.

(*Schepler*)
Lemma interior_intersection: forall S T:Ensemble (point_set X),
  interior (Intersection S T) =
  Intersection (interior S) (interior T).
Proof.
intros.
apply Extensionality_Ensembles; split.
assert (Included (interior (Intersection S T)) (interior S)).
apply interior_increasing.
auto with sets.
assert (Included (interior (Intersection S T)) (interior T)).
apply interior_increasing.
auto with sets.
auto with sets.

apply interior_maximal.
apply open_intersection2; apply interior_open.
pose proof (interior_deflationary S).
pose proof (interior_deflationary T).
red; intros x H1; constructor; (apply H || apply H0);
destruct H1; trivial.
Qed.

(*Schepler*)
Lemma interior_union: forall S T:Ensemble (point_set X),
  Included (Union (interior S) (interior T))
           (interior (Union S T)).
Proof.
intros.
apply interior_maximal.
apply open_union2; apply interior_open.
pose proof (interior_deflationary S).
pose proof (interior_deflationary T).
auto with sets.
Qed.

(*Schepler*)
Lemma interior_complement: forall S:Ensemble (point_set X),
  interior (Ensembles.Complement S) = Ensembles.Complement (closure S).
Proof.
intros.
apply Extensionality_Ensembles; split.
rewrite <- Complement_Complement with (A:=interior (Ensembles.Complement S)).
apply complement_inclusion.
apply closure_minimal.
red.
rewrite Complement_Complement.
apply interior_open.
pattern S at 1;
rewrite <- Complement_Complement with (A:=S).
apply complement_inclusion.
apply interior_deflationary.

apply interior_maximal.
apply closure_closed.
apply complement_inclusion.
apply closure_inflationary.
Qed.

(*Schepler*)
Lemma closure_increasing: forall S T:Ensemble (point_set X),
  Included S T -> Included (closure S) (closure T).
Proof.
intros.
apply closure_minimal.
apply closure_closed.
pose proof (closure_inflationary T).
auto with sets.
Qed.

(*Schepler*)
Lemma closure_complement: forall S:Ensemble (point_set X),
  closure (Ensembles.Complement S) = Ensembles.Complement (interior S).
Proof.
intros.
pose proof (interior_complement (Ensembles.Complement S)).
rewrite Complement_Complement in H.
rewrite H.
rewrite Complement_Complement; reflexivity.
Qed.

(*Schepler*)
Lemma closure_union: forall S T:Ensemble (point_set X),
  closure (Union S T) = Union (closure S) (closure T).
Proof.
intros.
apply Extensionality_Ensembles; split.
apply closure_minimal.
apply closed_union2; apply closure_closed.
pose proof (closure_inflationary S).
pose proof (closure_inflationary T).
auto with sets.

assert (Included (closure S) (closure (Union S T))).
apply closure_increasing; auto with sets.
assert (Included (closure T) (closure (Union S T))).
apply closure_increasing; auto with sets.
auto with sets.
Qed.

(*Schepler*)
Lemma closure_union_finite : forall (F:Family (point_set X)), Finite F ->
  closure (FamilyUnion F) = FamilyUnion (Im F (@closure X)).
intros F h1.
induction h1.
(*Empty case*)
assert (h2: FamilyUnion (Empty_set (Ensemble (point_set X))) - = Empty_set _).
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red. intros x h3.
  pose proof (empty_family_union (point_set X)) as h4.
  rewrite h4 in h3.

  pose proof (closure_fixes_closed (Empty_set (point_set X))) as h5.
  pose proof (h5 (closed_empty X)) as h6. 
  rewrite h6 in h3. contradiction.
  (* >= *)
  red. intros x h1. contradiction.
rewrite h2.
apply Extensionality_Ensembles.
red. split.
(*<=*)
red. intros; contradiction.
(*>=*)
red. intros x h3.
inversion h3 as [S y h4].
pose proof (image_empty _ _ (@closure X)) as h5.
rewrite h5 in h4. contradiction.
(*"Add" case*)
rename x into S. rename A into F.
pose proof (family_union_add F S) as h2.
pose proof (f_equal (@closure X) h2) as h3.
pose proof (closure_union S (FamilyUnion F)) as h4.
rewrite IHh1 in h4.
rewrite h3. rewrite h4.
pose proof (family_union_add (Im F (@closure X))  (S -)) as h5.
rewrite <- h5.
pose proof (Im_add _ _ F S (@closure X)) as h6.
apply f_equal; symmetry; assumption. 
Qed.
 
(*Schepler*)
Lemma closure_intersection: forall S T:Ensemble (point_set X),
  Included (closure (Intersection S T))
           (Intersection (closure S) (closure T)).
Proof.
intros.
assert (Included (closure (Intersection S T)) (closure S)).
apply closure_increasing; auto with sets.
assert (Included (closure (Intersection S T)) (closure T)).
apply closure_increasing; auto with sets.
auto with sets.
Qed.

(*Schepler*)
Lemma meets_every_open_neighborhood_impl_closure:
  forall (S:Ensemble (point_set X)) (x:point_set X),
  (forall U:Ensemble (point_set X), open U -> In U x ->
                                Inhabited (Intersection S U)) ->
  In (closure S) x.
Proof.
intros.
apply NNPP; intro.
pose proof (H (Ensembles.Complement (closure S))).
destruct H1.
apply closure_closed.
exact H0.
destruct H1.
contradiction H2.
apply closure_inflationary.
assumption.
Qed.

(*Schepler*)
Lemma closure_impl_meets_every_open_neighborhood:
  forall (S:Ensemble (point_set X)) (x:point_set X),
  In (closure S) x ->
  forall U:Ensemble (point_set X), open U -> In U x ->
    Inhabited (Intersection S U).
Proof.
intros.
apply NNPP; intro.
assert (Included (closure S) (Ensembles.Complement U)).
apply closure_minimal.
unfold closed.
rewrite Complement_Complement; assumption.
red; intros.
intro.
contradiction H2.
exists x0; constructor; trivial.

contradict H1.
apply H3.
assumption.
Qed.

(*Schepler*)
Definition dense (S:Ensemble (point_set X)) : Prop :=
  closure S = Full_set _.

(*Schepler*)
Lemma dense_meets_every_nonempty_open:
  forall S:Ensemble (point_set X), dense S ->
    forall U:Ensemble (point_set X), open U -> Inhabited U ->
    Inhabited (Intersection S U).
Proof.
intros.
destruct H1.
apply closure_impl_meets_every_open_neighborhood with x; trivial.
rewrite H; constructor.
Qed.

(*Schepler*)
Lemma meets_every_nonempty_open_impl_dense:
  forall S:Ensemble (point_set X),
  (forall U:Ensemble (point_set X), open U -> Inhabited U ->
   Inhabited (Intersection S U)) ->
  dense S.
Proof.
intros.
apply Extensionality_Ensembles; split; red; intros.
constructor.
apply meets_every_open_neighborhood_impl_closure.
intros.
apply H; trivial.
exists x; trivial.
Qed.


End interior_closure_relations.

Arguments interior_increasing [_] _ _ _ _ _.
Arguments interior_intersection [_] _ _.
Arguments interior_union [_] _ _ _ _.
Arguments interior_complement [_] _.
Arguments closure_increasing [_] _ _ _ _ _.
Arguments closure_complement [_] _.
Arguments closure_union [_] _ _.
Arguments closure_intersection [_] _ _ _ _.


(*From Chapter 9 in Givant/Halmos*)
Section more_interior_closure_relations1.
Variable X:TopologicalSpace.

(*Wyckoff*)
Lemma closure_eq1 : forall (P:Ensemble (point_set X)), P - = P ' o '.
intro P.
apply Extensionality_Ensembles. red. split.
(*left*)
pose proof (interior_open (P ')) as h1.
pose proof (interior_deflationary (P ')) as h2.
pose proof (complement_inclusion ((P ') o) (P ')) as h3.
pose proof (h3 h2) as h4.
pose proof (Complement_Complement _ P) as h5.
rewrite h5 in h4.
assert (h6:(((P ') o) ') - = ((P ') o) ').
  unfold closure.
  apply Extensionality_Ensembles.
  unfold Same_set. split.
    (*left*)
    red.
    intros x h7.
    inversion h7 as [y h8].
    pose proof (h8 P) as h9.
    assert (h10:closed (((P ') o )')).
      red.  
      pose proof (Complement_Complement _ ((P ') o)) as h11.
      rewrite h11.  assumption.
    assert (h11:Included (((P ') o) ') (((P ' ) o) ')).
      auto with sets.
    pose proof (conj h10 h11) as h12.
    assert (h13:In [F : Ensemble (point_set X) | closed F /\ Included (((P ') o) ') F] 
       (((P ') o) ')).
      constructor; assumption.    
    auto with sets.
    (*rigbt*)
    red. intros x h14.
    constructor.
    intros S h15.
    inversion h15 as [h16].
    destruct h16 as [h17 h18].
    auto with sets.
pose proof (closure_closed (((P ') o) ')) as h19.
rewrite h6 in h19.
apply (closure_minimal _ _ h19 h4).
(*right*)
unfold closure.
intros x h20.
constructor.
intros Q h21.
inversion h21 as [h22].
destruct h22 as [h23 h24].
pose proof (complement_inclusion _ _ h24) as h25.
red in h23.
pose proof (interior_maximal _ _ h23 h25) as h26.
pose proof (complement_inclusion _ _ h26) as h27.
pose proof (Complement_Complement _ Q) as h28.
rewrite h28 in h27.
auto with sets.
Qed.

(*Wyckoff*)
Lemma interior_eq: forall (P:Ensemble (point_set X)), P o = P ' - '.
intro P.
pose proof (closure_eq1 (P ')) as h1.
pose proof (Complement_Complement _ P) as h2.
rewrite h2 in h1.
rewrite h1.
simpl.
rewrite Complement_Complement.
reflexivity.
Qed.

Arguments dense [_] _.

(*Wyckoff*)
(*The reverse implication is above somewhere *)
Lemma dense_impl_meets_every_nonempty_open:
  forall S:Ensemble (point_set X),
  dense S ->
  (forall U:Ensemble (point_set X), open U -> Inhabited U ->
   Inhabited (Intersection S U)).
intros S h1 U h2 h3.
red in h1.
inversion h3 as [x h4].
pose proof (Full_intro _ x) as h5.
rewrite <- h1 in h5.
apply closure_impl_meets_every_open_neighborhood with x; assumption.
Qed.

(*Wyckoff*)
Definition dense_in (P Q:(Ensemble (point_set X))) := Included Q (P -).

Section Dense_Examples.
Variable T:Type.
(* In a discrete space, no proper subset of the space
is dense.*)

(*Wyckoff*)
Lemma dense_discrete_none : 
  let Xt := (point_set (X_dt T)) in
  forall S:(Ensemble Xt), S <> (Full_set Xt) -> ~dense S.
intros Xt S h0.
red.
intro h1.
red in h1.
unfold closure in h1.
unfold closed in h1.
simpl in h1.
assert (h2: open_dscrt _ (S ') /\ Included S S). split.
  (*left*)
  red; trivial.
  (*right*) 
  auto with sets.
assert (h4:Included (Full_set T) S).
rewrite <- h1.
red.
intros x h4.
inversion h4 as [y h5].
pose proof (h5 S) as h6.
assert (h7:In [F : Ensemble T | open_dscrt T (F ') /\ Included S F] S).
  constructor.
  apply h2.
apply (h6 h7).
assert (h8 : S = (Full_set T)).
  apply Extensionality_Ensembles. red. split.
  (*left*)
  red.
  intros x h9.
  constructor.
  (*right*)
  assumption.
contradiction.
Qed.

(*Wyckoff*)  
(*In an infinite space with cofinite topology, every infinite subset of
the space is dense.
*)
Lemma infnt_cof_infnt_dense : 
  let Xt := (point_set (X_ct T)) in 
  forall S:(Ensemble (point_set (X_ct T))), 
  ~ FiniteT T -> ~ Finite S -> dense S.
intros Xt S h1 h2.
red.
simpl in h2.
simpl.
apply cof_clo_eq_empty.
simpl.
assumption.
Qed.

End Dense_Examples.

(*Wyckoff*)
(*A set P is defined to
be nowhere dense if it is not dense in any non-empty open set.*)
Definition now_dense (P: (Ensemble (point_set X))) : Prop :=
  forall S:(Ensemble (point_set X)), open S -> Inhabited S -> ~ dense_in P S.

(*Wyckoff*)
Lemma closure_empty : (Empty_set (point_set X)) - = Empty_set (point_set X).
apply Extensionality_Ensembles.
red. split.
(*left*)
apply closure_minimal.
apply closed_empty.
auto with sets.
(*right*)
auto with sets.
Qed.

(*Wyckoff*)
Lemma interior_empty : (Empty_set (point_set X)) o = Empty_set (point_set X).
unfold interior.
apply Extensionality_Ensembles.
red. split.
(*left*)
red.
intros x h1.
inversion h1 as [U y h2 h3].
inversion h2 as [h4].
destruct h4 as [h5 h6].
auto with sets.
(*right*)
red.
intros x h7.
contradiction.
Qed.

(*Wyckoff*)
Lemma closure_full : (Full_set (point_set X)) - = Full_set (point_set X).
Proof.
apply Extensionality_Ensembles.
red.
split.
(*left*)
unfold closure.
red. intros.
constructor.
(*right*)
apply closure_inflationary.
Qed.

(*Wyckoff*)
Lemma interior_full : (Full_set (point_set X)) o = Full_set (point_set X).
apply Extensionality_Ensembles.
red.  split.
(*left*)
apply interior_deflationary.
(*right*)
unfold interior. red. intros x h1.
apply family_union_intro with (Full_set (point_set X)).
constructor.
split.
  (*left*)
  apply open_full.
  (*right*)
  auto with sets.
constructor.
Qed.



(*Wyckoff*)
(*Nowhere dense means that the interior of the closure of P is empty.*)

Lemma now_dense_iff1 : forall P : (Ensemble (point_set X)), now_dense P <->
  P - o = Empty_set (point_set X).
intros P.
split.
(*left*)
intro h1.  
apply Extensionality_Ensembles. red. split.
  (*left*)
  (*red. intros x h2. *)
  red in h1.
  unfold interior.
  unfold dense_in in h1.
  assert (h2:forall U:Ensemble (point_set X), open U -> Included U (P -) ->
    U = (Empty_set _)).
    intros U h3 h4.
    case (classic (U = (Empty_set _))); trivial.
      intro h5.
      pose proof (not_empty_Inhabited _ U h5) as h6.
      pose proof (h1 U h3 h6) as h7.
      contradiction.
  red.
  intros x h8.
  inversion h8 as [U y h9 h10].
  inversion h9 as [h11].
  destruct h11 as [h12  h13].
  pose proof (h2 U h12 h13) as h14.
  rewrite h14 in h10.
  contradiction.
  (*right*)
  auto with sets.
(*right*)
intro h11.
red.
intros S h1 h2.
pose proof (Inhabited_not_empty _ _ h2) as h3.
unfold dense_in.
intro h4.
unfold interior in h11.
assert (h12: In [U : Ensemble (point_set X) | open U /\ Included U (P -)] S).
  constructor. split; assumption.
inversion h2 as [x h13].
pose proof (family_union_intro _ _  _ x h12 h13) as h14.
rewrite h11 in h14.
contradiction.
Qed.

(*Wyckoff*)
Lemma now_dense_iff2 : forall P : (Ensemble (point_set X)), now_dense P <->
  (forall U:Ensemble (point_set X), open U -> Included U (P -) ->
    U = (Empty_set _)).
intros P. 
split. 
  (*left*)
  intros h3 U h5 h6.
  case (classic (U = (Empty_set _))); trivial.
      intro h7.
      pose proof (not_empty_Inhabited _ U h7) as h8.
      red in h3.
      pose proof (h3 U h5 h8) as h9.
      unfold dense_in in h9.
      contradiction.
  (*right*)
  intros h1. 
  red.
  intros S h2 h3.
  red. intro h4.
  red in h4.
  pose proof (h1 S h2 h4) as h5.
  rewrite h5 in h3.
  inversion h3 as [x h7].
contradiction.
Qed.

(*Wyckoff*)
(* "To say that no
non-empty open set is included in P − [equivalent to now_dense from above lemma]
is equivalent to saying that every non-
empty open set has a non-empty intersection with P - '" *)

Lemma now_dense_iff3 : forall P : (Ensemble (point_set X)), now_dense P <->
  forall (S : (Ensemble (point_set X))), open S -> Inhabited S -> 
    Inhabited (Intersection (P - ') S).
intro P. 
pose proof (now_dense_iff2 P) as h1.
rewrite h1. clear h1.
split.
(*left*)
intros h1 S h2 h3.
case (classic (Included S (P -))) as [h4 | h5].
  (*h4*)
  pose proof (h1 S h2 h4) as h6.
  rewrite h6 in h3.
  inversion h3 as [x h7].
  contradiction.
  (*h5*)
  apply complement_meets_non_subset; trivial.
(*right*)
intros h1 U h2 h3.
case (classic (Inhabited U)) as [h4 | h5].
(*h4*)
pose proof (h1 _ h2 h4) as h6.
pose proof (complement_meets_non_subset U (P -) h4) as h7.
rewrite <- h7 in h6.
contradiction.
(*h5*)
pose proof (not_empty_Inhabited (point_set X) U) as h6.
case (classic (U = Empty_set (point_set X))); trivial.
intro h7.
pose proof (h6 h7) as h8; contradiction.
Qed.

  
(*Wyckoff*)
(*Thus, P is nowhere dense if and only if P − 'is dense.*)
Lemma now_dense_iff4 : forall P: Ensemble (point_set X), now_dense P <->
  dense (P - ').
intro P.
rewrite now_dense_iff3.
pose proof (meets_every_nonempty_open_impl_dense _ ((P -) ')) as h1.
pose proof (dense_impl_meets_every_nonempty_open ((P -) ')) as h2.
tauto.
Qed.


End more_interior_closure_relations1.

Arguments now_dense [_] _.

Section nowhere_dense_examples.
Variable T:Type.

(*Wyckoff*)
(*In a discrete space only the empty set is nowhere dense.*)
Example now_dense_dscrt : forall S:(Ensemble (point_set (X_dt T))), 
  now_dense S -> S = (Empty_set _).
intros S h1.
pose proof (now_dense_iff2 (X_dt T) S) as h2.
rewrite h2 in h1.
simpl.
simpl in h1.
unfold open_dscrt in h1.
apply (h1 S I (closure_inflationary S)).
Qed.


(*Wyckoff*)
(*"In an infinite space with the cofinite topology,
a set is nowhere dense just in case it is finite."*)
Example now_dense_cof : ~FiniteT T -> forall S:(Ensemble (point_set (X_ct T))), 
  now_dense S <-> Finite S.
intros h1 S.
split.
(* -> *)
intro h2.
rewrite now_dense_iff1 in h2.
pose proof (cof_int_eq_empty _ (S -)) as h3.
destruct h3 as [h4  h5].
case (classic (Finite ((S -) '))) as [h6 | h7].
  (*h6*)
  pose proof (h4 h6) as h8.
  case (classic (Finite S)); trivial.
  intro h9.
  pose proof (cof_clo_eq_empty _ S) as h10.
  destruct h10 as [h11 h12].
  pose proof (h12 h9) as h13.
  assert (h14 : Full_set (point_set (X_ct T)) = Empty_set _).
    congruence.  
  pose proof (Empty_is_finite (point_set (X_ct T))) as h15.
  rewrite <- h14 in h15. 
  pose proof (Finite_FiniteT _ h15) as h16.
  contradiction.  
  (*h7*)
  pose proof (h5 h7) as h8.
  pose proof (closure_closed S) as h9.
  unfold closed in h9. simpl in h9. red in h9.
  case h9 as [h10 | h11].
    (*h10*)
    pose proof (complement_empty T) as h12.
    rewrite <- h10 in h12.
    pose proof (Complement_Complement _ (S -)) as h13.
    simpl in h12. simpl in h13.
    rewrite h13 in h12.
    pose proof (interior_full (X_ct T)) as h14.
    simpl in h14.
    rewrite <- h12 in h14 at 1.
    rewrite h8 in h14. simpl in h14.  
    pose proof (Empty_is_finite T) as h15.
    rewrite h14 in h15.
    pose proof (Finite_FiniteT _ h15) as h16.
    contradiction.
    (*h11*)
    pose proof (Complement_Complement _ (S -)) as h12.
    simpl in h11. simpl in h12.
    rewrite h12 in h11.
    simpl.
    apply Finite_downward_closed with (S -).
    simpl. assumption.
    apply closure_inflationary.
(* <- *)  
intro h2.
rewrite now_dense_iff1.
pose proof (cof_clo_eq_empty _ S) as h3.
destruct h3 as [h4 h4'].
pose proof (h4 h2) as h5.
pose proof (cof_int_eq_empty _ S) as h6.
destruct h6 as [h7  h8].
case (classic (Finite (S '))) as [h9 | h10].
  (*h9*)
  pose proof (h7 h9) as h10.
  pose proof (inf_comp_fin h1) as h11.
  pose proof (h11 S h9) as h12.
  simpl in h2.
  contradiction.
  (*h10*)
  pose proof (h8 h10) as h11.
  rewrite <- h5 in h11.
  assumption.
Qed.
End nowhere_dense_examples.

Section more_interior_closure_relations2.

(*Wyckoff*)
Lemma closure_eq2 : forall {X:TopologicalSpace}
  (S:Ensemble (point_set X)), closure S = 
    [x:(point_set X) | forall U:Ensemble (point_set X), 
      open U -> In U x -> Inhabited (Intersection S U)].
intros X S.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros x h1.
constructor.
intros U h2 h3.
apply closure_impl_meets_every_open_neighborhood with x; assumption.
(* >= *)
red.
intros x h2.
inversion h2 as [h3].
apply meets_every_open_neighborhood_impl_closure; assumption.
Qed.

(*Wyckoff*)
(* This is Exercise 13 c from Chapter 9. 
  "(P \/ Q)-' = P-' /\ Q-' "*)

Lemma closure_comp_distribute : forall {X:TopologicalSpace} 
  (P Q:Ensemble (point_set X)), (Union P Q) - ' = Intersection (P - ') (Q - ').
intros X P Q.
pose proof (closure_union P Q) as h1.
rewrite h1.
apply comp_union.
Qed.


(*Wyckoff*)
(* This is Exercise 13 d from Chapter 9. 
   "P /\ Q - <= (P /\ Q) - whenever P is open."*)
Lemma closure_over_intersection_open : forall {X:TopologicalSpace} 
  (P Q:Ensemble (point_set X)), open P -> 
  Included (Intersection P (Q -)) ((Intersection P Q) -).
intros X P Q h1.
red.
intros x h2.
destruct h2 as [x h2l h2r].
rewrite (closure_eq2 Q) in h2r.
inversion h2r as [h3].
rewrite (closure_eq2 (Intersection P Q)).
constructor.
intros U h5 h6.
pose proof (open_intersection2 _ _ _ h1 h5) as h7.
pose proof (Intersection_intro _ _ _ _ h2l h6) as h8.
pose proof (h3 _ h7 h8) as h9.
pose proof (assoc_prod_psa Q P U) as h10.
pose proof (comm_prod_psa P Q) as h11.
rewrite h11.
rewrite <- h10.
assumption.
Qed.

(*Wyckoff*)
Lemma union_now_dense : forall {X:TopologicalSpace} (P Q:Ensemble (point_set X)), 
  now_dense P -> now_dense Q -> now_dense (Union P Q).
intros X P Q h1 h2.
pose proof (now_dense_iff4 _ P) as h3.
pose proof (now_dense_iff4 _ Q) as h4.
rewrite h3 in h1.
rewrite h4 in h2.
red in h1. red in h2.
assert (h5: (Q - ') = Intersection (Full_set _) (Q - ')).
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red. intros. constructor. constructor. assumption.
  (* >= *)
  red. intros ? h6. inversion h6; assumption.
rewrite <- h1 in h5.
pose proof (closure_over_intersection_open (Q - ') (P - ')) as h6.
pose proof (Intersection_commutative _ (P - ' -) (Q - ')) as h7.
assert (h8:open (Q - ')).
  apply closed_complement_open.
  pose proof (Complement_Complement _ (Q -)) as h9.
  rewrite h9.
  apply closure_closed.
apply h6 in h8.
rewrite h7 in h5.
rewrite <- h5 in h8.
pose proof (closure_comp_distribute Q P) as h9.
pose proof (Union_commutative _ P Q) as h10.
rewrite h10.
apply now_dense_iff4.
red.
pose proof (f_equal (closure (X:=X)) h9) as h11.
rewrite h11.
pose proof (closure_closed (Intersection (Q - ') (P - '))) as h12.
pose proof (closure_minimal _ _ h12 h8) as h13.
rewrite h2 in h13.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. intros x ?. constructor.
(* >= *) assumption.
Qed.

(*Wyckoff*)
(*"A finite union of nowhere dense sets is again nowhere dense"*)
Lemma finite_union_now_dense : forall {X:TopologicalSpace} (F:Family (point_set X)), 
  Finite F -> (forall A, In F A -> now_dense A ) -> now_dense (FamilyUnion F).
intros X F h1 h2.
induction h1 as [|F A' h3 S h4].
(*Empty*)
pose proof (now_dense_iff1 X (Empty_set _)) as h3.
pose proof (empty_family_union (point_set X)) as h4.
rewrite h4.
rewrite h3.
pose proof (closure_empty X) as h5.
rewrite h5.
apply interior_empty.
(*Add*)
(*unfold Add. unfold Add in h2.*)
pose proof (family_union_add F S) as h5.
rewrite h5.
pose proof (Add_intro2 _ F S) as h6.
pose proof (h2 _ h6) as h7.
assert (h8:forall A:Ensemble (point_set X), In F A -> now_dense A).
  intros A h9.
  apply NNPP.
  intro h10.
  assert (h11: ~ In (Add F S) A). 
    pose proof (h2 A) as h11. tauto.
    pose proof (Add_intro1 _ F S A) as h12.
    apply h12 in h9.
    contradiction.
apply h3 in h8.
apply union_now_dense; assumption.
Qed.


End more_interior_closure_relations2.

(*Wyckoff*)
(*"A set is said to be meager if it is the countable union of nowhere dense sets."*)

Definition meager {X:TopologicalSpace} (S:Ensemble (point_set X)) := 
  exists (It:Type) (F:(IndexedFamily It (point_set X))), CountableT It /\
  (forall (i:It), now_dense (F i)) /\  (S = IndexedUnion F).

Section meager_examples.
Variable T:Type.

(*Wyckoff*)
(*"In a discrete space only the empty set is meager."*)
Example meager_empty_dscrt : forall (S:Ensemble (point_set (X_dt T))), 
  meager S -> S = (Empty_set _).
intros S h1.
(*simpl in S.
simpl.*)
red in h1.
elim h1. 
intros It h2.
elim h2.
intros F h3.
clear h1 h2.
destruct h3 as [h4 h5].
destruct h5 as [h6 h7].
assert (h8: forall i:It, (F i) = (Empty_set _)).
  intro i.
  pose proof (h6 i) as h9.
  apply (now_dense_dscrt _ (F i) h9).
rewrite h7.
case (classic (Inhabited S)) as [h10 | h11].
(*h10*)
inversion h10 as [x h11].
rewrite h7 in h11.
inversion h11 as [i y h12].
pose proof (h8 i) as h9.
rewrite h9 in h12.
contradiction.
(*h11*)
rewrite <- h7.
case (classic (S = Empty_set _)); trivial.
intro h12.
pose proof (not_empty_Inhabited _ _ h12) as h13.
contradiction.
Qed. 


(*Wyckoff*)
(*"In an infinite space with the cofinite topology, a set
is meager just in case it is countable."*)
Example meager_countable_cof_inf : (~FiniteT T) -> 
  forall S:(Ensemble (point_set (X_ct T))), (meager S <-> Countable S).
intros h1 S.
split.
(*->*)
intro h2.
unfold meager in h2.
destruct h2 as [It h3].
destruct h3 as [F h4].
destruct h4 as [h5 h6].
destruct h6 as [h7 h8].
rewrite h8.
simpl.
assert (h9:forall (i:It), Countable (F i)).
  intro i.
  pose proof (now_dense_cof _ h1 (F i)) as h10.
  apply h10 in h7.
  apply Finite_impl_Countable; assumption.
apply countable_union; assumption.
(* <- *)
intro h2.
red.
red in h2.
simpl in h2.
simpl.
pose proof CountableT_is_FiniteT_or_countably_infinite as h3.
destruct (h3 _ h2) as [h0|h4].
  (*h0 *)
  pose (SingF S) as F'.
  simpl in F'.
  pose (fun (s:{x:T | In S x}) => Singleton (proj1_sig s)) as F.
  exists {x:T | In S x}, F.
  split.
  apply FiniteT_impl_CountableT; assumption.
  split.
  intro s.
  pose proof (now_dense_cof _ h1 (F s)) as h4.
  rewrite h4.
  simpl.
  unfold F.
  apply Singleton_is_finite.
  pose proof (indexed_to_family_union _ _ F) as h4.
  rewrite h4.
  pose proof (union_singF_eq S) as h5.
  unfold SingF in h5.
  simpl in h5.
  unfold ImageFamily.
  unfold F.
  rewrite h5 at 1.
  assert (h6: Im S (Singleton (U:=T)) = Im (Full_set _) F).
    apply Extensionality_Ensembles.
    red. split.
    (* <= *)
    red.
    intros A h6.
    inversion h6 as [? h7 ? h8].
    simpl in x.  pose proof (Full_intro _ (exist _ x h7)) as h9.
    assert (h10:A = F (exist (In S) x h7)).
      unfold F.
      rewrite h8.
      simpl. reflexivity.
    rewrite h8 in h6.
    apply (Im_intro _ _ _ _ _ h9 _ h10).
    (* >= *)
    red. intros A h6.
    inversion h6 as [? ? ? h8].
    unfold F in h8.
    apply (Im_intro _ _ _ _ _ (proj2_sig x) _ h8).
  f_equal. assumption.
  (* h4 *)
  destruct h4 as [F h5].
  pose proof (bijective_impl_invertible _ h5) as h6.
  inversion h6 as [G h7 h8].
  assert (h9 : bijective G).
    apply invertible_impl_bijective.
    apply (intro_invertible G F h8 h7).
  pose (fun n:nat => Singleton (proj1_sig (G n))) as G'.
  exists _, G'.
  split. apply (intro_nat_injection _ (@id nat)).
    red. intros x1 x2 h10. unfold id in h10. assumption.
  split.
    intro n.
    pose proof (now_dense_cof _ h1 (G' n)) as h10.
    rewrite h10.
    apply Singleton_is_finite.
    pose proof (union_singF_eq S) as h10.
    unfold X_ct. simpl.
  simpl.
  pose proof (indexed_to_family_union _ _ G') as h11.
  rewrite h11. rewrite h10.
  unfold SingF. unfold ImageFamily.
  unfold X_ct. simpl.
  assert (h12: Im S (Singleton (U:=T)) = Im (Full_set nat) G').
    unfold G'.
    apply Extensionality_Ensembles. red. split.
    (* <= *)
    red. intros A h12.   
    inversion h12 as [t h13 ? h14 h15].
    simpl in t.
    red in h9.
    destruct h9 as [? h9].
    red in h9.
    pose proof (h9 (exist _ t h13)) as h16.
    destruct h16 as [n h17].
    pose proof (Full_intro _ n) as h18.
    symmetry in h17.
    pose proof (Im_intro _ _ _ _ _ h18 _ h17) as h19.
    pose proof (f_equal (@proj1_sig _ _ ) h17) as h20.
    unfold G'.
    eapply Im_intro. apply h18.
    rewrite <- h17.
    simpl. assumption.
    (* >= *)
    red. intros A h12.
    inversion h12 as [n ? B h14].
    eapply Im_intro.
    apply (proj2_sig (G n)). assumption.
f_equal; assumption.
Qed.

(*Wyckoff*)
(*"In particular, if the space is countably infinite, then every subset is meager."*)
Corollary meager_cof_countable : ~FiniteT T-> CountableT T -> 
  forall S:(Ensemble (point_set (X_ct T))), meager S.
intros h1 h2 S.
pose proof (meager_countable_cof_inf h1 S) as h3.
rewrite h3.
apply CountableT_Countable. simpl. assumption.
Qed.
    
End meager_examples.



Section Boundary.
Variable X:TopologicalSpace.

(*Wyckoff*)
(*"A point x is a boundary point of a set P if every neighborhood of x
contains points of P and points of P '."*)
Definition boundary_point (x:point_set X) (P:Ensemble (point_set X)) : Prop :=
  forall (U:Ensemble (point_set X)), open U -> In U x -> 
    Inhabited (Intersection P U) /\ Inhabited (Intersection (P ') U).

(* "In other words, x is a boundary point
  of P just in case it belongs to the closure of both P and P '." *)

(*Wyckoff*)
Lemma boundary_point_iff : forall (x:point_set X) (P:Ensemble (point_set X)), 
  boundary_point x P <-> (In (P -) x) /\ (In (P ' -) x).
intros x P.
rewrite closure_eq2.
rewrite closure_eq2.
split.
(* -> *)
intro h1.
unfold boundary_point in h1.
  split.
  (* P *)
  constructor. apply h1.
  (* P ' *)
  constructor. apply h1.
(* < - *)
intro h1.
destruct h1 as [h1l h1r].
red.
destruct h1l as [h1l].
destruct h1r as [h1r].
intros. 
split.
apply h1l; assumption.
apply h1r; assumption.
Qed.


(*Wyckoff*)
(* "The boundary of P is the set of its boundary points" *)
Definition boundary (P:Ensemble (point_set X)) : Ensemble (point_set X) := 
  [x:(point_set X) | boundary_point x P].

(*Wyckoff*)
(* " . . . that is to say, it is the set P - /\ (P ' − ) " *)
Lemma boundary_eq : forall (P:Ensemble (point_set X)), 
  boundary P = Intersection (P -) (P ' -).
intro P.
unfold boundary.
apply Extensionality_Ensembles.
red. intros.
split.
(*left*)
red. intros x h1.
destruct h1 as [h1].
apply boundary_point_iff in h1.
constructor; apply h1.
(*right*)
red.
intros x h1.
destruct h1 as [x h1 h2].
pose proof (conj h1 h2) as h3.
constructor.
rewrite boundary_point_iff; assumption.
Qed.

(*Wyckoff*)
(*"The boundary of a set is always closed, because it is the intersection of
two closed sets."*)
Lemma boundary_closed : forall (P:Ensemble (point_set X)), 
  closed (boundary P).
intros.
rewrite boundary_eq.
apply closed_intersection2; apply closure_closed.
Qed.

(*Wyckoff*)
(*"If a set P is open, then its complement P is closed, and therefore P ' - = P ' ;
in this case, the boundary of P is just the set-theoretic difference P - (- P)."*)
Lemma open_boundary_difference : forall P:(Ensemble (point_set X)), open P ->
  boundary P = Setminus (P -) P.
intros P h1.
pose proof (Complement_Complement _ P) as h2.
rewrite <- h2 in h1.
change (open ((P ') ')) with (closed (P ')) in h1.
rewrite boundary_eq.
pose proof (closure_closed (P ')).
assert (h3: (P ') - = P ').
  apply Extensionality_Ensembles.
  split. 
  (* <= *)
  simpl in h1.
  apply closure_minimal. assumption. auto with sets.
  (* >= *)
  apply closure_inflationary.
rewrite h3.
rewrite setminus_int_complement; reflexivity.
Qed.

(*Wyckoff*)
(*"More generally, the boundary of any nowhere
dense set P in a topological space is just the closure P - "*)
Lemma boundary_now_dense : forall (P:Ensemble (point_set X)), now_dense P ->
  boundary P = P -.
intros P h1.
rewrite now_dense_iff4 in h1.
pose proof (closure_inflationary P) as h2.
pose proof (complement_inclusion _ _ h2) as h3.
pose proof (closure_increasing _ _  h3) as h4.
red in h1.
rewrite h1 in h4.
assert (h5:(Full_set _ = P ' -)).
  apply Extensionality_Ensembles.
  red. split; auto with sets. red. intros; constructor.
rewrite boundary_eq.
rewrite <- h5.
apply Extensionality_Ensembles.
red.
split; red. 
(* <= *) intros x h6. destruct h6; assumption. 
(* >= *) intros x h6. constructor; [assumption|constructor].
Qed.



(*Wyckoff*)
(*"The closure of a set P ought to be, and is, the union of P with its
boundary. "*)
Lemma closure_union_boundary : forall P:Ensemble (point_set X), 
  let Q := (boundary P) in P - = Union P Q.
intros P Q.
apply Extensionality_Ensembles.
red. split.
(* >= *)
Focus 2. 
pose proof (boundary_eq P) as h1.
change (boundary P) with Q in h1.
pose proof (closure_inflationary P) as h2.
assert (h3: Included Q (P -)).
  red.  intros x h3. rewrite h1 in h3. destruct h3; assumption. 
apply included2_impl_union_included. constructor; assumption.
(* <= *)
red. intros x h1.
case (classic (In P x)). auto with sets.
intro h2.
change (~ In P x) with (In (P ') x) in h2.
assert (h3:In Q x).
  unfold Q.
  rewrite boundary_eq.
  pose proof (closure_inflationary (P ')).
  constructor; auto with sets.
auto with sets.
Qed.


End Boundary.

Arguments boundary [_] _ _.

Section Boundary_Examples.
Variable T:Type.

(*Wyckoff*)
(*" In a discrete
space every set of points has an empty boundary."*)

Example boundary_discrete : forall (S:Ensemble (point_set (X_dt T))), 
  boundary S = Empty_set _.
intro S.
(*simpl in S.
simpl.*)
rewrite open_boundary_difference.
assert (h1: closed S).
  apply all_closed_dscrt.
pose proof (closure_minimal S _ h1) as h2.
assert (h3:Included S S); auto with sets.
apply h2 in h3.
apply setminus_sub_sup; assumption.
simpl. red. trivial.
Qed.

(*Wyckoff*)
(* "In an infinite space with
the cofinite topology, the boundary of a finite set is itself, the boundary of
a cofinite set is its complement, and the boundary of an infinite set with an
infinite complement is the whole space."*)
Example boundary_inf_cof : ~FiniteT T -> forall S:(Ensemble (point_set (X_ct T))),
  ((Finite S) -> boundary S = S) /\ 
  ((Finite (S ')) -> boundary S = S ') /\
  ((~ Finite S) -> (~Finite (S ')) -> boundary S = (Full_set _)).
intros h1 S.
repeat split.
(* Conclusion I *)
intro h2.
pose proof closed_cof_fin as h3.
pose proof (or_introl (S = Full_set _) h2) as h4.
rewrite <- h3 in h4.
rewrite boundary_eq.
pose proof (closure_fixes_closed _ h4) as h5.
rewrite h5.
pose proof (Complement_Complement _ S) as h6.
rewrite <- h6 in h2.
pose proof (inf_comp_fin h1 _ h2) as h7. 
pose proof (infnt_cof_infnt_dense T _ h1 h7) as h8.
red in h8.
rewrite h8.
apply intersection_full_set.
(* Conclusion II *)
intro h2.
rewrite boundary_eq.
assert (h3: open S).
  simpl. red. right. assumption.
pose proof (Complement_Complement _ S) as h4.
rewrite <- h4 in h3.
assert (h5:closed (S ')). assumption.
pose proof (closure_fixes_closed _ h5) as h6.
rewrite h6.
pose proof (inf_comp_fin h1 _ h2) as h7.
pose proof (infnt_cof_infnt_dense _ _ h1 h7) as h8.
red in h8.
rewrite h8.
rewrite Intersection_commutative.
apply intersection_full_set.
(* Conclusion III *)
intros h2 h3.
rewrite boundary_eq.
pose proof (infnt_cof_infnt_dense _ _ h1 h2) as h4.
pose proof (infnt_cof_infnt_dense _ _ h1 h3) as h5.
red in h4. red in h5.
rewrite h4. rewrite h5.
apply intersection_full_set.
Qed.

End Boundary_Examples.



Section Regular.
Variable X:TopologicalSpace.

(*Wyckoff*)
(*"An open set is said to be regular if it coincides with the interior of its
own closure."*)
Definition regular (S:(Ensemble (point_set X))) : Prop := S = S - o.

(*Wyckoff*)
Lemma regular_open : forall (S:Ensemble (point_set X)), regular S -> open S.
intros S h1.
red in h1.
rewrite h1.
apply interior_open.
Qed.

(*Wyckoff*)
Lemma regular_iff : forall (P:Ensemble (point_set X)), 
  regular P <-> P = P - ' - '.
intro P.
unfold regular.
rewrite interior_eq. 
tauto.
Qed.

(*Wyckoff*)
Example example_open_not_regular : 
  (exists (P:Ensemble (point_set X)), Inhabited P /\ now_dense P) -> 
   exists (S:Ensemble (point_set X)), open S /\ ~ regular S.
intro h1.
destruct h1 as [P h1].
destruct h1 as [h1l h1r].
assert (h2:open (P - ')).
  change (open (P - ')) with (closed (P -)).
  apply closure_closed.
assert (h3: (P - ') <> Full_set _).
  intro h4.
  pose proof (complement_full (point_set X)) as h5.
  rewrite <- h4 in h5.
  pose proof (Complement_Complement _ (P -)) as h6.
  rewrite h6 in h5.
  pose proof (closure_inflationary P) as h7.
  rewrite h5 in h7.
  assert (h8:P = Empty_set _). 
    apply Extensionality_Ensembles. red. 
    split; [assumption | auto with sets].
  inversion h1l as [x h9].
  rewrite h8 in h9.
  contradiction.
exists (P - ').
split; try assumption.
assert (h4:(P - ' - ' - ') = (Full_set (point_set X)) -> 
  ~ regular (P - ')).
  intro h4.
  rewrite <- h4 in h3.
  unfold regular.
  rewrite interior_eq. 
  assumption.
assert (h5:P - ' - ' - ' = Full_set (point_set X)).
  apply now_dense_iff4 in h1r.
  red in h1r.
  rewrite h1r.
  pose proof (complement_full (point_set X)) as h5.
  rewrite h5.
  pose proof (closure_empty X) as h6.
  rewrite h6.
  apply complement_empty.
tauto.
Qed.
 
  


End Regular.







