/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            config.c
 *
 *  Sat Aug 21 23:59:26 2004
 *  Copyright  2004  Gergely Polonkai
 *  polesz@techinfo.hu
 ****************************************************************************/
/***************************************************************************
 * Configuration backends. Currently only GConf2 is supported.
 ****************************************************************************/

#include "botcommander.h"

#include <gconf/gconf-client.h>
#include <gtk/gtk.h>
#include <string.h>

#include "typedefs.h"

GConfClient *gconf_client;
OldTConfigData ConfigData;

/*
 * GConfSanityCheckString()
 * Copied from gnome-mud source code
 * Checks if gconf is available and operational
 */
gboolean
GConfSanityCheckString(GConfClient *client, const gchar *key)
{
	gchar *string;
	GError *error = NULL;
	
	string = gconf_client_get_string(client, key, &error);
	
	if (error)
	{
		int err = 0;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "allow_bold", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_float(client, GCONF_CONFIG_ROOT "background_saturation", 1.0, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "change_to_message", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "debug", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "default_mode", "B", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "sanity", "Needed for sanity check, never delete!", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "scroll_on_output", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "tearoff_menus", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "transparent_background", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "vte_font", "Courier 10 Pitch 12", &error);
		if (error)
			err = 1;

		if (err == 1)
		{
			GtkWidget *dialog;
		
			dialog = gtk_message_dialog_new(NULL,
																			0,
																			GTK_MESSAGE_ERROR,
																			GTK_BUTTONS_OK,
																			"There was an error accessing GConf: %s",
																			error->message);
			gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
			gtk_dialog_run(GTK_DIALOG(dialog));
		
			return FALSE;
		}
	}
	
	if (!string)
	{
		int err = 0;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "allow_bold", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_float(client, GCONF_CONFIG_ROOT "background_saturation", 1.0, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "change_to_message", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "debug", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "default_mode", "B", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "sanity", "Needed for sanity check, never delete!", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "scroll_on_output", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "tearoff_menus", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "transparent_background", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "vte_font", "Courier 10 Pitch 12", &error);
		if (error)
			err = 1;
		
		if (err == 1)
		{
			GtkWidget *dialog;
		
			dialog = gtk_message_dialog_new(NULL,
																			0,
																			GTK_MESSAGE_ERROR,
																			GTK_BUTTONS_OK,
																			"<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
																			"The default configuration values could not be retrieved correctly.",
																			"Please check your GConf configuration, specifically the schemam have been installed correctly."
																			);
			gtk_label_set_use_markup(GTK_LABEL(GTK_MESSAGE_DIALOG(dialog)->label), TRUE);
			gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
			gtk_dialog_run(GTK_DIALOG(dialog));
			gtk_widget_destroy(dialog);
		
			return FALSE;
		}
	}
	
	g_free(string);
	
	return TRUE;
}

/*
 * GetStringValue()
 * Get a string value from GConf.
 */
gboolean
GetStringValue(gchar *key, gchar **retval)
{
	GError *error = NULL;
	GConfValue *value = NULL;
	
	value = gconf_client_get(gconf_client, key, &error);
	
	if (error || !value) {
		return FALSE;
	}

	if (value) {
		if (value->type == GCONF_VALUE_STRING) {
			if (retval)
				*retval = g_strdup(gconf_client_get_string(gconf_client, key, &error));
			return TRUE;
		}
	}
	return FALSE;
}

/*
 * GetBooleanValue()
 * Get a boolean value from GConf.
 */
gboolean
GetBooleanValue(gchar *key, gboolean *retval)
{
	GError *error = NULL;
	GConfValue *value = NULL;
	
	value = gconf_client_get(gconf_client, key, &error);
	
	if ((error != NULL) || (value == NULL)) {
		return FALSE;
	}

	if (value) {
		if (value->type == GCONF_VALUE_BOOL) {
			if (retval)
				*retval = gconf_client_get_bool(gconf_client, key, &error);
			return TRUE;
		}
	}
	return FALSE;
}

/*
 * GetFloatValue()
 * Get a float value from GConf.
 */
gboolean
GetFloatValue(gchar *key, gfloat *retval)
{
	GError *error = NULL;
	GConfValue *value = NULL;
	
	value = gconf_client_get(gconf_client, key, &error);
	
	if ((error != NULL) || (value == NULL)) {
		return FALSE;
	}

	if (value) {
		if (value->type == GCONF_VALUE_FLOAT) {
			if (retval)
				*retval = gconf_client_get_float(gconf_client, key, &error);
			return TRUE;
		}
	}
	return FALSE;
}

/*
 * ConfigInit()
 * Initiates a connection to the gconf engine.
 */
gboolean
ConfigInit(void)
{
	gboolean setting_b;
	gfloat setting_f;
	gchar *setting_s;

	gconf_client = gconf_client_get_default();
	if (!GConfSanityCheckString(gconf_client, GCONF_CONFIG_ROOT "sanity"))
	{
		return FALSE;
	}
	
	/* Get the configuration data from GConf. */
	if (GetBooleanValue(GCONF_CONFIG_ROOT "scroll_on_output", &setting_b))
		ConfigData.scroll_on_output = setting_b;
	else
		ConfigData.scroll_on_output = DEFAULT_SCROLL_ON_OUTPUT;
	
	if (GetBooleanValue(GCONF_CONFIG_ROOT "transparent_background", &setting_b))
		ConfigData.transparent_background = setting_b;
	else
		ConfigData.transparent_background = DEFAULT_TRANSPARENT_BACKGROUND;
	
	if (GetFloatValue(GCONF_CONFIG_ROOT "background_saturation", &setting_f))
		ConfigData.background_saturation = setting_f;
	else
		ConfigData.background_saturation = DEFAULT_BACKGROUND_SATURATION;
	
	if (GetBooleanValue(GCONF_CONFIG_ROOT "debug", &setting_b))
		ConfigData.debug_mode = setting_b;
	else
		ConfigData.debug_mode = DEFAULT_DEBUG_MODE;
	
	if (GetStringValue(GCONF_CONFIG_ROOT "vte_font", &setting_s))
		ConfigData.vte_font = setting_s;
	else
		ConfigData.vte_font = g_strdup(DEFAULT_VTE_FONT);
	
	if (GetBooleanValue(GCONF_CONFIG_ROOT "change_to_message", &setting_b))
		ConfigData.change_to_message = setting_b;
	else
		ConfigData.change_to_message = DEFAULT_CHANGE_TO_MESSAGE;
	
	return TRUE;
}

/*
 * CreateVersionString()
 * Creates a version string. If need_feed is true, it will append a
 * linefeed to the end of the string, which is accepted by the vte.
 */
gchar *
CreateVersionString(gboolean need_feed)
{
	int len;
	gchar *ret;
	
	len = strlen(PACKAGE) + 1 + strlen(VERSION) + 1;
	if (need_feed)
		len += 2;
	
	if ((ret = g_malloc(len)) == NULL)
	{
		GtkWidget *dialog;
		
		dialog = gtk_message_dialog_new(NULL,
		                                0,
		                                GTK_MESSAGE_ERROR,
		                                GTK_BUTTONS_OK,
		                                "Cannot allocate enough memory. This is not good. Exiting.");
		gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
		gtk_dialog_run(GTK_DIALOG(dialog));
		
		gtk_main_quit();
	}
	
	strcpy(ret, PACKAGE);
	strcat(ret, " ");
	strcat(ret, VERSION);
	if (need_feed)
		strcat(ret, "\r\n");
	
	return ret;
}
